/*
 * Copyright (c) 2013-2018 Meltytech, LLC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtQuick.Controls 1.0
import QtQuick.Controls.Styles 1.0
import QtQuick.Layouts 1.0

Rectangle {
    id: trackHeadRoot
    property string trackName: ''
    property bool isMute
    property bool isHidden
    property bool isComposite
    property bool isLocked
    property bool isVideo
    property bool isFiltered
    property bool isBottomVideo
    property bool selected: false
    property bool current: false
    signal clicked()

    function pulseLockButton() {
        lockButtonAnim.restart();
    }

    SystemPalette { id: activePalette }
    color: selected ? selectedTrackColor : (index % 2)? activePalette.alternateBase : activePalette.base
    border.color: selected? 'red' : 'transparent'
    border.width: selected? 1 : 0
    clip: true
    state: 'normal'
    states: [
        State {
            name: 'selected'
            when: trackHeadRoot.selected
            PropertyChanges {
                target: trackHeadRoot
                color: isVideo? root.shotcutBlue : 'darkseagreen'
            }
        },
        State {
            name: 'current'
            when: trackHeadRoot.current
            PropertyChanges {
                target: trackHeadRoot
                color: selectedTrackColor
            }
        },
        State {
            when: !trackHeadRoot.selected && !trackHeadRoot.current
            name: 'normal'
            PropertyChanges {
                target: trackHeadRoot
                color: (index % 2)? activePalette.alternateBase : activePalette.base
            }
        }
    ]
    transitions: [
        Transition {
            to: '*'
            ColorAnimation { target: trackHeadRoot; duration: 100 }
        }
    ]

    MouseArea {
        anchors.fill: parent
        acceptedButtons: Qt.LeftButton | Qt.RightButton
        onClicked: {
            parent.clicked()
            nameEdit.visible = false
            if (mouse.button == Qt.RightButton)
                menu.popup()
        }
    }
    Column {
        id: trackHeadColumn
        spacing: (trackHeadRoot.height < 50)? 0 : 6
        anchors {
            top: parent.top
            left: parent.left
            margins: (trackHeadRoot.height < 50)? 0 : 4
        }

        Rectangle {
            color: 'transparent'
            width: trackHeadRoot.width - trackHeadColumn.anchors.margins * 2
            radius: 2
            border.color: (!timeline.isFloating() && trackNameMouseArea.containsMouse)? activePalette.shadow : 'transparent'
            height: nameEdit.height
            MouseArea {
                id: trackNameMouseArea
                height: parent.height
                width: nameEdit.width
                hoverEnabled: true
                onClicked: if (!timeline.isFloating()) {
                    nameEdit.focus = true
                    nameEdit.selectAll()
                }
            }
            Label {
                text: trackName
                color: activePalette.windowText
                elide: Qt.ElideRight
                x: 4
                y: 3
                width: parent.width - 8
            }
            TextField {
                id: nameEdit
                visible: focus
                width: trackHeadRoot.width - trackHeadColumn.anchors.margins * 2
                text: trackName
                onAccepted: {
                    timeline.setTrackName(index, text)
                    focus = false
                }
            }
        }
        RowLayout {
            spacing: 8
            ToolButton {
                id: muteButton
                implicitWidth: 20
                implicitHeight: 20
                iconName: isMute ? 'audio-volume-muted' : 'audio-volume-high'
                iconSource: isMute ? 'qrc:///icons/oxygen/32x32/status/audio-volume-muted.png' : 'qrc:///icons/oxygen/32x32/status/audio-volume-high.png'
                onClicked: timeline.toggleTrackMute(index)
                tooltip: isMute? qsTr('Unmute') : qsTr('Mute')
            }

            ToolButton {
                id: hideButton
                visible: isVideo
                implicitWidth: 20
                implicitHeight: 20
                iconName: isHidden ? 'layer-visible-off' : 'layer-visible-on'
                iconSource: isHidden? 'qrc:///icons/oxygen/32x32/actions/layer-visible-off.png' : 'qrc:///icons/oxygen/32x32/actions/layer-visible-on.png'
                onClicked: timeline.toggleTrackHidden(index)
                tooltip: isHidden? qsTr('Show') : qsTr('Hide')
            }

            ToolButton {
                id: compositeButton
                visible: isVideo
                enabled: !isBottomVideo
                opacity: enabled? 1.0 : 0.5
                implicitWidth: 20
                implicitHeight: 20
                iconName: isComposite ? 'merge' : 'split'
                iconSource: isComposite ? 'qrc:///icons/oxygen/32x32/actions/merge.png' : 'qrc:///icons/oxygen/32x32/actions/split.png'
                onClicked: timeline.setTrackComposite(index, !isComposite)
                tooltip: isComposite? qsTr('Disable compositing') : qsTr('Enable compositing')
            }

            ToolButton {
                id: lockButton
                implicitWidth: 20
                implicitHeight: 20
                iconName: isLocked ? 'object-locked' : 'object-unlocked'
                iconSource: isLocked ? 'qrc:///icons/oxygen/32x32/status/object-locked.png' : 'qrc:///icons/oxygen/32x32/status/object-unlocked.png'
                onClicked: timeline.setTrackLock(index, !isLocked)
                tooltip: isLocked? qsTr('Unlock track') : qsTr('Lock track')
            }

            ToolButton {
                visible: isFiltered
                anchors.right: parent.right
                implicitWidth: 20
                implicitHeight: 20
                iconName: 'view-filter'
                iconSource: 'qrc:///icons/oxygen/32x32/status/view-filter.png'
                tooltip: qsTr('Filters')
                onClicked: {
                    trackHeadRoot.clicked()
                    nameEdit.visible = false
                    timeline.filteredClicked()
                }
            }
        }
    }
}
