// $codepro.audit.disable
/**
 * Aptana Studio
 * Copyright (c) 2005-2012 by Appcelerator, Inc. All Rights Reserved.
 * Licensed under the terms of the GNU Public License (GPL) v3 (with exceptions).
 * Please see the license.html included with this distribution for details.
 * Any modifications to this file must keep this entire header intact.
 */
package com.aptana.editor.js.parsing;

import java.io.Reader;
import java.io.StringReader;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import beaver.Symbol;
import beaver.Scanner;

import org.eclipse.core.internal.utils.StringPool;

import com.aptana.editor.common.parsing.FlexScannerCharArrayReader;
import com.aptana.editor.js.parsing.lexer.JSTokenType;

%%

%public
%class JSFlexScanner
%extends Scanner
%type Symbol
%yylexthrow Scanner.Exception
%eofval{
	return newToken(Terminals.EOF, "end-of-file");
%eofval}
%unicode
%char

//%switch
//%table
//%pack

%{
	// last token used for look behind. Also needed when implementing the ITokenScanner interface
	private Symbol _lastToken;
	
	private StringPool _stringPool;

	// flag indicating if we should collect comments or not
	private boolean _collectComments = true;
	//flag if enable JSX_TAG lexer
	private boolean _enablejsx = false;
	
	private JsxElement _jsx_current_element = null;
	private Stack<JsxElement> _jsx_stack = new Stack<JsxElement>();
	
	private boolean _inTemplate = false;

	// accumulator of consecutive vsdoc lines, later added to vsdocComments as a single entity
	private List<Symbol> _vsdocAccumulator = new ArrayList<Symbol>();

	// comment collections, by type
	private List<Symbol> _sdocComments = new ArrayList<Symbol>();
	private List<Symbol> _vsdocComments = new ArrayList<Symbol>();
	private List<Symbol> _singleLineComments = new ArrayList<Symbol>();
	private List<Symbol> _multiLineComments = new ArrayList<Symbol>();

	public JSFlexScanner()
	{
		this((Reader) null);
	}
	
	private static class JsxElement{
	
		private static final int NONE = -1;
		
		private boolean opening;
		private boolean closing;
		
		private JsxElement parent;
		private int exprDepth = NONE;
		private JsxElement(JsxElement parent){
			this.parent = parent;
		}
		
		private boolean isRoot() {
			return parent == null;
		}
	}
	
	
	private boolean isEnablejsx(){
		return _enablejsx;
	}
	
	private void pushJsxElement(){
		_jsx_current_element = new JsxElement(_jsx_current_element);
	}
	
	private JsxElement getJsxElement(){
		return _jsx_current_element;
	}
	
	private void popJsxElement(){
		if(_jsx_current_element == null){
			System.err.println("[JSX_PARSER]can not match JSX_TAG tags!");
			return ;
		}
		_jsx_current_element.closing = false;
		_jsx_current_element = _jsx_current_element.parent;
	}

	public Symbol getLastToken()
	{
		return _lastToken;
	}

	public List<Symbol> getSDocComments()
	{
		return _sdocComments;
	}

	public List<Symbol> getVSDocComments()
	{
		return _vsdocComments;
	}

	public List<Symbol> getSingleLineComments()
	{
		return _singleLineComments;
	}

	public List<Symbol> getMultiLineComments()
	{
		return _multiLineComments;
	}

	private Symbol newToken(JSTokenType type, Object value)
	{
		return newToken(type.getIndex(), value);
	}

	private Symbol newToken(JSTokenType type)
	{
		return newToken(type.getIndex(), type.getName());
	}
	
	private String pool(String value)
	{
		return _stringPool.add(value);
	}

	private Symbol newToken(short id, Object value)
	{
		return new Symbol(id, yychar, yychar + yylength() - 1, value);
	}

	public Symbol nextToken() throws java.io.IOException, Scanner.Exception
	{
		// clear accumulators
		_vsdocAccumulator.clear();

		try
		{
			// get next token
			_lastToken = yylex();
		} 
		catch (Scanner.Exception e)
		{
			// create default token type
			String text = yytext();
			int end = yychar + text.length() - 1;

			_lastToken = new Symbol(JSTokenType.EOF.getIndex(), yychar, end, text);
		}
		finally
		{
			// process any accumulated vsdoc lines
			if (!_vsdocAccumulator.isEmpty())
			{
				Symbol vsdoc = newToken(JSTokenType.VSDOC, new ArrayList<Symbol>(_vsdocAccumulator));

				_vsdocComments.add(vsdoc);
			}
		}

		return _lastToken;
	}
	
	private boolean isValidLessStart(){//
		if(_lastToken == null){
			return false;
		}
		
		switch(_lastToken.getId()){
			case Terminals.LPAREN:
			case Terminals.EQUAL:
			case Terminals.RETURN:
			case Terminals.YIELD:
			case Terminals.LBRACKET:
			case Terminals.COLON:
			case Terminals.QUESTION:
			case Terminals.ARROW:
			case Terminals.COMMA:
				return false;
		}
		return true;
	}

	private boolean isValidDivisionStart()
	{
		if (_lastToken != null)
		{
			switch (_lastToken.getId())
			{
				case Terminals.IDENTIFIER:
				case Terminals.NUMBER:
				case Terminals.REGEX:
				case Terminals.STRING:
				case Terminals.RPAREN:
				case Terminals.PLUS_PLUS:
				case Terminals.MINUS_MINUS:
				case Terminals.RBRACKET:
				case Terminals.RCURLY:
				case Terminals.FALSE:
				case Terminals.NULL:
				case Terminals.THIS:
				case Terminals.TRUE:
					return true;
			}
		}

		return false;
	}

	public void setCollectComments(boolean flag)
	{
		_collectComments = flag;
	}
	
	public void setEnablejsx(boolean enable){
		_enablejsx = enable;
	}

	public void setSource(String source)
	{
		yyreset(new StringReader(source));

		_stringPool = new StringPool();

		// clear last token
		_lastToken = null;

		// reset comment collection lists
		_singleLineComments.clear();
		_multiLineComments.clear();
		_sdocComments.clear();
		_vsdocComments.clear();
	}
	
	public void setSource(char[] source, int offset, int length, int skipStart, int skipEnd){
		  
		yyreset(new FlexScannerCharArrayReader(source, offset, length, skipStart, skipEnd));

		_stringPool = new StringPool();

		// clear last token
		_lastToken = null;

		// reset comment collection lists
		_singleLineComments.clear();
		_multiLineComments.clear();
		_sdocComments.clear();
		_vsdocComments.clear();
	}
%}

LineTerminator = \r|\n|\r\n
RubyBlock = "<%" ~"%>"
PHPBlock = "<?" ~"?>"
DjangoBlock = "{%" ~"%}"
Whitespace = {LineTerminator} | [ \t\f] | {RubyBlock} | {DjangoBlock}

//Identifier = [a-zA-Z_$][a-zA-Z0-9_$]*
Identifier = ([:jletter:]|\$)([:jletterdigit:]|\$)*

Integer = [:digit:][:digit:]*
Hex = "0" [xX] [a-fA-F0-9]+
Float = ({Integer} \.[:digit:]*) | (\.[:digit:]+)
Scientific = ({Integer} | {Float}) [eE][-+]?[:digit:]+
Number = {Integer} | {Hex} | {Float} | {Scientific}

DoubleQuotedString = \"([^\\\"\r\n]|\\[^]|\\\r\n|\\\r|\\\n)*\"
SingleQuotedString = '([^\\'\r\n]|\\[^]|\\\r\n|\\\r|\\\n)*'

TemplateCharacter = [^\\`\$]|\\[^]|\$[^\{]

NoSubstitutionTemplate = "`" ({TemplateCharacter})* "`"
TemplateHead = "`" ({TemplateCharacter})* "${"
TemplateMiddle = "}" ({TemplateCharacter})* "${"
TemplateTail = "}" ({TemplateCharacter})* "`"

Strings = {DoubleQuotedString} | {SingleQuotedString}

SingleLineComment = "//" [^\r\n]*
MultiLineComment = "/*" ~"*/"
VSDocComment = "///" [^\r\n]*

CharClass = "[" ([^\]\\\r\n]|\\[^\r\n])* "]"
Character = ([^\[\\\/\r\n]|\\[^\r\n])+
Regex = "/" ({CharClass}|{Character})+ "/" [a-z]*

jsx_text = [^<>\{\}]*
JSXIdentifier = {Identifier}

%state DIVISION, REGEX, TEMPLATE ,JSX_TAG,JSX_TAG_NAME,JSX_ATTR

%%

<YYINITIAL> {
	{Whitespace}+	    { /* ignore */ }
	// comments
	{VSDocComment}		{
							if (_collectComments)
							{
								_vsdocAccumulator.add(newToken(JSTokenType.VSDOC, yytext()));
							}
						}
	{SingleLineComment}	{
							if (_collectComments)
							{
								_singleLineComments.add(newToken(JSTokenType.SINGLELINE_COMMENT, yytext()));
							}
						}
	{MultiLineComment}	{
							if (_collectComments)
							{	
								String text = yytext();
								if(text.startsWith("/**") && !("/**/".equals(text))){
									_sdocComments.add(newToken(JSTokenType.SDOC, text));
								}else{
									_multiLineComments.add(newToken(JSTokenType.MULTILINE_COMMENT, text));
								}
							}
						}

	// numbers
	{Number}		{ return newToken(Terminals.NUMBER, pool(yytext())); }

	// strings
	{Strings}		{ return newToken(Terminals.STRING, pool(yytext())); }
	
	{PHPBlock}		{ return newToken(Terminals.PHPBLOCK, pool(yytext())); }

	// keywords
	"as"			{ return newToken(JSTokenType.AS); }
	"break"			{ return newToken(JSTokenType.BREAK); }
	"case"			{ return newToken(JSTokenType.CASE); }
	"catch"			{ return newToken(JSTokenType.CATCH); }
	"const"			{ return newToken(JSTokenType.VAR, pool(yytext())); }
	"continue"		{
						if(yycharat(yylength())=='('){
        		  			return newToken(JSTokenType.IDENTIFIER,JSTokenType.CONTINUE.getName());
        	  			}
					 	return newToken(JSTokenType.CONTINUE); 
					}
	"class"			{ return newToken(JSTokenType.CLASS); }
	"default"		{ return newToken(JSTokenType.DEFAULT); }
	"debugger"		{ return newToken(JSTokenType.DEBUGGER); }
	"delete"		{
						char nextChar = Character.MIN_VALUE;
			        	int length = yylength();
			        	int offset = 0;
			        	while(true){
			        		nextChar = yycharat(length + offset);
			        		if(Character.isWhitespace(nextChar)){
			        			offset++;
			        			continue;
			        		}
			        		if(nextChar==':'){
			        			return newToken(JSTokenType.IDENTIFIER,yytext());
			        		}
			        		break;
			        	}
						return newToken(JSTokenType.DELETE); 
					}
	"do"			{ return newToken(JSTokenType.DO); }
	"else"			{ return newToken(JSTokenType.ELSE); }
	"export"		{ return newToken(JSTokenType.EXPORT); }
	"extends"		{ return newToken(JSTokenType.EXTENDS); }
	"false"			{ return newToken(JSTokenType.FALSE); }
	"finally"		{ return newToken(JSTokenType.FINALLY); }
	"for"			{ return newToken(JSTokenType.FOR); }
	"from"			{ 
						
						return newToken(JSTokenType.FROM);
					}
	"function"		{ return newToken(JSTokenType.FUNCTION); }
	"if"			{ return newToken(JSTokenType.IF); }
	"import"		{ return newToken(JSTokenType.IMPORT); }
	"instanceof"	{ return newToken(JSTokenType.INSTANCEOF); }
	"in"			{ return newToken(JSTokenType.IN); }
	"let"			{ return newToken(JSTokenType.VAR, pool(yytext())); }
	
	"new"			{ return newToken(JSTokenType.NEW); }
	"null"			{ return newToken(JSTokenType.NULL); }
	
	"of"			{ return newToken(JSTokenType.OF); }
	
	"return"		{ return newToken(JSTokenType.RETURN); }
	"static"		{ return newToken(JSTokenType.STATIC); }
	"super"			{ return newToken(JSTokenType.SUPER); }
	
	"switch"		{ return newToken(JSTokenType.SWITCH); }
	"this"			{ return newToken(JSTokenType.THIS); }
	"throw"			{ return newToken(JSTokenType.THROW); }
	"true"			{ return newToken(JSTokenType.TRUE); }
	"try"			{ return newToken(JSTokenType.TRY); }
	"typeof"		{ return newToken(JSTokenType.TYPEOF); }
	"var"			{ return newToken(JSTokenType.VAR, pool(yytext())); }
	"void"			{ return newToken(JSTokenType.VOID); }
	"while"			{ return newToken(JSTokenType.WHILE); }
	"with"			{ return newToken(JSTokenType.WITH); }
	"yield"			{ return newToken(JSTokenType.YIELD); }
	"get"			{ return newToken(JSTokenType.GET); }
	"set"			{ return newToken(JSTokenType.SET); }

	// identifiers
	{Identifier}	{ return newToken(Terminals.IDENTIFIER, pool(yytext())); }

	// operators
	">>>="			{ return newToken(JSTokenType.GREATER_GREATER_GREATER_EQUAL); }
	">>>"			{ return newToken(JSTokenType.GREATER_GREATER_GREATER); }

	"=>"			{ return newToken(JSTokenType.ARROW); }
	
	"<<="			{ return newToken(JSTokenType.LESS_LESS_EQUAL); }
	"<<"			{ return newToken(JSTokenType.LESS_LESS); }
	"<="			{ return newToken(JSTokenType.LESS_EQUAL); }
	"</"			{ return newToken(Terminals.CLOSE_TAG_BEGIN,pool(yytext())); }
	"<"				{
						if(!isEnablejsx() || isValidLessStart()){
							return newToken(JSTokenType.LESS);
						}else{
							yypushback(1);
							yybegin(JSX_TAG);
							if(_jsx_current_element != null){
								_jsx_stack.push(_jsx_current_element);
								_jsx_current_element = null;
							}
						}
					}

	">>="			{ return newToken(JSTokenType.GREATER_GREATER_EQUAL); }
	">>"			{ return newToken(JSTokenType.GREATER_GREATER); }
	">="			{ return newToken(JSTokenType.GREATER_EQUAL); }
	">"				{ return newToken(JSTokenType.GREATER); }

	"==="			{ return newToken(JSTokenType.EQUAL_EQUAL_EQUAL); }
	"=="			{ return newToken(JSTokenType.EQUAL_EQUAL); }
	"="				{ return newToken(JSTokenType.EQUAL); }

	"!=="			{ return newToken(JSTokenType.EXCLAMATION_EQUAL_EQUAL); }
	"!="			{ return newToken(JSTokenType.EXCLAMATION_EQUAL); }
	"!"				{ return newToken(JSTokenType.EXCLAMATION); }

	"&&"			{ return newToken(JSTokenType.AMPERSAND_AMPERSAND); }
	"&="			{ return newToken(JSTokenType.AMPERSAND_EQUAL); }
	"&"				{ return newToken(JSTokenType.AMPERSAND); }

	"||"			{ return newToken(JSTokenType.PIPE_PIPE); }
	"|="			{ return newToken(JSTokenType.PIPE_EQUAL); }
	"|"				{ return newToken(JSTokenType.PIPE); }

	"**="			{ return newToken(JSTokenType.STAR_STAR_EQUAL); }
	"*="			{ return newToken(JSTokenType.STAR_EQUAL); }
	"*"				{ return newToken(JSTokenType.STAR); }

	"/"				{
						yypushback(1);
						if (isValidDivisionStart())
						{
							yybegin(DIVISION);
						}
						else
						{
							yybegin(REGEX);
						}
					}
					
	"`"				{
						yypushback(1);
						yybegin(TEMPLATE);
					}
					

	"%="			{ return newToken(JSTokenType.PERCENT_EQUAL); }
	"%"				{ return newToken(JSTokenType.PERCENT); }

	"--"			{ return newToken(JSTokenType.MINUS_MINUS); }
	"-="			{ return newToken(JSTokenType.MINUS_EQUAL); }
	"-"				{ return newToken(JSTokenType.MINUS); }

	"++"			{ return newToken(JSTokenType.PLUS_PLUS); }
	"+="			{ return newToken(JSTokenType.PLUS_EQUAL); }
	"+"				{ return newToken(JSTokenType.PLUS); }

	"^="			{ return newToken(JSTokenType.CARET_EQUAL); }
	"^"				{ return newToken(JSTokenType.CARET); }
 
	"?"				{ return newToken(JSTokenType.QUESTION); }
	"~"				{ return newToken(JSTokenType.TILDE); }
	";"				{ return newToken(JSTokenType.SEMICOLON); }
	"("				{ return newToken(JSTokenType.LPAREN); }
	")"				{ return newToken(JSTokenType.RPAREN); }
	"["				{ return newToken(JSTokenType.LBRACKET); }
	"]"				{ return newToken(JSTokenType.RBRACKET); }
	"{"				{
						if(getJsxElement() != null && getJsxElement().exprDepth >= 0){
							getJsxElement().exprDepth++;
						}
						return newToken(JSTokenType.LCURLY);
					}
	"}"				{ 
						if(_inTemplate){
							yypushback(1);
							yybegin(TEMPLATE);
						}else if(getJsxElement() != null && getJsxElement().exprDepth >= 0){
							if(getJsxElement().exprDepth == 0){
								if(getJsxElement().opening){
									yybegin(JSX_ATTR);
								}else{
									yybegin(JSX_TAG);
								}
								getJsxElement().exprDepth = JsxElement.NONE;
								return newToken(Terminals.JSX_RCURLY,pool(yytext()));
							}
							getJsxElement().exprDepth--;
						}
						return newToken(JSTokenType.RCURLY);
					}
	","				{ return newToken(JSTokenType.COMMA); }
	":"				{ return newToken(JSTokenType.COLON); }
	
	"..."			{ return newToken(JSTokenType.ELLIPSIS); }
	"."				{ return newToken(JSTokenType.DOT); }
	"<!--"			{ /* ignore 忽略html的注释 */ }
	 .				{ /* ignore 其他不认识的字符都忽略 */ }
}

<JSX_TAG>{
	{Whitespace}+	{ /* ignore */ }	
	"</"			{
						yybegin(JSX_TAG_NAME);
						getJsxElement().closing = true;
						return newToken(Terminals.CLOSE_TAG_BEGIN,pool(yytext())); 
					}
	"<"				{
						yybegin(JSX_TAG_NAME);
						pushJsxElement();
						getJsxElement().opening = true;
						return newToken(Terminals.OPEN_TAG_BEGIN,pool(yytext())); 
					}
	">"				{
						if(getJsxElement().opening){
							getJsxElement().opening = false;
						}else if(getJsxElement().closing){
							if(getJsxElement().isRoot()){
								yybegin(YYINITIAL);
								if(!_jsx_stack.isEmpty()){
									_jsx_current_element = _jsx_stack.pop();
								}
							}else{
								popJsxElement();
							}
						}
						return newToken(Terminals.TAG_END,pool(yytext())); 
					}
	"/>"			{
						if(getJsxElement().isRoot()){
							yybegin(YYINITIAL);
							if(!_jsx_stack.isEmpty()){
								_jsx_current_element = _jsx_stack.pop();
							}
						}else{
							popJsxElement();
						}
						return newToken(Terminals.SELF_CLOSE,pool(yytext())); 
					}
	
	"{"				{
						yybegin(YYINITIAL);
						getJsxElement().exprDepth = 0;
						return newToken(Terminals.JSX_LCURLY,pool(yytext()));
					}
	{jsx_text}		{
						return newToken(Terminals.JSX_TEXT,pool(yytext())); 
					}
}

<JSX_TAG_NAME>{

	{JSXIdentifier} { return newToken(Terminals.JSX_TAG_NAME, pool(yytext())); }
	"."				{ return newToken(JSTokenType.DOT); }
	":"				{ return newToken(JSTokenType.COLON); }
	">"				{
						yypushback(1);
						yybegin(JSX_TAG);
					}
					
	{Whitespace}+	{ yybegin(JSX_ATTR); }
}


<JSX_ATTR>{
	{Whitespace}+	{ /* ignore */ }	
	{Strings}		{ return newToken(Terminals.STRING, pool(yytext())); }
	
	{JSXIdentifier}	{ return newToken(Terminals.JSX_ATTR_NAME, pool(yytext())); }
	
	"{"				{
						yybegin(YYINITIAL);
						getJsxElement().exprDepth = 0;
						return newToken(Terminals.JSX_LCURLY,pool(yytext()));
					}
	"="				{ return newToken(JSTokenType.EQUAL); }
	":"				{ return newToken(JSTokenType.COLON); }
	"<"				{
						yypushback(1);
						yybegin(JSX_TAG);
					}
	"/>"			{
						yypushback(2);
						yybegin(JSX_TAG);
					}
	
	">"				{
						yypushback(1);
						yybegin(JSX_TAG);
					}
	 .				{
	 					yybegin(YYINITIAL);
						throw new Scanner.Exception("Unexpected character '" + yytext() + "' around offset " + yychar);
	 				}
}

<DIVISION> {
	"/="			{
						yybegin(YYINITIAL);
						return newToken(JSTokenType.FORWARD_SLASH_EQUAL);
					}
	"/"				{
						yybegin(YYINITIAL);
						return newToken(JSTokenType.FORWARD_SLASH);
					}
}

<REGEX> {
	{Regex}			{
						yybegin(YYINITIAL);
						return newToken(Terminals.REGEX, pool(yytext()));
					}
	"/="			{
						yybegin(YYINITIAL);
						return newToken(JSTokenType.FORWARD_SLASH_EQUAL);
					}
	"/"				{
						yybegin(YYINITIAL);
						return newToken(JSTokenType.FORWARD_SLASH);
					}
}


<TEMPLATE>{
	{Whitespace}+		{ /* ignore */ }	
	{TemplateHead}		{
							yybegin(YYINITIAL);
							_inTemplate = true;
							return newToken(Terminals.TemplateHead,pool(yytext()));
						}
	{TemplateMiddle} 	{
							yybegin(YYINITIAL);
							_inTemplate = true;
							return newToken(Terminals.TemplateMiddle,pool(yytext()));
						}
	{TemplateTail}		{
							yybegin(YYINITIAL);
							_inTemplate = false;
							return newToken(Terminals.TemplateTail,pool(yytext()));
						}
	{NoSubstitutionTemplate}	{
									yybegin(YYINITIAL);
									_inTemplate = false;
									return newToken(Terminals.NoSubstitutionTemplate,pool(yytext()));
								}
	"`"					{
							yybegin(YYINITIAL);
							_inTemplate = false;
							return newToken(Terminals.TemplateTail,pool(yytext()));
						}
}



// \"[^\"\r\n\ ]+|.

[\"']\\?|.	{
				// make sure we reset the lexer state for next (potential) scan
				yybegin(YYINITIAL);
				throw new Scanner.Exception("Unexpected character '" + yytext() + "' around offset " + yychar);
			}
