/*****************************************************************************
//		Copyright (c) 1988-2000 Microsoft Corporation.
//              All rights reserved.
******************************************************************************/


USE master
go

-- Make sure catalog updates are enabled (server started in single-user-mode usually)
update sysobjects set status = status where id = 0
go
if @@error <> 0
	raiserror('Upgrade.sql failed catalog updates: restart the server with the ''-f'' option.',22,127) with log
go

DBCC TRACEON(1717)	-- Mark objects as system objects (sp_MS_upd_sysobj_category not yet created!)
go

-- CREATE A HELPER PROC USED BELOW --
if object_id(N'sp_msupg_removesystemcomputedcolumns') is not null
	drop proc sp_msupg_removesystemcomputedcolumns
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_removesystemcomputedcolumns
	@tab sysname
as
	begin tran
	dbcc lockobjectschema (@tab)
	declare @objid int
	select @objid = object_id(@tab)
	-- Delete dependencies of computed columns on other objects
	DELETE sysdepends FROM sysdepends d 
	WHERE d.id = @objid and d.number in (select colid from syscolumns where id = @objid and (colstat&4)=4)
	delete syscolumns where id = @objid and (colstat&4)=4
	delete syscomments where id = @objid
	update sysobjects
		set info = (select max(colid) from syscolumns where id = @objid)
		where id = @objid
	commit tran
go
deny execute on sp_msupg_removesystemcomputedcolumns to public
go

-------------------------------------------------------------------------------
----------------- Remove computed columns for master upgrade ------------------
-------------------------------------------------------------------------------
EXEC sp_msupg_removesystemcomputedcolumns sysdatabases
EXEC sp_msupg_removesystemcomputedcolumns sysservers
EXEC sp_msupg_removesystemcomputedcolumns sysxlogins
EXEC sp_msupg_removesystemcomputedcolumns syscharsets
go

-------------------------------------------------------------------------------
--------------- Upgrade system tables in master (NON-COMPUTED!) ---------------
-------------------------------------------------------------------------------

-- sysservers.srvcollation
IF NOT EXISTS (SELECT * FROM syscolumns WHERE id = 40 AND colid = 13)
	BEGIN
	BEGIN TRAN
	ALTER TABLE dbo.sysservers ADD srvcollation int null
	-- update sysserver.srvcollation to be the collation of local server if "collation 
	-- compatible" is set; otherwise set to NULL
	EXEC ('UPDATE dbo.sysservers SET srvcollation = convert(int, serverproperty(''collationid'')) WHERE srvstatus & 256 = 256')
	EXEC ('UPDATE dbo.sysservers SET srvcollation = NULL WHERE srvstatus & 256 = 0')
	COMMIT TRAN
	END
go

-- sysservers.srvcollation.connecttimeout/querytimeout
IF NOT EXISTS (SELECT * FROM syscolumns WHERE id = 40 AND colid = 14)
BEGIN
	BEGIN TRAN
	EXEC ('ALTER TABLE dbo.sysservers ADD connecttimeout int NOT NULL, querytimeout int NOT NULL')
	EXEC ('UPDATE dbo.sysservers SET connecttimeout = 0, querytimeout = 0')
	COMMIT TRAN
END
go

-- sysaltfiles.createlsn & droplsn
IF NOT EXISTS (
	select * from syscolumns where name='createlsn' and id = (select id from sysobjects where name='sysaltfiles'))
	BEGIN
	BEGIN TRAN
	ALTER TABLE dbo.sysaltfiles ADD createlsn binary(10) null
	ALTER TABLE dbo.sysaltfiles ADD droplsn binary(10) null
	COMMIT TRAN
	END
go

-- Add system_function_schema to master
IF NOT EXISTS ( SELECT * FROM sysusers WHERE uid = 4 )
	INSERT sysusers VALUES (4, 0, 'system_function_schema', NULL, 0x00, getdate(), getdate(), 0, NULL)
go

-- MAKE 'sysdatabases,sid' ANSI-PADDED.
UPDATE syscolumns SET typestat = typestat | 2
    WHERE id = object_id('sysdatabases') AND number = 0 AND name = 'sid'
go

checkpoint
go

-------------------------------------------------------------------------------
----------------- Add/Upgrade data in system tables in master -----------------
-------------------------------------------------------------------------------

		-----------------------
		---- CONFIG VALUES ----
		-----------------------

-- EXISTING CONFIG OPTIONS GET NEW comment AND/OR status
--  By existing, we mean options that were there in RTM version of Sphinx.
--	NOTE: If you are changing the comment, value and status, then this should
--  be considered a *new* option, and you should DELETE then ADD it.
--

-- 6/30/99 'Maximum worker threads' is not really dynamic
UPDATE sysconfigures
	SET status = 2
	where config = 503
go

UPDATE sysconfigures	-- bug 233058
	SET value = 600
	where config = 1520
go

-- collation related option 1123/12124/1125 are replaced by serverproperty('sqlsortorder'/'lcid'/'compflags')
delete sysconfigures where config in (1123, 1124, 1125)

-- 15Jul99 Remove 'language neutral full-text'
if not exists (SELECT * FROM dbo.sysconfigures WHERE config = 1126 AND 
			   comment = N'default full-text language' )
begin
	BEGIN TRAN
	
	-- if 'language neutral full-text indexing' was set to 1, use language_neutral (0).
	-- otherwise, use the lcid from the server 'collation' property if there it has
	-- a corresponding word breaker.
	-- if there isn't a word breaker associated with this lcid, use neutral.
	DECLARE @neutral int
	SELECT @neutral = value FROM sysconfigures WHERE config = 1126 AND
			comment = N'language neutral full-text indexing'

	DECLARE @srvlang int
	SELECT @srvlang = convert(int, CollationPropertyFromID(convert(int, ServerProperty('collationid')), 'lcid'))

	if @neutral <> 0
	begin
		SELECT @srvlang = 0
	end
	else
		begin
			-- langauges that have word breakers today
			if @srvlang not in ( 
							 0x0409,	-- English-US
							 0x0407,	-- German
							 0x040c,	-- French
							 0x0411,	-- Japanese
							 0x0c0a,	-- Spanish
							 0x0410,	-- Italian
							 0x0413		-- Dutch
							   )
			begin
				SELECT @srvlang = 0
			end
		end

	UPDATE sysconfigures
		SET value = @srvlang,
			comment = 'default full-text language'
		WHERE config = 1126 AND 
			  comment = N'language neutral full-text indexing'

	COMMIT TRAN
end
go

-- ADD ANY CONFIG NOT YET THERE
-- NOTE: If you are adding an option, also add it to the delete guard below the inserts
--
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 101)
	INSERT sysconfigures (config, status, value, comment) VALUES (  101, 3 ,     0, N'Maximum recovery interval in minutes' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 102)
	INSERT sysconfigures (config, status, value, comment) VALUES (  102, 1 ,     1, N'Allow updates to system tables' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 103)
	INSERT sysconfigures (config, status, value, comment) VALUES (  103, 2 ,     0, N'Number of user connections allowed' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 106)
	INSERT sysconfigures (config, status, value, comment) VALUES (  106, 2 ,  	 0, N'Number of locks for all users' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 107)
	INSERT sysconfigures (config, status, value, comment) VALUES (  107, 2 ,     0, N'Number of open database objects' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 109)
	INSERT sysconfigures (config, status, value, comment) VALUES (  109, 2 ,     0, N'Default fill factor percentage' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 115)
	INSERT sysconfigures (config, status, value, comment) VALUES (  115, 1 ,     1, N'Allow triggers to be invoked within triggers' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 117)
	INSERT sysconfigures (config, status, value, comment) VALUES (  117, 0 ,     1, N'Allow remote access' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 124)
	INSERT sysconfigures (config, status, value, comment) VALUES (  124, 1 ,     0, N'default language' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 503)
	INSERT sysconfigures (config, status, value, comment) VALUES (  503, 2 ,   255, N'Maximum worker threads' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 505)
	INSERT sysconfigures (config, status, value, comment) VALUES (  505, 3 ,  4096, N'Network packet size' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 518)
	INSERT sysconfigures (config, status, value, comment) VALUES (  518, 1 ,     0, N'show advanced options' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 542)
	INSERT sysconfigures (config, status, value, comment) VALUES (  542, 1 ,     0, N'Create DTC transaction for remote procedures' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 544)
	INSERT sysconfigures (config, status, value, comment) VALUES (  544, 2 ,     0, N'c2 audit mode' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1126)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1126, 3 ,  1033, N'default full-text language' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1127)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1127, 3 ,  2049, N'two digit year cutoff' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1505)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1505, 3 ,  	 0, N'Memory for index create sorts (kBytes)' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1517)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1517, 2 ,     0, N'Priority boost' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1519)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1519, 1 ,    20, N'remote login timeout' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1520)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1520, 1 ,   600, N'remote query timeout' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1531)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1531, 3 ,    -1, N'cursor threshold' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1532)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1532, 2 ,     0, N'set working set size' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1534)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1534, 1 ,     0, N'user options' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1535)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1535, 2 ,     0, N'affinity mask' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1536)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1536, 1 , 65536, N'Maximum size of a text field in replication.' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1537)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1537, 2 ,     0, N'Tape retention period in days' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1538)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1538, 3 ,     5, N'cost threshold for parallelism' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1539)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1539, 3 ,     0, N'maximum degree of parallelism' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1540)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1540, 3 ,  1024, N'minimum memory per query (kBytes)' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1541)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1541, 3 ,   -1,  N'maximum time to wait for query memory (s)' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1543)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1543, 3 ,     0, N'Minimum size of server memory (MB)' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1544)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1544, 3 ,2147483647, N'Maximum size of server memory (MB)' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1545)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1545, 3 ,     0, N'Maximum estimated cost allowed by query governor' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1546)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1546, 2 ,     0, N'User mode scheduler uses lightweight pooling' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1547)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1547, 2 ,     0, N'scan for startup stored procedures' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1548)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1548, 2 ,     0, N'AWE enabled in the server' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 1549)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 1549, 2 ,     0, N'affinity64 mask' )
IF NOT EXISTS (SELECT * FROM sysconfigures WHERE config = 400)
	INSERT sysconfigures (config, status, value, comment) VALUES ( 400,  1 ,     0, N'Cross DB Ownership Chaining' )
go



-- GUARD against arbitrary addition of options
-- Deleting all except those we know are present
-- If you are adding an option, also add the config number to list below
-- DO NOT ADD any options beyond this point !!!

delete sysconfigures
	where config not in
		(101,  102,  103,  106,  107,  109, 115,  117,	 124,
		 400,
		 503, 505, 518, 542, 544, 
		 1126, 1127,
		 1505, 1517, 1519, 1520,
		 1531, 1532, 1534, 1535, 1536, 1537, 1538, 1539,
		 1540, 1541, 1543, 1544, 1545, 1546, 1547, 1548,
		 1549
		)
go

checkpoint
go

-------------------------------------------------------------------------------
--------------------------------- BUILD SYSCHARSETS ---------------------------
-------------------------------------------------------------------------------
--
-- 08Feb98 Added all the SQL Server supported DBCS sort orders.
-- Note: case-sensitivity of far east sort orders are hard coded in server
--	to avoid Sphinx beta2 - beta3 on disk format changes.
--  Ranges 1-240 are documented to be reserved by SQL Server for system use
--

DELETE syscharsets where id between 1 and 240
go

INSERT syscharsets VALUES ( 1001,   1, 0, 0, N'iso_1', N'ISO 8859-1 (Latin-1) - Western European 8-bit character set.', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f010150700101517101015272010153730101547401)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x0155750101567601015777010158780101597901015a7a01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001008181010082820100838301008484010085850100868601008787010088880100898901008a8a01008b8b01008c8c01008d8d01008e8e01088f8f0108909001009191010092920100939301009494010095950100969601009797010098980100999901009a9a01009b9b01009c9c01009d9d01009e9e01009f9f0100a0a00108a1a10120a2a20120a3a34020a4a44020a5a54020a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0108aeae0120afaf0120b0b00120b1b10120b2
	+	0xb20120b3b30120b4b40120b5b50120b6b60120b7b70120b8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bebe0120bfbf0120c0e00101c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2f20101d3f30101d4f40101d5f50101d6f60101d7d70120d8f80101d9f90101dafa0101dbfb0101dcfc0101ddfd0101defe0101dfdf0101c0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102cded0102ceee0102cfef0102d0f00102d1f10102d2f20102d3f30102d4f40102d5f50102d6f60102f7f70120d8f80102d9f90102dafa0102dbfb0102dcfc0102ddfd0102defe0102ffff0102000080000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
	+	0x000000000000303132333031323030323234353530313236323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003030303030303030303030303030303030303030303030003030303030303030303030303030303030303030303030303030303030303000303030303030303000
WHERE id = 1
go

INSERT syscharsets VALUES ( 1001,   2, 0, 0, N'cp850', N'Code Page 850 (Multilingual) character set.', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f010150700101517101015272010153730101547401015575010156760101)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x5777010158780101597901015a7a01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808701019a81010290820102b68301028e840102b78501028f86010280870102d2880102d3890102d48a0102d88b0102d78c0102de8d01028e8401018f860101908201019291010292910101e293010299940102e3950102ea960102eb97010259980102999401019a8101019d9b01029c9c40209d9b01019e9e01209f9f0120b5a00102d6a10102e0a20102e9a30102a5a40102a5a40101a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120b3b30120b4b4
	+	0x0120b5a00101b6830101b7850101b8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bebe4020bfbf0120c0c00120c1c10120c2c20120c3c30120c4c40120c5c50120c7c60102c7c60101c8c80120c9c90120caca0120cbcb0120cccc0120cdcd0120cece0120cfcf4020d1d00102d1d00101d2880101d3890101d48a010149d50102d6a10101d78c0101d88b0101d9d90120dada0120dbdb0120dcdc0120dddd0120de8d0101dfdf0120e0a20101e1e10102e2930101e3950101e5e40102e5e40101e6e60102e8e70102e8e70101e9a30101ea960101eb970101edec0102edec0101eeee0120efef0120f0f00120f1f10120f2f20120f3f30120f4f40120f5f50120f6f60120f7f70120f8f80120f9f90120fafa0120fbfb0120fcfc0120fdfd0120fefe0120ffff0120000080000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000303132
	+	0x333031323030323234353530313236323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000303030303030303030303030303030303030303030303030303030300030000030303030303000000000000000000000000000000030303000000000000000000000000000003030000000000000000030303030303030303000000000003000300030303030003030303030303000000000000000000000000000000000000000
WHERE id = 2
go

INSERT syscharsets VALUES ( 1001,   3, 0, 0, N'cp437', N'Code Page 437 (United States) character set.', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015777010158780101597901015a7a01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808701019a81010290820102838301028e840102858501028f8601028087010288880102898901028a8a01028b8b01028c8c01028d8d01028e8401018f860101908201019291010292910101939301029994010295950102969601029797010259980102999401019a8101019b9b01209c9c40209d9d40209e9e01209f9f0120a0a00102a1a10102a2a20102a3a30102a5a40102a5a40101a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120b3b30120b4
	+	0xb40120b5b50120b6b60120b7b70120b8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bebe0120bfbf0120c0c00120c1c10120c2c20120c3c30120c4c40120c5c50120c6c60120c7c70120c8c80120c9c90120caca0120cbcb0120cccc0120cdcd0120cece0120cfcf0120d0d00120d1d10120d2d20120d3d30120d4d40120d5d50120d6d60120d7d70120d8d80120d9d90120dada0120dbdb0120dcdc0120dddd0120dede0120dfdf0120e0e00102e1e10102e2e20101e3e30102e4e50101e4e50102e6e60102e7e70102e8e80101e9e90101eaea0101ebeb0102ecec0102eded0102eeee0102efef0120f0f00120f1f10120f2f20120f3f30120f4f40120f5f50120f6f60120f7f70120f8f80120f9f90120fafa0120fbfb0120fcfc0120fdfd0120fefe0120ffff01080000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031
	+	0x32333031323030323234353530313236323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000303030303030303030303030303030303030303030303030303030000000000030303030303000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 3
go

INSERT syscharsets VALUES ( 1001,   4, 0, 0, N'cp1250', N'Code page 1250', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181012082820120838301208484012085850120868601208787012088880120898901208a9a01018b8b01208c9c01018d9d01018e9e01018f9f0101909001209191012092920120939301209494012095950120969601209797012098980120999901208a9a01029b9b01208c9c01028d9d01028e9e01028f9f0102a0a00108a1a10120a2a20120a3b30101a4a44020a5b90101a6a60120a7a70120a8a80120a9a90120aaba0101abab0120acac0120adad0120aeae0120afbf0101b0b00120b1b10120b2b20120a3b30102b4b40120b5b50120b6b60120b7b70120
	+	0xb8b80120a5b90102aaba0102bbbb0120bcbe0101bdbd0120bcbe0102afbf0102c0e00101c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2f20101d3f30101d4f40101d5f50101d6f60101d7d70120d8f80101d9f90101dafa0101dbfb0101dcfc0101ddfd0101defe0101dfdf0102c0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102cded0102ceee0102cfef0102d0f00102d1f10102d2f20102d3f30102d4f40102d5f50102d6f60102f7f70120d8f80102d9f90102dafa0102dbfb0102dcfc0102ddfd0102defe0102ffff01200000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 4
go

INSERT syscharsets VALUES ( 1001, 5, 0, 0, N'cp1251', N'Code Page 1251', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008809001018183010182820120818301028484012085850120868601208787012088880120898901208a9a01018b8b01208c9c01018d9d01018e9e01018f9f0101809001029191012092920120939301209494012095950120969601209797012098980120999901208a9a01029b9b01208c9c01028d9d01028e9e01028f9f0102a0a00108a1a20101a1a20102a3bc0101a4a44020a5b40101a6a60120a7a70120a8b80101a9a90120aaba0101abab0120acac0120adad0120aeae0120afbf0101b0b00120b1b10120b2b30101b2b30102a5b40102b5b50120b6b60120b7b70120a8
	+	0xb80102b9b90120aaba0102bbbb0120a3bc0102bdbe0101bdbe0102afbf0102c0e00101c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2f20101d3f30101d4f40101d5f50101d6f60101d7f70101d8f80101d9f90101dafa0101dbfb0101dcfc0101ddfd0101defe0101dfff0101c0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102cded0102ceee0102cfef0102d0f00102d1f10102d2f20102d3f30102d4f40102d5f50102d6f60102d7f70102d8f80102d9f90102dafa0102dbfb0102dcfc0102ddfd0102defe0102dfff0102000080000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000303132333031323030323234353530313236
	+	0x323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 5
go

INSERT syscharsets VALUES ( 1001,   6, 0, 0, N'cp1253', N'Code page 1253', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181012082820120838301018484012085850120868601208787012088880120898901208a8a01208b8b01208c9c01208d9d01208e9e01208f8f0120909001209191012092920120939301209494012095950120969601209797012098980120999901209a9a01209b9b01209c9c01209d9d01209e9e01209f9f0120a0a00108a1a10120a2dc0101a3a30120a4a44020a5a54020a6a60120a7a70120a8a80120a9a90120aaba0100abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120b3b30120b4b40120b5b50120b6b60120b7b70120
	+	0xb8dd0101b9de0101badf0101bbbb0120bcfc0101bdbd0120befd0101bffe0101c0c00102c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2d20100d3f30101d4f40101d5f50101d6f60101d7f70101d8f80101d9f90101dafa0101dbfb0101a2dc0102b8dd0102b9de0102badf0102e0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102cded0102ceee0102cfef0102d0f00102d1f10102d3f20102d3f30102d4f40102d5f50102d6f60102d7f70102d8f80102d9f90102dafa0102dbfb0102bcfc0102befd0102bffe0102ffff01000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 6
go

INSERT syscharsets VALUES ( 1001,   7, 0, 0, N'cp1254', N'Code page 1254', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181012082820120838301018484012085850120868601208787012088880120898901208a9a01018b8b01208c9c01018d8d01208e8e01208f8f0120909001209191012092920120939301209494012095950120969601209797012098980120999901208a9a01029b9b01208c9c01029d9d01209e9e01029fff0101a0a00120a1a10120a2a20120a3a30120a4a44020a5a54020a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120b3b30120b4b40120b5b50120b6b60120b7b70120
	+	0xb8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bebe0120bfbf0120c0e00101c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2f20101d3f30101d4f40101d5f50101d6f60101d7d70120d8f80101d9f90101dafa0101dbfb0101dcfc0101dddd0101defe0101dfdf0102c0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102eddd0102ceee0102cfef0102d0f00102d1f10102d2f20102d3f30102d4f40102d5f50102d6f60102f7f70120d8f80102d9f90102dafa0102dbfb0102dcfc0102fdfd0102defe01029fff01020000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 7
go

INSERT syscharsets VALUES ( 1001,   8, 0, 0, N'cp1255', N'Code page 1255', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181012082820120838301018484012085850120868601208787012088880120898901208a8a01208b8b01208c8c01208d8d01208e8e01208f8f0120909001209191012092920120939301209494012095950120969601209797012098980120999901209a9a01209b9b01209c9c01209d9d01209e9e01209f9f0120a0a00108a1a10120a2a20120a3a34020a4a44020a5a54020a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120b3b30120b4b40120b5b50120b6b60120b7b70120
	+	0xb8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bcbc0120bfbf0120c0c00120c1c10120c2c20120c3c30120c4c40120c5c50120c6c60120c7c70120c8c80120c9c90120caca0120cbcb0120cccc0120cdcd0120cece0120cfcf0120d0d00120d1d10120d2d20120d3d30120d4d40100d5d50100d6d60100d7d70100d8d80100d9d90100dada0100dbdb0100dcdc0100dddd0100dede0100dfdf0100e0e00101e1e10101e2e20101e3e30101e4e40101e5e50101e6e60101e7e70101e8e80101e9e90101eaeb0101eaeb0102ecec0101edee0101edee0102f0ef0101f0ef0102f1f10101f2f20101f3f40101f3f40102f5f60101f5f60102f7f70101f8f80102f9f90101fafa0101dbfb0100dcfc0100ddfd0100defe0100ffff01000000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 8
go

INSERT syscharsets VALUES ( 1001,   9, 0, 0, N'cp1256', N'Code page 1256', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181010182820120838301028484012085850120868601208787012088880120898901208a8a01208b8b01208c9c01018d8d01018e8e01018f8f0120909001019191012092920120939301209494012095950120969601209797012098980120999901209a9a01209b9b01208c9c01029d9d01208e9e01029f9f0120a0a00108a1a10120a2a20120a3a30120a4a40120a5a50120a6a60120a7a70120a8a80120a9a90120aaaa0120abab0120acac0120adad0120aeae0120afaf0120b0b00120b1b10120b2b20120a3b30102b4b40120b5b50120b6b60120b7b70120
	+	0xb8b80120b9b90120baba0120bbbb0120bcbc0120bdbd0120bebe0120bfbf0120c0c00100c1c10101c2e20101c3c30101c4c40101c5c50101c6c60101c7c70101c8c80101c9c90101caca0101cbcb0101cccc0101cdcd0101cece0101cfcf0101d0d00101d1d10101d2d20101d3d30101d4d40101d5d50101d6d60101d7d70120d8d80101d9d90101dada0101dbdb0101dcdc0101dddd0101dede0101dfdf010141e00102e1e1010241e20102e3e30102e4e40102e5e50102e6e6010243e7010245e8010245e9010245ea010245eb0102ecec0102eded010249ee010249ef0102f0f00120f1f10120f2f20120f3f301204ff40102f5f50120f6f60120f7f70120f8f8012055f90102fafa012055fb010255fc0102ddfd0102defe0102ffff01200000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 9
go

INSERT syscharsets VALUES ( 1001,  10, 0, 0, N'cp1257', N'Code page 1257', 0x000008000101202002022020030320200404202005052020060620200707082008080820090908080a0a08080b0b08080c0c08080d0d08080e0e20200f0f2020101020201111202012122020131320201414012015150120161620201717202018182020191920201a1a20201b1b20201c1c08201d1d20201e1e20201f1f2020202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x01015b5b20205c5c20205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2008808001208181012082820120838301208484012085850120868601208787012088880120898901208a8a01208b8b01208c8c01208d8d01208e8e01208f8f0120909001209191012092920120939301209494012095950120969601209797012098980120999901209a9a01209b9b01209c9c01209d9d01209e9e01209f9f0120a0a00108a1a10120a2a20120a3a30120a4a40120a5a50120a6a60120a7a70120a8b80101a9a90120aaba0101abab0120acac0120adad0120aeae0120afbf0101b0b00120b1b10120b2b20120b3b30120b4b40120b5b50120b6b60120b7b70120
	+	0xa8b80102b9b90120aaba0102bbbb0120bcbc0120bdbd0120bebe0120afbf0102c0e00101c1e10101c2e20101c3e30101c4e40101c5e50101c6e60101c7e70101c8e80101c9e90101caea0101cbeb0101ccec0101cded0101ceee0101cfef0101d0f00101d1f10101d2f20101d3f30101d4f40101d5f50101d6f60101d7d70120d8f80101d9f90101dafa0101dbfb0101dcfc0101ddfd0101defe0101dfdf0101c0e00102c1e10102c2e20102c3e30102c4e40102c5e50102c6e60102c7e70102c8e80102c9e90102caea0102cbeb0102ccec0102cded0102ceee0102cfef0102d0f00102d1f10102d2f20102d3f30102d4f40102d5f50102d6f60102f7f70120d8f80102d9f90102dafa0102dbfb0102dcfc0102ddfd0102defe0102ffff01200000800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003031323330313230303232343535303132
	+	0x36323330313032303200000000000030313233303132303032323435353031323632333031303230320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 10
go

INSERT syscharsets VALUES ( 1001,  12, 0, 0, N'cp949', N'Code page 949', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100
	+	0x00010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101ffff0101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 12
go

INSERT syscharsets VALUES ( 1001,  13, 0, 0, N'cp950', N'Code page 950', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100
	+	0x00010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101ffff0101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 13
go

INSERT syscharsets VALUES ( 1001,  14, 0, 0, N'cp936', N'Code page 936', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001018181010182820101838301018484010185850101868601018787010188880101898901018a8a01018b8b01018c8c01018d8d01018e8e01018f8f0101909001019191010192920101939301019494010195950101969601019797010198980101999901019a9a01019b9b01019c9c01019d9d01019e9e01019f9f0101a0a00101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100
	+	0x00010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101ffff0101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 14
go

INSERT syscharsets VALUES ( 1001, 109, 0, 0, N'cp932', N'Code page 932', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f20008080200000000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101a0a00100a1a12020a2a22020a3a32020a4a42020a5a52020a6a60101a7a70101a8a80101a9a90101aaaa0101abab0101acac0101adad0101aeae0101afaf0101b0b00101b1b10101b2b20101b3b30101b4b40101b5b50101b6b60101b7b70101b8
	+	0xb80101b9b90101baba0101bbbb0101bcbc0101bdbd0101bebe0101bfbf0101c0c00101c1c10101c2c20101c3c30101c4c40101c5c50101c6c60101c7c70101c8c80101c9c90101caca0101cbcb0101cccc0101cdcd0101cece0101cfcf0101d0d00101d1d10101d2d20101d3d30101d4d40101d5d50101d6d60101d7d70101d8d80101d9d90101dada0101dbdb0101dcdc0101dddd0101de000101df0001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101fdfd2000fefe2000ffff2000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 109
go




--  'CP850BIN.850'
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001,  40, 2, 0, N'bin_cp850', N'Binary Sort Order for the CodePage 850 Character Set', 0x0000)
go
--  'NOCASE.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001,  52, 1, 0, N'nocase_iso', N'Case-insensitive dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the ISO 8859-1 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3a00fe030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a8400013185000132860001338700013488000135890001368a0001378b0001388c0001398d0001413000013b3100013c3200013d3300013e3400013f350001403600015b8e0001428f00014390000144910001459200014693000147
	+	0x94000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000157a4000158a5000159a600015aa70001d03700015c3800015d3900015e3a00015f3b0001603c00017b8e0002428f0002439000024491000245920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b00024f9c0002509d0002519e0002529f000253a0000254a1000255a2000256a3000257a4000258a5000259a600025aa70002d03d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018a4c00018b4d00018c4e00018d4f00018e5000018f5100019052000191530001925400019355000194560001955700019658000197590001985a0001995b00019a5c00019b5d00019c5e00019d5f00019e6000019f610001a0620001a1630001a2640001a3650001a46600
	+	0x01a5670001a6680001a7690001a86a0001a96b0001aa6c0001ab6d0001ac6e0001ad6f0001ae700001af710001b0720001b1730001b2740001b3750001b4760001b5770001b6780001b7790001b87a0001b97b0001ba7c0001bb7d0001bc7e0001bd7f0001be800001bf810001d78e0201428e0301428e0401428e0501428e0601428e0701420000000090010144920201469203014692040146920501469601014a9602014a9603014a9604014aa80001de9b01014f9c0101509c0201509c0301509c0401509c050150820001f79c060150a2010156a2020156a2030156a2040156a601015aa9000100000000028e0202428e0302428e0402428e0502428e0602428e0702420000000190010244920202469203024692040246920502469601024a9602024a9603024a9604024aa80002de9b01024f9c0102509c0202509c0302509c0402509c050250830001309c060250a2010256a2020256a2030256a2040256a601025aa9000200a602015aff0300778e010142
	+	0x92010146ff0300318e01024292010246ff030000a0010154a0010154ff0000000000000000000000ff0000000000000000000000ff0000002a004000d0f81200ff000000bcf81200f280f277fff9120048f8120040000000ff000000e001000050006100ff0069006600690063002000ff00740061006e0064006100ff0064002000540069006d00ff0000000000000000000000ff0000000000000000000000ff000a000500020000000000ff0000000000000050006100ff0069006600690063002000ff00610079006c0069006700ff0074002000540069006d00ff0000000000000000000000ff0000000000000000000000ff0004000100020000000000ff000000c4ffffff80e94386ff1ebb01009e8df25121bb01ff40230e430000001f10dc77ff3914007cf912009eaadb77ff001400783914001810db77ff0100005000610070391400ff0b0000e95cf87748051400ff5cf877583a140000000000ff0c1400b805b07700000000ff0c14007012f877ffff
	+	0xffffff5cf87748051400ea5cf877ff20ad7700000000500c1400ff0000007cf26131e9c3db77ff00000058f912008cf91200ff39140033000000d1c6db77ff5cf87748052f00ea5cf877ff1000100cfb1200a0081400fff912001c10db770c07dd77ff626231c0f9120039cfdb77fff91200010000002f52db77fff912001810db77c0022e00ff052f0000802f0003000000ff000000600000005073f277ff00000080052f0001000000fff91200c673f277a8391400fff9120000800000b8052f00ff052f0080052f0080050010fffa1200314cf87700002f00ff052f00dd94f977b08dfb77ff94f9770000000050002f00ff2000000700b000acfa1200ff2af8779cfa1200645bfa77ff13f877ffffffffacfa1200ff2bf87700002f0000100010ff000000a008140000000000ff001000a0032e0098e50310ff081400a008140000f00700ff0000006889fb77a0081400ff0000000000000008000000fffa12000c001400e00e1400ff081400f4fa1200a65ef877ff0014005160f87748051400
where id = 52
go
--  'CP437BIN.437'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values  (2001, 30, 3, 0, N'bin_cp437', N'Binary Sort Order for the Code Page 437 Character Set', 0x0000)
go
--  'DICTION.437'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 31, 3, 0, N'dictionary_437', N'General purpose dictionary sort order for use in the US.  Uses the Code Page 437 character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition =
		0x3200ee020000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a9400013195000132960001339700013498000135990001369a0001379b0001389c0001399d0001413000013b3100013c3200013d3300013e3400013f350001403600015b9e0001429f000143a0000144a1000145a2000146a3000147a4000148a5000149a600014aa700014ba800014ca900014daa00014eab00014fac000150ad000151ae000152af000153b0000154b1000155b2000156b3000157b4000158b5000159b600015ab70001e03700015c3800015d3900015e3a00015f3b0001603c00017b9e0201429f010143
	+	0xa0010144a1010145a2020146a3010147a4010148a5010149a601014aa701014ba801014ca901014daa01014eab01014fac010150ad010151ae010152af010153b0010154b1010155b2010156b3010157b4010158b5010159b601015ab70101e03d00017c3e00017d3f00017e4000017f4100019ba0020144b2060156a20601469e0601429e0801429e0401429e0a0142a0030144a2070146a2080146a2040146a605014aa604014aa602014a9e0701429e090142a20501460000000100000000ac040150ac060150ac020150b2040156b2020156b602015aac050150b20501564200019c4300019d4400019e4500019f460001a69e050142a603014aac030150b2030156ab03014fab02014f470001a7480001a8490001a94a0001aa4b0001ab4c0001ac4d0001ad4e0001ae4f0001af500001b0510001b1520001b2530001b3540001b4550001b5560001b6570001b7580001b8590001b95a0001ba5b0001bb5c0001bc5d0001bd5e0001be5f0001bf600001c0610001c1620001c2630001c3640001c4650001c5660001c6670001c7
	+	0x680001c8690001c96a0001ca6b0001cb6c0001cc6d0001cd6e0001ce6f0001cf700001d0710001d1720001d2730001d3740001d4750001d5760001d6770001d7780001d8790001d97a0001da7b0001db7c0001dc7d0001dd7e0001de7f0001df800001ecb80001e1b90001e2ba0001e3bb0001e4bc0001e6bc0101e6bd0001e7be0001e8bf0001e9c00001eac10001ebc20001ee810001ed820001efc3000100830001f0840001f1850001f2860001f3870001f4880001f5890001f68a0001f78b0001f88c0001f98d0001fa8e0001fb8f0001fc900001fd910001fe920001ff93000130ff0300009e010142a2010146ff0300009e030142a2030146ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff0034000200000095000000ff000000748dcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 31
go
--  'NOACCENT.437'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)	
	values  (2001, 34, 3, 0, N'noaccents_437', N'Case-insensitive dictionary sort order for use in the US.  Uses the Code Page 437 character set.  Diacritical marks are ignored.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00ee020000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a9400013195000132960001339700013498000135990001369a0001379b0001389c0001399d0001413000013b3100013c3200013d3300013e3400013f350001403600015b9e0001429f000143a0000144a1000145a2000146a3000147a4000148a5000149a600014aa700014ba800014ca900014daa00014eab00014fac000150ad000151ae000152af000153b0000154b1000155b2000156b3000157b4000158b5000159b600015ab70001e03700015c3800015d3900015e3a00015f3b0001603c00017b9e0002429f000243
	+	0xa0000244a1000245a2000246a3000247a4000248a5000249a600024aa700024ba800024ca900024daa00024eab00024fac000250ad000251ae000252af000253b0000254b1000255b2000256b3000257b4000258b5000259b600025ab70002e03d00017c3e00017d3f00017e4000017f4100019ba0000344b2000756a20005469e0005429e0007429e0003429e000942a0000444a2000646a2000746a2000346a600064aa600054aa600034a9e0006429e000842a20004460000000100000000ac000550ac000750ac000350b2000556b2000356b600035aac000650b20006564200019c4300019d4400019e4500019f460001a69e000442a600044aac000450b2000456ab00044fab00034f470001a7480001a8490001a94a0001aa4b0001ab4c0001ac4d0001ad4e0001ae4f0001af500001b0510001b1520001b2530001b3540001b4550001b5560001b6570001b7580001b8590001b95a0001ba5b0001bb5c0001bc5d0001bd5e0001be5f0001bf600001c0610001c1620001c2630001c3640001c4650001c5660001c6670001c7
	+	0x680001c8690001c96a0001ca6b0001cb6c0001cc6d0001cd6e0001ce6f0001cf700001d0710001d1720001d2730001d3740001d4750001d5760001d6770001d7780001d8790001d97a0001da7b0001db7c0001dc7d0001dd7e0001de7f0001df800001ecb80001e1b90001e2ba0001e3bb0001e4bc0001e6bc0002e6bd0001e7be0001e8bf0001e9c00001eac10001ebc20001ee810001ed820001efc3000100830001f0840001f1850001f2860001f3870001f4880001f5890001f68a0001f78b0001f88c0001f98d0001fa8e0001fb8f0001fc900001fd910001fe920001ff93000130ff0300009e010142a2010146ff0300009e010242a2010246ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000de000000ff000000928dcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 34
go
--  'NOCASE.437'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values  (2001, 32, 3, 0, N'nocase_437', N'Case-insensitive dictionary sort order for use in the US.  Uses the Code Page 437 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3a00ee020000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a9400013195000132960001339700013498000135990001369a0001379b0001389c0001399d0001413000013b3100013c3200013d3300013e3400013f350001403600015b9e0001429f000143a0000144a1000145a2000146a3000147a4000148a5000149a600014aa700014ba800014ca900014daa00014eab00014fac000150ad000151ae000152af000153b0000154b1000155b2000156b3000157b4000158b5000159b600015ab70001e03700015c3800015d3900015e3a00015f3b0001603c00017b9e0002429f000243
	+	0xa0000244a1000245a2000246a3000247a4000248a5000249a600024aa700024ba800024ca900024daa00024eab00024fac000250ad000251ae000252af000253b0000254b1000255b2000256b3000257b4000258b5000259b600025ab70002e03d00017c3e00017d3f00017e4000017f4100019ba0010144b2040256a20302469e0401429e0502429e0201429e060242a0010244a2040146a2050146a2020146a604014aa603014aa601014a9e0501429e060142a20301460000000100000000ac030150ac040250ac010150b2030156b2010156b601015aac040150b20401564200019c4300019d4400019e4500019f460001a69e030142a602014aac020150b2020156ab01024fab01014f470001a7480001a8490001a94a0001aa4b0001ab4c0001ac4d0001ad4e0001ae4f0001af500001b0510001b1520001b2530001b3540001b4550001b5560001b6570001b7580001b8590001b95a0001ba5b0001bb5c0001bc5d0001bd5e0001be5f0001bf600001c0610001c1620001c2630001c3640001c4650001c5660001c6670001c7
	+	0x680001c8690001c96a0001ca6b0001cb6c0001cc6d0001cd6e0001ce6f0001cf700001d0710001d1720001d2730001d3740001d4750001d5760001d6770001d7780001d8790001d97a0001da7b0001db7c0001dc7d0001dd7e0001de7f0001df800001ecb80001e1b90001e2ba0001e3bb0001e4bc0001e6bc0002e6bd0001e7be0001e8bf0001e9c00001eac10001ebc20001ee810001ed820001efc3000100830001f0840001f1850001f2860001f3870001f4880001f5890001f68a0001f78b0001f88c0001f98d0001fa8e0001fb8f0001fc900001fd910001fe920001ff93000130ff0300009e010142a2010146ff0300009e010242a2010246ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000d2000000ff000000c58dcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 32
go
--  'NOCASEPR.437'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values  (2001, 33, 3, 0, N'nocasepref_437', N'Case-insensitive dictionary sort order, with case preference for collating purposes, for use in the US.  Uses the Code Page 437 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00ee020000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a9400013195000132960001339700013498000135990001369a0001379b0001389c0001399d0001413000013b3100013c3200013d3300013e3400013f350001403600015b9e0001429f000143a0000144a1000145a2000146a3000147a4000148a5000149a600014aa700014ba800014ca900014daa00014eab00014fac000150ad000151ae000152af000153b0000154b1000155b2000156b3000157b4000158b5000159b600015ab70001e03700015c3800015d3900015e3a00015f3b0001603c00017b9e0002429f000243
	+	0xa0000244a1000245a2000246a3000247a4000248a5000249a600024aa700024ba800024ca900024daa00024eab00024fac000250ad000251ae000252af000253b0000254b1000255b2000256b3000257b4000258b5000259b600025ab70002e03d00017c3e00017d3f00017e4000017f4100019ba0010144b2040256a20302469e0401429e0502429e0201429e060242a0010244a2040146a2050146a2020146a604014aa603014aa601014a9e0501429e060142a20301460000000100000000ac030150ac040250ac010150b2030156b2010156b601015aac040150b20401564200019c4300019d4400019e4500019f460001a69e030142a602014aac020150b2020156ab01024fab01014f470001a7480001a8490001a94a0001aa4b0001ab4c0001ac4d0001ad4e0001ae4f0001af500001b0510001b1520001b2530001b3540001b4550001b5560001b6570001b7580001b8590001b95a0001ba5b0001bb5c0001bc5d0001bd5e0001be5f0001bf600001c0610001c1620001c2630001c3640001c4650001c5660001c6670001c7
	+	0x680001c8690001c96a0001ca6b0001cb6c0001cc6d0001cd6e0001ce6f0001cf700001d0710001d1720001d2730001d3740001d4750001d5760001d6770001d7780001d8790001d97a0001da7b0001db7c0001dc7d0001dd7e0001de7f0001df800001ecb80001e1b90001e2ba0001e3bb0001e4bc0001e6bc0002e6bd0001e7be0001e8bf0001e9c00001eac10001ebc20001ee810001ed820001efc3000100830001f0840001f1850001f2860001f3870001f4880001f5890001f68a0001f78b0001f88c0001f98d0001fa8e0001fb8f0001fc900001fd910001fe920001ff93000130ff0300009e010142a2010146ff0300009e010242a2010246ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000b0000000ff000000e38dcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 33
go
--  'DICTION.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 41, 2, 0, N'dictionary_850', N'General purpose dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition =
		0x3200e8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f00014490000145910001469200014793000148940001499500014a9600014b9700014c9800014d9900014e9a00014f9b0001509c0001519d0001529e0001539f000154a0000155a1000156a2000157a3000158a4000159a500015aa60001d13700015c3800015d3900015e3a00015f3b0001603c00017b8d0201428e010143
	+	0x8f01014490010145910201469201014793010148940101499501014a9601014b9701014c9801014d9901014e9a01014f9b0101509c0101519d0101529e0101539f010154a0010155a1010156a2010157a3010158a4010159a501015aa60101d13d00017c3e00017d3f00017e4000017f4100019c8f020144a1090156910701468d0901428d0d01428d0501428d0f01428f03014491090146910b0146910501469509014a9507014a9503014a8d0c01428d0e01429106014600000001000000009b0701509b0b01509b030150a1070156a1030156a504015a9b0a0150a10801569b0d01504200019e9b0c01504300019f440001a68d0701429505014a9b050150a10501569a03014f9a02014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0601428d0801428d040142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d0b0142
	+	0x8d0a0142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d9a70101e7a70001e791080146910a014691040146950a014a9504014a9506014a9508014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9502014a6f0001e69b040150000000029b0601509b0201509b0901509b080150700001eea8000100a8010100a1040156a1060156a1020156a503015aa502015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014291010146ff0300008d03014291030146ff0300009f0201549f020154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff003400020000009e000000ff000000528ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 41
go
--  'NOACCENT.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 44, 2, 0, N'noaccents_850', N'Case-insensitive dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set.  Diacritical marks are ignored.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00e8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f00014490000145910001469200014793000148940001499500014a9600014b9700014c9800014d9900014e9a00014f9b0001509c0001519d0001529e0001539f000154a0000155a1000156a2000157a3000158a4000159a500015aa60001d13700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f00024490000245910002469200024793000248940002499500024a9600024b9700024c9800024d9900024e9a00024f9b0002509c0002519d0002529e0002539f000254a0000255a1000256a2000257a3000258a4000259a500025aa60002d13d00017c3e00017d3f00017e4000017f4100019c8f000344a1000a56910006468d0008428d000c428d0004428d000e428f0004449100084691000a469100044695000a4a9500084a9500044a8d000b428d000d429100054600000001000000009b0008509b000c509b000450a1000856a1000456a500055a9b000b50a10009569b000e504200019e9b000d504300019f440001a68d0006429500064a9b000650a10006569a00044f9a00034f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0005428d0007428d000342540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d000a42
	+	0x8d000942620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d9a70002e7a70001e791000746910009469100034695000b4a9500054a9500074a9500094a6a0001da6b0001db6c0001dc6d0001dd6e0001df9500034a6f0001e69b000550000000029b0007509b0003509b000a509b000950700001eea8000100a8000200a1000556a1000756a1000356a500045aa500035a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014291010146ff0300008d01024291010246ff0300009f0101549f010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000b6000000ff000000708ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 44
go
--  'NOCASE.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 42, 2, 0, N'nocase_850', N'Case-insensitive dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3a00e8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f00014490000145910001469200014793000148940001499500014a9600014b9700014c9800014d9900014e9a00014f9b0001509c0001519d0001529e0001539f000154a0000155a1000156a2000157a3000158a4000159a500015aa60001d13700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f00024490000245910002469200024793000248940002499500024a9600024b9700024c9800024d9900024e9a00024f9b0002509c0002519d0002529e0002539f000254a0000255a1000256a2000257a3000258a4000259a500025aa60002d13d00017c3e00017d3f00017e4000017f4100019c8f010144a1040256910302468d0402428d0602428d0202428d0702428f0102449104024691050246910202469504024a9503024a9501024a8d0601428d0701429103014600000001000000009b0302509b0502509b010250a1030256a1010256a502015a9b050150a10401569b0602504200019e9b0601504300019f440001a68d0302429502024a9b020250a10202569a01024f9a01014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0301428d0401428d020142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d050242
	+	0x8d050142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d9a70002e7a70001e79104014691050146910201469505014a9502014a9503014a9504014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9501014a6f0001e69b020150000000029b0301509b0101509b0402509b040150700001eea8000100a8000200a1020156a1030156a1010156a501025aa501015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014291010146ff0300008d01024291010246ff0300009f0101549f010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000ae000000ff000000a38ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 42
go
--  'NOCASE34.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 49, 2, 0, N'caseless_34', N'Case-insensitive sort order for 3.4.1 caseless database.', 0x0000)
update syscharsets set binarydefinition =
		0x0c00ff000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f0001303000013131000132320001333300013434000135350001363600013737000138380001393900013a3a00013b3b00013c3c00013d3d00013e3e00013f3f0001404000014141000142420001434300014444000145450001464600014747000148480001494900014a4a00014b4b00014c4c00014d4d00014e4e00014f4f0001505000015151000152520001535300015454000155550001565600015757000158580001595900015a5a00015b5b00015c5c00015d5d00015e5e00015f5f0001606000017b4100024242000243
	+	0x4300024444000245450002464600024747000248480002494900024a4a00024b4b00024c4c00024d4d00024e4e00024f4f0002505000025151000252520002535300025454000255550002565600025757000258580002595900025a5a00025b6100017c6200017d6300017e6400017f65000180660001816700018268000183690001846a0001856b0001866c0001876d0001886e0001896f00018a7000018b7100018c7200018d7300018e7400018f75000190760001917700019278000193790001947a0001957b0001967c0001977d0001987e0001997f00019a8000019b8100019c8200019d8300019e8400019f850001a0860001a1870001a2880001a3890001a48a0001a58b0001a68c0001a78d0001a88e0001a98f0001aa900001ab910001ac920001ad930001ae940001af950001b0960001b1970001b2980001b3990001b49a0001b59b0001b69c0001b79d0001b89e0001b99f0001baa00001bba10001bca20001bda30001bea40001bfa50001c0a60001c1a70001c2a80001c3a90001c4aa0001c5ab0001c6ac0001c7
	+	0xad0001c8ae0001c9af0001cab00001cbb10001ccb20001cdb30001ceb40001cfb50001d0b60001d1b70001d2b80001d3b90001d4ba0001d5bb0001d6bc0001d7bd0001d8be0001d9bf0001dac00001dbc10001dcc20001ddc30001dec40001dfc50001e0c60001e1c70001e2c80001e3c90001e4ca0001e5cb0001e6cc0001e7cd0001e8ce0001e9cf0001ead00001ebd10001ecd20001edd30001eed40001efd50001f0d60001f1d70001f2d80001f3d90001f4da0001f5db0001f6dc0001f7dd0001f8de0001f9df0001fae00001fbe10001fce20001fde30001fee40001ffe5000100ff006e00670020004f007200ff0065007200000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000c4000000ff000000c18ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 49
go
--  'NOCASEPR.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 43, 2, 0, N'nocasepref_850', N'Case-insensitive dictionary sort order, with case preference for collating purposes, for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00e8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f00014490000145910001469200014793000148940001499500014a9600014b9700014c9800014d9900014e9a00014f9b0001509c0001519d0001529e0001539f000154a0000155a1000156a2000157a3000158a4000159a500015aa60001d13700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f00024490000245910002469200024793000248940002499500024a9600024b9700024c9800024d9900024e9a00024f9b0002509c0002519d0002529e0002539f000254a0000255a1000256a2000257a3000258a4000259a500025aa60002d13d00017c3e00017d3f00017e4000017f4100019c8f010144a1040256910302468d0402428d0602428d0202428d0702428f0102449104024691050246910202469504024a9503024a9501024a8d0601428d0701429103014600000001000000009b0302509b0502509b010250a1030256a1010256a502015a9b050150a10401569b0602504200019e9b0601504300019f440001a68d0302429502024a9b020250a10202569a01024f9a01014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0301428d0401428d020142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d050242
	+	0x8d050142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d9a70002e7a70001e79104014691050146910201469505014a9502014a9503014a9504014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9501014a6f0001e69b020150000000029b0301509b0101509b0402509b040150700001eea8000100a8000200a1020156a1030156a1010156a501025aa501015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014291010146ff0300008d01024291010246ff0300009f0101549f010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000d0000000ff000000df8ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 43
go
--  'ALTDICT.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 55, 2, 0, N'alt_dictionary', N'General purpose dictionary sort order for use with several Western-European languages. Uses the Code Page 850 character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition =
		0x32009b030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b0001a59d0001509e0001519f000152a0000153a1000154a2000155a3000156a4000157a5000158a6000159a700015aa80001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0801428e010143
	+	0x8f020144900101d1920601469301014794010148950101499605014a9701014b9801014c9901014d9a01014e9b0101a59d0601509e0101519f010152a0010153a1010154a2010155a3050156a4010157a5010158a6010159a702015aa80101e73d00017c3e00017d3f00017e4000017f4100019c8f010144a3090156920901468d0c01428d0e01428d0a01428d0f01428f030144920a0146920b0146920801469609014a9608014a9606014a8d0601428d0701429203014600000001000000009d0901509d0b01509d070150a3080156a3060156a704015a9d050150a3040156aa0101004200019eaa0001004300019f440001a68d0b01429607014a9d080150a30701569c01014f9c00014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0301428d0401428d020142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d0d0142
	+	0x8d050142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d99101014591000145920401469205014692020146960a014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69d020150000000029d0301509d0101509d0a01509d040150700001eea900019da901019da3020156a3030156a3010156a703015aa701015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014292010146ff0300008d09014292070146ff030000a1020154a1020154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000bc000000ff000000fd8ecd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 55
go
--  'ALTNOACC.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 57, 2, 0, N'alt_noaccents', N'Case-insensitive dictionary sort order for most of the European languages.  Uses the Code Page 850 character set.  Diacritical marks are ignored.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00e8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b0001a59d0001509e0001519f000152a0000153a1000154a2000155a3000156a4000157a5000158a6000159a700015aa80001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f000244900002d1920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b0002a59d0002509e0002519f000252a0000253a1000254a2000255a3000256a4000257a5000258a6000259a700025aa80002e73d00017c3e00017d3f00017e4000017f4100019c8f000344a3000a56920006468d0008428d000c428d0004428d000e428f0004449200084692000a469200044696000a4a9600084a9600044a8d000b428d000d429200054600000001000000009d0008509d000c509d000450a3000856a3000456a700055a9d000b50a30009569d000e504200019e9d000d504300019f440001a68d0006429600064a9d000650a30006569c00024f9c00014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0005428d0007428d000342540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d000a42
	+	0x8d000942620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d9910002459100014592000746920009469200034696000b4a9600054a9600074a9600094a6a0001da6b0001db6c0001dc6d0001dd6e0001df9600034a6f0001e69d000550000000029d0007509d0003509d000a509d000950700001eea9000100a9000200a3000556a3000756a3000356a700045aa700035a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014292010146ff0300008d01024292010246ff030000a1010154a1010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000cb000000ff0000001b8fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 57
go
--  'ALTNOCSP.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 56, 2, 0, N'alt_nocasepref', N'Case-insensitive dictionary sort order, with case preference for collating purposes, for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3c009b030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b0001a59d0001509e0001519f000152a0000153a1000154a2000155a3000156a4000157a5000158a6000159a700015aa80001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f000244900002d1920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b0002a59d0002509e0002519f000252a0000253a1000254a2000255a3000256a4000257a5000258a6000259a700025aa80002e73d00017c3e00017d3f00017e4000017f4100019c8f010144a3040256920302468d0402428d0602428d0202428d0702428f0102449204024692050246920202469604024a9603024a9601024a8d0601428d0701429203014600000001000000009d0302509d0502509d010250a3030256a3010256a702015a9d050150a3040156aa0002004200019eaa0001004300019f440001a68d0302429602024a9d020250a30202569c00024f9c00014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0301428d0401428d020142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d050242
	+	0x8d050142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d991000245910001459204014692050146920201469605014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69d020150000000029d0301509d0101509d0402509d040150700001eea900019da900029da3020156a3030156a3010156a701025aa701015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014292010146ff0300008d01024292010246ff030000a1010154a1010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000dd000000ff0000004e8fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 56
go
--  'ALTNOCS.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 61, 2, 0, N'alt_nocase', N'Case-insensitive dictionary sort order, without case preference for collating purposes, for use with several Western-European languages including English, French, and German.  Uses the Code Page 850 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3a009b030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b0001a59d0001509e0001519f000152a0000153a1000154a2000155a3000156a4000157a5000158a6000159a700015aa80001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f000244900002d1920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b0002a59d0002509e0002519f000252a0000253a1000254a2000255a3000256a4000257a5000258a6000259a700025aa80002e73d00017c3e00017d3f00017e4000017f4100019c8f010144a3040256920302468d0402428d0602428d0202428d0702428f0102449204024692050246920202469604024a9603024a9601024a8d0601428d0701429203014600000001000000009d0302509d0502509d010250a3030256a3010256a702015a9d050150a3040156aa0002004200019eaa0001004300019f440001a68d0302429602024a9d020250a30202569c00024f9c00014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0301428d0401428d020142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d050242
	+	0x8d050142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d991000245910001459204014692050146920201469605014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69d020150000000029d0301509d0101509d0402509d040150700001eea900019da900029da3020156a3030156a3010156a701025aa701015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff0300008d01014292010146ff0300008d01024292010246ff030000a1010154a1010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000dc000000ff0000006c8fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 61
go
--  'SCANDICT.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 59, 2, 0, N'scand_dictionary', N'General purpose Scandinavian dictionary sort order.  Uses the Code Page 850 character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition =
		0x320094010000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000158a3020158a4000159a500015aa60001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0501428e010143
	+	0x8f020144900101d1920501469301014794010148950101499605014a9701014b9801014c9901014d9a01014e9b02014f9c0501509d0101519e0101529f010153a0010154a1010155a2050156a3010158a3030158a4010159a502015aa60101e73d00017c3e00017d3f00017e4000017f4100019c8f010144a2090156920701468d080142ab0101998d060142aa01018e8f0301449208014692090146920601469609014a9608014a9606014aab000199aa00018e92020146a801019da800019d9c080150ac0101009c060150a2080156a2060156a504015aac000100a2040156a901018f4200019ea900018f4300019f440001a68d0701429607014a9c070150a20701569b03014f9b01014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0201428d0301428d010142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d090142
	+	0x8d040142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d99101014591000145920301469204014692010146960a014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69c020150000000009c0301509c0101509c0901509c040150700001eea7000192a7010192a2020156a2030156a2010156a503015aa501015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff030000a0020154a0020154ff0065007200000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000a0000000ff0000008a8fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 59
go
--  'SCANNOCP.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 58, 2, 0, N'scand_nocasepref', N'Case-insensitive Scandinavian dictionary sort order, with  case preference for collating purposes.  Uses the Code Page 850  character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3c0094010000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000158a3000358a4000159a500015aa60001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f000244900002d1920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b00024f9c0002509d0002519e0002529f000253a0000254a1000255a2000256a3000258a3000458a4000259a500025aa60002e73d00017c3e00017d3f00017e4000017f4100019c8f010144a2040256920202468d030242ab0002998d010242aa00028e8f0102449203024692040246920102469604024a9603024a9601024aab000199aa00018e92020146a800029da800019d9c030250ac0002009c010250a2030256a2010256a502015aac000100a2040156a900028f4200019ea900018f4300019f440001a68d0202429602024a9c020250a20202569b01024f9b01014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0201428d0301428d010142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d040242
	+	0x8d040142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d991000245910001459203014692040146920101469605014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69c020150000000009c0301509c0101509c0402509c040150700001eea7000192a7000292a2020156a2030156a2010156a501025aa501015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff030000a0010154a0010154ff0065007200000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000be000000ff000000a88fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 58
go
--  'SCANNOCS.850'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 60, 2, 0, N'scand_nocase', N'Case-insensitive Scandinavian dictionary sort order, without  case preference for collating purposes.  Uses the Code Page 850  character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3a0094010000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a830001318400013285000133860001348700013588000136890001378a0001388b0001398c0001413000013b3100013c3200013d3300013e3400013f350001403600015b8d0001428e0001438f000144900001d1920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000158a3000358a4000159a500015aa60001e73700015c3800015d3900015e3a00015f3b0001603c00017b8d0002428e000243
	+	0x8f000244900002d1920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b00024f9c0002509d0002519e0002529f000253a0000254a1000255a2000256a3000258a3000458a4000259a500025aa60002e73d00017c3e00017d3f00017e4000017f4100019c8f010144a2040256920202468d030242ab0002998d010242aa00028e8f0102449203024692040246920102469604024a9603024a9601024aab000199aa00018e92020146a800029da800019d9c030250ac0002009c010250a2030256a2010256a502015aac000100a2040156a900028f4200019ea900018f4300019f440001a68d0202429602024a9c020250a20202569b01024f9b01014f450001a7460001a8470001a9480001aa490001ab4a0001ac4b0001ad4c0001ae4d0001af4e0001b04f0001b1500001b2510001b3520001b4530001b88d0201428d0301428d010142540001b9550001ba560001bb570001bc580001bd590001be5a0001bf5b0001c05c0001c15d0001c25e0001c35f0001c4600001c5610001c88d040242
	+	0x8d040142620001c9630001ca640001cb650001cc660001cd670001ce680001cf690001d991000245910001459203014692040146920101469605014a9602014a9603014a9604014a6a0001da6b0001db6c0001dc6d0001dd6e0001df9601014a6f0001e69c020150000000009c0301509c0101509c0402509c040150700001eea7000192a7000292a2020156a2030156a2010156a501025aa501015a710001ef720001f0730001f1740001f2750001f3760001f4770001f5780001f6790001f77a0001f87b0001f97c0001fa7d0001fb7e0001fc7f0001fd800001fe810001ff82000130ff030000a0010154a0010154ff0065007200000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff003400020000005d000000ff000000c68fcd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 60
go
--  'ISO_1BIN.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 50, 1, 0, N'bin_iso_1', N'Binary Sort Order for the ISO 8859-1 (Latin 1) Character Set', 0x0000)
go
--  'DICTION.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 51, 1, 0, N'dictionary_iso', N'General purpose dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the ISO 8859-1 character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition =
		0x3200fe030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a8400013185000132860001338700013488000135890001368a0001378b0001388c0001398d0001413000013b3100013c3200013d3300013e3400013f350001403600015b8e0001428f0001439000014491000145920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000157a4000158a5000159a600015aa70001d03700015c3800015d3900015e3a00015f3b0001603c00017b8e0201428f010143
	+	0x9001014491010145920201469301014794010148950101499601014a9701014b9801014c9901014d9a01014e9b01014f9c0101509d0101519e0101529f010153a0010154a1010155a2010156a3010157a4010158a5010159a601015aa70101d03d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018a4c00018b4d00018c4e00018d4f00018e5000018f5100019052000191530001925400019355000194560001955700019658000197590001985a0001995b00019a5c00019b5d00019c5e00019d5f00019e6000019f610001a0620001a1630001a2640001a3650001a4660001a5670001a6680001a7690001a86a0001a96b0001aa6c0001ab6d0001ac6e0001ad6f0001ae700001af710001b0720001b1730001b2740001b3750001b4760001b5770001b6780001b7790001b87a0001b97b0001ba7c0001bb7d0001bc7e0001bd7f0001be800001bf810001d78e0401428e0601428e0801428e0a01428e0c01428e0e014200000000
	+	0x90020144920401469206014692080146920a01469602014a9604014a9606014a9608014aa80001de9b02014f9c0201509c0401509c0601509c0801509c0a0150820001f79c0c0150a2020156a2040156a2060156a2080156a602015aa9000100000000028e0501428e0701428e0901428e0b01428e0d01428e0f01420000000190030144920501469207014692090146920b01469603014a9605014a9607014a9609014aa80101de9b03014f9c0301509c0501509c0701509c0901509c0b0150830001309c0d0150a2030156a2050156a2070156a2090156a603015aa9010100a604015aff0300008e01014292010146ff0300008e03014292030146ff030000a0020154a0020154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff0034000200000088000000ff0000004a90cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 51
go
--  'NOACCENT.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 54, 1, 0, N'noaccents_iso', N'Case-insensitive dictionary sort order for use with several Western-European languages including English, French, and German.  Uses the ISO 8859-1 character set.  Diacritical marks are ignored.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00fe030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a8400013185000132860001338700013488000135890001368a0001378b0001388c0001398d0001413000013b3100013c3200013d3300013e3400013f350001403600015b8e0001428f0001439000014491000145920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000157a4000158a5000159a600015aa70001d03700015c3800015d3900015e3a00015f3b0001603c00017b8e0002428f000243
	+	0x9000024491000245920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b00024f9c0002509d0002519e0002529f000253a0000254a1000255a2000256a3000257a4000258a5000259a600025aa70002d03d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018a4c00018b4d00018c4e00018d4f00018e5000018f5100019052000191530001925400019355000194560001955700019658000197590001985a0001995b00019a5c00019b5d00019c5e00019d5f00019e6000019f610001a0620001a1630001a2640001a3650001a4660001a5670001a6680001a7690001a86a0001a96b0001aa6c0001ab6d0001ac6e0001ad6f0001ae700001af710001b0720001b1730001b2740001b3750001b4760001b5770001b6780001b7790001b87a0001b97b0001ba7c0001bb7d0001bc7e0001bd7f0001be800001bf810001d78e0003428e0005428e0007428e0009428e000b428e000d4200000000
	+	0x90010144920003469200054692000746920009469600034a9600054a9600074a9600094aa80001de9b00034f9c0003509c0005509c0007509c0009509c000b50820001f79c000d50a2000356a2000556a2000756a2000956a600035aa9000100000000028e0004428e0006428e0008428e000a428e000c428e000e42000000019001024492000446920006469200084692000a469600044a9600064a9600084a96000a4aa80002de9b00044f9c0004509c0006509c0008509c000a509c000c50830001309c000e50a2000456a2000656a2000856a2000a56a600045aa9000200a600055aff0300008e01014292010146ff0300008e01024292010246ff030000a0010154a0010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff0034000200000095000000ff0000006890cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 54
go
--  'NOCASEPR.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 53, 1, 0, N'nocasepref_iso', N'Case-insensitive dictionary sort order, with case preference for collating purposes, for use with several Western-European languages including English, French, and German.  Uses the ISO 8859-1 character set.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00fe030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a8400013185000132860001338700013488000135890001368a0001378b0001388c0001398d0001413000013b3100013c3200013d3300013e3400013f350001403600015b8e0001428f0001439000014491000145920001469300014794000148950001499600014a9700014b9800014c9900014d9a00014e9b00014f9c0001509d0001519e0001529f000153a0000154a1000155a2000156a3000157a4000158a5000159a600015aa70001d03700015c3800015d3900015e3a00015f3b0001603c00017b8e0002428f000243
	+	0x9000024491000245920002469300024794000248950002499600024a9700024b9800024c9900024d9a00024e9b00024f9c0002509d0002519e0002529f000253a0000254a1000255a2000256a3000257a4000258a5000259a600025aa70002d03d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018a4c00018b4d00018c4e00018d4f00018e5000018f5100019052000191530001925400019355000194560001955700019658000197590001985a0001995b00019a5c00019b5d00019c5e00019d5f00019e6000019f610001a0620001a1630001a2640001a3650001a4660001a5670001a6680001a7690001a86a0001a96b0001aa6c0001ab6d0001ac6e0001ad6f0001ae700001af710001b0720001b1730001b2740001b3750001b4760001b5770001b6780001b7790001b87a0001b97b0001ba7c0001bb7d0001bc7e0001bd7f0001be800001bf810001d78e0201428e0301428e0401428e0501428e0601428e07014200000000
	+	0x90010144920201469203014692040146920501469601014a9602014a9603014a9604014aa80001de9b01014f9c0101509c0201509c0301509c0401509c050150820001f79c060150a2010156a2020156a2030156a2040156a601015aa9000100000000028e0202428e0302428e0402428e0502428e0602428e0702420000000190010244920202469203024692040246920502469601024a9602024a9603024a9604024aa80002de9b01024f9c0102509c0202509c0302509c0402509c050250830001309c060250a2010256a2020256a2030256a2040256a601025aa9000200a602015aff0300008e01014292010146ff0300008e01024292010246ff030000a0010154a0010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000d2000000ff000000a490cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 53
go
--  'DANONOCP.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 183, 1, 0, N'danno_nocasepref', N'Case-insensitive Danish/Norwegian dictionary sort order, with case preference for collating purposes. Uses the ISO 8859-1 character set with Windows 3.1 additional characters.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00e5050000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a7f0001318000013281000133820001348300013584000136850001378600013887000139880001413000013b3100013c3200013d3300013e3400013f350001403600015b890001428a0001438b0001448c0001458d0001468e0001478f000148900001499100014a9200014b9300014c9400014d9500014e9600014f9700015098000151990001529a0001539b0001549c0001559d0001569e0001579f000158a0000159a100015aa20001c63700015c3800015d3900015e3a00015f3b0001603c00017b890002428a000243
	+	0x8b0002448c0002458d0002468e0002478f000248900002499100024a9200024b9300024c9400024d9500024e9600024f9700025098000251990002529a0002539b0002549c0002559d0002569e0002579f000258a0000259a100025aa20002c63d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018b9b0301544c00018d000000004d00018e4e00018f4f0001905000019151000192520001935300019454000195550001965600019757000198580001995900019b9b0302545a00019d000000015b00019e5c0001a0a103015a5d0001a15e0001a25f0001a3600001a4610001a5620001a6630001a7640001a8650001a9660001aa670001ab680001ac690001ad6a0001ae6b0001af6c0001b06d0001b16e0001b26f0001b3700001b4710001b5720001b6730001b7740001b8750001b9760001ba770001bb780001bc790001bd7a0001be7b0001bf7c0001d789020142890101428903014289040142a30101d8a5000100a30001d8
	+	0x8b0101448d0301468d0201468d0401468d0501469102014a9101014a9103014a9104014a8c0101459601014f97030150970201509704015097050150a40101c57d0001f7a40001c59d0201569d0101569d030156a101015aa102015a000000030000000289020242890102428903024289040242a30102d8a5000200a30002d88b0102448d0302468d0202468d0402468d0502469102024a9101024a9103024a9104024a8c0102459601024f97030250970202509704025097050250a40102c57e000130a40002c59d0202569d0102569d030256a101025aa102025a00000004a103025aff030000970101508d010146ff030000970102508d010246ff0300009b0101549b010154ff0300009c01015590010149ff0300009c01025590010249ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000b0000000ff000000d790cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 183
go
--  'ISLANOCP.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 186, 1, 0, N'islan_nocasepref', N'Case-insensitive Icelandic dictionary sort order, with case preference for collating purposes. Uses the ISO 8859-1 character set with Windows 3.1 additional characters.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00f8030000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a7f0001318000013281000133820001348300013584000136850001378600013887000139880001413000013b3100013c3200013d3300013e3400013f350001403600015b890001c18b0001438c0001448d0001d08f0001c9910001479200014893000149940001cd9600014b9700014c9800014d9900014e9a00014f9b0001d39d0001519e0001529f000153a0000154a1000155a20001daa4000157a5000158a6000159a70001dda90001de3700015c3800015d3900015e3a00015f3b0001603c00017b890002c18b000243
	+	0x8c0002448d0002d08f0002c9910002479200024893000249940002cd9600024b9700024c9800024d9900024e9a00024f9b0002d39d0002519e0002529f000253a0000254a1000255a20002daa4000257a5000258a6000259a70002dda90002de3d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018ba00301544c00018d000000004d00018e4e00018f4f0001905000019151000192520001935300019454000195550001965600019757000198580001995900019ba00302545a00019d000000015b00019e5c0001a0a70101dd5d0001a15e0001a25f0001a3600001a4610001a5620001a6630001a7640001a8650001a9660001aa670001ab680001ac690001ad6a0001ae6b0001af6c0001b06d0001b16e0001b26f0001b3700001b4710001b5720001b6730001b7740001b8750001b9760001ba770001bb780001bc790001bd7a0001be7b0001bf7c0001d7890101c18a000142890201c1890401c1890301c1890501c1ab0001d6
	+	0x8c0101448f0201c9900001468f0301c98f0401c9940101cd9500014a940201cd940301cd8e0001459a01014f9b0201d39c0001509b0301d39b0401d3ac0001007d0001f7ac010100a20101daa3000156a20201daa20301daa800015aaa0001c600000002890102c18a000242890202c1890402c1890302c1890502c1ab0002d68c0102448f0202c9900002468f0302c98f0402c9940102cd9500024a940202cd940302cd8e0002459a01024f9b0202d39c0002509b0302d39b0402d3ac0002007e000130ac010200a20102daa3000256a20202daa20302daa800025aaa0002c6a70102ddff0300009b0101d38f0101c9ff0300009b0102d38f0102c9ff030000a0010154a0010154ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000d4000000ff000000f590cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 186
go
--  'SVF1NOCP.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 184, 1, 0, N'svfi1_nocasepref', N'Case-insensitive Swedish/Finnish dictionary sort order, with case preference for collating purposes. Uses the ISO 8859-1 character set with Windows 3.1 additional characters. The letters V and W are treated as variants of the same letter.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00f8070000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a7f0001318000013281000133820001348300013584000136850001378600013887000139880001413000013b3100013c3200013d3300013e3400013f350001403600015b890001428a0001438b0001448c0001458d0001468e0001478f000148900001499100014a9200014b9300014c9400014d9500014e9600014f9700015098000151990001529a0001539b0001549c0001559d0001569e0001589e0101589f000159a000015aa10001c53700015c3800015d3900015e3a00015f3b0001603c00017b890002428a000243
	+	0x8b0002448c0002458d0002468e0002478f000248900002499100024a9200024b9300024c9400024d9500024e9600024f9700025098000251990002529a0002539b0002549c0002559d0002569e0002589e0102589f000259a000025aa10002c53d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018b9b0301544c00018d000000024d00018e4e00018f4f0001905000019151000192520001935300019454000195550001965600019757000198580001995900019b9b0302545a00019d000000035b00019e5c0001a0a003015a5d0001a15e0001a25f0001a3600001a4610001a5620001a6630001a7640001a8650001a9660001aa670001ab680001ac690001ad6a0001ae6b0001af6c0001b06d0001b16e0001b26f0001b3700001b4710001b5720001b6730001b7740001b8750001b9760001ba770001bb780001bc790001bd7a0001be7b0001bf7c0001d789030142890201428904014289050142a30001d6a20001c400000000
	+	0x8b0101448d0401468d0301468d0501468d0601469102014a9101014a9103014a9104014a8c0101459601014f97030150970201509704015097050150a40001007d0001f7a40101009d0201569d0101569d030156a001015aa002015a000000050000000489030242890202428904024289050242a30002d6a20002c4000000018b0102448d0402468d0302468d0502468d0602469102024a9101024a9103024a9104024a8c0102459601024f97030250970202509704025097050250a40002007e000130a40102009d0202569d0102569d030256a001025aa002025a00000006a003025aff030000890101428d010146ff030000890102428d010246ff030000970101508d020146ff030000970102508d020246ff0300009b0101549b010154ff0300009c01015590010149ff0300009c01025590010249ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff003400020000009e000000ff0000001391cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 184
go
--  'SVF2NOCP.ISO'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 185, 1, 0, N'svfi2_nocasepref', N'Case-insensitive Swedish/Finnish dictionary sort order, with case preference for collating purposes. Uses the ISO 8859-1 character set with Windows 3.1 additional characters. The letters V and W are treated as separate letters.', 0x0000)
update syscharsets set binarydefinition =
		0x3c00f8070000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f0001202000012121000122220001232300012424000125250001262600012727000128280001292900012a2a00012b2b00012c2c00012d2d00012e2e00012f2f00013a7f0001318000013281000133820001348300013584000136850001378600013887000139880001413000013b3100013c3200013d3300013e3400013f350001403600015b890001428a0001438b0001448c0001458d0001468e0001478f000148900001499100014a9200014b9300014c9400014d9500014e9600014f9700015098000151990001529a0001539b0001549c0001559d0001569e0001579f000158a0000159a100015aa20001c53700015c3800015d3900015e3a00015f3b0001603c00017b890002428a000243
	+	0x8b0002448c0002458d0002468e0002478f000248900002499100024a9200024b9300024c9400024d9500024e9600024f9700025098000251990002529a0002539b0002549c0002559d0002569e0002579f000258a0000259a100025aa20002c53d00017c3e00017d3f00017e4000017f4100018042000181430001824400018345000184460001854700018648000187490001884a0001894b00018b9b0301544c00018d000000024d00018e4e00018f4f0001905000019151000192520001935300019454000195550001965600019757000198580001995900019b9b0302545a00019d000000035b00019e5c0001a0a103015a5d0001a15e0001a25f0001a3600001a4610001a5620001a6630001a7640001a8650001a9660001aa670001ab680001ac690001ad6a0001ae6b0001af6c0001b06d0001b16e0001b26f0001b3700001b4710001b5720001b6730001b7740001b8750001b9760001ba770001bb780001bc790001bd7a0001be7b0001bf7c0001d789030142890201428904014289050142a40001d6a30001c400000000
	+	0x8b0101448d0401468d0301468d0501468d0601469102014a9101014a9103014a9104014a8c0101459601014f97030150970201509704015097050150a50001007d0001f7a50101009d0201569d0101569d030156a101015aa102015a000000050000000489030242890202428904024289050242a40002d6a30002c4000000018b0102448d0402468d0302468d0502468d0602469102024a9101024a9103024a9104024a8c0102459601024f97030250970202509704025097050250a50002007e000130a50102009d0202569d0102569d030256a101025aa102025a00000006a103025aff030000890101428d010146ff030000890102428d010246ff030000970101508d020146ff030000970102508d020246ff0300009b0101549b010154ff0300009c01015590010149ff0300009c01025590010249ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000
	+	0x0000000000000000ff0000000000000060e3f777ff1c14009c2deb7718000000ff0014000000000000000000ff4af6774805240050060000ffe3f777d819140000000000ff0200000000140000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000b805240000002400ff0000004000000000000000ff000000000000008cf31200ff4bf67700002400b8052400ff4cf6774805240000000000fffa1200c0052400b4f31200fff412004901000070f41200ff91f6770000240000000000ff0524001a8cf677c0052400ff87f87700f0fd7f00000000ff00340002000000b6000000ff0000003191cd0000000000ff78d1fb0100000000000000ff69f47702000000e0052400ff060000b069f477a05afa77ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff000000000000004ec1f677ff55fa774fc1f67700000000ffffffff9c2deb7700001400ff1c140028001400a81c1400ffbef87728000000
	+	0x00000000ff0414001800000067000000ff030000d4f2120028001400fff412009c2deb7718000000ff001400f0f41200f0f41200ff9df877579ef87748051400ff2deb770000000000001400ff0200000000140018f51200fff51200e79df877579ef877ff0514000000000000000000ff001400bbf81200ff000000ff00000004f51200a81c1400ff0014008cf4120006133000fff6120094fef87740b7f977ffffffffa8f612000ed7f777ff0014000101071018000000ff2deb770000000000001400
where id = 185
go
--  'BINARY.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 80, 4, 0, N'bin_cp1250', N'Binary Sort Order for the Code Page 1250 Character Set', 0x0000)
go
--  'DICTION.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 81, 4, 0, N'dictionary_1250', N'Dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008E0100000101 
	+ 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101627B0101637C0101647D0101657E0101667F01016780010168810101698201016A8301016B8401016C8501016D8601016E8701016F88010170890101718A0101728B0101738C0301748D0101998F0101769001017791010178920101799301017A940101003D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001647D0001657E0001667F00016780000168810001698200016A8300016B8400016C8500016D8600016E8700016F88000170890001718A0001728B0001738C0001748D0001998F0001769000017791000178920001799300017A940001004300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001808C0701745600019B8C0501748D03019994070100940301001F000198500001925100018253000194540001846C0001852400019725000120200001278E0001758C0601745700012B8C0401748D0201999406010094020100270001094B0001A24C0001FF8507016D610001A77A0B0162470001A8620001A9480001B4630001AC8C0901745D0001BB640001AE23000196650001B094050100660001B55C0001AB4E0001BD8506016D490001B8670001B6680001B7690001864A0001A17A0A01628C0801745E0001D78505016D4F0001918504016D940401008B030173 
	+ 0x7A0301627A0501627A0901627A0701628503016D7C0301647C0701647C0501647E0301667E0901667E0501667E0701668203016A8205016A7D0301657D0501658703016F8705016F880301708805017088090170880701705F0001F78B0501738F0701768F0301768F0901768F0501769303017A8D050199000000008B0201737A0201627A0401627A0801627A0601628502016D7C0201647C0601647C0401647E0201667E0801667E0401667E0601668202016A8204016A7D0201657D0401658702016F8704016F88020170880401708808017088060170600001A48B0401738F0601768F0201768F0801768F0401769302017A8D0401994D0001B2FF030000 
	+ 0x8C0101748C010174FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C5000000FF000000EA00B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 81
go
--  'NOCASE.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 82, 4, 0, N'nocase_1250', N'Dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008E0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002627B0002637C0002647D0002657E0002667F00026780000268810002698200026A8300026B8400026C8500026D8600026E8700026F88000270890002718A0002728B0002738C0002748D0002998F0002769000027791000278920002799300027A940002003D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001647D0001657E0001667F00016780000168810001698200016A8300016B8400016C8500016D8600016E8700016F88000170890001718A0001728B0001738C0001748D0001998F0001769000017791000178920001799300017A940001004300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001808C0402745600019B8C0302748D01029994030200940102001F000198500001925100018253000194540001846C0001852400019725000120200001278E0001758C0401745700012B8C0301748D0101999403010094010100270001094B0001A24C0001FF8503026D610001A77A050262470001A8620001A9480001B4630001AC8C0502745D0001BB640001AE23000196650001B094020200660001B55C0001AB4E0001BD8503016D490001B8670001B6680001B7690001864A0001A17A0501628C0501745E0001D78502026D4F0001918502016D940201008B010273 
	+ 0x7A0102627A0202627A0402627A0302628501026D7C0102647C0302647C0202647E0102667E0402667E0202667E0302668201026A8202026A7D0102657D0202658701026F8702026F880102708802027088040270880302705F0001F78B0202738F0302768F0102768F0402768F0202769301027A8D020299000000008B0101737A0101627A0201627A0401627A0301628501016D7C0101647C0301647C0201647E0101667E0401667E0201667E0301668201016A8202016A7D0101657D0201658701016F8702016F88010170880201708804017088030170600001A48B0201738F0301768F0101768F0401768F0201769301017A8D0201994D0001B2FF030000 
	+ 0x8C0101748C010174FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000CA000000FF000000FB02B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 82
go
--  'CSYDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 83, 4, 0, N'csydic', N'Czech dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008E0600000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101627B010163000000027E0101657F0101668001016781010168820101698401016A8501016B8601016C8701016D8801016E8901016F8A0101708B0101718C0101728D0101F88F03019A91010199930101769401017795010178960101799701017A9801019E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B000163000000007E0001657F0001668000016781000168820001698400016A8500016B8600016C8700016D8800016E8900016F8A0001708B0001718C0001728D0001F88F00019A91000199930001769400017795000178960001799700017A9800019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180900101745600019B8F05019A91030199990101009805019E1F000198500001925100018253000194540001846C00018524000197250001202000012792000175900001745700012B8F04019A91020199990001009804019E270001094B0001A24C0001FF8707016D610001A77A0B0162470001A8620001A9480001B4630001AC8F07019A5D0001BB640001AE23000196650001B09803019E660001B55C0001AB4E0001BD8706016D490001B8670001B6680001B7690001864A0001A17A0A01628F06019A5E0001D78705016D4F0001918704016D9802019E8D0301F8 
	+ 0x7A0501627A0701627A0901627A0301628703016D7C0301E87C0501E87D0101647F0501667F0901667F0301667F0701668403016A8405016A7E0301657E0501658903016F8905016F8A0501708A0701708A0901708A0301705F0001F78E010173930701769305017693090176930301769703017A91050199000000038D0201F87A0401627A0601627A0801627A0201628702016D7C0201E87C0401E87D0001647F0401667F0801667F0201667F0601668402016A8404016A7E0201657E0401658902016F8904016F8A0401708A0601708A0801708A020170600001A48E000173930601769304017693080176930201769702017A910401994D0001B205010100 
	+ 0x6800200083000169040101006800000083010169010101004800000083020169FF0300008F01019A8F01019AFF0000007C0101E800000000FF0000007C0001E800000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000CB000000FF0000004DFFB10200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 83
go
--  'CSYNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 84, 4, 0, N'csync', N'Czech dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008E0600000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002627B000263000000027E0002657F0002668000026781000268820002698400026A8500026B8600026C8700026D8800026E8900026F8A0002708B0002718C0002728D0002F88F00029A91000299930002769400027795000278960002799700027A9800029E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B000163000000007E0001657F0001668000016781000168820001698400016A8500016B8600016C8700016D8800016E8900016F8A0001708B0001718C0001728D0001F88F00019A91000199930001769400017795000178960001799700017A9800019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180900002745600019B8F03029A91010299990002009802029E1F000198500001925100018253000194540001846C00018524000197250001202000012792000175900001745700012B8F03019A91010199990001009802019E270001094B0001A24C0001FF8703026D610001A77A050262470001A8620001A9480001B4630001AC8F04029A5D0001BB640001AE23000196650001B09801029E660001B55C0001AB4E0001BD8703016D490001B8670001B6680001B7690001864A0001A17A0501628F04019A5E0001D78702026D4F0001918702016D9801019E8D0102F8 
	+ 0x7A0202627A0302627A0402627A0102628701026D7C0102E87C0202E87D0002647F0202667F0402667F0102667F0302668401026A8402026A7E0102657E0202658901026F8902026F8A0202708A0302708A0402708A0102705F0001F78E000273930302769302027693040276930102769701027A91020299000000038D0101F87A0201627A0301627A0401627A0101628701016D7C0101E87C0201E87D0001647F0201667F0401667F0101667F0301668401016A8402016A7E0101657E0201658901016F8902016F8A0201708A0301708A0401708A010170600001A48E000173930301769302017693040176930101769701017A910201994D0001B205010100 
	+ 0x6800200083000169040101006800000083000269010101004800000083000369FF0300008F01019A8F01019AFF0000007C0002E800000000FF0000007C0001E800000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF0034000200000086000000FF0000004200B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 84
go
--  'HUNDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 85, 4, 0, N'hundic', N'Hungarian dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32005A2900000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101627B0101630000000200000005800101668101016700000008840101698501016A8601016B8701016C0000000B8A01016E0000000E8D0101F68F01017190010172910101730000001100000014970101FC990101779A0101789B0101799C01017A000000173D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001630000000000000003800001668100016700000006840001698500016A8600016B8700016C000000098A00016E0000000C8D0001F68F00017190000172910001730000000F00000012970001FC990001779A0001789B0001799C00017A000000154300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180920701745600019B92050174940301999D0701009D0301001F000198500001925100018253000194540001846C00018524000197250001202000012795000175920601745700012B92040174940201999D0601009D020100270001094B0001A24C0001FF8807016D610001A77A0B0162470001A8620001A9480001B4630001AC920901745D0001BB640001AE23000196650001B09D050100660001B55C0001AB4E0001BD8806016D490001B8670001B6680001B7690001864A0001A17A0A0162920801745E0001D78805016D4F0001918804016D9D04010091030173 
	+ 0x7A0301627A0501627A0901627A0701628803016D7C0301647C0701647C050164800301668009016680050166800701668503016A8505016A7E0301657E0501658B03016F8B05016F8D0301F68D0501F68E0301708E0101705F0001F791050173970501FC970301FC98030176980101769C03017A9405019900000018910201737A0201627A0401627A0801627A0601628802016D7C0201647C0601647C040164800201668008016680040166800601668502016A8504016A7E0201657E0401658B02016F8B04016F8D0201F68D0401F68E0201708E000170600001A491040173970401FC970201FC98020176980001769C02017A940401994D0001B221010100 
	+ 0x730020007D00016419010100730000007D01016401010100530000007D020164220101007A0000007F0001651A0101007A0000007F010165040101005A0000007F0201652301010079000000830001681B010100790000008301016807010100590000008302016824010100790000008900016D1C010100790000008901016D0A010100590000008902016D25010100790000008C00016F1D010100790000008C01016F0D010100590000008C02016F260101007A000000930001741E0101007A2DEB7793010174100101005A000000930201742701017779052400960001751F01017779191400960101751301010059001400960201752801010073000000 
	+ 0x9E00010020010100730000009E01010016010100530000009E020100FF0300009201017492010174FF0000007C01016400002400FF0000007E01016500000000FF000000820101688CF31200FF0000778801016DB8052400FF0000778B01016F00000000FF00000092030174B4F31200FF0000009401019970F41200FF0000779D01010000000000FF0000007C000164C0052400FF0000777E00016500000000FF00000082000168D7000000FF0000008800016D00000000FF0000FB8B00016F00000000FF00007792000174E0052400FF00000094000199A05AFA77FF0000009D00010000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 85
go
--  'HUNNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 86, 4, 0, N'hunnc', N'Hungarian dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C005A2900000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002627B0002630000000200000005800002668100026700000008840002698500026A8600026B8700026C0000000B8A00026E0000000E8D0002F68F00027190000272910002730000001100000014970002FC990002779A0002789B0002799C00027A000000173D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001630000000000000003800001668100016700000006840001698500016A8600016B8700016C000000098A00016E0000000C8D0001F68F00017190000172910001730000000F00000012970001FC990001779A0001789B0001799C00017A000000154300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180920402745600019B92030274940102999D0302009D0102001F000198500001925100018253000194540001846C00018524000197250001202000012795000175920401745700012B92030174940101999D0301009D010100270001094B0001A24C0001FF8803026D610001A77A050262470001A8620001A9480001B4630001AC920502745D0001BB640001AE23000196650001B09D020200660001B55C0001AB4E0001BD8803016D490001B8670001B6680001B7690001864A0001A17A050162920501745E0001D78802026D4F0001918802016D9D02010091010273 
	+ 0x7A0102627A0202627A0402627A0302628801026D7C0102647C0302647C020264800102668004026680020266800302668501026A8502026A7E0102657E0202658B01026F8B02026F8D0102F68D0202F68E0102708E0002705F0001F791020273970202FC970102FC98010276980002769C01027A9402029900000018910101737A0101627A0201627A0401627A0301628801016D7C0101647C0301647C020164800101668004016680020166800301668501016A8502016A7E0101657E0201658B01016F8B02016F8D0101F68D0201F68E0101708E000170600001A491020173970201FC970101FC98010176980001769C01017A940201994D0001B221010100 
	+ 0x730020007D00016419010100730000007D00026401010100530000007D000364220101007A0000007F0001651A0101007A0000007F000265040101005A0000007F0003652301010079000000830001681B010100790000008300026807010100590000008300036824010100790000008900016D1C010100790000008900026D0A010100590000008900036D25010100790000008C00016F1D010100790000008C00026F0D010100590000008C00036F260101007A000000930001741E0101007A2DEB7793000274100101005A000000930003742701017779052400960001751F01017779191400960002751301010059001400960003752801010073000000 
	+ 0x9E00010020010100730000009E00020016010100530000009E000300FF0300009201017492010174FF0000007C00026400002400FF0000007E00026500000000FF000000820002688CF31200FF0000778800026DB8052400FF0000778B00026F00000000FF00000092000274B4F31200FF0000009400029970F41200FF0000779D00020000000000FF0000007C000164C0052400FF0000777E00016500000000FF000000820001689F000000FF0000008800016D00000000FF0000FB8B00016F00000000FF00007792000174E0052400FF00000094000199A05AFA77FF0000009D00010000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 86
go
--  'PLKDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 87, 4, 0, N'plkdic', N'Polish dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3200AF0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101B97C0101637D0101E67F010165800101EA8201016783010168840101698501016A8601016B8701016C880101B38A01016E8B0101F18D0101F38F01017190010172910101739203019C94010199960101769701017798010178990101799A01017A9B01019F3D00015C3E00015D3F00015E4000015F410001604200017B7A0001B97C0001637D0001E67F000165800001EA8200016783000168840001698500016A8600016B8700016C880001B38A00016E8B0001F18D0001F38F00017190000172910001739200019C94000199960001769700017798000178990001799A00017A9B00019F4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001809205019C5600019B93010174940301999B03019F9C0101BF1F000198500001925100018253000194540001846C000185240001972500012020000127950001759204019C5700012B93000174940201999B02019F9C0001BF270001094B0001A24C0001FF8901016D610001A77B010162470001A8620001A9480001B4630001AC9207019C5D0001BB640001AE23000196650001B09D010100660001B55C0001AB4E0001BD8900016D490001B8670001B6680001B7690001864A0001A17B0001629206019C5E0001D7880501B34F000191880401B39D00010091030173 
	+ 0x7A0301B97A0501B97A0901B97A0701B9880301B37E0101647D0501E67D0301E6800301EA81010166800501EA800701EA8503016A8505016A7F0301657F0501658C01016F8B0301F18E0101708D0301F38D0701F38D0501F35F0001F791050173960701769603017696090176960501769A03017A9405019900000000910201737A0201B97A0401B97A0801B97A0601B9880201B37E0001647D0401E67D0201E6800201EA81000166800401EA800601EA8502016A8504016A7F0201657F0401658C00016F8B0201F18E0001708D0201F38D0601F38D0401F3600001A491040173960601769602017696080176960401769A02017A940401994D0001B2FF030000 
	+ 0x9201019C9201019CFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF0034000200000098000000FF000000BB03B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 87
go
--  'PLKNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 88, 4, 0, N'plknc', N'Polish dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C00AF0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002B97C0002637D0002E67F000265800002EA8200026783000268840002698500026A8600026B8700026C880002B38A00026E8B0002F18D0002F38F00027190000272910002739200029C94000299960002769700027798000278990002799A00027A9B00029F3D00015C3E00015D3F00015E4000015F410001604200017B7A0001B97C0001637D0001E67F000165800001EA8200016783000168840001698500016A8600016B8700016C880001B38A00016E8B0001F18D0001F38F00017190000172910001739200019C94000199960001769700017798000178990001799A00017A9B00019F4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001809203029C5600019B93000274940102999B01029F9C0002BF1F000198500001925100018253000194540001846C000185240001972500012020000127950001759203019C5700012B93000174940101999B01019F9C0001BF270001094B0001A24C0001FF8900026D610001A77B000262470001A8620001A9480001B4630001AC9204029C5D0001BB640001AE23000196650001B09D000200660001B55C0001AB4E0001BD8900016D490001B8670001B6680001B7690001864A0001A17B0001629204019C5E0001D7880202B34F000191880201B39D00010091010273 
	+ 0x7A0102B97A0202B97A0402B97A0302B9880102B37E0002647D0202E67D0102E6800102EA81000266800202EA800302EA8501026A8502026A7F0102657F0202658C00026F8B0102F18E0002708D0102F38D0302F38D0202F35F0001F791020273960302769601027696040276960202769A01027A9402029900000000910101737A0101B97A0201B97A0401B97A0301B9880101B37E0001647D0201E67D0101E6800101EA81000166800201EA800301EA8501016A8502016A7F0101657F0201658C00016F8B0101F18E0001708D0101F38D0301F38D0201F3600001A491020173960301769601017696040176960201769A01017A940201994D0001B2FF030000 
	+ 0x9201019C9201019CFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000AC000000FF0000006304B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 88
go
--  'ROMDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 89, 4, 0, N'romdic', N'Romanian dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008E0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101E37C0101637D0101647E0101657F010166800101678101016882010169830101EE8501016B8601016C8701016D8801016E8901016F8A0101708B0101718C0101728D0101738E0301BA90010199930101769401017795010178960101799701017A980101003D00015C3E00015D3F00015E4000015F410001604200017B7A0001E37C0001637D0001647E0001657F000166800001678100016882000169830001EE8500016B8600016C8700016D8800016E8900016F8A0001708B0001718C0001728D0001738E0001BA90000199930001769400017795000178960001799700017A980001004300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001808E0701BA5600019B8E0501BA9003019998070100980301001F000198500001925100018253000194540001846C000185240001972500012020000127910001FE8E0601BA5700012B8E0401BA900201999806010098020100270001094B0001A24C0001FF8707016D610001A77A0901E3470001A8620001A9480001B4630001AC8F0101745D0001BB640001AE23000196650001B098050100660001B55C0001AB4E0001BD8706016D490001B8670001B6680001B7690001864A0001A17A0801E38F0001745E0001D78705016D4F0001918704016D980401008D030173 
	+ 0x7A0301E37A0501E37B0101627A0701E38703016D7D0301647D0701647D0501647F0301667F0901667F0501667F070166830301EE8401016A7E0301657E0501658903016F8905016F8A0301708A0501708A0901708A0701705F0001F78D050173930701769303017693090176930501769703017A92010175000000008D0201737A0201E37A0401E37B0001627A0601E38702016D7D0201647D0601647D0401647F0201667F0801667F0401667F060166830201EE8400016A7E0201657E0401658902016F8904016F8A0201708A0401708A0801708A060170600001A48D040173930601769302017693080176930401769702017A920001754D0001B2FF030000 
	+ 0x8E0101BA8E0101BAFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000BC000000FF0000002305B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 89
go
--  'ROMNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 90, 4, 0, N'romnc', N'Romanian dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008E0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002E37C0002637D0002647E0002657F000266800002678100026882000269830002EE8500026B8600026C8700026D8800026E8900026F8A0002708B0002718C0002728D0002738E0002BA90000299930002769400027795000278960002799700027A980002003D00015C3E00015D3F00015E4000015F410001604200017B7A0001E37C0001637D0001647E0001657F000166800001678100016882000169830001EE8500016B8600016C8700016D8800016E8900016F8A0001708B0001718C0001728D0001738E0001BA90000199930001769400017795000178960001799700017A980001004300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001808E0402BA5600019B8E0302BA9001029998030200980102001F000198500001925100018253000194540001846C000185240001972500012020000127910001FE8E0401BA5700012B8E0301BA900101999803010098010100270001094B0001A24C0001FF8703026D610001A77A0402E3470001A8620001A9480001B4630001AC8F0002745D0001BB640001AE23000196650001B098020200660001B55C0001AB4E0001BD8703016D490001B8670001B6680001B7690001864A0001A17A0401E38F0001745E0001D78702026D4F0001918702016D980201008D010273 
	+ 0x7A0102E37A0202E37B0002627A0302E38701026D7D0102647D0302647D0202647F0102667F0402667F0202667F030266830102EE8400026A7E0102657E0202658901026F8902026F8A0102708A0202708A0402708A0302705F0001F78D020273930302769301027693040276930202769701027A92000275000000008D0101737A0101E37A0201E37B0001627A0301E38701016D7D0101647D0301647D0201647F0101667F0401667F0201667F030166830101EE8400016A7E0101657E0201658901016F8902016F8A0101708A0201708A0401708A030170600001A48D020173930301769301017693040176930201769701017A920001754D0001B2FF030000 
	+ 0x8E0101BA8E0101BAFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000A8000000FF000000CB05B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 90
go
--  'SHLDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 91, 4, 0, N'shldic', N'Croatian dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008E1000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101627B0101637C0101E800000002820101668301016784010168850101698601016A8701016B8801016C000000058B01016E000000088E0101708F01017190010172910101739203019A94010199960101769701017798010178990101799A01017A9B01019E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001E800000000820001668300016784000168850001698600016A8700016B8800016C000000038B00016E000000068E0001708F00017190000172910001739200019A94000199960001769700017798000178990001799A00017A9B00019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180930101745600019B9205019A940301999C0101009B03019E1F000198500001925100018253000194540001846C00018524000197250001202000012795000175930001745700012B9204019A940201999C0001009B02019E270001094B0001A24C0001FF8907016D610001A77A0B0162470001A8620001A9480001B4630001AC9207019A5D0001BB640001AE23000196650001B09B05019E660001B55C0001AB4E0001BD8906016D490001B8670001B6680001B7690001864A0001A17A0A01629206019A5E0001D78903016D4F0001918902016D9B04019E91050173 
	+ 0x7A0501627A0701627A0301627A0901628905016D7E0101647C0301E87D0101E6820501668209016682070166820301668603016A8605016A7F0301F0800101658C05016F8C03016F8E0301708E0501708E0901708E0701705F0001F791030173960701769603017696090176960501769A03017A9405019900000009910401737A0401627A0601627A0201627A0801628904016D7E0001647C0201E87D0001E6820401668208016682060166820201668602016A8604016A7F0201F0800001658C04016F8C02016F8E0201708E0401708E0801708E060170600001A491020173960601769602017696080176960401769A02017A940401994D0001B20D010100 
	+ 0x9E002000810001650A0101009E00000081010165010101008E000000810201650E0101006A0000008A00016D0B0101006A0000008A01016D040101004A0000008A02016D0F0101006A0000008D00016F0C0101006A0000008D01016F070101004A0000008D02016FFF0300009201019A9201019AFF0000007F0101F000000000FF0000008901016D00000000FF0000008C01016F00000000FF0000007F0001F000000000FF0000008900016D00000000FF0000008C00016F60E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C1000000FF0000007406B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 91
go
--  'SHLNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 92, 4, 0, N'shlnc', N'Croatian dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008E1000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002627B0002637C0002E800000002820002668300026784000268850002698600026A8700026B8800026C000000058B00026E000000088E0002708F00027190000272910002739200029A94000299960002769700027798000278990002799A00027A9B00029E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001E800000000820001668300016784000168850001698600016A8700016B8800016C000000038B00016E000000068E0001708F00017190000172910001739200019A94000199960001769700017798000178990001799A00017A9B00019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180930002745600019B9203029A940102999C0002009B01029E1F000198500001925100018253000194540001846C00018524000197250001202000012795000175930001745700012B9203019A940101999C0001009B01019E270001094B0001A24C0001FF8903026D610001A77A050262470001A8620001A9480001B4630001AC9204029A5D0001BB640001AE23000196650001B09B02029E660001B55C0001AB4E0001BD8903016D490001B8670001B6680001B7690001864A0001A17A0501629204019A5E0001D78901026D4F0001918901016D9B02019E91020273 
	+ 0x7A0202627A0302627A0102627A0402628902026D7E0002647C0102E87D0002E6820202668204026682030266820102668601026A8602026A7F0102F0800002658C02026F8C01026F8E0102708E0202708E0402708E0302705F0001F791010273960302769601027696040276960202769A01027A9402029900000009910201737A0201627A0301627A0101627A0401628902016D7E0001647C0101E87D0001E6820201668204016682030166820101668601016A8602016A7F0101F0800001658C02016F8C01016F8E0101708E0201708E0401708E030170600001A491010173960301769601017696040176960201769A01017A940201994D0001B20D010100 
	+ 0x9E002000810001650A0101009E00000081000265010101008E000000810003650E0101006A0000008A00016D0B0101006A0000008A00026D040101004A0000008A00036D0F0101006A0000008D00016F0C0101006A0000008D00026F070101004A0000008D00036FFF0300009201019A9201019AFF0000007F0002F000000000FF0000008900026D00000000FF0000008C00026F00000000FF0000007F0001F000000000FF0000008900016D00000000FF0000008C00016F60E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF0034000200000068000000FF0000003507B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 92
go
--  'SKYDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 93, 4, 0, N'skydic', N'Slovak dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008E0600000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101E47C010163000000027F010165800101668101016782010168830101698501016A8601016B8701016C8801016D8901016E8A01016F8B0101F48D0101718E0101728F0101F89103019A93010199950101769601017797010178980101799901017A9A01019E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001E47C000163000000007F000165800001668100016782000168830001698500016A8600016B8700016C8800016D8900016E8A00016F8B0001F48D0001718E0001728F0001F89100019A93000199950001769600017797000178980001799900017A9A00019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180920101745600019B9105019A930301999B0101009A05019E1F000198500001925100018253000194540001846C00018524000197250001202000012794000175920001745700012B9104019A930201999B0001009A04019E270001094B0001A24C0001FF8807016D610001A77A0901E4470001A8620001A9480001B4630001AC9107019A5D0001BB640001AE23000196650001B09A03019E660001B55C0001AB4E0001BD8806016D490001B8670001B6680001B7690001864A0001A17A0801E49106019A5E0001D78805016D4F0001918804016D9A02019E8F0301F8 
	+ 0x7A0301E47A0501E47A0701E47B0101628803016D7D0301E87D0501E87E010164800501668009016680030166800701668503016A8505016A7F0301657F0501658A03016F8A05016F8B0501F48C0101708B0701F48B0301F45F0001F790010173950701769505017695090176950301769903017A93050199000000038F0201F87A0201E47A0401E47A0601E47B0001628802016D7D0201E87D0401E87E000164800401668008016680020166800601668502016A8504016A7F0201657F0401658A02016F8A04016F8B0401F48C0001708B0601F48B0201F4600001A490000173950601769504017695080176950201769902017A930401994D0001B205010100 
	+ 0x6800200084000169040101006800000084010169010101004800000084020169FF0300009101019A9101019AFF0000007D0101E800000000FF0000007D0001E800000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000BB000000FF000000DD07B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 93
go
--  'SKYNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 94, 4, 0, N'skync', N'Slovak dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008E0600000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002E47C000263000000027F000265800002668100026782000268830002698500026A8600026B8700026C8800026D8900026E8A00026F8B0002F48D0002718E0002728F0002F89100029A93000299950002769600027797000278980002799900027A9A00029E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001E47C000163000000007F000165800001668100016782000168830001698500016A8600016B8700016C8800016D8900016E8A00016F8B0001F48D0001718E0001728F0001F89100019A93000199950001769600017797000178980001799900017A9A00019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E000180920002745600019B9103029A930102999B0002009A02029E1F000198500001925100018253000194540001846C00018524000197250001202000012794000175920001745700012B9103019A930101999B0001009A02019E270001094B0001A24C0001FF8803026D610001A77A0402E4470001A8620001A9480001B4630001AC9104029A5D0001BB640001AE23000196650001B09A01029E660001B55C0001AB4E0001BD8803016D490001B8670001B6680001B7690001864A0001A17A0401E49104019A5E0001D78802026D4F0001918802016D9A01019E8F0102F8 
	+ 0x7A0102E47A0202E47A0302E47B0002628801026D7D0102E87D0202E87E000264800202668004026680010266800302668501026A8502026A7F0102657F0202658A01026F8A02026F8B0202F48C0002708B0302F48B0102F45F0001F790000273950302769502027695040276950102769901027A93020299000000038F0101F87A0101E47A0201E47A0301E47B0001628801016D7D0101E87D0201E87E000164800201668004016680010166800301668501016A8502016A7F0101657F0201658A01016F8A02016F8B0201F48C0001708B0301F48B0101F4600001A490000173950301769502017695040176950101769901017A930201994D0001B205010100 
	+ 0x6800200084000169040101006800000084000269010101004800000084000369FF0300009101019A9101019AFF0000007D0002E800000000FF0000007D0001E800000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000B9000000FF0000009E08B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 94
go
--  'SLVDIC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 95, 4, 0, N'slvdic', N'Slovenian dictionary case-sensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x32008F0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0101627B0101637C0101E87F0101F0810101668201016783010168840101698501016A8601016B8701016C8801016D8901016E8A01016F8B0101708C0101718D0101728E0101738F03019A92010199940101769501017796010178970101799801017A9901019E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001E87F0001F0810001668200016783000168840001698500016A8600016B8700016C8800016D8900016E8A00016F8B0001708C0001718D0001728E0001738F00019A92000199940001769500017796000178970001799800017A9900019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001809001019C5600019B91010174920301999A01019F9B0101001F000198500001925100018253000194540001846C000185240001972500012020000127930001759000019C5700012B91000174920201999A00019F9B000100270001094B0001A24C0001FF8807016D610001A77A0B0162470001A8620001A9480001B4630001AC8F05019A5D0001BB640001AE23000196650001B09903019E660001B55C0001AB4E0001BD8806016D490001B8670001B6680001B7690001864A0001A17A0A01628F04019A5E0001D78805016D4F0001918804016D9902019E8E030173 
	+ 0x7A0301627A0501627A0901627A0701628803016D7E0101647C0301E87D0101E6810301668109016681050166810701668503016A8505016A7F0301F0800101658A03016F8A05016F8B0301708B0501708B0901708B0701705F0001F78E050173940701769403017694090176940501769803017A92050199000000008E0201737A0201627A0401627A0801627A0601628802016D7E0001647C0201E87D0001E6810201668108016681040166810601668502016A8504016A7F0201F0800001658A02016F8A04016F8B0201708B0401708B0801708B060170600001A48E040173940601769402017694080176940401769802017A920401994D0001B2FF030000 
	+ 0x8F01019A8F01019AFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF003400020000008B000000FF0000004709B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 95
go
--  'SLVNC.250'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 96, 4, 0, N'slvnc', N'Slovenian dictionary case-insensitive sort order for use with code page 1250', 0x0000)
update syscharsets set binarydefinition = 0x3C008F0100000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2800010A2900010B2A00010C2B00010D2C0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F260001A02D0001222E0001232F0001243000012531000126320001282100012D330001293400012A3500012C5800013C3600012E220001AD3700012F3800013A700001317100013272000133730001347400013575000136760001377700013878000139790001613900013B3A00013F5900013D5A00013E5B0001B13B0001403C00015B 
	+ 0x7A0002627B0002637C0002E87F0002F0810002668200026783000268840002698500026A8600026B8700026C8800026D8900026E8A00026F8B0002708C0002718D0002728E0002738F00029A92000299940002769500027796000278970002799800027A9900029E3D00015C3E00015D3F00015E4000015F410001604200017B7A0001627B0001637C0001E87F0001F0810001668200016783000168840001698500016A8600016B8700016C8800016D8900016E8A00016F8B0001708C0001718D0001728E0001738F00019A92000199940001769500017796000178970001799800017A9900019E4300017C4400017D4500017E460001A61B0001816F000130 
	+ 0x1C000183520001931D0001885500018B6D0001896A0001876B0001951E0001906E0001809000029C5600019B91000274920102999A00029F9B0002001F000198500001925100018253000194540001846C000185240001972500012020000127930001759000019C5700012B91000174920101999A00019F9B000100270001094B0001A24C0001FF8803026D610001A77A050262470001A8620001A9480001B4630001AC8F03029A5D0001BB640001AE23000196650001B09901029E660001B55C0001AB4E0001BD8803016D490001B8670001B6680001B7690001864A0001A17A0501628F03019A5E0001D78802026D4F0001918802016D9901019E8E010273 
	+ 0x7A0102627A0202627A0402627A0302628801026D7E0002647C0102E87D0002E6810102668104026681020266810302668501026A8502026A7F0102F0800002658A01026F8A02026F8B0102708B0202708B0402708B0302705F0001F78E020273940302769401027694040276940202769801027A92020299000000008E0101737A0101627A0201627A0401627A0301628801016D7E0001647C0101E87D0001E6810101668104016681020166810301668501016A8502016A7F0101F0800001658A01016F8A02016F8B0101708B0201708B0401708B030170600001A48E020173940301769401017694040176940201769801017A920201994D0001B2FF030000 
	+ 0x8F01019A8F01019AFF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C8000000FF000000F009B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 96
go
--  'BINARY.251'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 104, 5, 0, N'bin_cp1251', N'Binary Sort Order for the Code Page 1251 Character Set', 0x0000)
go
--  'DICTION.251'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 105, 5, 0, N'dictionary_251', N'Dictionary case-sensitive sort order for use with code page 1251', 0x0000)
update syscharsets set binarydefinition = 0x0200DF0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2400010A2500010B2600010C2700010D280001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F220001A0290001222A0001232B0001242C0001252D0001262E0001281D00012D2F0001293000012A3100012C4C00013C3200012E1E0001AD3300012F3400013A62000131630001326400013365000134660001356700013668000137690001386A0001396B0001613500013B3600013F4D00013D4E00013E4F0001B1370001403800015B 
	+ 0x6C0101626D0101636E0101646F010165700101667101016772010168730101697401016A7501016B7601016C7701016D7801016E790101B97B0101707C0101717D0101727E0101737F01017480010199820101768301017784010178850101798601017A870101E03900015C3A00015D3B00015E3C00015F3D0001603E00017B6C0001626D0001636E0001646F000165700001667100016772000168730001697400016A7500016B7600016C7700016D7800016E790001B97B0001707C0001717D0001727E0001737F00017480000199820001768300017784000178850001798600017A870001E03F00017C4000017D4100017E420001A61B0001988D0101E5 
	+ 0x8B0501E4460001938B0401E44900018B5F0001895C0001875D00019561000130600001889A0101EC4A00019B9D0101EE980301EBA30101F3AA0101F88D0001E5440001924500018247000194480001845E00018520000197210001201C000127810001759A0001EC4B00012B9D0001EE980201EBA30001F3AA0001F823000109A50101F4A50001F4970101EA530001A78B0301E443000191540001A98E0301BA550001AC8F0101E6510001BB560001AE1F000196570001B0950101E9580001B5500001AB940101BF940001BF8B0201E4590001B65A0001B75B0001868E0201BA7A00016F8F0001E6520001A4970001EA920101E8920001E8950001E9880101E1 
	+ 0x890101E28A0101E38B0101E48C0101908E0101BA900101E7910101BE930101B3960101BC980101EB9901019A9B0101ED9C01019C9E0101EF9F0101F0A00101F1A10101F2A201019EA40101A2A60101F5A70101F6A80101F7A901019FAB0101F9AC0101FAAD0101FBAE0101FCAF0101FDB00101FEB10101FFB2010100880001E1890001E28A0001E38B0001E48C0001908E0001BA900001E7910001BE930001B3960001BC980001EB9900019A9B0001ED9C00019C9E0001EF9F0001F0A00001F1A10001F2A200019EA40001A2A60001F5A70001F6A80001F7A900019FAB0001F9AC0001FAAD0001FBAE0001FCAF0001FDB00001FEB10001FFB2000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C5000000FF0000008D0BB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 105
go
--  'NOCASE.251'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 106, 5, 0, N'nocase_1251', N'Dictionary case-insensitive sort order for use with code page 1251', 0x0000)
update syscharsets set binarydefinition = 0x0C00DF0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2400010A2500010B2600010C2700010D280001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F220001A0290001222A0001232B0001242C0001252D0001262E0001281D00012D2F0001293000012A3100012C4C00013C3200012E1E0001AD3300012F3400013A62000131630001326400013365000134660001356700013668000137690001386A0001396B0001613500013B3600013F4D00013D4E00013E4F0001B1370001403800015B 
	+ 0x6C0002626D0002636E0002646F000265700002667100026772000268730002697400026A7500026B7600026C7700026D7800026E790002B97B0002707C0002717D0002727E0002737F00027480000299820002768300027784000278850002798600027A870002E03900015C3A00015D3B00015E3C00015F3D0001603E00017B6C0001626D0001636E0001646F000165700001667100016772000168730001697400016A7500016B7600016C7700016D7800016E790001B97B0001707C0001717D0001727E0001737F00017480000199820001768300017784000178850001798600017A870001E03F00017C4000017D4100017E420001A61B0001988D0002E5 
	+ 0x8B0202E4460001938B0201E44900018B5F0001895C0001875D00019561000130600001889A0002EC4A00019B9D0002EE980102EBA30002F3AA0002F88D0001E5440001924500018247000194480001845E00018520000197210001201C000127810001759A0001EC4B00012B9D0001EE980101EBA30001F3AA0001F823000109A50002F4A50001F4970002EA530001A78B0102E443000191540001A98E0102BA550001AC8F0002E6510001BB560001AE1F000196570001B0950002E9580001B5500001AB940002BF940001BF8B0101E4590001B65A0001B75B0001868E0101BA7A00016F8F0001E6520001A4970001EA920002E8920001E8950001E9880002E1 
	+ 0x890002E28A0002E38B0002E48C0002908E0002BA900002E7910002BE930002B3960002BC980002EB9900029A9B0002ED9C00029C9E0002EF9F0002F0A00002F1A10002F2A200029EA40002A2A60002F5A70002F6A80002F7A900029FAB0002F9AC0002FAAD0002FBAE0002FCAF0002FDB00002FEB10002FFB2000200880001E1890001E28A0001E38B0001E48C0001908E0001BA900001E7910001BE930001B3960001BC980001EB9900019A9B0001ED9C00019C9E0001EF9F0001F0A00001F1A10001F2A200019EA40001A2A60001F5A70001F6A80001F7A900019FAB0001F9AC0001FAAD0001FBAE0001FCAF0001FDB00001FEB10001FFB2000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000D0000000FF000000350CB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 106
go
--  'UKRDIC.251'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 107, 5, 0, N'ukrdic', N'Ukrainian dictionary case-sensitive sort order for use with code page 1251', 0x0000)
update syscharsets set binarydefinition = 0x0200DC0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2400010A2500010B2600010C2700010D280001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F220001A0290001222A0001232B0001242C0001252D0001262E0001281D00012D2F0001293000012A3100012C4C00013C3200012E1E0001AD3300012F3400013A62000131630001326400013365000134660001356700013668000137690001386A0001396B0001613500013B3600013F4D00013D4E00013E4F0001B1370001403800015B 
	+ 0x6C0101626D0101636E0101646F010165700101667101016772010168730101697401016A7501016B7601016C7701016D7801016E790101B97B0101707C0101717D0101727E0101737F01017480010199820101768301017784010178850101798601017A870101E03900015C3A00015D3B00015E3C00015F3D0001603E00017B6C0001626D0001636E0001646F000165700001667100016772000168730001697400016A7500016B7600016C7700016D7800016E790001B97B0001707C0001717D0001727E0001737F00017480000199820001768300017784000178850001798600017A870001E03F00017C4000017D4100017E420001A61B0001988D0101E5 
	+ 0x8B0501E4460001938B0401E44900018B5F0001895C0001875D00019561000130600001889A0101EC4A00019B9D0101EE980301EBA30101F3AA0101F88D0001E5440001924500018247000194480001845E00018520000197210001201C000127810001759A0001EC4B00012B9D0001EE980201EBA30001F3AA0001F823000109A50101F4A50001F4970101EA530001A78B0301E443000191540001A98E0301BA550001AC8F0101E6510001BB560001AE1F000196570001B0950101E9580001B5500001AB940101BF940001BF8B0201E4590001B65A0001B75B0001868E0201BA7A00016F8F0001E6520001A4970001EA920101E8920001E8950001E9880101E1 
	+ 0x890101E28A0101E38B0101E48C0101908E0101BA900101E7910101BE930101B3960101BC980101EB9901019A9B0101ED9C01019C9E0101EF9F0101F0A00101F1A10101F2A201019EA40101A2A60101F5A70101F6A80101F7A901019FAB0101F9AC0101FAAD0101FBAE0101FDB2010100AF0101FEB00101FFB10101FC880001E1890001E28A0001E38B0001E48C0001908E0001BA900001E7910001BE930001B3960001BC980001EB9900019A9B0001ED9C00019C9E0001EF9F0001F0A00001F1A10001F2A200019EA40001A2A60001F5A70001F6A80001F7A900019FAB0001F9AC0001FAAD0001FBAE0001FDB2000100AF0001FEB00001FFB10001FCFF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000DB000000FF000000F60CB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 107
go
--  'UKRNC.251'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 108, 5, 0, N'ukrnc', N'Ukrainian dictionary case-insensitive sort order for use with code page 1251', 0x0000)
update syscharsets set binarydefinition = 0x0C00DC0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2400010A2500010B2600010C2700010D280001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F220001A0290001222A0001232B0001242C0001252D0001262E0001281D00012D2F0001293000012A3100012C4C00013C3200012E1E0001AD3300012F3400013A62000131630001326400013365000134660001356700013668000137690001386A0001396B0001613500013B3600013F4D00013D4E00013E4F0001B1370001403800015B 
	+ 0x6C0002626D0002636E0002646F000265700002667100026772000268730002697400026A7500026B7600026C7700026D7800026E790002B97B0002707C0002717D0002727E0002737F00027480000299820002768300027784000278850002798600027A870002E03900015C3A00015D3B00015E3C00015F3D0001603E00017B6C0001626D0001636E0001646F000165700001667100016772000168730001697400016A7500016B7600016C7700016D7800016E790001B97B0001707C0001717D0001727E0001737F00017480000199820001768300017784000178850001798600017A870001E03F00017C4000017D4100017E420001A61B0001988D0002E5 
	+ 0x8B0202E4460001938B0201E44900018B5F0001895C0001875D00019561000130600001889A0002EC4A00019B9D0002EE980102EBA30002F3AA0002F88D0001E5440001924500018247000194480001845E00018520000197210001201C000127810001759A0001EC4B00012B9D0001EE980101EBA30001F3AA0001F823000109A50002F4A50001F4970002EA530001A78B0102E443000191540001A98E0102BA550001AC8F0002E6510001BB560001AE1F000196570001B0950002E9580001B5500001AB940002BF940001BF8B0101E4590001B65A0001B75B0001868E0101BA7A00016F8F0001E6520001A4970001EA920002E8920001E8950001E9880002E1 
	+ 0x890002E28A0002E38B0002E48C0002908E0002BA900002E7910002BE930002B3960002BC980002EB9900029A9B0002ED9C00029C9E0002EF9F0002F0A00002F1A10002F2A200029EA40002A2A60002F5A70002F6A80002F7A900029FAB0002F9AC0002FAAD0002FBAE0002FDB2000200AF0002FEB00002FFB10002FC880001E1890001E28A0001E38B0001E48C0001908E0001BA900001E7910001BE930001B3960001BC980001EB9900019A9B0001ED9C00019C9E0001EF9F0001F0A00001F1A10001F2A200019EA40001A2A60001F5A70001F6A80001F7A900019FAB0001F9AC0001FAAD0001FBAE0001FDB2000100AF0001FEB00001FFB10001FCFF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000BD000000FF0000009E0DB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 108
go
--  'BINARY.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 112, 6, 0, N'bin_cp1253', N'Binary Sort Order for the Code Page 1253 Character Set', 0x0000)
go
--  'DICTION.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 113, 6, 0, N'dictionary_1253', N'Greek case sensitive dictionary sort order for use with code page 1253', 0x0000)
update syscharsets set binarydefinition = 0x0E00FF0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E3200010A3300010B3400010C3500010D360001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F300001A03700012238000123390001243A0001253B0001263C0001282A00012D3D0001293E00012A3F00012C5D00013C4000012E2B0001AD4100012F4200013A750001BD77000132780001B27A0001B37C0001357D0001367E0001377F00013880000139810001614300013B4400013F5E00013D5F00013E600001B1450001404600015B 
	+ 0x82010162830101638401016485010165860101668701016788010168890101698A01016A8B01016B8C01016C8D01016D8E01016E8F01016F9001017091010171920101729301017394010174950101999701017698010177990101789A0101799B01017A9C0101E14700015C4800015D4900015E4A00015F4B0001604C00017B82000162830001638400016485000165860001668700016788000168890001698A00016A8B00016B8C00016C8D00016D8E00016E8F00016F9000017091000171920001729300017394000174950001999700017698000177990001789A0001799B00017A9C0001E14D00017C4E00017D4F00017E500001A61B00018174000130 
	+ 0x1C00018857000193870201675A00018B720001896F000187700001951D00018A730001801E00018C5B00019B1F00018D2000018E2100018F220001902300019855000192560001825800019459000184710001852D0001972E0001AF2400019A960001752500019C5C00012B2600019D2700019E2800019F2900012731000109540001919D0301E2640001A4650001A5660001A7510001A8670001A9520001B4680001ACB50001D2620001BB690001AE2C0001966A0001B02F0001206B0001B5610001AB790001337B000134530001A16C0001B66D0001B76E000186A10301E6A30301E8A50301EA630001A3AB0301F076000131B00301F6B40301AAA50601EA 
	+ 0x9D0101E29E0101E39F0101E4A00101E5A10101E6A20101E7A30101E8A40101E9A50101EAA60101EBA70101ECA80101EDA90101EEAA0101EFAB0101F0AC0101F1AD0101F3B60001FFAE0101F4AF0101F5B00101F6B10101F7B20101F8B30101F9B40101AAA50501EAB00501F69D0201E2A10201E6A30201E8A50201EAB00601F69D0001E29E0001E39F0001E4A00001E5A10001E6A20001E7A30001E8A40001E9A50001EAA60001EBA70001ECA80001EDA90001EEAA0001EFAB0001F0AC0001F1AD0001F3AE0002F4AE0001F4AF0001F5B00001F6B10001F7B20001F8B30001F9B40001AAA50401EAB00401F6AB0201F0B00201F6B40201AAB7000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000A0000000FF000000390FB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 113
go
--  'NOCASE.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 114, 6, 0, N'nocase_1253', N'Greek case insensitive dictionary sort order for use with code page 1253', 0x0000)
update syscharsets set binarydefinition = 0x0C00FF0000000101 + 0x010001020200010303000104040001050500010606000107070001080800010E3200010A3300010B3400010C3500010D360001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F300001A03700012238000123390001243A0001253B0001263C0001282A00012D3D0001293E00012A3F00012C5D00013C4000012E2B0001AD4100012F4200013A750001BD77000132780001B27A0001B37C0001357D0001367E0001377F00013880000139810001614300013B4400013F5E00013D5F00013E600001B1450001404600015B 
	+ 0x82000262830002638400026485000265860002668700026788000268890002698A00026A8B00026B8C00026C8D00026D8E00026E8F00026F9000027091000271920002729300027394000274950002999700027698000277990002789A0002799B00027A9C0002E14700015C4800015D4900015E4A00015F4B0001604C00017B82000162830001638400016485000165860001668700016788000168890001698A00016A8B00016B8C00016C8D00016D8E00016E8F00016F9000017091000171920001729300017394000174950001999700017698000177990001789A0001799B00017A9C0001E14D00017C4E00017D4F00017E500001A61B00018174000130 
	+ 0x1C00018857000193870101675A00018B720001896F000187700001951D00018A730001801E00018C5B00019B1F00018D2000018E2100018F220001902300019855000192560001825800019459000184710001852D0001972E0001AF2400019A960001752500019C5C00012B2600019D2700019E2800019F2900012731000109540001919D0102E2640001A4650001A5660001A7510001A8670001A9520001B4680001ACB50001D2620001BB690001AE2C0001966A0001B02F0001206B0001B5610001AB790001337B000134530001A16C0001B66D0001B76E000186A10102E6A30102E8A50102EA630001A3AB0102F076000131B00102F6B40102AAA50301EA 
	+ 0x9D0002E29E0002E39F0002E4A00002E5A10002E6A20002E7A30002E8A40002E9A50002EAA60002EBA70002ECA80002EDA90002EEAA0002EFAB0002F0AC0002F1AD0002F3B60001FFAE0003F4AF0002F5B00002F6B10002F7B20002F8B30002F9B40002AAA50202EAB00202F69D0101E2A10101E6A30101E8A50101EAB00301F69D0001E29E0001E39F0001E4A00001E5A10001E6A20001E7A30001E8A40001E9A50001EAA60001EBA70001ECA80001EDA90001EEAA0001EFAB0001F0AC0001F1AD0001F3AE0002F4AE0001F4AF0001F5B00001F6B10001F7B20001F8B30001F9B40001AAA50201EAB00201F6AB0101F0B00101F6B40101AAB7000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000DD000000FF000000F311B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 114

--  'GRALTDCT.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 121, 6, 0, N'greek_altdictionary', N'Greek dictionary sort order for use with Greek and English. Uses the ISO 8859-7 (ANSI 1253) Greek Windows character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition = 0x0200FE0000000101 + 0x01000102020001030300010404000105050001060600010707000108080001090900010A0A00010B0B00010C0C00010D0D00010E0E00010F0F0001101000011111000112120001131300011414000115150001161600011717000118180001191900011A1A00011B1B00011C1C00011D1D00011E1E00011F1F0001202000012121000122220001232300012424000125250001262600012727000128280001292900012A2A00012B2B00012C2C00012D2D00012E2E00012F2F00013A7D0001317E0001327F000133800001348100013582000136830001378400013885000139860001413000013B3100013C3200013D3300013E3400013F350001403600015B 
	+ 0x8700014288000143890001448A0001458B0001468C0001478D0001488E0001498F00014A9000014B9100014C9200014D9300014E9400014F95000150960001519700015298000153990001549A0001559B0001569C0001579D0001589E0001599F00015AA00001C13700015C3800015D3900015E3A00015F3B0001603C00017B8701014288010143890101448A0101458B0101468C0101478D0101488E0101498F01014A9001014B9101014C9201014D9301014E9401014F95010150960101519701015298010153990101549A0101559B0101569C0101579D0101589E0101599F01015AA00101C13D00017C3E00017D3F00017E4000017F4100018042000181 
	+ 0x430001824400018345000184460001854700018648000187490001884A0001894B00018A4C00018B4D00018C4E00018D4F00018E5000018F5100019052000191530001925400019355000194560001955700019658000197590001985A0001995B00019A5C00019B5D00019C5E00019D5F00019E6000019F610001A0620001A1630001A3A10101C2640001A4650001A5660001A6670001A7680001A8690001A96A0001AA6B0001AB6C0001AC6D0001AD6E0001AE6F0001AF700001B0710001B1720001B2730001B3740001B4750001B5760001B6770001B7780001BBA50101C6A70101C8A90101CA790001BDAF0101D07A0001D2B40101D6B8010100A90601CA 
	+ 0xA10001C2A20001C3A30001C4A40001C5A50001C6A60001C7A70001C8A80001C9A90001CAAA0001CBAB0001CCAC0001CDAD0001CEAE0001CFAF0001D0B00001D1B10001D37B0001FFB20001D4B30001D5B40001D6B50001D7B60001D8B70001D9B8000100A90501CAB40201D6A10301C2A50301C6A70301C8A90401CAB40601D6A10201C2A20101C3A30101C4A40101C5A50201C6A60101C7A70201C8A80101C9A90301CAAA0101CBAB0101CCAC0101CDAD0101CEAE0101CFAF0201D0B00101D1B10101D3B20201D4B20101D4B30101D5B40301D6B50101D7B60101D8B70101D9B8020100A90201CAB40501D6AF0301D0B40401D6B80301007C000130FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF003400020000008C000000FF000000E10FB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 121
go
--  'GRMXTDCT.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 120, 6, 0, N'greek_mixeddictionary', N'Greek English mixed dictionary sort order. Uses the ISO 8859-7 (ANSI 1253) Greek Windows character set and is case-sensitive.', 0x0000)
update syscharsets set binarydefinition = 0x0200FE0000000101 + 0x01000102020001030300010404000105050001060600010707000108080001090900010A0A00010B0B00010C0C00010D0D00010E0E00010F0F0001101000011111000112120001131300011414000115150001161600011717000118180001191900011A1A00011B1B00011C1C00011D1D00011E1E00011F1F0001202000012121000122220001232300012424000125250001262600012727000128280001292900012A2A00012B2B00012C2C00012D2D00012E2E00012F2F00013A7D0001317E0001327F000133800001348100013582000136830001378400013885000139860001C13000013B3100013C3200013D3300013E3400013F350001403600015B 
	+ 0x870401C2880201C3900401CB8A0201C58B0401C69B0201D7890201C49C0201D88F0701CA990401D5900201CB910201CC920201CD930201CE950401D0960201D1900601CB970201D3980301D4990201D59A0901D6880401C3880601C3940201CF9A0701D68C0201C73700015C3800015D3900015E3A00015F3B0001603C00017B870501C2880301C3900501CB8A0301C58B0501C69B0301D7890301C49C0301D88F0801CA990501D5900301CB910301CC920301CD930301CE950501D0960301D1900701CB970301D3980401D4990301D59A0A01D6880501C3880701C3940301CF9A0801D68C0301C73D00017C3E00017D3F00017E4000017F4100018042000181 
	+ 0x430001824400018345000184460001854700018648000187490001884A0001894B00018A4C00018B4D00018C4E00018D4F00018E5000018F5100019052000191530001925400019355000194560001955700019658000197590001985A0001995B00019A5C00019B5D00019C5E00019D5F00019E6000019F610001A0620001A1630001A3870101C2640001A4650001A5660001A6670001A7680001A8690001A96A0001AA6B0001AB6C0001AC6D0001AD6E0001AE6F0001AF700001B0710001B1720001B2730001B3740001B4750001B5760001B6770001B7780001BB8B0101C68D0101C88F0101CA790001BD950101D07A0001D29A0101D69E0101008F0601CA 
	+ 0x870001C2880001C3890001C48A0001C58B0001C68C0001C78D0001C88E0001C98F0001CA900001CB910001CC920001CD930001CE940001CF950001D0960001D1970001D37B0001FF980001D4990001D59A0001D69B0001D79C0001D89D0001D99E0001008F0501CA9A0201D6870301C28B0301C68D0301C88F0401CA9A0601D6870201C2880101C3890101C48A0101C58B0201C68C0101C78D0201C88E0101C98F0301CA900101CB910101CC920101CD930101CE940101CF950201D0960101D1970101D3980201D4980101D4990101D59A0301D69B0101D79C0101D89D0101D99E0201008F0201CA9A0501D6950301D09A0401D69E0301007C000130FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000BE000000FF000000A110B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 120
go
--  'GRNOACCE.253'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 124, 6, 0, N'greek_nocasedict', N'Greek dictionary sort order for use with Greek and English. Uses the ISO 8859-7 (ANSI 1253) Greek Windows character set and is not case-sensitive.', 0x0000)
update syscharsets set binarydefinition = 0x0900FE0000000101 + 0x01000102020001030300010404000105050001060600010707000108080001090900010A0A00010B0B00010C0C00010D0D00010E0E00010F0F0001101000011111000112120001131300011414000115150001161600011717000118180001191900011A1A00011B1B00011C1C00011D1D00011E1E00011F1F0001202000012121000122220001232300012424000125250001262600012727000128280001292900012A2A00012B2B00012C2C00012D2D00012E2E00012F2F00013A7D0001317E0001327F000133800001348100013582000136830001378400013885000139860001413000013B3100013C3200013D3300013E3400013F350001403600015B 
	+ 0x8700014288000143890001448A0001458B0001468C0001478D0001488E0001498F00014A9000014B9100014C9200014D9300014E9400014F95000150960001519700015298000153990001549A0001559B0001569C0001579D0001589E0001599F00015AA00001C13700015C3800015D3900015E3A00015F3B0001603C00017B8700024288000243890002448A0002458B0002468C0002478D0002488E0002498F00024A9000024B9100024C9200024D9300024E9400024F95000250960002519700025298000253990002549A0002559B0002569C0002579D0002589E0002599F00025AA00002C13D00017C3E00017D3F00017E4000017F4100018042000181 
	+ 0x430001824400018345000184460001854700018648000187490001884A0001894B00018A4C00018B4D00018C4E00018D4F00018E5000018F5100019052000191530001925400019355000194560001955700019658000197590001985A0001995B00019A5C00019B5D00019C5E00019D5F00019E6000019F610001A0620001A1630001A3A10002C2640001A4650001A5660001A6670001A7680001A8690001A96A0001AA6B0001AB6C0001AC6D0001AD6E0001AE6F0001AF700001B0710001B1720001B2730001B3740001B4750001B5760001B6770001B7780001BBA50002C6A70002C8A90002CA790001BDAF0002D07A0001D2B40002D6B8000200A90007CA 
	+ 0xA10001C2A20001C3A30001C4A40001C5A50001C6A60001C7A70001C8A80001C9A90001CAAA0001CBAB0001CCAC0001CDAD0001CEAE0001CFAF0001D0B00001D1B10001D37B0001FFB20001D4B30001D5B40001D6B50001D7B60001D8B70001D9B8000100A90006CAB40003D6A10004C2A50004C6A70004C8A90005CAB40007D6A10003C2A20002C3A30002C4A40002C5A50003C6A60002C7A70003C8A80002C9A90004CAAA0002CBAB0002CCAC0002CDAD0002CEAE0002CFAF0003D0B00002D1B10002D3B20003D4B20002D4B30002D5B40004D6B50002D7B60002D8B70002D9B8000300A90003CAB40006D6AF0004D0B40005D6B80004007C000130FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C7000000FF0000004B11B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 124
go
--  'BINARY.254'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 128, 7, 0, N'bin_cp1254', N'Binary Sort Order for the Code Page 1254 Character Set', 0x0000)
go
--  'DICTION.254'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 129, 7, 0, N'dictionary_1254', N'Case sensitive dictionary sort order for use with code page 1254', 0x0000)
update syscharsets set binarydefinition = 0x32005A0500000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2A00010A2B00010B2C00010C2D00010D2E0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F280001A02F00012230000123310001243200012533000126340001282300012D350001293600012A3700012C5A00013C3800012E240001AD3900012F3A00013A750001BC790001B97B0001B27D0001B37F00013580000136810001378200013883000139840001613B00013B3C00013F5B00013D5C00013E5D0001B13D0001403E00015B 
	+ 0x8502016286010163870101E7890101658A0301668B0101678C0101F08E0101FD8F01016A9001016B9101016C9201016D9301016E9401016F950201F69701017198010172990101739A0301FE9C0101999E0101FCA0010177A1010178A2010179A301017AA40101003F00015C4000015D4100015E42000188440001604500017B8500016286000163870001E7890001658A0001668B0001678C0001F08E0001FD8F02016A9000016B9100016C9200016D9300016E9400016F950001F69700017198000172990001739A0001FE9C0001999E0001FCA0000177A1000178A2000179A300017AA40001004600017C4700017D4800017E490001A11B00018174000130 
	+ 0x1C00018D540001938B0201675700018B720001896F000187700001954300015F730001809A0501FE5800019B000000031D00018E1E00018F1F0001902000019D52000192530001825500019456000184710001852600019727000120510001919D0001759A0401FE5900012B000000022100019E22000127A303017A290001094A0001A6630001A3640001A4650001A5660001A74B0001A8670001A94C0001AF680001AC850401625F0001BB690001AE250001966A0001B04D0001B46B0001B55E0001AB7C0001337E0001344E0001B86C0001B66D0001B76E0001864F0001BF7A000132950401F6600001D7760001BD770001BE780001315000019885080162 
	+ 0x85060162850A0162850E0162850C01628510016200000001880101648A0901668A0701668A0B01668A0D01668F07016A8F05016A8F09016A8F0B016A8D0101689403016F950801F6950601F6950A01F6950C01F696010170610001F7950E01F69E0501FC9E0301FC9E0701FC9F0101768F03016A9B01017400000004850701628505016285090162850D0162850B0162850F016200000000880001648A0801668A0601668A0A01668A0C01668F06016A8F04016A8F08016A8F0A016A8D0001689402016F950701F6950501F6950901F6950B01F696000170620001A2950D01F69E0401FC9E0201FC9E0601FC9F0001768F00016A9B000174A302017AFF030000 
	+ 0x850101628A010166FF030000850301628A040166FF030000950101F68A020166FF030000950301F68A050166FF0300009A0101FE9A0101FEFF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000B1000000FF0000008E13B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 129
go
--  'NOCASE.254'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 130, 7, 0, N'nocase_1254', N'Case insensitive dictionary order for use with code page 1254', 0x0000)
update syscharsets set binarydefinition = 0x3C005A0500000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2A00010A2B00010B2C00010C2D00010D2E0001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F280001A02F00012230000123310001243200012533000126340001282300012D350001293600012A3700012C5A00013C3800012E240001AD3900012F3A00013A750001BC790001B97B0001B27D0001B37F00013580000136810001378200013883000139840001613B00013B3C00013F5B00013D5C00013E5D0001B13D0001403E00015B
	+ 0x8500026286000263870002E7890002658A0002668B0002678C0002F08E0002FD8F0002699100026B9200026C9300026D9400026E9500026F960002F698000271990002729A0002739B0002FE9D0002999F0002FCA1000277A2000278A3000279A400027AA50002003F00015C4000015D4100015E42000188440001604500017B8500016286000163870001E7890001658A0001668B0001678C0001F08E0001FD9000016A9100016B9200016C9300016D9400016E9500016F960001F698000171990001729A0001739B0001FE9D0001999F0001FCA1000177A2000178A3000179A400017AA50001004600017C4700017D4800017E490001A11B00018174000130
	+ 0x1C00018D540001938B0101675700018B720001896F000187700001954300015F730001809B0302FE5800019B000000031D00018E1E00018F1F0001902000019D52000192530001825500019456000184710001852600019727000120510001919E0001759B0301FE5900012B000000022100019E22000127A401027A290001094A0001A6630001A3640001A4650001A5660001A74B0001A8670001A94C0001AF680001AC850201625F0001BB690001AE250001966A0001B04D0001B46B0001B55E0001AB7C0001337E0001344E0001B86C0001B66D0001B76E0001864F0001BF7A000132960201F6600001D7760001BD770001BE780001315000019885040262
	+ 0x850302628505026285070262850602628508026200000001880002648A0402668A0302668A0502668A0602669002026A9001026A9003026A9004026A8D0002689501026F960402F6960302F6960502F6960602F697000270610001F7960702F69F0202FC9F0102FC9F0302FCA00002769000026A9C0002740000000485040162850301628505016285070162850601628508016200000000880001648A0401668A0301668A0501668A0601669002016A9001016A9003016A9004016A8D0001689501016F960401F6960301F6960501F6960601F697000170620001A2960701F69F0201FC9F0101FC9F0301FCA00001768F0001699C000174A401017AFF030000
	+ 0x850101628A010166FF030000850102628A010266FF030000960101F68A020166FF030000960102F68A020266FF0300009B0101FE9B0101FEFF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1B14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7772819140000000000FF0200000000140000000000FF00000000000000
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000A9000000FF000000AEE2710200000000FFD2D2FC0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1B140028001400F81B1400FFBEF8772800000000000000FF001400180000007D000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200F81B1400FF0014008CF41200C6123000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400
where id = 130
go


--  'BINARY.255'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 136, 8, 0, N'bin_cp1255', N'Binary Sort Order for the Code Page 1255 Character Set', 0x0000)
go
--  'DICTION.255'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 137, 8, 0, N'dictionary_1255', N'Case-sensitive dictionary sort order for use with code page 1255', 0x0000)
update syscharsets set binarydefinition = 0x0200FF00000001C0 + 0x120001021300010314000104150001051600010617000107180001081900010E4000010A4100010B4200010C4300010D440001211A00010F1B0001101C0001111D0001121E0001131F00011420000115210001162200011723000118240001192500011A2600011B2700011C2800011D2900011E2A00011F2B00017F3E0001A045000122460001234700012448000125490001264A0001283900012D4B0001294C00012A4D00012C7200013C4E00012E3A0001AD4F00012F5000013A8D0001BC910001B9930001B2950001B39700013598000136990001379A0001389B0001399C0001615100013B5200013F7300013D7400013E750001B1530001405400015B 
	+ 0x9D0101629E0101639F010164A0010165A1010166A2010167A3010168A4010169A501016AA601016BA701016CA801016DA901016EAA01016FAB010170AC010171AD010172AE010173AF010174B0010199B2010176B3010177B4010178B5010179B601017AB70101E05500015C5600015D5700015E580001885A0001605B00017B9D0001629E0001639F000164A0000165A1000166A2000167A3000168A4000169A500016AA600016BA700016CA800016DA900016EAA00016FAB000170AC000171AD000172AE000173AF000174B0000199B2000176B3000177B4000178B5000179B600017AB70001E05C00017C5D00017D5E00017E5F0001A12C0001818C000130 
	+ 0x2D00018A6C000193A20201676F00018B8A00018987000187880001955900015F8B0001802E00018C7000019B2F00018D3000018E3100018F320001903300019A6A0001926B0001826D0001946E000184890001853C0001973D000120670001CEB10001753400019C7100012B3500019D3600019E3700019F380001273F000109600001A67B0001A37C0001A47D0001A57E0001A7610001A87F0001A9620001AF800001AC790001BA770001BB810001AE3B000196820001B0630001B4830001B5760001AB9400013396000134640001B8840001B6850001B786000186650001BF920001327A0001A2780001AA8E0001BD8F0001BE9000013166000198010001C1 
	+ 0x020001C2030001C3040001C4050001C5060001C6070001C7080001C8090001C90A0001CBCE0001D40B0001CC0C0001CD0D0001CF680001D30E0001D00F0001D1100001D21100010169000191CF0001D5D00001D6D10001D7D20001D8D30001D9D40001DAD50001DBD60001DCD70001DDD80001DED90001DFDA0001FBB80001E1B90001E2BA0001E3BB0001E4BC0001E5BD0001E6BE0001E7BF0001E8C00001E9C10001EBC20101ECC20001ECC30001EEC40101F0C40001F0C50101F1C50001F1C60001F2C70001F4C80101F6C80001F6C90101F7C90001F7CA0001F8CB0001F9CC0001FACD0001CADB0001FCDC0001FDDD0001FEDE0001FFDF000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C6000000FF000000D415B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 137
go
--  'NOCASE.255'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 138, 8, 0, N'nocase_1255', N'Case-insensitive dictionary sort order for use with code page 1255', 0x0000)
update syscharsets set binarydefinition = 0x0900FF00000001C0 + 0x120001021300010314000104150001051600010617000107180001081900010E4000010A4100010B4200010C4300010D440001211A00010F1B0001101C0001111D0001121E0001131F00011420000115210001162200011723000118240001192500011A2600011B2700011C2800011D2900011E2A00011F2B00017F3E0001A045000122460001234700012448000125490001264A0001283900012D4B0001294C00012A4D00012C7200013C4E00012E3A0001AD4F00012F5000013A8D0001BC910001B9930001B2950001B39700013598000136990001379A0001389B0001399C0001615100013B5200013F7300013D7400013E750001B1530001405400015B 
	+ 0x9D0002629E0002639F000264A0000265A1000266A2000267A3000268A4000269A500026AA600026BA700026CA800026DA900026EAA00026FAB000270AC000271AD000272AE000273AF000274B0000299B2000276B3000277B4000278B5000279B600027AB70002E05500015C5600015D5700015E580001885A0001605B00017B9D0001629E0001639F000164A0000165A1000166A2000167A3000168A4000169A500016AA600016BA700016CA800016DA900016EAA00016FAB000170AC000171AD000172AE000173AF000174B0000199B2000176B3000177B4000178B5000179B600017AB70001E05C00017C5D00017D5E00017E5F0001A12C0001818C000130 
	+ 0x2D00018A6C000193A20101676F00018B8A00018987000187880001955900015F8B0001802E00018C7000019B2F00018D3000018E3100018F320001903300019A6A0001926B0001826D0001946E000184890001853C0001973D000120670001CEB10001753400019C7100012B3500019D3600019E3700019F380001273F000109600001A67B0001A37C0001A47D0001A57E0001A7610001A87F0001A9620001AF800001AC790001BA770001BB810001AE3B000196820001B0630001B4830001B5760001AB9400013396000134640001B8840001B6850001B786000186650001BF920001327A0001A2780001AA8E0001BD8F0001BE9000013166000198010001C1 
	+ 0x020001C2030001C3040001C4050001C5060001C6070001C7080001C8090001C90A0001CBCE0001D40B0001CC0C0001CD0D0001CF680001D30E0001D00F0001D1100001D21100010169000191CF0001D5D00001D6D10001D7D20001D8D30001D9D40001DAD50001DBD60001DCD70001DDD80001DED90001DFDA0001FBB80001E1B90001E2BA0001E3BB0001E4BC0001E5BD0001E6BE0001E7BF0001E8C00001E9C10001EBC20002ECC20001ECC30001EEC40002F0C40001F0C50002F1C50001F1C60001F2C70001F4C80002F6C80001F6C90002F7C90001F7CA0001F8CB0001F9CC0001FACD0001CADB0001FCDC0001FDDD0001FEDE0001FFDF000100FF006E00 
	+ 0x670020004F007200FF0065007200000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C5000000FF0000009516B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 138
go
--  'BINARY.256'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 144, 9, 0, N'bin_cp1256', N'Binary Sort Order for the Code Page 1256 Character Set', 0x0000)
go
--  'DICTION.256'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 145, 9, 0, N'dictionary_1256', N'Case-sensitive dictionary order for use with code page 1256', 0x0000)
update syscharsets set binarydefinition = 0x3200FE03000001F8 + 0x020001020300010304000104050001050600010607000107080001080900010E3000010A3100010B3200010C3300010D3400019D0A00010F0B0001100C0001110D0001120E0001130F00011410000115110001161200011713000118140001191500011A1600011B1700011C1800011D1900011E1A00011F1B00017F2E0001A03700012238000123390001243A0001253B0001263C0001282200012D3D0001293E00012A3F00012C6200013C4000012E230001AD4100012F4200013A7D0001BC810001B9830001B2850001B38700013588000136890001378A0001388B0001398C0001614300013B4400013F6300013D6400013E650001B1450001404600015B 
	+ 0x8D0101628E0101638F01016490010165910201669201016793010168940101699501016A9601016B9701016C9801016D9901016E9A01016F9B0201709C0101719D0101729E0101739F010174A0010199A2010176A3010177A4010178A5010179A601017AA70101C14700015C4800015D4900015E4A0001884C0001604D00017B8D0001628E0001638F00016490000165910001669200016793000168940001699500016A9600016B9700016C9800016D9900016E9A00016F9B0001709C0001719D0001729E0001739F000174A0000199A2000176A3000177A4000178A5000179A600017AA70001C14E00017C4F00017D5000017E510001A61C00018A7C000130 
	+ 0xAB0001C95C000193920201675F00018B7A00018977000187780001954B00015F7B0001801D00018F6000019B00000002AF0001CDB60001D31E000198C20001E15A0001925B0001825D0001945E0001847900018525000197260001F01F00019AA10001752000019F6100012B000000013500019E36000121210001272F000109570001BA6B0001A36C0001A46D0001A56E0001A7520001A86F0001A9530001AF700001ACC90001C0670001BB710001AE24000196720001B0540001B4730001B5660001AB8400013386000134550001B8740001B6750001B776000186560001A182000132580001BF680001D77E0001BD7F0001BE8000013159000191CA0001FF 
	+ 0xA80001C700000000A80201C7A80301C7A80401C7A80501C7A90001C8AA000181AC0001CBAC0101CBAD0001CCAE00018DB00001CEB10001CFB20001D0B30001D1B40001D2B500018EB70001D4B80001D5B90001D6BA0001D8690001F7BB0001D9BC0001DABD0001DBBE0001DDCC0001FDBF0001DEC00001DFC10001908D020162C30001E38D030162C40001E4C50001E5C60001E6C70001EC8F02016491050166910401669106016691070166C80001AAC80101AA9502016A9503016A270001F1280001F2290001F32A0001F59B0401702B0001F62C0001FA6A0001A201000101A20201762D000120A2030176A2040176CD0001FECE000100CB0001DCFF030000 
	+ 0xA80101C7A90101C8FF0300009B01017091010166FF0300009B03017091030166FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000D7000000FF0000003318B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 145
go
--  'NOCASE.256'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 146, 9, 0, N'nocase_1256', N'Case-insensitive dictionary sort order for use with code page 1256', 0x0000)
update syscharsets set binarydefinition = 0x3C00FF03000001F8 + 0x020001020300010304000104050001050600010607000107080001080900010E3000010A3100010B3200010C3300010D3400019D0A00010F0B0001100C0001110D0001120E0001130F00011410000115110001161200011713000118140001191500011A1600011B1700011C1800011D1900011E1A00011F1B00017F2E0001A03700012238000123390001243A0001253B0001263C0001282200012D3D0001293E00012A3F00012C6200013C4000012E230001AD4100012F4200013A7D0001BC810001B9830001B2850001B38700013588000136890001378A0001388B0001398C0001614300013B4400013F6300013D6400013E650001B1450001404600015B 
	+ 0x8D0002628E0002638F00026490000265910002669200026793000268940002699500026A9600026B9700026C9800026D9900026E9A00026F9B0002709C0002719D0002729E0002739F000274A0000299A2000276A3000277A4000278A5000279A600027AA70002C14700015C4800015D4900015E4A0001884C0001604D00017B8D0001628E0001638F00016490000165910001669200016793000168940001699500016A9600016B9700016C9800016D9900016E9A00016F9B0001709C0001719D0001729E0001739F000174A0000199A2000176A3000177A4000178A5000179A600017AA70001C14E00017C4F00017D5000017E510001A61C00018A7C000130 
	+ 0xAB0001C95C000193920101675F00018B7A00018977000187780001954B00015F7B0001801D00018F6000019B00000002AF0001CDB60001D31E000198C30001E15A0001925B0001825D0001945E0001847900018525000197260001F01F00019AA10001752000019F6100012B000000013500019E36000121210001272F000109570001BA6B0001A36C0001A46D0001A56E0001A7520001A86F0001A9530001AF700001ACCA0001C0670001BB710001AE24000196720001B0540001B4730001B5660001AB8400013386000134550001B8740001B6750001B776000186560001A182000132580001BF680001D77E0001BD7F0001BE8000013159000191CB0001FD 
	+ 0xA80001C700000000A80201C7A80301C7A80401C7A80501C7A90001C8AA000181AC0001CBAC0002CBAD0001CCAE00018DB00001CEB10001CFB20001D0B30001D1B40001D2B500018EB70001D4B80001D5B90001D6BA0001D8690001F7BB0001D9BC0001DABD0001DBBE0001DCBF0001DDC00001DEC10001DFC20001908D010162C40001E38D020162C50001E4C60001E5C70001E6C80001EC8F01016491030166910201669104016691050166C90001AAC90101AA9501016A9502016A270001F1280001F2290001F32A0001F59B0201702B0001F62C0001FA6A0001A201000101A20101762D000120A2020176A2030176CC0001FECD0001FFCE000100FF030000 
	+ 0xA80101C7A90101C8FF0300009B01017091010166FF0300009B01027091010266FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF003400020000009F000000FF000000E518B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 146
go
--  'BINARY.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 152, 10, 0, N'bin_cp1257', N'Binary Sort Order for the Code Page 1257 Character Set', 0x0000)
go
--  'DICTION.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 153, 10, 0, N'dictionary_1257', N'Case sensitive sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3200A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x860201628701016388010164890101658A0201668B0101678C0101688D0101698E01016A8F01016B9001016C9101016D9201016E9301016F9401017095010171960101729701017398030174990101999B0101769C0101779D0101789E0101799F01017AA00101A14200015C4300015D4400015E4500015F460001604700017B860001628700016388000164890001658A0001668B0001678C0001688D0001698E00016A8F00016B9000016C9100016D9200016E9300016F9400017095000171960001729700017398000174990001999B0001769C0001779D0001789E0001799F00017AA00001A14800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9A0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A10001A5650001A3660001A4670001A7A20001004C00018D680001A9940B0170690001AC97030173610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186940A01707B00013297020173620001D7770001BD780001BE7900013100000000860B0162 
	+ 0x8E05016A860701628803016486050162860901628A0B01668A090166880501648A050166A00301A18A0701668C0301689003016C8E03016A9103016D980701749303016F9305016F94030170940701709409017094050170630001F79B0701769105016D980501749B0501769B030176A00501A1A00701A100000002860A01628E04016A860601628802016486040162860801628A0A01668A080166880401648A040166A00201A18A0601668C0201689002016C8E02016A9102016D980601749302016F9304016F94020170940601709408017094040170640001A29B0601769104016D980401749B0401769B020176A00401A1A00601A15200019EFF030000 
	+ 0x860101628A010166FF030000860301628A030166FF0300009801017498010174FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000A0000000FF000000931AB20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 153
go
--  'NOCASE.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 154, 10, 0, N'nocase_1257', N'Case insensitive dictionary order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3C00A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x860002628700026388000264890002658A0002668B0002678C0002688D0002698E00026A8F00026B9000026C9100026D9200026E9300026F9400027095000271960002729700027398000274990002999B0002769C0002779D0002789E0002799F00027AA00002A14200015C4300015D4400015E4500015F460001604700017B860001628700016388000164890001658A0001668B0001678C0001688D0001698E00016A8F00016B9000016C9100016D9200016E9300016F9400017095000171960001729700017398000174990001999B0001769C0001779D0001789E0001799F00017AA00001A14800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9A0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A10001A5650001A3660001A4670001A7A20001004C00018D680001A994050270690001AC97010273610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186940501707B00013297010173620001D7770001BD780001BE790001310000000086050262 
	+ 0x8E02026A860302628801026486020262860402628A0502668A040266880202648A020266A00102A18A0302668C0102689001026C8E01026A9101026D980402749301026F9302026F94010270940302709404027094020270630001F79B0302769102026D980302749B0202769B010276A00202A1A00302A100000002860501628E02016A860301628801016486020162860401628A0501668A040166880201648A020166A00101A18A0301668C0101689001016C8E01016A9101016D980401749301016F9302016F94010170940301709404017094020170640001A29B0301769102016D980301749B0201769B010176A00201A1A00301A15200019EFF030000 
	+ 0x860101628A010166FF030000860102628A010266FF0300009801017498010174FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C8000000FF000000A326B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 154
go
--  'ETIDIC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 155, 10, 0, N'etidic', N'Estonian case sensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3200A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x860201628701016388010164890101658A0201668B0101678C0101688D0101698E01016A8F01016B9001016C9101016D9201016E9301016F94010170950101719601017297010173980301F09C0101999E0101769F0101F59F0301F5A4010179A50101A19A0101FE4200015C4300015D4400015E4500015F460001604700017B860001628700016388000164890001658A0001668B0001678C0001688D0001698E00016A8F00016B9000016C9100016D9200016E9300016F94000170950001719600017297000173980001F09C0001999E0001769F0001F59F0201F5A4000179A50001A19A0001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9D0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A60001A5650001A3660001A4670001A7A70001004C00018D680001A994070170690001AC97030173610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186940601707B00013297020173620001D7770001BD780001BE790001310000000086090162 
	+ 0x8E05016A8605016288030164A10101F6860701628A0B01668A090166880501648A0501669A0301FE8A0701668C0301689003016C8E03016A9103016D9901017A9303016F9305016F9403017094050170A00101E4A20101FC630001F79E0501769105016D980501F09E030176A30101789A0501FE9B01017400000002860801628E04016A8604016288020164A10001F6860601628A0A01668A080166880401648A0401669A0201FE8A0601668C0201689002016C8E02016A9102016D9900017A9302016F9304016F9402017094040170A00001E4A20001FC640001A29E0401769104016D980401F09E020176A30001789A0401FE9B0001745200019EFF030000 
	+ 0x860101628A010166FF030000860301628A030166FF030000980101F0980101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000BE000000FF000000B420B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 155
go
--  'ETINC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 156, 10, 0, N'etinc', N'Estonian case insensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3C00A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x860002628700026388000264890002658A0002668B0002678C0002688D0002698E00026A8F00026B9000026C9100026D9200026E9300026F94000270950002719600027297000273980002F09C0002999E0002769F0002F59F0102F5A4000279A50002A19A0002FE4200015C4300015D4400015E4500015F460001604700017B860001628700016388000164890001658A0001668B0001678C0001688D0001698E00016A8F00016B9000016C9100016D9200016E9300016F94000170950001719600017297000173980001F09C0001999E0001769F0001F59F0101F5A4000179A50001A19A0001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9D0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A60001A5650001A3660001A4670001A7A70001004C00018D680001A994030270690001AC97010273610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186940301707B00013297010173620001D7770001BD780001BE790001310000000086040262 
	+ 0x8E02026A8602026288010264A10002F6860302628A0502668A040266880202648A0202669A0102FE8A0302668C0102689001026C8E01026A9101026D9900027A9301026F9302026F9401027094020270A00002E4A20002FC630001F79E0202769102026D980302F09E010276A30002789A0202FE9B00027400000002860401628E02016A8602016288010164A10001F6860301628A0501668A040166880201648A0201669A0101FE8A0301668C0101689001016C8E01016A9101016D9900017A9301016F9302016F9401017094020170A00001E4A20001FC640001A29E0201769102016D980301F09E010176A30001789A0201FE9B0001745200019EFF030000 
	+ 0x860101628A010166FF030000860102628A010266FF030000980101F0980101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C7000000FF0000007521B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 156
go
--  'LTHDIC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 159, 10, 0, N'lthdic', N'Lithuanian case sensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3200A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x8602016287010163880101E88A0101658B0201668C0101678D0101688E0101698F01016A9001016B9101016C9201016D9301016E9401016F95010170960101719701017298010173990301F09B0101999D0101769E0101779F010178A001017A8F03016AA10101FE4200015C4300015D4400015E4500015F460001604700017B8600016287000163880001E88A0001658B0001668C0001678D0001688E0001698F00016A9000016B9100016C9200016D9300016E9400016F95000170960001719700017298000173990001F09B0001999D0001769E0001779F000178A000017A8F02016AA10001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9C0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A30001A5650001A3660001A4670001A7A40001004C00018D680001A9950B0170690001AC98030173610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186950A01707B00013298020173620001D7770001BD780001BE7900013100000000860B0162 
	+ 0x8F07016A86070162880301E886050162860901628B0B01668B090166890101648B050166A10301FE8B0701668D0301689103016C8F05016A9203016D9A0101749403016F9405016F95030170950701709509017095050170630001F79D0701769205016D990501F09D0501769D030176A10501FEA20101A100000002860A01628F06016A86060162880201E886040162860801628B0A01668B080166890001648B040166A10201FE8B0601668D0201689102016C8F04016A9202016D9A0001749402016F9404016F95020170950601709508017095040170640001A29D0601769204016D990401F09D0401769D020176A10401FEA20001A15200019EFF030000 
	+ 0x860101628B010166FF030000860301628B030166FF030000990101F0990101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000DD000000FF0000004D22B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 159
go
--  'LTHNC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 160, 10, 0, N'lthnc', N'Lithuanian case insensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3C00A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x8600026287000263880002E88A0002658B0002668C0002678D0002688E0002698F00026A9000026B9100026C9200026D9300026E9400026F95000270960002719700027298000273990002F09B0002999D0002769E0002779F000278A000027A8F01026AA10002FE4200015C4300015D4400015E4500015F460001604700017B8600016287000163880001E88A0001658B0001668C0001678D0001688E0001698F00016A9000016B9100016C9200016D9300016E9400016F95000170960001719700017298000173990001F09B0001999D0001769E0001779F000178A000017A8F01016AA10001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019A9C0001752300019C5B00012B2400019F4E0001B453000191250001272C000109A30001A5650001A3660001A4670001A7A40001004C00018D680001A995050270690001AC98010273610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186950501707B00013298010173620001D7770001BD780001BE790001310000000086050262 
	+ 0x8F03026A86030262880102E886020262860402628B0502668B040266890002648B020266A10102FE8B0302668D0102689101026C8F02026A9201026D9A0002749401026F9402026F95010270950302709504027095020270630001F79D0302769202026D990302F09D0202769D010276A10202FEA20002A100000002860501628F03016A86030162880101E886020162860401628B0501668B040166890001648B020166A10101FE8B0301668D0101689101016C8F02016A9201016D9A0001749401016F9402016F95010170950301709504017095020170640001A29D0301769202016D990301F09D0201769D010176A10201FEA20001A15200019EFF030000 
	+ 0x860101628B010166FF030000860102628B010266FF030000990101F0990101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000C2000000FF0000000D23B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 

where id = 160
go
--  'LVIDIC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 157, 10, 0, N'lvidic', N'Latvian case sensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3200A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x8602016287010163880101E88A0101658B0201668C0101678D0101EC8F0101699001016A9101016B920101ED940101EF9601016E970101F2990101709A0101719B0101729C0101739D0301F09F010199A1010176A2010177A3010178A4010179A501017AA60101FE4200015C4300015D4400015E4500015F460001604700017B8600016287000163880001E88A0001658B0001668C0001678D0001EC8F0001699000016A9100016B920001ED940001EF9600016E970001F2990001709A0001719B0001729C0001739D0001F09F000199A1000176A2000177A3000178A4000179A500017AA60001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019AA00001752300019C5B00012B2400019F4E0001B453000191250001272C000109A80001A5650001A3660001A4670001A7A90001004C00018D680001A9990B0170690001AC9C030173610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186990A01707B0001329C020173620001D7770001BD780001BE7900013100000000860B0162 
	+ 0x9005016A86070162880301E886050162860901628B0B01668B090166890101648B050166A60301FE8B0701668E0101689301016C9003016A9501016D9E010174970301F29801016F99030170990701709909017099050170630001F7A1070176940301EF9D0501F0A1050176A1030176A60501FEA70101A100000002860A01629004016A86060162880201E886040162860801628B0A01668B080166890001648B040166A60201FE8B0601668E0001689300016C9002016A9500016D9E000174970201F29800016F99020170990601709908017099040170640001A2A1060176940201EF9D0401F0A1040176A1020176A60401FEA70001A15200019EFF030000 
	+ 0x860101628B010166FF030000860301628B030166FF0300009D0101F09D0101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000B1000000FF000000E623B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 157
go
--  'LVINC.257'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 158, 10, 0, N'lvinc', N'Latvian case insensitive dictionary sort order for use with code page 1257', 0x0000)
update syscharsets set binarydefinition = 0x3C00A50300000101 + 0x010001020200010303000104040001050500010606000107070001080800010E2D00010A2E00010B2F00010C3000010D310001210900010F0A0001100B0001110C0001120D0001130E0001140F000115100001161100011712000118130001191400011A1500011B1600011C1700011D1800011E1900011F1A00017F2B0001A03200012233000123340001243500012536000126370001282600012D380001293900012A3A00012C5C00013C3B00012E270001AD3C00012F3D00013A760001BC7A0001B97C0001B27E0001B38000013581000136820001378300013884000139850001613E00013B3F00013F5D00013D5E00013E5F0001B1400001404100015B 
	+ 0x8600026287000263880002E88A0002658B0002668C0002678D0002EC8F0002699000026A9100026B920002ED940002EF9600026E970002F2990002709A0002719B0002729C0002739D0002F09F000299A1000276A2000277A3000278A4000279A500027AA60002FE4200015C4300015D4400015E4500015F460001604700017B8600016287000163880001E88A0001658B0001668C0001678D0001EC8F0001699000016A9100016B920001ED940001EF9600016E970001F2990001709A0001719B0001729C0001739D0001F09F000199A1000176A2000177A3000178A4000179A500017AA60001FE4800017C4900017D4A00017E4B0001A61B00018175000130 
	+ 0x1C000183560001931D0001885900018B7300018970000187710001951E00018A740001801F00018C5A00019B200001904D00019D510001FF5000018E210001985400019255000182570001945800018472000185290001972A0001202200019AA00001752300019C5B00012B2400019F4E0001B453000191250001272C000109A80001A5650001A3660001A4670001A7A90001004C00018D680001A999050270690001AC9C010273610001BB6A0001AE280001966B0001B0000000016C0001B5600001AB7D0001337F0001344F00018F6D0001B66E0001B76F000186990501707B0001329C010173620001D7770001BD780001BE790001310000000086050262 
	+ 0x9002026A86030262880102E886020262860402628B0502668B040266890002648B020266A60102FE8B0302668E0002689300026C9001026A9500026D9E000274970102F29800026F99010270990302709904027099020270630001F7A1030276940102EF9D0302F0A1020276A1010276A60202FEA70002A100000002860501629002016A86030162880101E886020162860401628B0501668B040166890001648B020166A60101FE8B0301668E0001689300016C9001016A9500016D9E000174970101F29800016F99010170990301709904017099020170640001A2A1030176940101EF9D0301F0A1020176A1010176A60201FEA70001A15200019EFF030000 
	+ 0x860101628B010166FF030000860102628B010266FF0300009D0101F09D0101F0FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF0000000000000060E3F777FF1C14009C2DEB7718000000FF0014000000000000000000FF4AF6774805240050060000FFE3F7773819140000000000FF0200000000140000000000FF00000000000000 
	+ 0x00000000FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000FF000000B805240000002400FF0000004000000000000000FF000000000000008CF31200FF4BF67700002400B8052400FF4CF6774805240000000000FFFA1200C0052400B4F31200FFF412004901000070F41200FF91F6770000240000000000FF0524001A8CF677C0052400FF87F87700F0FD7F00000000FF00340002000000A5000000FF000000A724B20200000000FF22EDFB0100000000000000FF69F47702000000E0052400FF060000B069F477A05AFA77FF0000000000000000000000FF0000000000000000000000FF0000000000000000000000 
	+ 0xFF000000000000004EC1F677FF55FA774FC1F67700000000FFFFFFFF9C2DEB7700001400FF1C140028001400081C1400FFBEF8772800000000000000FF001400180000007B000000FF030000D4F2120028001400FFF412009C2DEB7718000000FF001400F0F41200F0F41200FF9DF877579EF87748051400FF2DEB770000000000001400FF0200000000140018F51200FFF51200E79DF877579EF877FF0514000000000000000000FF001400BBF81200FF000000FF00000004F51200081C1400FF0014008CF41200E6133000FFF6120094FEF87740B7F977FFFFFFFFA8F612000ED7F777FF0014000101071018000000FF2DEB770000000000001400 
where id = 158
go



-- These are sort orders for which SQL Server uses NT NLS sorting.  Encode their
-- LCIDs in master..syscharsets.binarydefinition.

--  'CP932BIN.932'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 192, 109, 0, N'bin_cp932', N'Binary Sort Order for the Code Page 932 Character Set',
	0x0000)

--  'CP932NLS.932'.
-- Sort order 193: LCID = 0x411, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 193, 109, 0, N'nls_cp932', N'NLS Case-Insensitive Sort Order for the Code Page 932 Character Set',
	0x800000001104000001000300)

--  'cp932cls.932'.
-- Sort order 200: LCID = 0x411, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 200, 109, 0, N'nls_cp932_cs', N'NLS Case-Sensitive Sort Order for the Code Page 932 Character Set',
	0x800000001104000000000300)

--  'CP936BIN.936'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 198, 14, 0, N'bin_cp936', N'Binary Sort Order for the Code Page 936 Character Set',
	0x0000)

--  'CP936NLS.936'.
-- Sort order 199: LCID = 0x804, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 199, 14, 0, N'nls_cp936', N'NLS Case-Insensitive Sort Order for the Code Page 936 Character Set',
	0x800000000408000001000300)

--  'cp936cls.936'.
-- Sort order 203: LCID = 0x804, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 203, 14, 0, N'nls_cp936_cs', N'NLS Case-Sensitive Sort Order for the Code Page 936 Character Set',
	0x800000000408000000000300)

--  'CP949BIN.949'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 194, 12, 0, N'bin_cp949', N'Binary Sort Order for the Code Page 949 Character Set',
	0x0000)

--  'CP949NLS.949'.
-- Sort order 195: LCID = 0x412, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 195, 12, 0, N'nls_cp949', N'NLS Case-Insensitive Sort Order for the Code Page 949 Character Set',
	0x800000001204000001000300)

--  'cp949cls.949'.
-- Sort order 201: LCID = 0x412, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 201, 12, 0, N'nls_cp949_cs', N'NLS Case-Sensitive Sort Order for the Code Page 949 Character Set',
	0x800000001204000000000300)

--  'CP950BIN.950'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 196, 13, 0, N'bin_cp950', N'Binary Sort Order for the Code Page 950 Character Set',
	0x0000)

--  'CP950NLS.950'.
-- Sort order 197: LCID = 0x404, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 197, 13, 0, N'nls_cp950', N'NLS Case-Insensitive Sort Order for the Code Page 950 Character Set',
	0x800000000404000001000300)

--  'cp950cls.950'.
-- Sort order 202: LCID = 0x404, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 202, 13, 0, N'nls_cp950_cs', N'NLS Case-Sensitive Sort Order for the Code Page 950 Character Set',
	0x800000000404000000000300)
go

--  'CP874BIN.874'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 204, 15, 0, N'bin_cp874', N'Binary Sort Order for the Code Page 874 Character Set',
	0x0000)

--  'CP874NLS.874'.
-- Sort order 205: LCID = 0x41e, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 205, 15, 0, N'nls_cp874', N'NLS Case-Insensitive Sort Order for the Code Page 874 Character Set',
	0x800000001e04000001000300)

--  'cp874cls.874'.
-- Sort order 206: LCID = 0x41e, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 206, 15, 0, N'nls_cp874_cs', N'NLS Case-Sensitive Sort Order for the Code Page 874 Character Set',
	0x800000001e04000000000300)
go

-- Code Page 874 (Thai)
INSERT syscharsets VALUES ( 1001,  15, 0, 0, N'cp874', N'Code page 874', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100
	+	0x00010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101ffff0101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 15
go

-- Code Page 1258 (Vietnamese)
INSERT syscharsets VALUES ( 1001,  16, 0, 0, N'cp1258', N'Code page 1258', 0x000008000101080002020800030308000404080005050800060608000707080008080800090908080a0a08080b0b08080c0c08080d0d08080e0e08080f0f0808101008001111080012120800131308001414080015150800161608001717080018180800191908001a1a08001b1b08001c1c08001d1d08001e1e08001f1f0800202008082121202022221020232301202424402025252020262620202727102028282020292920202a2a20202b2b20202c2c20202d2d20202e2e20202f2f2020303002143131021432320214333302143434021435350214363602143737021438380214393902143a3a20203b3b20203c3c20203d3d20203e3e20203f3f2020404001204161011142620111436301114464011145650111466601114767010148680101496901014a6a01014b6b01014c6c01014d6d01014e6e01014f6f0101507001015171010152720101537301015474010155750101567601015777010158780101597901015a7a01)
UPDATE syscharsets set binarydefinition = binarydefinition
	+	0x015b5b20205c5c40205d5d20205e5e20205f5f0120606020204161011242620112436301124464011245650112466601124767010248680102496901024a6a01024b6b01024c6c01024d6d01024e6e01024f6f0102507001025171010252720102537301025474010255750102567601025777010258780102597901025a7a01027b7b20207c7c20207d7d20207e7e20207f7f2000808001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100
	+	0x00010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101000001010000010100000101ffff0101000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000030313233303132303032323435353031323632333031
	+	0x303230320000000000003031323330313230303232343535303132363233303130323032000000000000000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
WHERE id = 16
go

--  New sort orders for code page 1252 that uses NT extended sorting
-- Sort order 71: LCID = 0x409, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 71, 1, 0, N'dictionary_1252', N'NT Latin-1 case sensitive sort order for code page 1252',
	0x800000000904000000000300)
go

-- Sort order 72: LCID = 0x409, NORM_IGNOREKANATYPE | NORM_IGNOREWIDTH | NORM_IGNORECASE
insert syscharsets (type, id, csid, status, name, description, binarydefinition)
	values (2001, 72, 1, 0, N'nocase_1252', N'NT Latin-1 case insensitive sort order for code page 1252',
	0x800000000904000001000300)
go

-- EBCDIC sort orders

checkpoint
go

--  'EBCDIC 037'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 210, 1, 0, N'EBCDIC_037', N'EBCDIC 037 code page sort - United States/Canada', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a05b000124800001d87c0001405c00012a6d00015f510001e97e00013d4e00012b5e00013b5d0001294f00017c6c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001235f0001ac4d0001287f0001226f
	+ 0x00013f700001f87d000127c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b2bb00015de10001f7bc0001af290001836e00013e7a00013a82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1c100014150000126d100014a31000191200001
	+ 0x813b00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189b10001a32f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fa20001733800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bf4b00012eb20001a5a00001b5b30001b76b00012cb60001b6be0001b4b50001a79b0001ba8b0001bb60000197cb0001f4b0000188bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a1000198b70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba00015bac0001d0650001c1660001
	+ 0xc3630001c4670001c5640001c0680001c79f0001a4690001d1750001cd720001ca730001cb740001c879000160760001ce770001cf780001ccad0001dd6a0001a6ee0001d3ef0001d5ec0001d6f0000130ed0001d2c000017b81000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a000121450001e1460001e3430001e4470001e5440001e0480001e79d000182490001f1550001ed520001ea530001eb540001e8590001df560001ee570001ef580001ec8d0001fd4a0001a2ce0001f3cf0001f5cc0001f6d000017dcd0001f2e2000153710001c9de0001fadf0001ffdc0001fcdd0001f98e0001fe8f0001b1e000015cff068e000818000000000000fff8120080f81200
	+ 0x80f81200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff
	+ 0x018e00881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c00
	+ 0x5300ff007300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 210
go

--  'EBCDIC 273'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 211, 1, 0, N'EBCDIC_273', N'EBCDIC 273 code page sort - Austria/Germany F.R.', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a050000126800001d87c0001a75c00012a6d00015f510001e97e00013d4e00012b5e00013b5d0001294f0001216c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001235f0001884d0001287f0001226f
	+ 0x00013f700001f8b60001b6c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b2640001c0ed0001d2fd0001d9290001836e00013e7a00013a82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1440001e0bc0001afdd0001f93100019120000181
	+ 0x3b00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189600001972f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012f5a0001dc3800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001a3b20001a5a00001b5b30001b7cd0001f27d000127be0001b4b50001409b0001ba8b0001bbbb00017ccb0001f4b00001a2bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a10001dfb70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba0001acac0001d0650001c1660001c363
	+ 0x00015b670001c54b00012e680001c79f0001a4690001d1750001cd720001ca730001cb740001c879000160760001ce770001cf780001ccad0001dd6a0001f6ee0001d3ef0001d5ec00015cf0000130e10001f7c00001e481000161fe0001daff000100fc00015d5b000124ae0001deaf0001aea2000173450001e1460001e34300017b470001e5c1000141480001e79d000182490001f1550001ed520001ea530001eb540001e859000198560001ee570001ef580001ec8d0001fd4a0001c4ce0001f3cf0001f5cc0001a6d00001fc6b00012ce2000153710001c9de0001fadf0001ffdc00017dd100014a8e0001fe8f0001b1e00001d6ff068e000818000000000000fff8120080f8120080f8
	+ 0x1200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff018e
	+ 0x00881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c005300
	+ 0xff007300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 211
go

--  'EBCDIC 277'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 212, 1, 0, N'EBCDIC_277', N'EBCDIC 277 code page sort - Denmark/Norway', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a050000126800001404b00012e680001c76d00015f510001e97e00013d4e00012b5e00013b5d0001294f0001216c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001c65f0001884d0001287f0001226f
	+ 0x00013f700001a681000161c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b29f00015de10001f7a00001b5290001836e00013e7a00013a82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a19d000182bc0001af480001e731000191200001
	+ 0x813b00018a2100018d9e00015b2a0001842b0001852c0001862d0001872e000189600001972f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fdd0001f93800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001a3b20001a55b0001c5b30001b7710001c9b60001b6be0001b4b50001a79b0001ba8b0001bbbb00017ccb0001f4b00001a2bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a10001fcb70001bcb40001a93a000180db0001fb9c00017b8c0001f0b80001bdb90001beba0001acac0001d0650001c16600
	+ 0x01c3630001c467000124640001c05c00012a7c0001d8690001d1750001cd720001ca730001cb740001c879000160760001ce770001cf780001ccad0001dd6a0001f8ee0001d3ef0001d5ec0001d6f0000130ed0001d2c00001e67d000127fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a0001a4450001e1460001e3430001e44700017d440001e0d100014ac1000141490001f1550001ed520001ea530001eb540001e8590001df560001ee570001ef580001ec8d0001fd4a000123ce0001f3cf0001f5cc0001f6d00001e5cd0001f2e20001536b00012cde0001fadf0001ffdc000198a20001738e0001fe8f0001b1e000015cff068e000818000000000000fff8120080f812
	+ 0x0080f81200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e9
	+ 0x77ff018e00881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054
	+ 0x005c005300ff007300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 212
go

--  'EBCDIC 278'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 213, 1, 0, N'EBCDIC_278', N'EBCDIC 278 code page sort - Finland/Sweden', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a050000126800001d8640001c0680001c76d00015f510001607e00013d4e00012b5e00013b5d0001294f0001216c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001c45f0001884d0001287f0001226f
	+ 0x00013f700001f8ed0001d2c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b2b60001b6720001caa00001b5290001836e00013e520001ea82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1440001e0bc0001af480001e731000191200001
	+ 0x813b00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189600001972f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fdd0001f93800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001a3b20001a55b0001c5b30001b7cd0001f24b00012ebe0001b4b500015b9b0001ba8b0001bbbb00017ccb0001f4b00001a2bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a10001fcb70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba0001acac0001d0650001c1660001
	+ 0xc363000123670001247c0001d65c00012a9f00015d690001d1750001cde10001f7730001cb740001c8790001e9760001ce770001cf780001ccad0001dd6a0001f6ee0001d3ef0001d5ec000140f00001307d000127c00001e481000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a0001a4450001e1460001e34300017b4700017dc1000141d100014a9d000182490001f1550001ed7a00013a530001eb540001e8590001df560001ee570001ef580001ec8d0001fd4a0001a7ce0001f3cf0001f5cc0001a6d00001e56b00012ce20001537100015cde0001fadf0001ffdc000198a20001738e0001fe8f0001b1e00001c9ff068e000818000000000000fff8120080f812
	+ 0x0080f81200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e9
	+ 0x77ff018e00881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054
	+ 0x005c005300ff007300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 213
go

--  'EBCDIC 280'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 214, 1, 0, N'EBCDIC_280', N'EBCDIC 280 code page sort - Italy', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a050000126800001d8b20001a55c00012a6d00015f5100015d7e00013d4e00012b5e00013b5d0001294f0001216c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001a35f0001884d0001287f0001226f
	+ 0x00013f700001f8b60001b6c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b29100016a490001f1520001ea290001836e00013ede0001fa82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1450001e1bc0001af550001ed31000191200001
	+ 0x813b00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189600001972f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012f590001df3800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001237c0001a7a00001b5b30001b7ce0001f37d000127be0001b4b50001409b0001ba8b0001bbbb00017ccb0001f4b00001a2bd0001a84b00012e9000015beb0001d4fb0001dbbf0001d7a10001ecb70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba0001acac0001d0650001c1660001
	+ 0xc3630001c4670001c5640001c0680001c79f0001a4690001d1750001cd720001ca730001cb740001c8790001f9760001ce770001cf780001ccad0001dd6a0001f2ee0001d3ef0001d5ec0001d6f0000130ed0001d2c00001e081000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a0001e9c1000141460001e3430001e4470001e54400017b4800015c9d000182e10001f7d100014a5b000124530001eb5400017da2000173560001ee570001ef580001988d0001fd4a0001b06b00012ccf0001f5cc0001f6d00001e8cd0001a6e2000153710001c97a00013adf0001ffdc0001fcdd0001608e0001fe8f0001b1e00001e7ff068e000818000000000000fff8120080f81200
	+ 0x80f81200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff
	+ 0x018e00881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c00
	+ 0x5300ff007300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 214
go

--  'EBCDIC 284'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 215, 1, 0, N'EBCDIC_284', N'EBCDIC 284 code page sort - Latin American/Spain', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a0bc0001af800001d86a0001f15c00012a6d00015f510001e97e00013d4e00012b5e00013b5d0001294f00017c6c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001d15f0001ac4d0001287f0001226f00
	+ 0x013f700001f87d000127c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b24b00012ee10001f75b000124290001836e00013e7a00013a82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1c100014150000126d100014a3100019120000181
	+ 0x3b00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189bb0001212f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fa20001733800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001a3b20001a5a00001b5b30001b74a00015bb60001b6be0001b4b50001a79b0001ba8b0001bb60000197cb0001f4b00001a2bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a1000198b70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba000188ac0001d0650001c1660001c3
	+ 0x630001c4670001c5640001c0680001c79f0001a469000123750001cd720001ca730001cb740001c879000160760001ce770001cf780001ccad0001dd7c000140ee0001d3ef0001d5ec0001d6f0000130ed0001d2c000017b81000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a00015d450001e1460001e3430001e4470001e5440001e0480001e79d000182490001a6550001ed520001ea530001eb540001e8590001df560001ee570001ef580001ec8d0001fd6b00012cce0001f3cf0001f5cc0001f6d000017dcd0001f2e2000153710001c9de0001fadf0001ffdc0001fcdd0001f98e0001fe8f0001b1e000015cff068e000818000000000000fff8120080f8120080f8
	+ 0x1200ff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff018e00
	+ 0x881e8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c005300ff00
	+ 0x7300740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 215
go

--  'EBCDIC 285'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 216, 1, 0, N'EBCDIC_285', N'EBCDIC 285 code page sort - United Kingdom', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a05b0001a3800001d87c0001404b00012e6d00015f510001e97e00013d4e00012b5e00013b5d0001294f00017c6c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001235f0001ac4d0001287f0001226f00
	+ 0x013f700001f87d000127c2000142c3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b2b20001a5e10001f7bc0001af290001836e00013e7a00013a82000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1c100014150000126d100014a31000191200001813b
	+ 0x00018a2100018d9e0001c62a0001842b0001852c0001862d0001872e000189bb00015d2f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fa20001733800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb100015b5c00012aa00001b5b30001b76b00012cb60001b6be0001b4b50001a79b0001ba8b0001bb60000197cb0001f4b00001a2bd0001a89100016a900001b0eb0001d4fb0001dbbf0001d7a1000198b70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba000188ac0001d0650001c1660001c36300
	+ 0x01c4670001c5640001c0680001c79f0001a4690001d1750001cd720001ca730001cb740001c879000160760001ce770001cf780001ccad0001dd6a0001a6ee0001d3ef0001d5ec0001d6f0000130ed0001d2c000017b81000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a000121450001e1460001e3430001e4470001e5440001e0480001e79d000182490001f1550001ed520001ea530001eb540001e8590001df560001ee570001ef580001ec8d0001fd4a000124ce0001f3cf0001f5cc0001f6d000017dcd0001f2e2000153710001c9de0001fadf0001ffdc0001fcdd0001f98e0001fe8f0001b1e000015cff068e000818000000000000fff8120080f8120080f81200
	+ 0xff78fb77700ff977fffffffffff812001795fc774795fc77ff1e8e002000000000100010ff00010000000000b8f71200ff4ee87764000000dcf71200ff721675c84d130078018e00ff018e0078018e0078018e00ffe50310084f130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f9770000130038381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff0813004038130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff018e00881e
	+ 0x8e00901e8e00ff010000010000006cf91200fff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130078381300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200402b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c005300ff007300
	+ 0x740065006d003300ff005c006e00740064006c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 216
go

--  'EBCDIC 297'.
insert syscharsets (type, id, csid, status, name, description, binarydefinition)			
	values  (2001, 217, 1, 0, N'EBCDIC_297', N'EBCDIC 297 code page sort - France', 0x0000)
update syscharsets set binarydefinition = 
	0x0100da000000010101000102020001030300010404000105050001060600010707000108080001090900010a0a00010b0b00010c0c00010d0d00010e0e00010f0f0001101000011111000112120001131300011414000115150001161600011717000118180001191900011a1a00011b1b00011c1c00011d1d00011e1e00011f1f00017f410001a050000126800001d8b20001a55c00012a6d00015f5100017b7e00013d4e00012b5e00013b5d0001294f0001216c0001252800015e4c00013c620001c2f1000131f2000132f3000133f4000134f5000135f6000136f7000137f8000138f9000139fa0001b37b0001a35f0001884d0001287f0001226f00013f700001f8450001e1c2000142
	+ 0xc3000143c4000144c5000145c6000146c7000147c8000148c9000149ca0001add200014bd300014cd400014dd500014ed600014fd7000150d8000151d9000152da0001b9e3000154e4000155e5000156e6000157e7000158e8000159e900015aea0001b29100016a490001f1b60001b6290001836e00013ea10001a882000162830001638400016485000165860001668700016788000168890001698a0001ab9200016b9300016c9400016d9500016e9600016f9700017098000171990001729a0001aaa3000174a4000175a5000176a6000177a7000178a8000179a900017aaa0001a1520001eabc0001af550001ed31000191200001813b00018a2100018d9e0001c62a000184
	+ 0x2b0001852c0001862d0001872e000189600001972f00018b3c00019a3000017e3e00019c2200018e2300018f240001902500019d3200019233000193340001943500019536000196370001996100012fbe0001b43800019b3d00018c390001b83f00019f2600019e2700012d40000120420001e2ab0001bfb10001237c0001e0a0000160b30001b7de0001fa5b000124a2000173b500015d9b0001ba8b0001bbbb00017ccb0001f4b00001a2bd0001984b00012e9000015beb0001d4fb0001dbbf0001d77a00013ab70001bcb40001a93a000180db0001fb9c0001e68c0001f0b80001bdb90001beba0001acac0001d0650001c1660001c3630001c4670001c5640001c0680001c79f00
	+ 0x01a4690001d1750001cd720001ca730001cb740001c8790001b5760001ce770001cf780001ccad0001dd6a0001f9ee0001d3ef0001d5ec0001d6f0000130ed0001d2c00001e981000161fe0001daff000100fc0001dcfd0001d9ae0001deaf0001ae5a0001a77d000127460001e3430001e4470001e5440001404800015c9d000182e10001f7d100014ac1000141530001eb5400017d590001df560001ee570001ef580001ec8d0001fd4a0001b0ce0001f3cf0001f5cc0001f6d00001e8cd0001f2e2000153710001c96b00012cdf0001ffdc0001fcdd0001a68e0001fe8f0001b1e00001e7ff068e000818000000000000fff8120080f8120080f81200ff78fb77700ff977fffffffffff812001795fc77
	+ 0x4795fc77ff1e8e002000000000100010ff000100000000000d000000ff000000d8041300dcf71200ff721675e84e130078018e00ff018e0078018e0078018e00ffe503102850130020d3fc77ffd3fc770010000000000100ffeffd7f00f0070000000000ff0000000000000000000000ff000000000000005a620200ff1000008a7f000001000000ff7f00000000010000000100fffffe7f03000000026f1675ff0000005cf8120000000000ff42f97700001300e0381300ff00000038f8120088061300fff81200e378fb772842f977fffffffffcf81200599dfc77ff081300e838130000100010ff2b130000000000001af977ff000000901f8e00b2d0e977ff018e00881e8e00901e8e00ff010000010000006cf91200
	+ 0xfff912006cf91200e378fb77ff00000000100010f0f81200fff812000011001007000000ff1e8e000000130008000000ff000000901e8e00e9431275fff812004fe5f8776cf81200fff912006cf91200e378fb77ff0ff977ffffffff7cf91200ff45f8770000130000000000ff381300462df877301f1300fff0fd7fe41e130020391300ffe0fd7f0900000000e0fd7fffe0fd7f54f9120000f0fd7fff00f877010000000e000000fff91200802b130000100010ff381300000000000000f877ff0000000cfd12000cfd1200fffd1200e378fb77a87bf877ffffffff98fc12009551f877fffd120000e0fd7f00f0fd7fff00000043003a005c005700ff004e004e0054005c005300ff007300740065006d003300ff005c006e0074006400
	+ 0x6c00ff002e0064006c006c000000ff0049006d00610067006500ff00460069006c0065002000ff0078006500630075007400ff006f006e0020004f007000ff0069006f006e0073005c00ff0048004100520053004500ff002e004500580045000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000ff0000000000000000000000
where id = 217
go

-------------------------------------------------------------------------------
---------------- Computed Columns for backward-compat (MASTER) ----------------
-------------------------------------------------------------------------------

ALTER TABLE sysdatabases ADD
    version AS convert(smallint, databaseproperty (name, 'version'))
go

ALTER TABLE sysservers ADD
    srvnetname AS convert(char(30),
            CASE WHEN srvproduct = N'SQL Server' THEN datasource ELSE NULL END),
	isremote AS convert(bit,(srvstatus & 32)/32),
	rpc AS convert(bit,(srvstatus & 1)),
	pub AS convert(bit,(srvstatus & 2)/2),
	sub AS convert(bit,(srvstatus & 4)/4),
	dist AS convert(bit,(srvstatus & 8)/8),
	dpub AS convert(bit,(srvstatus & 16)/16),
	rpcout AS convert(bit,(srvstatus & 64)/64),
	dataaccess AS convert(bit,(srvstatus & 128)/128),
	collationcompatible AS convert(bit,(srvstatus & 256)/256),
	system AS convert(bit,(srvstatus & 512)/512),
	useremotecollation AS convert(bit,(srvstatus & 1024)/1024),
	lazyschemavalidation AS convert(bit,(srvstatus & 2048)/2048),
	collation AS convert(sysname, CollationPropertyFromID(srvcollation, 'name'))
go

GRANT SELECT ON sysservers TO PUBLIC
go

DENY SELECT (providerstring) ON sysservers TO PUBLIC
go


ALTER TABLE sysxlogins ADD
    isrpcinmap AS convert(smallint, CASE WHEN (xstatus&32)=32 THEN 1 ELSE 0 END),
    ishqoutmap AS convert(smallint, CASE WHEN (xstatus&64)=64 THEN 1 ELSE 0 END),
    selfoutmap AS convert(smallint, CASE WHEN (xstatus&192)=192 THEN 1 ELSE 0 END)
go

ALTER TABLE syscharsets ADD
    definition AS convert(image, binarydefinition)
go


-------------------------------------------------------------------------------
--------------------------- System views for MASTER ---------------------------
-------------------------------------------------------------------------------

IF OBJECT_ID('syslogins') IS NOT NULL
	DROP VIEW syslogins
go
CREATE VIEW syslogins AS SELECT
    sid = convert(varbinary(85), sid),
    status = convert(smallint, 8 +
            CASE WHEN (xstatus & 2)=0 THEN 1 ELSE 2 END),
	createdate = convert(datetime, xdate1),
	updatedate = convert(datetime, xdate2),
    accdate = convert(datetime, xdate1),
    totcpu = convert(int, 0),
    totio = convert(int, 0),
    spacelimit = convert(int, 0),
    timelimit = convert(int, 0),
    resultlimit = convert(int, 0),
    name = convert(sysname, name),
    dbname = convert(sysname, db_name(dbid)),
    password = convert(sysname, password),
    language = convert(sysname, language),
    denylogin = convert(int, CASE WHEN (xstatus&1)=1 THEN 1 ELSE 0 END),
    hasaccess = convert(int, CASE WHEN (xstatus&2)=2 THEN 1 ELSE 0 END),
    isntname = convert(int, CASE WHEN (xstatus&4)=4 THEN 1 ELSE 0 END),
    isntgroup = convert(int, CASE WHEN (xstatus&12)=4 THEN 1 ELSE 0 END),
    isntuser = convert(int, CASE WHEN (xstatus&12)=12 THEN 1 ELSE 0 END),
    sysadmin = convert(int, CASE WHEN (xstatus&16)=16 THEN 1 ELSE 0 END),
    securityadmin = convert(int, CASE WHEN (xstatus&32)=32 THEN 1 ELSE 0 END),
    serveradmin = convert(int, CASE WHEN (xstatus&64)=64 THEN 1 ELSE 0 END),
    setupadmin = convert(int, CASE WHEN (xstatus&128)=128 THEN 1 ELSE 0 END),
    processadmin = convert(int, CASE WHEN (xstatus&256)=256 THEN 1 ELSE 0 END),
    diskadmin = convert(int, CASE WHEN (xstatus&512)=512 THEN 1 ELSE 0 END),
    dbcreator = convert(int, CASE WHEN (xstatus&1024)=1024 THEN 1 ELSE 0 END),
	bulkadmin = convert(int, CASE WHEN (xstatus&4096)=4096 THEN 1 ELSE 0 END),
    loginname = convert(sysname, name)
FROM sysxlogins WHERE srvid IS NULL
go
GRANT SELECT ON syslogins TO PUBLIC
DENY SELECT (password) ON syslogins TO PUBLIC
go

IF OBJECT_ID('sysremotelogins') IS NOT NULL
	DROP VIEW sysremotelogins
go
CREATE VIEW sysremotelogins AS SELECT
    remoteserverid = convert(smallint, srvid),
    remoteusername = convert(sysname, name),
    status = convert(smallint,
            CASE WHEN (xstatus & 16)!=0 THEN 1 ELSE 0 END),
    sid = convert(varbinary(85), sid),
	changedate = convert(datetime, xdate1)
FROM sysxlogins
WHERE srvid IS NOT NULL AND isrpcinmap = 1
go
GRANT SELECT ON sysremotelogins TO PUBLIC
go

IF OBJECT_ID('sysoledbusers') IS NOT NULL
	DROP VIEW sysoledbusers
go
CREATE VIEW sysoledbusers AS SELECT
    rmtsrvid = convert(smallint, srvid),
    rmtloginame = convert(sysname, name),
    rmtpassword = convert(sysname, password),
    loginsid = convert(varbinary(85), sid),
    status = convert(smallint,
            CASE WHEN xstatus&192=192 THEN 1 ELSE 0 END),
	changedate = convert(datetime, xdate1)
FROM sysxlogins
WHERE srvid IS NOT NULL AND ishqoutmap = 1
go
GRANT SELECT ON sysremotelogins TO PUBLIC
DENY SELECT (rmtpassword) ON sysoledbusers TO PUBLIC
go

IF OBJECT_ID('sysopentapes') IS NOT NULL
	DROP VIEW sysopentapes
go
CREATE VIEW sysopentapes AS SELECT * from OpenRowset (OpenTapes)
go
GRANT SELECT ON sysopentapes TO PUBLIC
go

checkpoint
go


-------------------------------------------------------------------------------
---------------------- CREATE PER-DATABASE FAKE TABLES ------------------------
-------------------------------------------------------------------------------

if object_id('sp_msupg_recreatecatalogfaketables') is not null
	drop proc sp_msupg_recreatecatalogfaketables
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_recreatecatalogfaketables
as
begin
	IF OBJECT_ID('sysindexkeys', 'S') IS NOT NULL
		DROP TABLE sysindexkeys
	CREATE TABLE sysindexkeys (
		id              int NOT NULL,
		indid           smallint NOT NULL,
		colid           smallint NOT NULL,
		keyno           smallint NOT NULL
		)
	GRANT SELECT ON sysindexkeys TO PUBLIC

	IF OBJECT_ID('sysforeignkeys', 'S') IS NOT NULL
		DROP TABLE sysforeignkeys
	CREATE TABLE sysforeignkeys (
		constid         int NOT NULL,
		fkeyid          int NOT NULL,
		rkeyid          int NOT NULL,
		fkey            smallint NOT NULL,
		rkey            smallint NOT NULL,
		keyno           smallint NOT NULL
		)
	GRANT SELECT ON sysforeignkeys TO PUBLIC

	IF OBJECT_ID('sysmembers', 'S') IS NOT NULL
		DROP TABLE sysmembers
	CREATE TABLE sysmembers (
		memberuid		smallint NOT NULL,
		groupuid		smallint NOT NULL
		)
	GRANT SELECT ON sysmembers TO PUBLIC

	IF OBJECT_ID('sysprotects', 'S') IS NOT NULL
		DROP TABLE sysprotects
	CREATE TABLE sysprotects (
		id              int NOT NULL,
		uid             smallint NOT NULL,
		action          tinyint NOT NULL,
		protecttype     tinyint NOT NULL,
		columns         varbinary(4000) NULL,
		grantor         smallint NOT NULL
		)
	GRANT SELECT ON sysprotects TO PUBLIC

	IF OBJECT_ID('sysfiles', 'S') IS NOT NULL
		DROP TABLE sysfiles
	CREATE TABLE sysfiles (
		fileid		smallint NOT NULL,
		groupid		smallint NOT NULL,
		size		int NOT NULL,
		maxsize		int NOT NULL,
		growth		int NOT NULL,
		status		int NOT NULL,
		perf		int NOT NULL,
		name		nchar (128) NOT NULL,
		filename	nchar (260) NOT NULL
		)
	GRANT SELECT ON sysfiles TO PUBLIC
end

go
deny execute on sp_msupg_recreatecatalogfaketables to public
go


-------------------------------------------------------------------------------
-------------------------- DATABASE UPGRADE STEPS -----------------------------
-------------------------------------------------------------------------------

if object_id('sp_msupg_dosystabcatalogupgrades') is not null
	drop proc sp_msupg_dosystabcatalogupgrades
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_dosystabcatalogupgrades
as
	-- 29Jan99 column level collations
	IF NOT EXISTS (SELECT * FROM syscolumns WHERE id = 3 AND colid = 20)
		BEGIN
		BEGIN TRAN
		ALTER TABLE dbo.syscolumns ADD collationid int null 
		EXEC (N'UPDATE dbo.syscolumns SET collationid = null WHERE xtype NOT IN (35, 99, 167, 175, 231, 239)')
		COMMIT TRAN
		END
	ELSE
		BEGIN
		UPDATE dbo.syscolumns SET name = N'collationid' WHERE id = 3 AND colid = 20 AND name = 'collation'
		END

	IF NOT EXISTS (SELECT * FROM syscolumns WHERE id = 4 AND colid = 12)
		BEGIN
		BEGIN TRAN
		ALTER TABLE dbo.systypes ADD collationid int null
		EXEC (N'UPDATE dbo.systypes SET collationid = null WHERE xtype NOT IN (35, 99, 167, 175, 231, 239)')
		COMMIT TRAN
		END
	ELSE
		BEGIN
		UPDATE dbo.syscolumns SET name = N'collationid' WHERE id = 4 AND colid = 12 and name = N'collation'
		END

	-- 18Jan99 Add sql_variant to systypes if not already present
	if not exists (select * from systypes where name = N'sql_variant')
	begin
		insert systypes (name, xtype, status, xusertype, length, xprec, xscale,
			tdefault, domain, uid, reserved)
			values (N'sql_variant', 98, 0, 98, 8016, 0, 0, 0, 0, 1, 0)
	end
	else
	begin
		update systypes set
				xtype = 98, status = 0, xusertype = 98, length = 8016,
				xprec = 0, xscale = 0, tdefault = 0, domain = 0, uid = 1,
				reserved = 0
			where name = N'sql_variant'
	end

	--20Jan99
	IF NOT EXISTS (SELECT * FROM systypes WHERE xtype = 127)
		INSERT systypes(name,xtype,status,xusertype,length,xprec,xscale,tdefault,domain,reserved,uid)
		VALUES ('bigint',127,0,127,8,19,0,0,0,0,1)
	ELSE
		update systypes set 
			name = 'bigint', status = 0, xusertype = 127, length = 8,
			xprec = 19, xscale = 0, tdefault = 0, domain = 0,
			reserved = 0,  uid = 1		
		where xtype = 127

	--20Jan99 change sysindexes rowcnt column type from binary to bigint
	UPDATE syscolumns SET xtype = 127,xusertype = 127, length = 8, xprec = 19, xscale = 0
	WHERE id = 2 AND colid = 12

	-- 18Feb99 Add sysfulltextnotify if not already present
	if object_id('sysfulltextnotify', 'S') is null
	begin
		BEGIN TRAN

		-- Upgrade from 7.0 requires rebuilding full-text catalogs.  Turn off sysdatabases
		-- flag indicating this database is enabled for full-text.
		update master.dbo.sysdatabases set status2 = status2 & ~536870912 where dbid = db_id()

		CREATE TABLE sysfulltextnotify (
				tableid			int NOT NULL,
				rowinfo			smallint NOT NULL,
				ftkey			varbinary(482) NOT NULL
				)
		CREATE UNIQUE CLUSTERED INDEX sysfulltextnotify
				ON sysfulltextnotify ( tableid, ftkey )	
		COMMIT TRAN
	end

	-- 2Apr99 Remap sysobjects::status bits
	if not exists (select id from sysobjects where id = 1 and (0x20000000 = status & 0x20000000))
	begin
		BEGIN TRAN
		-- remap A_N and Q_I for proc, func, view, trig, repl proc
		update sysobjects
		set status = (status & ~24)                                         | -- mask out old A_N, Q_I
					 (case (status & 16) when 16 then 536870912 else 0 end) | -- add new A_N if required
					 (case (status & 8) when 8 then 1073741824 else 0 end)  	 -- add new Q_I if required
		where xtype in ('P', 'FN', 'V', 'TR', 'RF')

		-- mask out TAB_ROWGUIDCOL, TAB_SELINTO for all tables
		-- set A_N and Q_I for all existing tables
		update sysobjects
		set status = (status & ~(64|1048576)) | 1610612736
		where xtype in ('S', 'U')

		COMMIT TRAN
	end

	-- 050499 Add sysproperties if not already present
	if object_id('sysproperties', 'S') is null
	begin
		BEGIN TRAN
		EXEC ('CREATE TABLE sysproperties (
			id				int			NOT NULL,
			smallid			smallint	NOT NULL,
			type			tinyint		NOT NULL,
			name			sysname		NOT NULL,
			value			sql_variant		NULL
				)'	)
		CREATE UNIQUE CLUSTERED INDEX sysproperties 
			ON sysproperties (id,type,smallid,name)
		COMMIT TRAN
	end

	-- 27APR1999 Drop sysallocations which was never put to use
	if object_id('sysallocations') is not null and
	   object_id('sysallocations') = 98
	begin
	    EXEC ('DROP TABLE sysallocations')
	end

	-- 30/04/1999 Reset PRC_SYSCATUP bit on system views.
	UPDATE sysobjects 
	SET status = status & ~1
	WHERE xtype = 'V' and status & 0x80000000 = 0x80000000

	-- 05/07/1999 Add column to sysdepends, change clustered index and clean up
	--         sysdepends
	if col_length('sysdepends', 'deptype') is null
	begin
		BEGIN TRAN

		-- new column
		EXEC ('ALTER TABLE sysdepends ADD deptype tinyint not null')
		EXEC ('UPDATE sysdepends set deptype = 0')

		-- redefined clustered index
		EXEC ('
		CREATE UNIQUE CLUSTERED INDEX sysdepends
			ON sysdepends ( deptype, depid, id, depnumber, number )
			WITH DROP_EXISTING, IGNORE_DUP_KEY
			')

		-- new nonclustered index
		EXEC ('
		CREATE UNIQUE NONCLUSTERED INDEX ncsysdepends1 
			ON sysdepends ( id, number, deptype, depid, depnumber ) 
			WITH IGNORE_DUP_KEY
			')

		COMMIT TRAN
	end

	-- cleanup sysdepends
	DELETE sysdepends FROM sysdepends d WHERE d.id not in (select id from sysobjects)
	DELETE sysdepends FROM sysdepends d WHERE d.depid not in (select id from sysobjects) and d.depid != 0

	-- 10Jul99 Add LCID column to syscolumns
	if not exists (SELECT * FROM dbo.syscolumns WHERE id = 3 AND name = 'language')
	begin
	 	BEGIN TRAN

		-- default to LANG_NEUTRAL for ALTER TABLE purposes.  This value will
		-- be updated later
	 	EXEC ('
		ALTER TABLE dbo.syscolumns ADD language int NOT null
			')

		EXEC ('
		UPDATE syscolumns SET language = 
					CASE 
						WHEN colstat & 16 <> 0 THEN (SELECT value FROM master.dbo.sysconfigures WHERE config = 1126) 
						ELSE 0 
					END
		    ')

		COMMIT TRAN
	end

	-- 22-Jul-1999: Remove INFORMATION_SCHEMA views from all databases but master
	if db_id() <> 1
	begin
		if object_id('INFORMATION_SCHEMA.SCHEMATA', 'V') is not NULL
			drop view INFORMATION_SCHEMA.SCHEMATA
		if object_id('INFORMATION_SCHEMA.TABLES', 'V') is not NULL
			drop view INFORMATION_SCHEMA.TABLES
		if object_id('INFORMATION_SCHEMA.TABLE_CONSTRAINTS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.TABLE_CONSTRAINTS
		if object_id('INFORMATION_SCHEMA.TABLE_PRIVILEGES', 'V') is not NULL
			drop view INFORMATION_SCHEMA.TABLE_PRIVILEGES
		if object_id('INFORMATION_SCHEMA.COLUMNS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.COLUMNS
		if object_id('INFORMATION_SCHEMA.COLUMN_DOMAIN_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.COLUMN_DOMAIN_USAGE
		if object_id('INFORMATION_SCHEMA.COLUMN_PRIVILEGES', 'V') is not NULL
			drop view INFORMATION_SCHEMA.COLUMN_PRIVILEGES
		if object_id('INFORMATION_SCHEMA.DOMAINS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.DOMAINS
		if object_id('INFORMATION_SCHEMA.DOMAIN_CONSTRAINTS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.DOMAIN_CONSTRAINTS
		if object_id('INFORMATION_SCHEMA.KEY_COLUMN_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.KEY_COLUMN_USAGE
		if object_id('INFORMATION_SCHEMA.REFERENTIAL_CONSTRAINTS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.REFERENTIAL_CONSTRAINTS
		if object_id('INFORMATION_SCHEMA.CHECK_CONSTRAINTS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.CHECK_CONSTRAINTS
		if object_id('INFORMATION_SCHEMA.CONSTRAINT_TABLE_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.CONSTRAINT_TABLE_USAGE
		if object_id('INFORMATION_SCHEMA.CONSTRAINT_COLUMN_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.CONSTRAINT_COLUMN_USAGE
		if object_id('INFORMATION_SCHEMA.VIEWS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.VIEWS
		if object_id('INFORMATION_SCHEMA.VIEW_TABLE_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.VIEW_TABLE_USAGE
		if object_id('INFORMATION_SCHEMA.VIEW_COLUMN_USAGE', 'V') is not NULL
			drop view INFORMATION_SCHEMA.VIEW_COLUMN_USAGE
		if object_id('INFORMATION_SCHEMA.ROUTINES', 'V') is not NULL
			drop view INFORMATION_SCHEMA.ROUTINES
		if object_id('INFORMATION_SCHEMA.PARAMETERS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.PARAMETERS
		if object_id('INFORMATION_SCHEMA.ROUTINE_COLUMNS', 'V') is not NULL
			drop view INFORMATION_SCHEMA.ROUTINE_COLUMNS
		delete sysusers where uid = 3 and name = 'INFORMATION_SCHEMA'
	end

	UPDATE dbo.syscolumns SET name = N'reserved3' WHERE id = 2 AND colid = 14 and name = N'soid'
	UPDATE dbo.syscolumns SET name = N'reserved4' WHERE id = 2 AND colid = 15 and name = N'csid'
	EXEC('UPDATE dbo.sysindexes SET reserved3 = 0')
	EXEC('UPDATE dbo.sysindexes SET reserved4 = 0')

	-- Reset bits used to record determinism (objects & computed-columns) in interim builds
	UPDATE sysobjects SET status = status & ~268435456
	UPDATE syscolumns SET colstat = colstat & ~32 WHERE id IN (SELECT id FROM sysobjects WHERE xtype = 'U')
go
deny execute on sp_msupg_dosystabcatalogupgrades to public
go


-------------------------------------------------------------------------------
---------------- ADD COMPUTED-COLUMNS TO PER-DATABASE CATALOG -----------------
-------------------------------------------------------------------------------

if object_id('sp_msupg_dropcatalogcomputedcols') is not null
	drop proc sp_msupg_dropcatalogcomputedcols
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_dropcatalogcomputedcols
as
	EXEC sp_msupg_removesystemcomputedcolumns sysobjects
	EXEC sp_msupg_removesystemcomputedcolumns sysindexes
	EXEC sp_msupg_removesystemcomputedcolumns syscolumns
	EXEC sp_msupg_removesystemcomputedcolumns systypes
	EXEC sp_msupg_removesystemcomputedcolumns syscomments
	EXEC sp_msupg_removesystemcomputedcolumns sysdepends
	EXEC sp_msupg_removesystemcomputedcolumns sysreferences
	EXEC sp_msupg_removesystemcomputedcolumns sysusers
go
deny execute on sp_msupg_dropcatalogcomputedcols to public
go

if object_id('sp_msupg_createcatalogcomputedcols') is not null
	drop proc sp_msupg_createcatalogcomputedcols
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_createcatalogcomputedcols
as
	ALTER TABLE sysobjects ADD
		schema_ver AS (base_schema_ver & ~15),
		stats_schema_ver AS (base_schema_ver & 15),

		type AS convert(char(2),
				CASE
					WHEN xtype in ('PK','UQ') THEN 'K'
					ELSE xtype
				END),
		userstat AS convert(smallint,
			CASE WHEN xtype in ('U','S') THEN 1 ELSE 0 END),	-- rowlocking!
		sysstat AS convert(smallint,
				CASE xtype
					WHEN 'S' THEN 1 WHEN 'V' THEN 2 WHEN 'U' THEN 3
					WHEN 'P' THEN 4 WHEN 'RF' THEN 4 WHEN 'X' THEN 4
					WHEN 'L' THEN 5 WHEN 'D' THEN 6 WHEN 'R' THEN 7
					WHEN 'TR' THEN 8 WHEN 'PK' THEN 9 WHEN 'UQ' THEN 9
					WHEN 'C' THEN 10 WHEN 'F' THEN 11 WHEN 'AP' THEN 13
					ELSE 0 END
				+ CASE WHEN xtype IN ('U', 'S') THEN
					CASE WHEN (status & 1)!=0				-- O_CLUST
							THEN (16)ELSE 0 END
					+ CASE WHEN (status & 2)!=0				-- O_NONCLUST
							THEN (32) ELSE 0 END
					+ (64)									-- O_LOGGED
					+ CASE WHEN (substring(name,1,1)='#')	-- O_TMPOBJECT
							THEN (256) ELSE 0 END
					+ CASE WHEN (status & 0x1000000)!=0		-- O_INRAM
							THEN (512) ELSE 0 END
					+ CASE WHEN (status & 0x200000)!=0		-- O_FAKE
							THEN (1024) ELSE 0 END
					+ CASE WHEN (status & 0x4000000)!=0		-- O_UNDO_NOLOG
							THEN (2048) ELSE 0 END
					+ CASE WHEN (status & 4)!=0				-- O_TEXTIMAGE
							THEN (8192) ELSE 0 END
					+ CASE WHEN (substring(name,1,2)='##')	-- O_GLOBALTMP
							THEN (-32768) ELSE 0 END
				  ELSE 0 END),
		indexdel AS convert(smallint, (base_schema_ver & ~15) / 65536),
		refdate AS convert(datetime, crdate),
		version AS convert(int, 0),
		deltrig AS convert(int,
				CASE WHEN xtype in ('U','S')
						THEN ObjectProperty(id, 'TableDeleteTrigger')
					WHEN xtype = 'TR' THEN parent_obj
					ELSE 0 END),
		instrig AS convert(int,
				CASE WHEN xtype in ('U','S')
						THEN ObjectProperty(id, 'TableInsertTrigger')
					ELSE 0 END),
		updtrig AS convert(int,
				CASE WHEN xtype in ('U','S')
						THEN ObjectProperty(id, 'TableUpdateTrigger')
					ELSE 0 END),
		seltrig AS convert(int, 0),
		category AS convert(int,
				CASE WHEN (status & 0x80000000)!=0		-- CAT_MSOBJECT
						THEN (2) ELSE 0 END
				+ CASE WHEN (replinfo & 1)!=0			-- CAT_PUBLISHED_X
						THEN (32) ELSE 0 END
				+ CASE WHEN (replinfo & 2)!=0			-- CAT_REPLICATE_X
						THEN (64) ELSE 0 END
				+ CASE WHEN (replinfo & 4)!=0			-- CAT_REPLICA_X
						THEN (256) ELSE 0 END
				+ CASE WHEN xtype='P' AND (status & 2)!=0	-- CAT_STARTUP
						THEN (16) ELSE 0 END
				+ CASE WHEN xtype='D' AND parent_obj!=0	-- CAT_DEFAULT
						THEN (2048) ELSE 0 END
				+ CASE WHEN xtype in ('U','S') THEN
					CASE WHEN (status & 0x20)!=0		-- CAT_IDENTITY
							THEN (1)ELSE 0 END
					+ CASE WHEN (status & 0x400)!=0		-- CAT_FKEY
							THEN (4) ELSE 0 END
					+ CASE WHEN (status & 0x800)!=0		-- CAT_REF
							THEN (8) ELSE 0 END
					+ CASE WHEN (status & 0x1000)!=0	-- CAT_CHECK
							THEN (128) ELSE 0 END
					+ CASE WHEN (status & 0x100)!=0		-- CAT_PRIMARY
							THEN (512) ELSE 0 END
					+ CASE WHEN (status & 0x200)!=0		-- CAT_UNIQUE
							THEN (1024) ELSE 0 END
					+ CASE WHEN (status & 0x2000)!=0	-- CAT_DEFAULT
							THEN (2048) ELSE 0 END
					+ CASE WHEN (status & 0x4000)!=0	-- CAT_REPLCNST
							THEN (4096) ELSE 0 END
					+ CASE WHEN (status & 0x10)!=0		-- CAT_TIMESTAMP
							THEN (16384) ELSE 0 END
				  ELSE 0 END),
		cache AS convert(smallint, 0)

	ALTER TABLE sysindexes ADD
		  maxlen	AS (8000),
			-- rows is computed as an int4 from rowcnt which is internally and int8.
			-- If rowcnt > 2^31-1 then rows should be 2^31-1 otherwise it should be
			-- rowcnt converted to an int.
			  rows		AS
				CASE WHEN rowcnt > 2147483647 THEN 2147483647
				ELSE (convert(int, rowcnt))
				END

	ALTER TABLE syscolumns ADD
		status AS convert(tinyint,
				(bitpos & 7)
				+ CASE WHEN (typestat&1)=0
						THEN 8 ELSE 0 END			-- STATNULL
				+ CASE WHEN ((typestat & 2)!=0 OR
							(type_name(xtype) IN ('text', 'image')
								AND (colstat & 0x1000)!=0))
						THEN 16 ELSE 0 END			-- COL_NOTRIM, COL_REPLICATE
				+ CASE WHEN (type_name(xtype) IN ('text', 'image')
								AND (colstat & 0x2000)!=0)
							OR (type_name(xtype) IN ('binary', 'char', 'timestamp')
								AND (typestat&1)=0 )
							THEN 32 ELSE 0 END		-- COL_FIXEDNULL, COL_NONSQLSUB
				+ CASE WHEN (colstat & 4)!=0
						THEN 64 ELSE 0 END			-- COL_OUTPUTPARAM
				+ CASE WHEN (colstat & 1)!=0
						THEN 128 ELSE 0 END),		-- COL_IDENTITY
		type AS convert(tinyint, XTypeToTds(xtype, 1 - (typestat & 1))),
		usertype AS convert(smallint, ColumnProperty(id, name, 'oldusertype')),
		printfmt AS convert(varchar(255), autoval),
		prec AS convert(smallint,
				CASE WHEN type_name(xtype) in ('text', 'image', 'ntext') THEN NULL
				WHEN type_name(xtype) in ('uniqueidentifier') THEN xprec
				ELSE OdbcPrec(xtype,length,xprec)
				END),
		scale AS OdbcScale(xtype,xscale),
		iscomputed AS convert(int, sign(colstat & 4)),
		isoutparam AS convert(int, sign(colstat & 4)),
		isnullable AS convert(int, 1 - (typestat & 1)),
		collation AS convert(sysname, CollationPropertyFromID(collationid, 'name')),
		tdscollation AS convert(binary(5), CollationPropertyFromID(collationid, 'tdscollation'))


	ALTER TABLE systypes ADD
		usertype AS convert(smallint, TypeProperty(name, 'oldusertype')),
		variable AS convert(bit,
				CASE WHEN type_name(xtype) IN
						('varbinary', 'varchar', 'nvarchar')
						THEN 1 ELSE 0 END),
		allownulls AS convert(bit, 1-(status & 1)),
		type AS convert(tinyint, XTypeToTds(xtype, 0)),
		printfmt AS convert(varchar(255), NULL),
		prec AS convert(smallint,
				CASE WHEN type_name(xtype) IN ('text', 'image', 'ntext') THEN NULL
					ELSE TypeProperty(name, 'precision')
				END),
		scale AS convert(tinyint, TypeProperty(name, 'scale')),
		collation AS  convert(sysname, CollationPropertyFromID(collationid, 'name'))


	ALTER TABLE syscomments ADD
		texttype AS convert(smallint, 2 + 4 * (status & 1)),
		language AS convert(smallint, 0),
		encrypted AS convert(bit, status & 1),
		compressed AS convert(bit, status & 2),
		text AS convert(nvarchar(4000),
						CASE WHEN (status&2=2)
							THEN uncompress(ctext)
							ELSE ctext END)

	ALTER TABLE sysdepends ADD
		depdbid AS convert(smallint, 0),
		depsiteid AS convert(smallint, 0),
		selall AS convert(bit, (status & 2)),
		resultobj AS convert(bit, (status & 4)),
		readobj AS convert(bit, (status & 8))

	ALTER TABLE sysreferences ADD
		fkeydbid AS convert(smallint, 0),
		rkeydbid AS convert(smallint, 0),
		fkey1 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys,  1, 2))), 0)),
		fkey2 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys,  3, 2))), 0)),
		fkey3 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys,  5, 2))), 0)),
		fkey4 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys,  7, 2))), 0)),
		fkey5 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys,  9, 2))), 0)),
		fkey6 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 11, 2))), 0)),
		fkey7 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 13, 2))), 0)),
		fkey8 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 15, 2))), 0)),
		fkey9 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 17, 2))), 0)),
		fkey10 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 19, 2))), 0)),
		fkey11 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 21, 2))), 0)),
		fkey12 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 23, 2))), 0)),
		fkey13 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 25, 2))), 0)),
		fkey14 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 27, 2))), 0)),
		fkey15 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 29, 2))), 0)),
		fkey16 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(forkeys, 31, 2))), 0)),
		rkey1 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys,  1, 2))), 0)),
		rkey2 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys,  3, 2))), 0)),
		rkey3 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys,  5, 2))), 0)),
		rkey4 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys,  7, 2))), 0)),
		rkey5 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys,  9, 2))), 0)),
		rkey6 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 11, 2))), 0)),
		rkey7 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 13, 2))), 0)),
		rkey8 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 15, 2))), 0)),
		rkey9 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 17, 2))), 0)),
		rkey10 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 19, 2))), 0)),
		rkey11 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 21, 2))), 0)),
		rkey12 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 23, 2))), 0)),
		rkey13 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 25, 2))), 0)),
		rkey14 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 27, 2))), 0)),
		rkey15 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 29, 2))), 0)),
		rkey16 AS convert(smallint,
				isnull(convert(binary(2),reverse(substring(refkeys, 31, 2))), 0))

	ALTER TABLE sysusers ADD
		gid AS convert(smallint,
				CASE WHEN uid >= 16400 THEN uid
				WHEN datalength(roles) IS NULL OR datalength(roles) <= 2 THEN 0
				ELSE 16384 - 8 + (datalength(roles) * 8) +
					CASE WHEN (convert(tinyint,roles)&1)!=0 THEN 0
						WHEN (convert(tinyint,roles)&2)!=0 THEN 1
						WHEN (convert(tinyint,roles)&4)!=0 THEN 2
						WHEN (convert(tinyint,roles)&8)!=0 THEN 3
						WHEN (convert(tinyint,roles)&16)!=0 THEN 4
						WHEN (convert(tinyint,roles)&32)!=0 THEN 5
						WHEN (convert(tinyint,roles)&64)!=0 THEN 6
						WHEN (convert(tinyint,roles)&128)!=0 THEN 7
						ELSE NULL END END),
		environ AS convert(varchar(255), NULL),
		hasdbaccess AS convert(int, CASE WHEN (status & 2)=2 THEN 1 ELSE 0 END),
		islogin AS convert(int, CASE WHEN (status&32)=0 and uid > 0 and uid < 16384 THEN 1 ELSE 0 END),
		isntname AS convert(int, CASE WHEN (status&4)=4 THEN 1 ELSE 0 END),
		isntgroup AS convert(int, CASE WHEN (status&12)=4 THEN 1 ELSE 0 END),
		isntuser AS convert(int, CASE WHEN (status&12)=12 THEN 1 ELSE 0 END),
		issqluser AS convert(int, CASE WHEN (status&60)=0 and uid > 0 and uid < 16384 THEN 1 ELSE 0 END),
		isaliased AS convert(int, CASE WHEN (status&16)=16 THEN 1 ELSE 0 END),
		issqlrole AS convert(int, CASE WHEN uid >= 16384 or uid = 0 THEN 1 ELSE 0 END),
		isapprole AS convert(int, CASE WHEN (status&32)=32 THEN 1 ELSE 0 END)
go
deny execute on sp_msupg_createcatalogcomputedcols to public
go


-------------------------------------------------------------------------------
---------------------- CREATE PER-DATABASE SYSTEM VIEWS -----------------------
-------------------------------------------------------------------------------

if object_id('sp_msupg_recreatesystemviews') is not null
	drop proc sp_msupg_recreatesystemviews
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_recreatesystemviews
as
	IF OBJECT_ID('syssegments') IS NOT NULL
		DROP VIEW syssegments
	EXEC (
'CREATE VIEW syssegments (segment, name, status) AS
	SELECT  0, ''system''     , 0  UNION
	SELECT	1, ''default''    , 1  UNION
	SELECT	2, ''logsegment'' , 0
')
	-- Mark this user db view as being system object
	UPDATE sysobjects set status = status|0x80000000 
	WHERE name='syssegments' and uid=1
	GRANT SELECT ON syssegments TO PUBLIC

	IF OBJECT_ID('sysconstraints') IS NOT NULL
		DROP VIEW sysconstraints
	EXEC (
'CREATE VIEW sysconstraints AS SELECT
	constid = convert(int, id),
	id = convert(int, parent_obj),
	colid = convert(smallint, info),
	spare1 = convert(tinyint, 0),
	status = convert(int,
			CASE xtype
				WHEN ''PK'' THEN 1 WHEN ''UQ'' THEN 2 WHEN ''F'' THEN 3
				WHEN ''C'' THEN 4 WHEN ''D'' THEN 5 ELSE 0 END
			+ CASE WHEN info != 0			-- CNST_COLUMN / CNST_TABLE
					THEN (16) ELSE (32) END
			+ CASE WHEN (status & 16)!=0	-- CNST_CLINDEX
					THEN (512) ELSE 0 END
			+ CASE WHEN (status & 32)!=0	-- CNST_NCLINDEX
					THEN (1024) ELSE 0 END
			+ (2048)						-- CNST_NOTDEFERRABLE
			+ CASE WHEN (status & 256)!=0	-- CNST_DISABLE
					THEN (16384) ELSE 0 END
			+ CASE WHEN (status & 512)!=0	-- CNST_ENABLE
					THEN (32767) ELSE 0 END
			+ CASE WHEN (status & 4)!=0		-- CNST_NONAME
					THEN (131072) ELSE 0 END
			+ CASE WHEN (status & 1)!=0		-- CNST_NEW
					THEN (1048576) ELSE 0 END
			+ CASE WHEN (status & 1024)!=0	-- CNST_REPL
					THEN (2097152) ELSE 0 END),
	actions = convert(int,  4096),
	error = convert(int, 0)
FROM sysobjects WHERE xtype in (''C'', ''F'', ''PK'', ''UQ'', ''D'')
					AND (status & 64) = 0
')
	-- Mark this user db view as being system object
	UPDATE sysobjects set status = status|0x80000000 
	WHERE name='sysconstraints' and uid=1
	GRANT SELECT ON sysconstraints TO PUBLIC

	IF OBJECT_ID('sysalternates') IS NOT NULL
		DROP VIEW sysalternates
go
deny execute on sp_msupg_recreatesystemviews to public
go

-------------------------------------------------------------------------------
---------------------- PULL TOGETHER ALL PER-DB UPGRADES ----------------------
-------------------------------------------------------------------------------

if object_id('sp_msupg_upgradecatalog') is not null
	drop proc sp_msupg_upgradecatalog
go
-- INTERNAL PROC: DO NOT DOCUMENT
create proc sp_msupg_upgradecatalog @upgradeVersion int
as
	-- CHECK UPGRADE VERSION GIVEN (MUST MATCH EXPECTED) --
	if @upgradeVersion <> 536
		-- NOTE: This "error" is really an "Assertion", so no sysmessages...
		raiserror('sp_msupg_upgradecatalog detected inconsistent versioning!',22,127) with log

	-- CHECK PERMS: Db must require upgrade OR must be sysadmin
	if is_srvrolemember('sysadmin') = 0 and databaseproperty (db_name(), 'version') > @upgradeVersion
		-- NOTE: This "error" is really an "Assertion", so no sysmessages...
		raiserror('Permissions denied on sp_msupg_upgradecatalog!',22,127) with log

	-- EXEC THE UPGRADE PROCS (UNDONE: MAKE A SUBSET POSSIBLE?) --
	exec sp_msupg_dropcatalogcomputedcols
	exec sp_msupg_dosystabcatalogupgrades
	exec sp_msupg_createcatalogcomputedcols
	exec sp_msupg_recreatecatalogfaketables
	exec sp_msupg_recreatesystemviews
go
grant execute on sp_msupg_upgradecatalog to public
go


-------------------------------------------------------------------------------
------------------------- WRAP UP: Upgrade & Shutdown -------------------------
-------------------------------------------------------------------------------

checkpoint
go

--- DO THE PER-CATALOG UPGRADE FOR MASTER AND MODEL
exec master..sp_msupg_upgradecatalog 536
go
exec model..sp_msupg_upgradecatalog 536
go


-------------------------------------------------------------------------------
------------------------ Create fake tables in master -------------------------
-------------------------------------------------------------------------------

IF OBJECT_ID('sysprocesses', 'S') IS NOT NULL
	DROP TABLE sysprocesses
go
CREATE TABLE sysprocesses (     -- see ntdbms\storeng\dfs\startup\dataserv.c
    spid            smallint NOT NULL,
    kpid            smallint NOT NULL,
    blocked         smallint NOT NULL,
    waittype        binary(2) NOT NULL,
	waittime		int NOT NULL,
	lastwaittype	nchar(32) NOT NULL,
	waitresource	nchar(256) NOT NULL,
    dbid            smallint NOT NULL,
    uid             smallint NOT NULL,
    cpu             int NOT NULL,
    physical_io     bigint NOT NULL,
    memusage        int NOT NULL,
    login_time      datetime NOT NULL,
    last_batch      datetime NOT NULL,
    ecid            smallint NOT NULL,
	open_tran		smallint NOT NULL,
    status          nchar(30) NOT NULL,
    sid             binary(86) NOT NULL,
    hostname        nchar(128) NOT NULL,
    program_name    nchar(128) NOT NULL,
    hostprocess     nchar(8) NOT NULL,
    cmd             nchar(16) NOT NULL,
    nt_domain       nchar(128) NOT NULL,
    nt_username     nchar(128) NOT NULL,
    net_address     nchar(12) NOT NULL,
    net_library     nchar(12) NOT NULL,
    loginame        nchar(128) NOT NULL,
    context_info	binary(128) NOT NULL,
    sql_handle		binary(20) NOT NULL,
    stmt_start		int NOT NULL,
    stmt_end		int NOT NULL
   )
go
GRANT SELECT ON sysprocesses TO PUBLIC
go

IF OBJECT_ID('syslocks', 'S') IS NOT NULL
	DROP TABLE syslocks
go
CREATE TABLE syslocks (         -- see ntdbms\storeng\dfs\manager\lockmgr.c
    id              int NOT NULL,
    dbid            smallint NOT NULL,
    page            int NOT NULL,
    type            smallint NOT NULL,
    spid            smallint NOT NULL
    )
go
GRANT SELECT ON syslocks TO PUBLIC
go

IF OBJECT_ID('sysperfinfo', 'S') IS NOT NULL
	DROP TABLE sysperfinfo
go
CREATE TABLE sysperfinfo (
   object_name   nchar(128) NOT NULL,
   counter_name  nchar(128) NOT NULL,
   instance_name nchar(128),
   cntr_value    int NOT NULL,
   cntr_type     int NOT NULL
   )
go
--GRANT SELECT ON sysperfinfo TO PUBLIC
--go

IF OBJECT_ID('syslockinfo', 'S') IS NOT NULL
	DROP TABLE syslockinfo
go
CREATE TABLE syslockinfo (
    rsc_text         nchar(32) NOT NULL,
    rsc_bin          binary(16) NOT NULL,
    rsc_valblk       binary(16) NOT NULL,
    rsc_dbid	     smallint NOT NULL,
    rsc_indid        smallint NOT NULL,
    rsc_objid        integer NOT NULL,
    rsc_type         tinyint NOT NULL,
    rsc_flag         tinyint NOT NULL,
    req_mode         tinyint NOT NULL,
    req_status       tinyint NOT NULL,
    req_refcnt       smallint NOT NULL,
    req_cryrefcnt    smallint NOT NULL,
    req_lifetime     integer NOT NULL,
    req_spid         integer NOT NULL,
    req_ecid         integer NOT NULL,
    req_ownertype    smallint NOT NULL,
	req_transactionID bigint,
	req_transactionUOW uniqueidentifier
    )
go
GRANT SELECT ON syslockinfo TO PUBLIC
go

IF OBJECT_ID('syscurconfigs', 'S') IS NOT NULL
	DROP TABLE syscurconfigs
go
CREATE TABLE syscurconfigs (
    value           int NOT NULL,
    config          smallint NOT NULL,
    comment         nvarchar(255) NOT NULL,
    status          smallint NOT NULL
    )
go
GRANT SELECT ON syscurconfigs TO PUBLIC
go

IF OBJECT_ID('syscursorrefs', 'S') IS NOT NULL
	DROP TABLE syscursorrefs
go
CREATE TABLE syscursorrefs (	-- see ntdbms\cursors\crsfktbl.cpp
    reference_name  sysname NULL,
    cursor_scope    tinyint NOT NULL,
    cursor_handl    int NOT NULL
    )
go
--GRANT SELECT ON syscursorrefs TO PUBLIC
--go

IF OBJECT_ID('syscursors', 'S') IS NOT NULL
	DROP TABLE syscursors
go
CREATE TABLE syscursors (	-- see ntdbms\cursors\crsfktbl.cpp
    cursor_handle   int NOT NULL,
    cursor_name     sysname NULL,
    status          int NOT NULL,
    model           tinyint NOT NULL,
    concurrency     tinyint NOT NULL,
    scrollable      tinyint NOT NULL,
    open_status     tinyint NOT NULL,
    cursor_rows     decimal(10),   -- minimum of 8 bytes of storage required
    fetch_status    smallint  NOT NULL,
    column_count    smallint NOT NULL,
    row_count       decimal(10),
    last_operation  tinyint NOT NULL
    )
go
--GRANT SELECT ON syscursors TO PUBLIC
--go

IF OBJECT_ID('syscursorcolumns', 'S') IS NOT NULL
	DROP TABLE syscursorcolumns
go
CREATE TABLE syscursorcolumns (		-- ssee ntdbms\cursors\crsfktbl.cpp
    cursor_handle   int NOT NULL,
    column_name     sysname NULL,
    ordinal_position     int NOT NULL,
    column_characteristics_flags	    int NOT NULL,
    column_size     int NOT NULL,
    data_type_sql  smallint NOT NULL,
    column_precision   tinyint NOT NULL,
    column_scale       tinyint NOT NULL,
    order_position      int NOT NULL,
    order_direction nvarchar(1) NULL,
    hidden_column   smallint NOT NULL,
    columnid        int NOT NULL,
    objectid        int NOT NULL,
    dbid            int NOT NULL,
    dbname	    sysname NULL
    )
go
--GRANT SELECT ON syscursorcolumns TO PUBLIC
--go

IF OBJECT_ID('syscursortables', 'S') IS NOT NULL
	DROP TABLE syscursortables
go
CREATE TABLE syscursortables (	-- see ntdbms\cursors\crsfktbl.cpp
    cursor_handle   	int NOT NULL,
    table_owner  	sysname NULL,
    table_name   	sysname NULL,
    optimizer_hint 	smallint NOT NULL,
    lock_type        	smallint NOT NULL,
    server_name		sysname NULL,
    objectid        	int NOT NULL,
    dbid            	int NOT NULL,
    dbname	    	sysname NULL
    )
go
--GRANT SELECT ON syscursortables TO PUBLIC
--go

IF OBJECT_ID('syscacheobjects', 'S') IS NOT NULL
	DROP TABLE syscacheobjects
go
CREATE TABLE syscacheobjects (
    bucketid        int NOT NULL,
    cacheobjtype    nvarchar(17) NOT NULL,
    objtype         nvarchar(8) NOT NULL,
    objid           int NOT NULL,
    dbid            smallint NOT NULL,
    dbidexec        smallint NOT NULL,
    uid             smallint NOT NULL,
    refcounts       int NOT NULL,
    usecounts       int NOT NULL,
    pagesused       int NOT NULL,
	lasttime		bigint NOT NULL,
	maxexectime		bigint NOT NULL,
	avgexectime		bigint NOT NULL,
	lastreads		bigint NOT NULL,
	lastwrites		bigint NOT NULL,
    setopts         int NOT NULL,
    langid          smallint NOT NULL,
    dateformat      smallint NOT NULL,
    status          int NOT NULL,
    sqlbytes        int NOT NULL,
    sql             nvarchar(3900) NULL,
   )
go
--no permissions decision yet
go

--- MAKE ANY NEW/CHANGED CONFIG VALUES TAKE EFFECT
RECONFIGURE WITH OVERRIDE
go

--- ALL DONE: SHUTDOWN: MUST RESTART TO RECOVER OTHER DATABASES
shutdown
go


