/**
 *  IFLYTEK Education Division Architecture Team
 *
 *  @author:        shenghe <shenghe@iflytek.com>
 *  @license:       (C) 2013, iFlyTEK CO.Ltd. All rights reserved.
IFLYTEK Education Division Architecture Team
 *  @version:       $Id$
 *  @module         speechui/cards
 *  @description    诗词对答模块
 */
define(function (require, exports, module) {
    require('../../view/css/global.css');
    require('../../view/css/poetry.css');

    var $ = require('jquery');
    var _speech = require('../common/speech');
    var _cardbase = require('./cardbase');
    var _define = require('../define');
    var tplparse = require('../thirdparty/template');
    var mask = require('../common/mask');

    var bkImg = [];


    function splitSentence(sentence) {
        var words = sentence.split('');
        var _words = [];
        var laststatus = true;

        for (var k = 0; k < words.length; k++) {
            var index = _words.length - 1;
            var isvalid = '[],.，。;'.indexOf(words[k]) >= 0 ? false : true;
            if (index === -1) {
                _words.push(words[k]);
            } else {
                if (isvalid !== laststatus) {
                    if (words[k - 1] === ']' || words[k - 1] === '，' ) {
                        _words.push(words[k]);
                    } else {
                        _words[index] += words[k];
                    }
                } else {
                    if (isvalid) {
                        _words.push(words[k]);
                    } else {
                        _words[index] += words[k];
                    }
                }
            }

            laststatus = isvalid;
        }
        return _words;
    }

    function getRecordID() {
        var id = 'poetry-';
        d = new Date();
        id += d.getHours();
        id += d.getMinutes();
        id += d.getSeconds();
        id += d.getMilliseconds();
        id += Math.ceil(Math.random() * 1000);
        return(id);
    }


    function getTimeline(poem) {
        var timeline = [];
        for(var i in poem.sentences) {
            var word = poem['sentences'][i]['word'];
            var begin = parseFloat(poem['sentences'][i]['begin']);
            var end = parseFloat(poem['sentences'][i]['end']);

            var words = word.replace(/[\[|\]|,|.|，|。]/g, '').split('');
            var interval = (end - begin) / words.length;
            var lastWordTime = begin;
            for(var j in words) {
                var o = {};
                o.word = words[j];
                o.begin = lastWordTime;
                o.end = lastWordTime + interval;
                lastWordTime = o.end;
                timeline.push(o);
            }
        }
        return timeline;
    }

    function getWordIndex(position, timeline) {
        for(var j in timeline) {
            if (position > timeline[j]['begin'] &&  position <= timeline[j]['end']) {
                return j;
            }
        }

        return undefined;
    }

    var Render = {
        colourWord : function(cardid, index, color) {
            color = (color === undefined) ? '#006400' : color;
            $('#poetry-'+ cardid +'-word'+ index).css('color', color);
            return Render;
        },

        showWord : function(cardid, index) {
            var node = $('#poetry-'+ cardid +'-word'+ index);
            node.show().text(node.data('word'));
            return Render;
        },

        hideWord : function(cardid, index) {
            $('#poetry-'+ cardid +'-word'+ i).text('*');
            return Render;
        },

        colourLine : function(cardid, line, color) {
            color = color === undefined ? '#FCFFCD' : color;
            $('#poetry-'+ cardid +'-line'+ line).css('background-color', color);
            return Render;
        },

        showLine : function(cardid, line) {
            $('#poetry-'+ cardid +'-line'+ line +'  span').each(function(i, val) {
                var word = $(val).data('word');
                $(val).text(word);
            });
            return Render;
        },

        hideLine : function(cardid, line) {
            $('#poetry-'+ cardid +'-line'+ line).css('display', 'none');
            return Render;
        },

        setBg : function(cardid, img) {
            if (img) {
                  $("#poetry-" + cardid + '-body').css('background-image', 'url(' + img + ')');
              }
            return Render;
        },

        reset : function(cardid, poem, hide, hideword) {
            var tpl = $("#poetry-elocntetpl").text();
            var view = {'sentences' : []};
            var wordIndex = 0;

            for (var i = 0, len = poem['sentences'].length; i < len; i++) {
                var sentence = {
                    "class" : 'poetry-content',
                    'html' : '',
                    'begin' : poem['sentences'][i]['begin'],
                    'end' : poem['sentences'][i]['end'],
                    'img' : poem['sentences'][i]['img'],
                    'index' : wordIndex,
                    'cardid': cardid,
                    'line': i,
                    'display' : 'inline'
                };
                if (i === 0) {
                    sentence['class'] = "poetry-title";
                } else if (i === 1) {
                    sentence['class'] = "poetry-author";
                }
                //合并特殊字符成数组,用于显示在页面上.例如: '[唐]孟浩然'  ->  ['[唐]', '孟', '浩', '然']
                var words = splitSentence(poem['sentences'][i]['word']);

                for(var index in words) {
                    var w = words[index];
                    if (i !== 0 && i !== 1) {
                        w = hideword ? '*' : w;
                    }

                    if (hide && i !== 0 && i !== 1) {
                        w = '&nbsp;&nbsp;';
                    }
                    sentence['html'] += '<span id="poetry-' + cardid + '-word' + wordIndex + '" data-word="'+ words[index] +'" >' + w + '</span>';
                    wordIndex += 1;
                }
                view['sentences'].push(sentence);
            }
            $("#poetry-" + cardid + '-body').empty().append(tplparse.render(tpl, view)).css('background-image', 'url(' + poem['background'] + ')');
            return Render;
        },

        tip : function(cardid, score) {
            var prefix = '#poetry-'+ cardid;
            var msg = '开始朗诵吧！';
            var img = './js-debug/speechui/view/images/normal.png';
            if (score !== undefined) {
                if (score < 60) {
                    msg = '还没有完全掌握这首诗，继续努力吧！';
                    img = './js-debug/speechui/view/images/bad.png';
                } else if(score >= 60 && score < 80) {
                    msg = '不错， 你能做的更好！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 80 && score < 90) {
                    msg = '你掌握的真好！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 90) {
                    msg = '太棒了， 非常准确！';
                    img = './js-debug/speechui/view/images/great.png';
                }
            }

            $(prefix +'-tip').text(msg);
            $(prefix +'-tipimg').attr("src", img);
            $(prefix +'-tipdiv').show().fadeOut(3000);
            return Render;
        },

        tipAnswer : function(cardid, isood, score, func) {
            var prefix = '#poetry-'+ cardid;
            var msg = '我说上句，你说下句';
            var img = './js-debug/speechui/view/images/good.png';
            if (!isood) {
                msg = '我说下句，你说上句';
            }

            if (score !== undefined) {
                if (score < 60) {
                    msg = '别灰心，加油！';
                    img = './js-debug/speechui/view/images/bad.png';
                } else if(score >= 60 && score < 80) {
                    msg = '相信你还能更好！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 80 && score < 90) {
                    msg = '你掌握的真好！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 90) {
                    msg = '太棒了， 非常准确！';
                    img = './js-debug/speechui/view/images/great.png';
                }
            }
            $(prefix +'-tip').text(msg);
            $(prefix +'-tipimg').attr("src", img);
            $(prefix +'-tipdiv').fadeIn(1000).fadeOut(1000, func);
            return Render;
        },

        tipRecite: function(cardid) {
            var prefix = '#poetry-'+ cardid;
            var msg = '开始背诗吧';
            var img = './js-debug/speechui/view/images/good.png';
            $(prefix +'-tip').text(msg);
            $(prefix +'-tipimg').attr("src", img);
            $(prefix +'-tipdiv').fadeIn(1000).fadeOut(1000);
        },

        tipOrderAnswer: function(cardid, score) {
            var prefix = '#poetry-'+ cardid;
            var msg = '我们来对诗吧';
            if (score !== undefined) {
                if (score < 60) {
                    msg = '没关系，加油！';
                    img = './js-debug/speechui/view/images/bad.png';
                } else if(score >= 60 && score < 80) {
                    msg = '继续努力吧！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 80 && score < 90) {
                    msg = '相信你还能更好！';
                    img = './js-debug/speechui/view/images/good.png';
                } else if(score >= 90) {
                    msg = '你掌握的真好！';
                    img = './js-debug/speechui/view/images/great.png';
                }
            }
            var img = './js-debug/speechui/view/images/good.png';
            $(prefix +'-tip').text(msg);
            $(prefix +'-tipimg').attr("src", img);
            $(prefix +'-tipdiv').fadeIn(1000).fadeOut(1000);
        }
    };

    var Speecher = {
        //player播放
        playAudio : function(mp3, begin, end) {
            _speech.speecher.Player.loadAudio(encodeURI(mp3));
            if (begin !== undefined && end !== undefined) {
                _speech.speecher.Player.playRange(begin, end);
            } else {
                _speech.speecher.Player.play();
            }
        },

        //recorder加载音频
        playRecord : function(recordid, begin, end) {
            _speech.speecher.Recorder.loadAudio(recordid);
        },

        //record播放
        record : function(recordid, content) {
            var success = false;
            try {
                success = _speech.speecher.Recorder.record({
                    evalType : SeEvalType.ReadSentenceCn,
                    evalText : content,
                    evalParams : "autoTrack=true,trackType=easy,vadEnable=true,vadSpeechTail=2500",
                    sndId : recordid
                });
                //console.log(content +'|'+ success +'|'+ recordid);
            } catch (e) {}

            return success;
        },

        //player停止
        stopAudio : function() {
            _speech.speecher.Player.stop();
        },

        //stop停止
        stopRecord : function() {
            _speech.speecher.Recorder.stop();
            _speech.speecher.Recorder.stopPlay();
        },

        //开始朗诵
        registeRecordEvent : function(cardid, ins, params) {
            var prefix = '#poetry-'+ cardid;

            function onEvalResult(data) {
                if (status === 'recited' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer') {
                    return;
                }

                if(data.resultType == EvalCallbackType.ENDPOINT)   {
                    _speech.speecher.Recorder.stop();
                } else {
                    $(prefix +'-recording').hide();
                    $(prefix +'-recorded').show();

                    Render.reset(cardid, ins.poem);

                    if (('result' in data) && ('Sentences' in data['result'])) {
                        ins.voice['words'] = [];
                        ins.voice['lines'] = [];

                        var sens = data['result']['Sentences'];
                        for (var i = 0, senlen = sens.length; i < senlen; i++) {
                            if ('Words' in sens[i]) {
                                var words = sens[i]['Words'];
                                for (var j = 0, wordslen = words.length; j < wordslen; j++) {
                                    var sylls = words[j]['Sylls'];
                                    for (var k = 0, syllslen = sylls.length; k < syllslen; k++) {
                                        var info = {'word': '', 'index': 0, 'begin': 0, 'end': 0};
                                        info.word = sylls[k]['Content'];
                                        info.index = sylls[k]['GlobalIndex'];
                                        info.begin = sylls[k]['BegPos'];
                                        info.end = sylls[k]['EndPos'];
                                        ins.voice['words'].push(info);
                                    }
                                }
                            }

                            ins.voice['lines'].push({'begin': sens[i]['BegPos'], 'end': sens[i]['EndPos']});
                        }

                        Render.tip(cardid, data['result']['TotalScore']);
                        ins.voice.id = ins.recordid;
                    } else {
                        Render.tip(cardid, 0);
                    }

                    if (ins.voice['id'] !== '') {
                        $(prefix +'-read').attr('class', 'btn_funread_nor');
                        $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    }

                    ins.status = "normal";
                    Poetry.recordTime = new Date().getTime();
                }
            }

            function onTrackInfo(data) {
                if (data.ReadStatus === 2) {
                    var img = $(prefix + "-word" + data.Index).parent().data('img');
                    Render.colourWord(cardid, data.Index).setBg(cardid, img);

                    if (data.Index === ins.wordscount - 1) {
                        Speecher.stopRecord();
                    }
                }
            }

            function onRecording(data) {
                var vol = data.energy * (280 / 100);
                var node = $(prefix +"-voicebar");

                if (vol > 280) {
                    vol = 280;
                }

                node.width(vol).css('background-color', '#006400');
                if (vol < 20) {
                    node.css('background-color', '#F50808');
                }
            }

            function onRecordError(data) {
                ins.voice = {'id' : '', 'info' : ''};
                onEvalResult(data);
            }

            _speech.registerEvent({
                recordEvents : {
                    onRecording : onRecording,
                    onTrackInfo : onTrackInfo,
                    onEvalResult : onEvalResult,
                    onError : onRecordError
                }
            });
        },

        //示范朗读
        registePlayEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;
            var timeline = getTimeline(ins.poem);

            function onPlayed(e) {
                if (ins.status === 'recited' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer') {
                    $(prefix +'-play > a').text('示范朗读');
                    $(prefix +'-play').attr('class', 'btn_funplay_nor');

                    $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                    if (ins.voice['id'] !== '') {
                        $(prefix +'-read').attr('class', 'btn_funread_nor');
                        $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    }
                    return;
                }
                $(prefix +'-play > a').text('示范朗读');
                $(prefix +'-play').attr('class', 'btn_funplay_nor');
                $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                if (ins.voice['id'] !== '') {
                    $(prefix +'-read').attr('class', 'btn_funread_nor');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                }

                ins.status = 'normal';
                Render.reset(cardid, ins.poem);

                $('.poetry-poem-border').bind('click', {'ins' : ins}, function(e) {
                    Handle['onClickLine'](e);
                });
            }

            function onPlaying(e) {
                var wordIndex = getWordIndex(e.position, timeline);
                if (wordIndex !== undefined) {
                    var img = $(prefix +'-word'+ wordIndex).parent().data('img');
                    Render.setBg(cardid, img).colourWord(cardid, wordIndex);
                }
            }

            _speech.registerEvent({
                playEvents : {
                    'onPlaying' : onPlaying,
                    'onStop' : onPlayed,
                    'onError' : onPlayed
                }
            });
        },

        //我的朗读
        registeReadEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;

            function onRecordPlaying(data) {
                var pos = data.position;
                var info = ins.voice['words'];
                var node;
                for (var i = 0, len = info.length; i < len; i++) {
                    if (pos >= info[i]['begin'] && pos < info[i]['end']) {
                        node = $(prefix + "-word" + info[i]['index']);
                        Render.colourWord(cardid, info[i]['index']);
                        break;
                    }
                }

                if (node !== undefined) {
                    var img = node.parent().data('img');
                    Render.setBg(cardid, img);
                }
            }

            function onRecordStop(data) {
                if (ins.status === 'recited' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer') {
                    $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                    $(prefix +'-play').attr('class', 'btn_funplay_nor');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    $(prefix +'-read').attr('class', 'btn_funread_nor');
                    $(prefix +'-read > a').text('我的朗诵');
                    return;
                }
                $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                $(prefix +'-play').attr('class', 'btn_funplay_nor');
                $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                $(prefix +'-read').attr('class', 'btn_funread_nor');
                $(prefix +'-read > a').text('我的朗诵');
                ins.status = 'normal';
                Render.reset(cardid, ins.poem);
            }

            function onAudioLoaded(data) {
                _speech.speecher.Recorder.play();
            }

            function onError(data){
                if( data.errorType == 2 && (data.errorCode == 10115 || data.errorCode == 10116) ){
                    var maskOption = {
                        self    : '',
                        type    : 'poetry',
                        msg     : '',
                        lagTime : 1,
                        flag    : true
                    };
                    var time = new Date().getTime();
                    if( time - Poetry.recordTime < 5000 ){
                        maskOption.msg = "正在上传语音文件";
                    }else{
                        maskOption.msg = "音频加载出错";
                    }
                    mask.showMask(maskOption);

                    setTimeout(function(){
                        maskOption.flag = false;
                        mask.showMask(maskOption);
                        onRecordStop();
                    },2000)
                }
            }

            _speech.registerEvent({
                recordEvents : {
                    'onPlaying' : onRecordPlaying,
                    'onPlayStop' : onRecordStop,
                    'onAudioLoaded' : onAudioLoaded,
                    'onError'       : onError
                }
            });
        },

        //对比朗读
        registeContrastEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;
            var line = 0;
            var timeline = getTimeline(ins.poem);

            function onPlaying(data) {
                var wordIndex = getWordIndex(data.position, timeline);
                var img = $(prefix +'-word'+ wordIndex).parent().data('img');
                Render.colourWord(cardid, wordIndex).setBg(cardid, img);
            }

            function onPlayed(data) {
                if (ins.status === 'recited' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer') {
                    return;
                }

                if (ins.status === 'stopContrast' || line > ins.voice['lines'].length - 1) {
                    $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                    $(prefix +'-play').attr('class', 'btn_funplay_nor');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    $(prefix +'-read').attr('class', 'btn_funread_nor');
                    $(prefix +'-contrast > a').text('对比朗诵');
                    Render.reset(cardid, ins.poem);
                    ins.status = 'normal';
                    return;
                }
                $(prefix +'-line'+ line +' > span').css('color', '#666');
                var begin = ins.voice['lines'][line]['begin'];
                var end = ins.voice['lines'][line]['end'];

                Speecher.playRecord(ins.voice['id'], begin, end);
            }

            function onRecordPlaying(data) {
                var pos = data.position;
                var info = ins.voice['words'];
                var node;
                for (var i = 0, len = info.length; i < len; i++) {
                    if (pos >= info[i]['begin'] && pos < info[i]['end']) {
                        node = $(prefix + "-word" + info[i]['index']);
                        Render.colourWord(cardid, info[i]['index'], '#1E90FF');
                        break;
                    }
                }

                if (node !== undefined) {
                    img = node.parent().data('img');
                    Render.setBg(cardid, img);
                }
            }

            function onRecordStop(data) {
                if (ins.status === 'recited' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer' || ins.status === 'stopContrast') {
                    $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                    $(prefix +'-play').attr('class', 'btn_funplay_nor');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    $(prefix +'-read').attr('class', 'btn_funread_nor');
                    $(prefix +'-contrast > a').text('对比朗诵');

                    if (ins.status === 'stopContrast') {
                        Render.reset(cardid, ins.poem);
                        ins.status = 'normal';
                    }
                    return;
                }
                if('resultType' in data && data.resultType == EvalCallbackType.ENDPOINT)   {
                    _speech.speecher.Recorder.stop();
                }

                if (line < ins.poem['sentences'].length - 1) {
                    line += 1;
                    var begin = ins.poem['sentences'][line]['begin'];
                    var end = ins.poem['sentences'][line]['end'];

                    Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
                } else {
                    $(prefix +'-startRecord').attr('class', 'btn_funstart_nor');
                    $(prefix +'-play').attr('class', 'btn_funplay_nor');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_nor');
                    $(prefix +'-read').attr('class', 'btn_funread_nor');
                    $(prefix +'-contrast > a').text('对比朗诵');
                    ins.status = 'normal';
                    Render.reset(cardid, ins.poem);
                }
            }

            function onAudioLoaded(data) {
                var begin = ins.voice['lines'][line]['begin'];
                var end = ins.voice['lines'][line]['end'];
                _speech.speecher.Recorder.playRange(begin, end);
            }

            function onError(data){
                if( data.errorType == 2 && (data.errorCode == 10115 || data.errorCode == 10116) ){
                    var maskOption = {
                        self    : '',
                        type    : 'poetry',
                        msg     : '',
                        lagTime : 1,
                        flag    : true
                    };
                    var time = new Date().getTime();
                    if( time - Poetry.recordTime < 5000 ){
                        maskOption.msg = "正在上传语音文件";
                    }else{
                        maskOption.msg = "音频加载出错";
                    }
                    mask.showMask(maskOption);

                    setTimeout(function(){
                        maskOption.flag = false;
                        mask.showMask(maskOption);
                        ins.status = 'stopContrast';
                        onRecordStop();
                    },2000)
                }
            }

            _speech.registerEvent({
                recordEvents : {
                    'onPlaying' : onRecordPlaying,
                    'onPlayStop' : onRecordStop,
                    'onAudioLoaded' : onAudioLoaded,
                    'onError' : onError
                },
                playEvents : {
                    'onPlaying' : onPlaying,
                    'onStop' : onPlayed,
                    'onError': onPlayed
                }
            });
        },

        //背诵
        registeReciteEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;

            function onEvalResult(data) {
                if (ins.status === 'normal' || ins.status === 'singleAnswer' || ins.status === 'orderAnswer') {
                    return;
                }

                if(data.resultType == EvalCallbackType.ENDPOINT)   {
                    _speech.speecher.Recorder.stop();
                } else if(data.resultType == EvalCallbackType.SCORE || 'errorType' in data) {
                    var score = 0;
                    if ('result' in data && 'TotalScore' in data.result) {
                        score = data.result.TotalScore;
                    }
                    Render.tip(cardid, score);
                    $(prefix +'stopRecord > a').text('停止背诵');
                    ins.status = 'recited';
                    Render.reset(cardid, ins.poem, false, true);
                    $('.poetry-'+ cardid +'-btns').hide();
                    $(prefix +'-reciteBtn').show();
                }
            }

            function onTrackInfo(data) {
                var node = $(prefix + "-word" + data.Index);
                if (data.ReadStatus === 2) {
                    var img = node.parent().data('img');
                    Render.setBg(cardid, img).showWord(cardid, data.Index).colourWord(cardid, data.Index);

                    if (data.Index === ins.wordscount - 1) {
                        Speecher.stopRecord();
                    }
                }
            }

            function onRecording(data) {
                var vol = data.energy * (280 / 100);
                var node = $(prefix +"-voicebar");

                if (vol > 280) {
                    vol = 280;
                }

                node.width(vol).css('background-color', '#006400');
                if (vol < 20) {
                    node.css('background-color', '#F50808');
                }
            }

            _speech.registerEvent({
                recordEvents : {
                    'onRecording' : onRecording,
                    'onTrackInfo' : onTrackInfo,
                    'onEvalResult' : onEvalResult,
                    'onError' : onEvalResult
                }
            });
        },

        //整句对答
        registeOrderAnswerEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;
            var line = 2;
            var status = 'playing';
            var timeline = getTimeline(ins.poem);

            function onEvalResult(data) {
                if (ins.status === 'normal' || ins.status === 'recited') {
                    return;
                }

                if('resultType' in data && data.resultType == EvalCallbackType.ENDPOINT)   {
                    _speech.speecher.Recorder.stop();
                    return;
                }

                if ('errorType' in data || data.resultType == EvalCallbackType.SCORE) {
                    $(prefix +'stopRecord > a').text('停止对答');

                    var score = 0;
                    if ('result' in data && data.result && 'TotalScore' in data.result) {
                        score = data['result']['TotalScore'];
                    }
                    Render.tipOrderAnswer(cardid, score);

                    Render.showLine(cardid, line);

                    $(prefix +'-voicebar').width(0);
                    $(prefix +'-line'+ line).css('background-color', '');


                    if (ins.isreset || line === ins.poem['sentences'].length - 1) {
                        $('.poetry-'+ ins.cardid +'-btns').hide();
                        $(prefix +'-answerBtn').show();
                        Render.reset(cardid, ins.poem);
                        ins.isreset = false;
                    } else {
                        line += 1;
                        var begin = ins.poem['sentences'][line]['begin'];
                        var end = ins.poem['sentences'][line]['end'];
                        Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
                    }
                }
            }

            function onTrackInfo(data) {
                if (data.ReadStatus === 2) {
                    var lIndex = $(prefix +'-line'+ line +' > div').data('index');
                    var index = lIndex + data.Index;
                    var img = $(prefix + "-word" + index).parent().data('img');
                    Render.setBg(cardid, img).showWord(cardid, index).colourWord(cardid, index, '#35A6FF');

                    var nIndex = $(prefix +'-line'+ (line + 1) +' > div').data('index');
                    if (index === ins.wordscount - 1 || (nIndex !== undefined && index === nIndex - 1)) {
                        Speecher.stopRecord();
                    }
                }
            }

            function onRecording(data) {
                var vol = data.energy * (280 / 100);
                var node = $(prefix +"-voicebar");

                if (vol > 280) {
                    vol = 280;
                }

                node.width(vol).css('background-color', '#006400');
                if (vol < 20) {
                    node.css('background-color', '#F50808');
                }
            }

            function onPlaying(data) {
                var wordIndex = getWordIndex(data.position, timeline);
                if (wordIndex !== undefined) {
                    var img = $(prefix +'-word'+ wordIndex).parent().data('img');
                    Render.setBg(cardid, img).showWord(cardid, wordIndex).colourWord(cardid, wordIndex);
                }
            }

            function onPlayed(data) {
                if (ins.status === 'normal' || ins.status === 'recited') {
                    return;
                }
                if (ins.isreset) {
                    $('.poetry-'+ ins.cardid +'-btns').hide();
                    $(prefix +'-answerBtn').show();
                    Render.reset(ins.cardid, ins.poem);
                    ins.isreset = false;
                } else {
                    var sens = ins.poem['sentences'];
                    if (line < sens.length - 1) {
                        line += 1;
                        Render.colourLine(ins.cardid, line);
                        if(!Speecher.record(getRecordID(), ins.poem['sentences'][line]['word'].replace(/[\[|\]|,|.|，|。]/ig, ''))) {
                            $('.poetry-'+ ins.cardid +'-btns').hide();
                            $(prefix +'-answerBtn').show();
                            Render.reset(ins.cardid, ins.poem);
//                            alert('网络错误，请检查网络！');
                        }
                    }
                }


            }

            _speech.registerEvent({
                recordEvents : {
                    'onRecording' : onRecording,
                    'onTrackInfo' : onTrackInfo,
                    'onEvalResult' : onEvalResult,
                    'onError' : onEvalResult
                },
                playEvents : {
                    'onPlaying' : onPlaying,
                    'onStop' : onPlayed,
                    'onError' : onPlayed
                }
            });
        },

        //单句对答
        registeSingleAnswerEvent : function(cardid, ins) {
            var prefix = "#poetry-"+ cardid;
            var timeline = getTimeline(ins.poem);
            var curLine = 0;
            function onPlaying(data) {
                var wordIndex = getWordIndex(data.position, timeline);

                if (wordIndex !== undefined) {
                    var img = $(prefix +'-word'+ wordIndex).parent().data('img');
                    Render.setBg(cardid, img).showWord(cardid, wordIndex).colourWord(cardid, wordIndex);
                }
            }

            function onPlayed(data) {
                if (ins.status === 'normal' || ins.status === 'recited') {
                    return;
                }

                if (ins.isreset) {
                    $('.poetry-'+ ins.cardid +'-btns').hide();
                    $(prefix +'-answerBtn').show();
                    Render.reset(ins.cardid, ins.poem);
                    ins.isreset = false;
                } else {
                    var sens = ins.poem['sentences'];
                    var line = ins.line.cur;
                    if (line % 2 === 1) {
                        line -= 1;
                        curLine = line;
                        Render.colourLine(ins.cardid, line);
                        if(!Speecher.record(getRecordID(), ins.poem['sentences'][line]['word'].replace(/[\[|\]|,|.|，|。]/ig, ''))){
                            $('.poetry-'+ ins.cardid +'-btns').hide();
                            $(prefix +'-answerBtn').show();
                            Render.reset(ins.cardid, ins.poem);
//                            alert('网络错误，请检查网络！');
                        }
                    } else if(ins.line.length !== ins.poem['sentences'].length){
                        line += 1;
                        curLine = line;
                        Render.colourLine(ins.cardid, line);
                        if(!Speecher.record(getRecordID(), ins.poem['sentences'][line]['word'].replace(/[\[|\]|,|.|，|。]/ig, ''))){
                            $('.poetry-'+ ins.cardid +'-btns').hide();
                            $(prefix +'-answerBtn').show();
                            Render.reset(ins.cardid, ins.poem);
//                            alert('网络错误，请检查网络！');
                        }
                    }
                }
            }

            function onEvalResult(data) {
                if (ins.status === 'normal' || ins.status === 'recited') {
                    return;
                }

                if('resultType' in data && data.resultType == EvalCallbackType.ENDPOINT)   {
                    _speech.speecher.Recorder.stop();
                    return;
                }

                if ('errorType' in data || data.resultType == EvalCallbackType.SCORE) {
                    $(prefix +'stopRecord > a').text('停止对答');
                    var score = 0;
                    if ('result' in data && data.result && 'TotalScore' in data['result']) {
                        score = data['result']['TotalScore'];
                    }

                    if (ins.isreset || ins.line.length === ins.poem['sentences'].length - 2) {
                        $('.poetry-'+ ins.cardid +'-btns').hide();
                        $(prefix +'-answerBtn').show();
                        Render.reset(ins.cardid, ins.poem);
                        ins.isreset = false;
                        Render.tipAnswer(cardid, false, score, function(){});
                    } else {
                        var line = ins.line.cur;
                        $(prefix +'-voicebar').width(0);
                        $(prefix +'-line'+ line).css('background-color', '');
                        Render.reset(ins.cardid, ins.poem, true, false);
                        while(true){
                            line = Math.round(Math.random()*(3-ins.poem['sentences'].length)+ins.poem['sentences'].length - 1);
                            if (line === 1 || line === 0) {
                                line = 2;
                            }

                            if (!(line in ins.line)) {
                                ins.line[line] = line;
                                ins.line.cur = line;
                                ins.line.length += 1;
                                break;
                            }
                        }

                        Render.tipAnswer(cardid, false, score, function(){
                            if (line % 2 === 0) {
                                Render.tipAnswer(ins.cardid, true);
                            } else {
                                Render.tipAnswer(ins.cardid, false);
                            }
                            var begin = ins.poem['sentences'][line]['begin'];
                            var end = ins.poem['sentences'][line]['end'];
                            Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
                        });

                    }
                }
            }

            function onTrackInfo(data) {
                if (data.ReadStatus === 2) {
                    var lIndex = $(prefix +'-line'+ curLine +' > div').data('index');
                    var index = lIndex + data.Index;
                    var img = $(prefix + "-word" + index).parent().data('img');
                    Render.setBg(cardid, img).showWord(cardid, index).colourWord(cardid, index, '#35A6FF');

                    var nIndex = $(prefix +'-line'+ (curLine + 1) +' > div').data('index');
                    if (index === ins.wordscount - 1 || (nIndex !== undefined && index === nIndex - 1)) {
                        Speecher.stopRecord();
                    }
                }
            }

            function onRecording(data) {
                var vol = data.energy * (280 / 100);
                var node = $(prefix +"-voicebar");

                if (vol > 280) {
                    vol = 280;
                }

                node.width(vol).css('background-color', '#006400');
                if (vol < 20) {
                    node.css('background-color', '#F50808');
                }
            }

            _speech.registerEvent({
                recordEvents : {
                    'onRecording' : onRecording,
                    'onTrackInfo' : onTrackInfo,
                    'onEvalResult' : onEvalResult,
                    'onError' : onEvalResult
                },
                playEvents : {
                    'onPlaying' : onPlaying,
                    'onStop' : onPlayed,
                    'onError' : onPlayed
                }
            });
        }
    };

    var Handle = {
        'onInit' : function(e) {
            var ins = e.data['ins'];
            var prefix = "#poetry-"+ ins.cardid;
            ins.status = 'normal';
            Speecher.stopRecord();
            Speecher.stopAudio();

            Render.reset(ins.cardid, ins.poem);
            if (ins.voice['id'] !== '') {
                $(prefix + '-read').attr('class', 'btn_funread_nor');
                $(prefix + '-contrast').attr('class', 'btn_fundb_nor');
            }
            $(prefix +'-tipholder').hide();
            $('.poetry-'+ ins.cardid +'-btns').hide();
            $(prefix +'-recorded').show();
        },

        'onInitRecite' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;
            ins.status = 'recited';
            Speecher.stopRecord();
            Speecher.stopAudio();

            Render.reset(ins.cardid, ins.poem, false, true);

            $('.poetry-'+ ins.cardid +'-btns').hide();
            $(prefix +'-reciteBtn').show();
            $(prefix +'-tipholder').hide();
        },

        'onInitAnswer' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;
            if ($(prefix +'-orderAnswer').attr('class') === 'poetry-answer') {
                ins.status = 'orderAnswer';
            } else {
                ins.status = 'singleAnswer';
            }

            Speecher.stopRecord();
            Speecher.stopAudio();

            Render.reset(ins.cardid, ins.poem);
            $(prefix +'-tipholder').toggle(10);
            $('.poetry-'+ ins.cardid +'-btns').hide();
            $(prefix +'-answerBtn').show();
        },

        'onStartRecord' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            if (ins.status === 'normal') {
                ins.recordid = getRecordID();

                if (Speecher.record(ins.recordid, ins.sentence)) {
                    $(prefix +'-recorded').hide();
                    $(prefix +'-recording').show();
                    $(prefix +'-stopRecord >a').text('停止朗诵');
                    Render.tip(ins.cardid);
                    ins.status = 'recording';
                    Speecher.registeRecordEvent(ins.cardid, ins);
                } else {
//                    alert('网络错误，请检查网络！');
                }
            }
        },

        'onStopRecord' : function(e) {
            var ins = e.data['ins'];
            if (ins.status === 'recording') {
                Speecher.stopRecord();
            }
        },

        'onStartPlay' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            if (ins.status === 'normal') {
                ins.status = 'playing';

                $(prefix +'-startRecord').attr('class', 'btn_funstart_dis');
                $(prefix +'-read').attr('class', 'btn_funread_dis');
                $(prefix +'-contrast').attr('class', 'btn_fundb_dis');
                $(prefix +'-play').attr('class', 'btn_funplay_hov');
                $(prefix + '-play > a').text('停止朗读');

                Speecher.registePlayEvent(ins.cardid, ins);
                Speecher.playAudio(ins.poem['mp3']);
            }

        },

        'onStopPlay' : function(e) {
            var ins = e.data['ins'];
            if (ins.status === 'playing' || ins.status === 'playLine') {
                Speecher.stopAudio();
            }
        },

        'onStartRead' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            if (ins.voice['id'] !== '' && ins.status === 'normal') {
                ins.status = 'playing';

                $(prefix +'-startRecord').attr('class', 'btn_funstart_dis');
                $(prefix +'-play').attr('class', 'btn_funplay_dis');
                $(prefix +'-contrast').attr('class', 'btn_fundb_dis');
                $(prefix +'-read').attr('class', 'btn_funread_hov');
                $(prefix +'-read > a').text('停止朗诵');

                Speecher.registeReadEvent(ins.cardid, ins);
                Speecher.playRecord(ins.voice['id']);
            }
        },

        'onStopRead' : function(e) {
            Speecher.stopRecord();
        },

        'onStartContrast' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            if (ins.voice['id'] !== '' && ins.status === 'normal') {
                ins.status = 'playing';
                $(prefix +'-startRecord').attr('class', 'btn_funstart_dis');
                $(prefix +'-play').attr('class', 'btn_funplay_dis');
                $(prefix +'-contrast').attr('class', 'btn_fundb_hov');
                $(prefix +'-read').attr('class', 'btn_funread_dis');
                $(prefix +'-contrast > a').text('停止朗诵');

                Speecher.registeContrastEvent(ins.cardid, ins);
                var begin = ins.poem['sentences'][0]['begin'];
                var end = ins.poem['sentences'][0]['end'];
                Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
            }
        },

        'onStopContrast' : function(e) {
            var ins = e.data['ins'];
            ins.status = 'stopContrast';
            Speecher.stopRecord();
            Speecher.stopAudio();
        },

        'onStartRecite' : function(e) {
            var ins = e.data['ins'];
            if (Speecher.record(getRecordID(), ins.sentence)) {
                var prefix = '#poetry-'+ ins.cardid;

                ins.status = 'reciting';

                $(prefix +'-stopRecord > a').text('停止背诵');
                $('.poetry-'+ ins.cardid +'-btns').hide();
                $(prefix +'-recording').show();
                Render.tipRecite(ins.cardid);
                Speecher.registeReciteEvent(ins.cardid, ins);
            } else {
 //               alert('网络错误，请检查网络！');
            }
        },

        'onStopRecite' : function(e) {
            Speecher.stopRecord();
        },

        'onStartAnswer' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            Render.reset(ins.cardid, ins.poem, true, false);
            $(prefix +'-answerBtn').hide();
            $(prefix +'-recording').show();
            $(prefix +'-stopRecord > a').text('停止对答');
            $(prefix +'-tipholder').hide();

            var begin = ins.poem['sentences'][2]['begin'];
            var end = ins.poem['sentences'][2]['end'];

            if (ins.status === 'orderAnswer') {
                Render.tipOrderAnswer(ins.cardid);
                Speecher.registeOrderAnswerEvent(ins.cardid, ins);
            } else if(ins.status === 'singleAnswer') {
                ins.line = {'length': 0, 'cur': 2};
                Speecher.registeSingleAnswerEvent(ins.cardid, ins);
                var line = Math.round(Math.random()*(3-ins.poem['sentences'].length)+ins.poem['sentences'].length-1);
                if (line === 1 || line === 0) {
                    line = 2;
                }
                ins.line[line] = line;
                ins.line.cur = line;
                ins.line.length += 1;
                begin = ins.poem['sentences'][line]['begin'];
                end = ins.poem['sentences'][line]['end'];
                if (line % 2 === 0) {
                    Render.tipAnswer(ins.cardid, true);
                } else {
                    Render.tipAnswer(ins.cardid, false);
                }
            }
            Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
        },

        'onStopAnswer' : function(e) {
            var ins = e.data['ins'];
            ins.isreset = true;
            Speecher.stopAudio();
            Speecher.stopRecord();
        },

        'onClickLine' : function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;
            var target = e.srcElement ? e.srcElement : e.target;
            var id = target.id;

            if (id !== '' && (ins.status === 'normal' || ins.status === 'playLine')) {
                var ret = true;
                var begin = 0;
                var end = 0;


                if($('#'+ id).attr('class') === 'poetry-poem-border')
                {
                    begin = $("#"+ id).data('begin');
                    end = $("#"+ id).data('end');
                } else if ($('#'+ id).parent().attr('class') === 'poetry-poem-border') {
                    begin = $("#"+ id).parent().data('begin');
                    end = $("#"+ id).parent().data('end');
                }
                if (begin !== 0 && end !== 0) {

                    $(prefix +'-startRecord').attr('class', 'btn_funstart_dis');
                    $(prefix +'-read').attr('class', 'btn_funread_dis');
                    $(prefix +'-contrast').attr('class', 'btn_fundb_dis');
                    $(prefix +'-play').attr('class', 'btn_funplay_hov');
                    $(prefix + '-play > a').text('停止朗读');
                    ins.status = 'playLine';

                    Speecher.registePlayEvent(ins.cardid, ins);
                    $(prefix + '-body span').css("color","#666");
                    Speecher.playAudio(ins.poem['mp3'], begin * 1000, end * 1000);
                }
            }
        },

        'onOrderAnswer': function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            $(prefix +'-orderAnswer').attr('class', 'poetry-answer');
            $(prefix +'-singleAnswer').removeClass('poetry-answer');
            $(prefix +'-tipholder').hide();

            ins.status = 'orderAnswer';

            Speecher.stopAudio();
            Speecher.stopRecord();
        },

        'onSingleAnswer': function(e) {
            var ins = e.data['ins'];
            var prefix = '#poetry-'+ ins.cardid;

            $(prefix +'-singleAnswer').attr('class', 'poetry-answer');
            $(prefix +'-orderAnswer').removeClass('poetry-answer');
            $(prefix +'-tipholder').hide();

            ins.status = 'singleAnswer';

            Speecher.stopAudio();
            Speecher.stopRecord();
        }
    };

    /**
     * 诗词对答卡片的构造函数,对外提供此类
     * @param object option 诗词定义格式,诗词的元数据,格式如下:
     * {
     *     "title": "春晓",
     *     "author": "[唐]孟浩然",
     *     "sentences": [
     *         {
     *             "word": "春眠不觉晓，",
     *             "begin": 0.024247,
     *             "end": 0.966497,
     *             "pitch": 0,
     *             "pronounce": "春眠不觉晓，",
     *             "lang": 1,
     *             "img": "http://example.com/春晓01.jpg"             //可以是本地地址,可以为空
     *         }
     *     ],
     *     "background": "http://example.com/春晓00.jpg",
     *     "mp3": "http://example.com/春晓.mp3"
     * }
     */
    function Poetry(option) {
        //保证外部必须new一个实例,而不是直接调用函数
        if (!(this instanceof Poetry)) {
            throw new Error('The object not instanceof Poetry, please use `new` create it.');
        }

        if (!('sentences' in option)) {
            alert('卡片无法正常工作，请检查网络！');
            return;
        }

        this.line = {};

        /**
         * 记录卡片信息，所有实例访问
         * @static
         */
        Poetry.cardsCount += 1;

        //诗词元数据
        this.poem = option;
        this.timeline = getTimeline(this.poem);

        //诗词去掉特殊符号后的内容
        this.sentence = '';

        var imgLoaded = [];
        var imgLoadedStatus = false;
        var sensLength = this.poem['sentences'].length;
        var maskOption = {
            self    : '',
            type    : 'poetry',
            msg     : '资源加载中，请稍候',
            lagTime : 1,
            flag    : true
        };
        //console.log("%c【chengyang】语句个数：%d","color:blue",sensLength);

        for (var i = 0; i < this.poem['sentences'].length; i++) {
            this.sentence += this.poem['sentences'][i]['word'].replace(/[,|.|\[|\]|，|。]/ig, '') +'，';
            bkImg[i] = new Image();
            bkImg[i].src = this.poem.sentences[i].img;
            imgLoaded[i] = false;

            $(function(i){
                bkImg[i].onload = function(){
                   // console.log("%c【chengyang】imgLoaded[%d]加载完成","color:blue",i);
                    imgLoaded[i] = true;
                    for(var k=0;k<sensLength;k++){
                     //   console.log("%c【chengyang】imgLoaded[%d]","color:blue",k,imgLoaded[k]);
                        if(imgLoaded[k] == false){
                            return;
                        }
                    }
                   // console.log("%c【chengyang】图片加载完成","color:blue");
                    imgLoadedStatus = true;
                    if(window.readyInfo.recorder == true && window.readyInfo.player == true){
                     //   console.log("%c【chengyang】图片加载完成,去遮罩","color:blue");
                        maskOption.flag = false;
                        mask.showMask(maskOption);
                    }
                }
            }(i));
        }

        this.wordscount = this.sentence.split('').length - this.poem['sentences'].length;
        //卡片的Id，为了防止各个卡片的干扰
        this.cardid = Poetry.cardsCount;

        //录音时生成的id和包含了字符的位置和编号的数据
        this.voice = {'id': '', 'words': [], 'lines': []};

        //卡片状态
        //normal, recording, playing, reading, contrasting, reciting, recited, answering, answered, answere-played, replying, replyed
        this.status = 'normal';
        this.render();
        this.bind();


        mask.showMask(maskOption);

        _speech.registerEvent({
            recordEvents : {
                onReady : function(){
                    window.readyInfo.recorder = true;
                 //   console.log("%c【chengyang】录音器ready","color:blue");
                    if( window.readyInfo.recorder == true && window.readyInfo.player == true && imgLoadedStatus == true){
                  //      console.log("%c【chengyang】录音器ready去遮罩","color:blue");
                        maskOption.flag = false;
                        setTimeout(function() {
                            mask.showMask(maskOption);
                        }, 1000);
                    }
                }
            },
            playEvents:{
                onReady : function(){
                    window.readyInfo.player = true;
                //    console.log("%c【chengyang】播放器ready","color:blue");
                    _speech.speecher.Player.loadAudio(encodeURI(option.mp3));
                    if( window.readyInfo.recorder == true &&  window.readyInfo.player == true  && imgLoadedStatus == true){
                 //       console.log("%c【chengyang】播放器ready去遮罩","color:blue");
                        maskOption.flag = false;
                        setTimeout(function() {
                            mask.showMask(maskOption);
                        }, 1000);
                    }
                }
            }
        });
    }

    Poetry.cardsCount = 0;
    Poetry.recordTime = 0;

    Poetry.prototype = new _cardbase.CardBase();
    Poetry.prototype.constructor = Poetry;

    Poetry.prototype.getCardName = function () {
        return _define.CardName.Poetry;
    };

    Poetry.prototype.render = function () {
        var body = $('body');
        if (this.cardid === 1) {
            body.append(this.getTemplate());
        }
        body.append(tplparse.render($('#poetry-containertpl').text(), {'cardid' : this.cardid }));
        Render.reset(this.cardid, this.poem);
        // $(".poetry-footer").tipTip({'content': '<div id="poetry-{{cardid}}-orderAnswer" class="poetry-answer">整诗对答</div> <div id="poetry-{{cardid}}-singleAnswer">单句对答</div>', 'defaultPosition': 'top', 'activation': 'click', 'keepAlive': true});
    };

    Poetry.prototype.bind = function() {
        //事件代理，避免一次绑定的标签太多
        $('.poetry-'+ this.cardid +'-btns').bind('click', {'ins' : this}, function(e) {
            var target = e.srcElement ? e.srcElement : e.target;
            var ins = e.data['ins'];
            var id = target.id;
            if (target.tagName === 'A') {
                id = target.parentNode.id;
            }
            var text = $('#' + id  +' > a').text();
            if (id === 'poetry-'+ ins.cardid +'-startRecord') {
                if (text === '开始朗诵') {
                    Handle['onStartRecord'](e);
                }
            } else if (id === 'poetry-'+ ins.cardid +'-play') {
                if (text === '示范朗读') {
                    Handle['onStartPlay'](e);
                } else {
                    setTimeout(function(){
                        Handle['onStopPlay'](e);
                    },600);
                }
            } else if (id === 'poetry-'+ ins.cardid +'-read') {
                if (text === '我的朗诵') {
                    Handle['onStartRead'](e);
                } else {
                    setTimeout(function(){
                        Handle['onStopRead'](e);
                    },600);
                }
            }  else if (id === 'poetry-'+ ins.cardid +'-contrast') {
                if (text === '对比朗诵') {
                    Handle['onStartContrast'](e);
                } else {
                    setTimeout(function(){
                        Handle['onStopContrast'](e);
                    },600);
                }
            } else if(id === 'poetry-'+ ins.cardid +'-stopRecord') {
                if(text === '停止背诵') {
                    Handle['onStopRecite'](e);
                } else if(text === '停止对答') {
                    Handle['onStopAnswer'](e);
                } else {
                    Handle['onStopRecord'](e);
                }
            } else if(id === 'poetry-'+ ins.cardid +'-startRecite') {
                Handle['onStartRecite'](e);
            }
        });

        $('#poetry-'+ this.cardid +'-elocnte').bind('click', {'ins' : this}, function(e) {
            Handle['onInit'](e);
        });

        $('#poetry-'+ this.cardid +'-answer').bind('click', {'ins' : this}, function(e) {
            Handle['onInitAnswer'](e);
        });

        $('#poetry-'+ this.cardid +'-recite').bind('click', {'ins' : this}, function(e) {
            Handle['onInitRecite'](e);
        });

        $('#poetry-'+ this.cardid +'-body').bind('click', {'ins' : this}, function(e) {
            Handle['onClickLine'](e);
        });

        $('#poetry-'+ this.cardid +'-orderAnswer').bind('click', {'ins':this}, function(e) {
            Handle['onOrderAnswer'](e);
        });

        $('#poetry-'+ this.cardid +'-singleAnswer').bind('click', {'ins':this}, function(e) {
            Handle['onSingleAnswer'](e);
        });

        $('#poetry-'+ this.cardid +'-startAnswer').bind('click', {'ins':this}, function(e) {
            Handle['onStartAnswer'](e);
        });
    };
    /**
     * @export
     */
    exports.Constructor = Poetry;
});
