/*
 * Copyright 2015-2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.querydsl.binding;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.springframework.test.util.ReflectionTestUtils.*;

import java.text.ParseException;
import java.util.List;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.junit.Before;
import org.junit.Test;
import org.springframework.data.querydsl.QSpecialUser;
import org.springframework.data.querydsl.QUser;
import org.springframework.data.querydsl.QUserWrapper;
import org.springframework.data.querydsl.SimpleEntityPathResolver;
import org.springframework.data.querydsl.User;
import org.springframework.data.querydsl.Users;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import com.querydsl.collections.CollQueryFactory;
import com.querydsl.core.types.Constant;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.StringPath;

/**
 * Unit tests for {@link QuerydslPredicateBuilder}.
 * 
 * @author Christoph Strobl
 * @author Oliver Gierke
 */
public class QuerydslPredicateBuilderUnitTests {

	static final ClassTypeInformation<User> USER_TYPE = ClassTypeInformation.from(User.class);
	static final QuerydslBindings DEFAULT_BINDINGS = new QuerydslBindings();

	QuerydslPredicateBuilder builder;
	MultiValueMap<String, String> values;

	@Before
	public void setUp() {
		this.builder = new QuerydslPredicateBuilder(new DefaultFormattingConversionService(),
				SimpleEntityPathResolver.INSTANCE);
		this.values = new LinkedMultiValueMap<String, String>();
	}

	@Test(expected = IllegalArgumentException.class) // DATACMNS-669
	public void rejectsNullConversionService() {
		new QuerydslPredicateBuilder(null, SimpleEntityPathResolver.INSTANCE);
	}

	@Test(expected = IllegalArgumentException.class) // DATACMNS-669
	public void getPredicateShouldThrowErrorWhenBindingContextIsNull() {
		builder.getPredicate(null, values, null);
	}

	@Test // DATACMNS-669
	public void getPredicateShouldReturnEmptyPredicateWhenPropertiesAreEmpty() {

		assertThat(builder.getPredicate(ClassTypeInformation.OBJECT, values, DEFAULT_BINDINGS), is(nullValue()));
	}

	@Test // DATACMNS-669
	public void resolveArgumentShouldCreateSingleStringParameterPredicateCorrectly() throws Exception {

		values.add("firstname", "Oliver");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		assertThat(predicate, is((Predicate) QUser.user.firstname.eq("Oliver")));

		List<User> result = CollQueryFactory.from(QUser.user, Users.USERS).where(predicate).fetchResults().getResults();

		assertThat(result, hasSize(1));
		assertThat(result, hasItem(Users.OLIVER));
	}

	@Test // DATACMNS-669
	public void resolveArgumentShouldCreateNestedStringParameterPredicateCorrectly() throws Exception {

		values.add("address.city", "Linz");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		assertThat(predicate, is((Predicate) QUser.user.address.city.eq("Linz")));

		List<User> result = CollQueryFactory.from(QUser.user, Users.USERS).where(predicate).fetchResults().getResults();

		assertThat(result, hasSize(1));
		assertThat(result, hasItem(Users.CHRISTOPH));
	}

	@Test // DATACMNS-669
	public void ignoresNonDomainTypeProperties() {

		values.add("firstname", "rand");
		values.add("lastname".toUpperCase(), "al'thor");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		assertThat(predicate, is((Predicate) QUser.user.firstname.eq("rand")));
	}

	@Test // DATACMNS-669
	public void forwardsNullForEmptyParameterToSingleValueBinder() {

		values.add("lastname", null);

		QuerydslBindings bindings = new QuerydslBindings();
		bindings.bind(QUser.user.lastname).first(new SingleValueBinding<StringPath, String>() {

			@Override
			public Predicate bind(StringPath path, String value) {
				return value == null ? null : path.contains(value);
			}
		});

		builder.getPredicate(USER_TYPE, values, bindings);
	}

	@Test // DATACMNS-734
	@SuppressWarnings("unchecked")
	public void resolvesCommaSeparatedArgumentToArrayCorrectly() {

		values.add("address.lonLat", "40.740337,-73.995146");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		Constant<Object> constant = (Constant<Object>) ((List<?>) getField(getField(predicate, "mixin"), "args")).get(1);

		assertThat(constant.getConstant(), instanceOf(Double[].class));
		assertThat((Double[]) (constant.getConstant()), arrayContaining(40.740337D, -73.995146D));
	}

	@Test // DATACMNS-734
	@SuppressWarnings("unchecked")
	public void leavesCommaSeparatedArgumentUntouchedWhenTargetIsNotAnArray() {

		values.add("address.city", "rivers,two");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		Constant<Object> constant = (Constant<Object>) ((List<?>) getField(getField(predicate, "mixin"), "args")).get(1);

		assertThat(constant.getConstant(), instanceOf(String.class));
		assertThat((String) (constant.getConstant()), equalTo("rivers,two"));
	}

	@Test // DATACMNS-734
	public void bindsDateCorrectly() throws ParseException {

		DateTimeFormatter format = DateTimeFormat.forPattern("yyyy-MM-dd");
		String date = format.print(new DateTime());

		values.add("dateOfBirth", format.print(new DateTime()));

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		assertThat(predicate, is((Predicate) QUser.user.dateOfBirth.eq(format.parseDateTime(date).toDate())));
	}

	@Test // DATACMNS-883
	public void automaticallyInsertsAnyStepInCollectionReference() {

		values.add("addresses.street", "VALUE");

		Predicate predicate = builder.getPredicate(USER_TYPE, values, DEFAULT_BINDINGS);

		assertThat(predicate, is((Predicate) QUser.user.addresses.any().street.eq("VALUE")));
	}

	@Test // DATACMNS-941
	public void buildsPredicateForBindingUsingDowncast() {

		values.add("specialProperty", "VALUE");

		QuerydslBindings bindings = new QuerydslBindings();
		bindings.bind(QUser.user.as(QSpecialUser.class).specialProperty)//
				.first(QuerydslBindingsUnitTests.ContainsBinding.INSTANCE);

		Predicate predicate = builder.getPredicate(USER_TYPE, values, bindings);

		assertThat(predicate, is((Predicate) QUser.user.as(QSpecialUser.class).specialProperty.contains("VALUE")));
	}

	@Test // DATACMNS-941
	public void buildsPredicateForBindingUsingNestedDowncast() {

		values.add("user.specialProperty", "VALUE");

		QUserWrapper $ = QUserWrapper.userWrapper;

		QuerydslBindings bindings = new QuerydslBindings();
		bindings.bind($.user.as(QSpecialUser.class).specialProperty)//
				.first(QuerydslBindingsUnitTests.ContainsBinding.INSTANCE);

		Predicate predicate = builder.getPredicate(USER_TYPE, values, bindings);

		assertThat(predicate, is((Predicate) $.user.as(QSpecialUser.class).specialProperty.contains("VALUE")));
	}
}
