/*
 * Copyright 2011-2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.geo;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;

import org.junit.Test;
import org.springframework.util.SerializationUtils;

/**
 * Unit tests for {@link Circle}.
 * 
 * @author Oliver Gierke
 * @author Thomas Darimont
 */
public class CircleUnitTests {

	@Test(expected = IllegalArgumentException.class) // DATACMNS-437
	public void rejectsNullOrigin() {
		new Circle(null, new Distance(0));
	}

	@Test(expected = IllegalArgumentException.class) // DATACMNS-437
	public void rejectsNegativeRadius() {
		new Circle(1, 1, -1);
	}

	@Test // DATACMNS-437
	public void considersTwoCirclesEqualCorrectly() {

		Circle left = new Circle(1, 1, 1);
		Circle right = new Circle(1, 1, 1);

		assertThat(left, is(right));
		assertThat(right, is(left));

		right = new Circle(new Point(1, 1), new Distance(1));

		assertThat(left, is(right));
		assertThat(right, is(left));
	}

	@Test // DATACMNS-437
	public void testToString() {

		assertThat(new Circle(1, 1, 1).toString(), is("Circle: [center=Point [x=1.000000, y=1.000000], radius=1.0]"));
	}

	@Test // DATACMNS-482
	public void testSerialization() {

		Circle circle = new Circle(1, 1, 1);

		Circle serialized = (Circle) SerializationUtils.deserialize(SerializationUtils.serialize(circle));
		assertThat(serialized, is(circle));
	}
}
