/*
 * Copyright 2012-2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package sample.parent;

import java.io.File;
import java.util.function.Consumer;

import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.integration.channel.DirectChannel;
import org.springframework.integration.dsl.IntegrationFlow;
import org.springframework.integration.dsl.IntegrationFlows;
import org.springframework.integration.dsl.Pollers;
import org.springframework.integration.dsl.SourcePollingChannelAdapterSpec;
import org.springframework.integration.file.FileReadingMessageSource;
import org.springframework.integration.file.FileWritingMessageHandler;

@SpringBootApplication
@EnableConfigurationProperties(ServiceProperties.class)
public class SampleParentContextApplication {

	@EnableAutoConfiguration
	protected static class Parent {

		@Bean
		public FileReadingMessageSource fileReader() {
			FileReadingMessageSource reader = new FileReadingMessageSource();
			reader.setDirectory(new File("target/input"));
			return reader;
		}

		@Bean
		public DirectChannel inputChannel() {
			return new DirectChannel();
		}

		@Bean
		public DirectChannel outputChannel() {
			return new DirectChannel();
		}

		@Bean
		public FileWritingMessageHandler fileWriter() {
			FileWritingMessageHandler writer = new FileWritingMessageHandler(
					new File("target/output"));
			writer.setExpectReply(false);
			return writer;
		}

		@Bean
		public IntegrationFlow integrationFlow(SampleEndpoint endpoint) {
			return IntegrationFlows.from(fileReader(), new FixedRatePoller())
					.channel(inputChannel()).handle(endpoint).channel(outputChannel())
					.handle(fileWriter()).get();
		}

		private static class FixedRatePoller
				implements Consumer<SourcePollingChannelAdapterSpec> {

			@Override
			public void accept(SourcePollingChannelAdapterSpec spec) {
				spec.poller(Pollers.fixedRate(500));
			}

		}

	}

	public static void main(String[] args) throws Exception {
		new SpringApplicationBuilder(Parent.class)
				.child(SampleParentContextApplication.class).run(args);
	}

}
