﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using MikuMikuPlugin;
using DxMath;

namespace SampleScenObjPlugin
{
    public class SampleSceneObjPlugin : IResidentPlugin
    {
        /// <summary>2Dスクリーンオブジェクト</summary>
        private ScreenImage_2D screen2D;
        /// <summary>3Dスクリーンオブジェクト</summary>
        private ScreenImage_3D screen3D;

        /// <summary>時間</summary>
        private float time = 0;

        /// <summary>マウスDrag開始フラグ</summary>
        private bool mouseDragging = false;
        /// <summary>マウスDown位置</summary>
        private Point mouseDownPosition;
        /// <summary>マウスDown時のオブジェクト位置</summary>
        private Point positionWhenMouseDown;

        ///////////////////////////////////////////////////////////////////////////
        // マウスイベント処理

        /// <summary>
        /// MouseDown
        /// </summary>
        void screenImage_MouseDown(object sender, MouseEventArgs e)
        {
            mouseDragging = true;
            mouseDownPosition = e.Location;
            positionWhenMouseDown = screen2D.Position;
        }

        /// <summary>
        /// MouseMove
        /// </summary>
        void screenImage_MouseMove(object sender, MouseEventArgs e)
        {
            if (mouseDragging)
            {
                screen2D.Position = new Point(positionWhenMouseDown.X + e.Location.X - mouseDownPosition.X,
                                                 positionWhenMouseDown.Y + e.Location.Y - mouseDownPosition.Y);
            }
        }

        /// <summary>
        /// MouseUp
        /// </summary>
        void screenImage_MouseUp(object sender, MouseEventArgs e)
        {
            mouseDragging = false;
        }

        /// <summary>
        /// 3DのMouseDown
        /// </summary>
        void screen3D_MouseDown(object sender, MouseEventArgs e)
        {
            MessageBox.Show(ApplicationForm, "MouseDown");
        }

        ///////////////////////////////////////////////////////////////////////////
        // プラグイン必須項目

        /// <summary>
        /// このプラグインのGUID
        /// </summary>
        public Guid GUID
        {
            get { return new Guid("FFE4FFBA-4A27-4D3D-84BA-E0625BC6A655"); }
        }

        /// <summary>
        /// メインフォーム
        /// MikuMikuMoving側から与えられます。
        /// ダイアログ表示やメッセージ表示に使用してください。
        /// </summary>
        public IWin32Window ApplicationForm { get; set; }

        /// <summary>
        /// シーンオブジェクト
        /// MikuMikuMoving側から与えられます。
        /// MikuMikuMovingのモデルやアクセサリといったオブジェクトにアクセスできます。
        /// </summary>
        public Scene Scene { get; set; }

        /// <summary>
        /// プラグインの名前や作者名、プラグインの説明
        /// </summary>
        public string Description
        {
            get { return "Sample SceenObj Plugin v0.1 by Mogg"; }
        }

        /// <summary>
        /// ボタンに表示するアイコン画像(32x32)
        /// nullだとデフォルト画像が表示されます。
        /// </summary>
        public Image Image
        {
            get { return null; }
        }

        /// <summary>
        /// 中コマンドバーに表示するアイコン画像(20x20)
        /// nullだとデフォルト画像が表示されます。
        /// </summary>
        public Image SmallImage
        {
            get { return null; }
        }

        /// <summary>
        /// ボタンに表示するテキスト
        /// 日本語環境で表示されるテキストです。改行する場合は Environment.NewLineを使用してください。
        /// </summary>
        public string Text
        {
            get { return "SceenObj" + Environment.NewLine + "サンプル"; }
        }

        /// <summary>
        /// ボタンに表示する英語テキスト
        /// 日本以外の環境で表示されるテキストです。
        /// </summary>
        public string EnglishText
        {
            get { return "SceenObj" + Environment.NewLine + "Sample"; }
        }

        /// <summary>
        /// Initialize
        /// プラグイン作成時に1度だけ呼ばれます。
        /// </summary>
        public void Initialize()
        {
            screen2D = new ScreenImage_2D(new Point(10, 10), Properties.Resources.Image);
            screen2D.MouseDown += new MouseEventHandler(screenImage_MouseDown);
            screen2D.MouseUp += new MouseEventHandler(screenImage_MouseUp);
            screen2D.MouseMove += new MouseEventHandler(screenImage_MouseMove);

            screen3D = new ScreenImage_3D(new Vector3(0.0f, 10.0f, 0.0f), Properties.Resources.Image);
            screen3D.MouseDown += new MouseEventHandler(screen3D_MouseDown);

            Scene.ScreenObjects.Add(screen2D);
            Scene.ScreenObjects.Add(screen3D);
        }

        /// <summary>
        /// Enabled
        /// プラグインが有効化されたときに呼び出されます。
        /// </summary>
        public void Enabled()
        {

        }

        /// <summary>
        /// 更新
        /// 毎フレームごとに呼び出されます。有効時にしか呼ばれません。
        /// </summary>
        /// <param name="Frame">現在のフレーム番号</param>
        /// <param name="ElapsedTime">前フレームからの差分実時間(秒)</param>
        public void Update(float Frame, float ElapsedTime)
        {
            time += ElapsedTime;
            //回転させる
            screen2D.Transform = Matrix.RotationZ(time * 90 * (float)(Math.PI / 180.0));
            screen3D.Rotation = Quaternion.RotationAxis(Vector3.UnitZ, time * 90 * (float)(Math.PI / 180.0));
        }

        /// <summary>
        /// Disabled
        /// プラグインが無効化されたときに呼び出されます。
        /// </summary>
        public void Disabled()
        {

        }

        /// <summary>
        /// プラグイン破棄処理
        /// もし解放しないといけないオブジェクトがある場合は、ここで解放してください。
        /// </summary>
        public void Dispose()
        {
            screen2D.Dispose();
            screen3D.Dispose();
        }
    }
}
