﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using MikuMikuPlugin;
using DxMath;

namespace SampleResidentPlugin
{
    public class SampleResidentPlugin : IResidentPlugin
    {
        /// <summary>
        /// ローカルで保存している時間
        /// </summary>
        private float time;

        /// <summary>
        /// このプラグインのGUID
        /// </summary>
        public Guid GUID
        {
            get { return new Guid("46729020-D555-4F72-ADDC-F99C1043DAF1"); }
        }

        /// <summary>
        /// メインフォーム
        /// MikuMikuMoving側から与えられます。
        /// ダイアログ表示やメッセージ表示に使用してください。
        /// </summary>
        public IWin32Window ApplicationForm { get; set; }

        /// <summary>
        /// シーンオブジェクト
        /// MikuMikuMoving側から与えられます。
        /// MikuMikuMovingのモデルやアクセサリといったオブジェクトにアクセスできます。
        /// </summary>
        public Scene Scene { get; set; }

        /// <summary>
        /// プラグインの名前や作者名、プラグインの説明
        /// </summary>
        public string Description
        {
            get { return "Sample Resident Plugin v0.1 by Mogg"; }
        }

        /// <summary>
        /// ボタンに表示するアイコン画像(32x32)
        /// nullだとデフォルト画像が表示されます。
        /// </summary>
        public Image Image
        {
            get { return null; }
        }

        /// <summary>
        /// 中コマンドバーに表示するアイコン画像(20x20)
        /// nullだとデフォルト画像が表示されます。
        /// </summary>
        public Image SmallImage
        {
            get { return null; }
        }

        /// <summary>
        /// ボタンに表示するテキスト
        /// 日本語環境で表示されるテキストです。改行する場合は Environment.NewLineを使用してください。
        /// </summary>
        public string Text
        {
            get { return "サンプル" + Environment.NewLine + "プラグイン"; }
        }

        /// <summary>
        /// ボタンに表示する英語テキスト
        /// 日本以外の環境で表示されるテキストです。
        /// </summary>
        public string EnglishText
        {
            get { return "Sample" + Environment.NewLine + "Plugin"; }
        }

        /// <summary>
        /// Initialize
        /// プラグイン作成時に1度だけ呼ばれます。
        /// </summary>
        public void Initialize()
        {
            time = 0;
        }

        /// <summary>
        /// Enabled
        /// プラグインが有効化されたときに呼び出されます。
        /// </summary>
        public void Enabled()
        {

        }

        /// <summary>
        /// 更新
        /// 毎フレームごとに呼び出されます。
        /// </summary>
        /// <param name="Frame">現在のフレーム番号</param>
        /// <param name="ElapsedTime">前フレームからの差分実時間(秒)</param>
        public void Update(float Frame, float ElapsedTime)
        {
            time += ElapsedTime;
            float r = 90 * time * (float)(Math.PI / 180.0);

            foreach (Model model in Scene.Models)
            {
                MotionData mdata = model.Bones.RootBone.CurrentLocalMotion;

                //毎秒90度Y軸周りに回転させる
                mdata.Rotation *= Quaternion.RotationAxis(Vector3.UnitY, r);
                model.Bones.RootBone.CurrentLocalMotion = mdata;

                //あモーフをつける
                if (model.Morphs["あ"] != null)
                {
                    model.Morphs["あ"].CurrentWeight = (float)(Math.Sin(r) + 1.0) / 2.0f;
                }
            }

            //カメラとライトを回転させる
            foreach (Camera camera in Scene.Cameras)
            {
                CameraMotionData mdata = camera.CurrentMotion;
                mdata.Angle = new Vector3(camera.CurrentMotion.Angle.X, camera.CurrentMotion.Angle.Y + r / 6, camera.CurrentMotion.Angle.Z);
                camera.CurrentMotion = mdata;
            }
            foreach (Light light in Scene.Lights)
            {
                LightMotionData mdata = light.CurrentMotion;
                mdata.Position = new Vector3((float)Math.Sin(r / 6) * 100.0f, 100.0f, (float)Math.Cos(r / 6) * 100.0f);
                light.CurrentMotion = mdata;
            }
        }

        /// <summary>
        /// Disabled
        /// プラグインが無効化されたときに呼び出されます。
        /// </summary>
        public void Disabled()
        {

        }

        /// <summary>
        /// プラグイン破棄処理
        /// もし解放しないといけないオブジェクトがある場合は、ここで解放してください。
        /// </summary>
        public void Dispose()
        {

        }
    }
}
