﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.Text;
using MikuMikuPlugin;
using DxMath;

namespace FrameInfoPlugin
{
    public class FrameInfoPlugin : ICommandPlugin
    {
        /// <summary>
        /// メインフォーム
        /// </summary>
        public IWin32Window ApplicationForm { get; set; }

        /// <summary>
        /// API群
        /// </summary>
        public ICommandAPI API { get; set; }

        /// <summary>
        /// プラグインの名前や作者名、プラグインの説明
        /// </summary>
        public string Description
        {
            get { return "FrameInfo Plugin v0.1 by Mogg"; }
        }

        /// <summary>
        /// ボタンに表示するアイコン画像(32x32)
        /// </summary>
        public Image Image
        {
            get { return Properties.Resources.FrameInfo_L; }
        }

        /// <summary>
        /// 中コマンドバーに表示するアイコン画像(20x20)
        /// </summary>
        public Image SmallImage
        {
            get { return Properties.Resources.FrameInfo; }
        }

        /// <summary>
        /// ボタンに表示するテキスト
        /// </summary>
        public string Text
        {
            get { return "フレーム" + Environment.NewLine + "情報"; }
        }

        /// <summary>
        /// ボタンに表示する英語テキスト
        /// </summary>
        public string EnglishText
        {
            get { return "Frame" + Environment.NewLine + "Info"; }
        }

        /// <summary>
        /// 実行
        /// </summary>
        public void Run()
        {
            //StringBuilder(数が多い場合があるので)
            StringBuilder builder = new StringBuilder();

            //現在有効なモデル/アクセサリ情報の取得
            ObjectType objtype = API.GetActivateObjectType();
            if (objtype == ObjectType.Model || objtype == ObjectType.Accessory)
            {
                //オブジェクト名
                if (API.Language == "en")
                    builder.AppendLine(API.GetCurrentObjectEnglishName());
                else
                    builder.AppendLine(API.GetCurrentObjectName());


                //ボーン一覧を取得する
                List<Bone> bones = API.GetBones();

                int scount;                         //多段数
                List<MotionFrameData> boneframes;   //モーションフレームリスト

                for (int bidx = 0; bidx < bones.Count; bidx++)
                {
                    //ボーン名を表示
                    if (API.Language == "en")
                        builder.AppendLine(bones[bidx].EnglishName);
                    else
                        builder.AppendLine(bones[bidx].Name);


                    //多段数を取得
                    scount = API.GetMotionLayerCount(bidx);
                    for (int sidx = 0; sidx < scount; sidx++)
                    {
                        //ボーンモーションフレームを取得
                        boneframes = API.GetMotionFrameData(bidx, sidx);

                        //各フレーム情報
                        foreach (MotionFrameData fdata in boneframes)
                        {
                            builder.AppendLine(" " + sidx + ":" + fdata.frameNumber + ":" + fdata.position.ToString() + ":" + fdata.quaternion.ToString());
                        }
                    }
                }

                //モーフ一覧を取得する
                List<Morph> morphs = API.GetMorphs();

                //モーフフレームリスト
                List<MorphFrameData> morphframes;   

                for (int midx = 0; midx < morphs.Count; midx++)
                {
                    //モーフ名を表示
                    if (API.Language == "en")
                        builder.AppendLine(morphs[midx].EnglishName);
                    else
                        builder.AppendLine(morphs[midx].Name);

                    //ボーンモーションフレームを取得
                    morphframes = API.GetMorphFrameData(midx);

                    //各フレーム情報
                    foreach (MorphFrameData fdata in morphframes)
                    {
                        builder.AppendLine(" " + fdata.frameNumber + ":" + fdata.weight.ToString());
                    }
                }
            }

            //カメラ情報
            List<CameraFrameData> cameraframes = API.GetCameraFrameData(0);

            if (cameraframes != null)
            {
                builder.AppendLine("");
                if (API.Language == "en")
                    builder.AppendLine("Camera");
                else
                    builder.AppendLine("カメラ");

                //各フレーム情報を書き出す
                foreach (CameraFrameData fdata in cameraframes)
                {
                    builder.AppendLine(" " + fdata.frameNumber + ":" + fdata.position + ":" + fdata.angle + ":" + fdata.radius + ":" + fdata.fov);
                }

                //ライト情報
                int lightcount = API.GetLightCount();
                List<LightFrameData> lightframes;

                for (int lidx = 0; lidx < lightcount; lidx++)
                {
                    //フレーム情報取得
                    lightframes = API.GetLightFrameData(lidx);

                    //ライト名
                    builder.AppendLine("");
                    if (API.Language == "en")
                        builder.AppendLine("Light" + lidx);
                    else
                        builder.AppendLine("ライト" + lidx);

                    //各フレーム情報を書き出す
                    foreach (LightFrameData fdata in lightframes)
                    {
                        builder.AppendLine(" " + fdata.frameNumber + ":" + fdata.position + ":" + fdata.color);
                    }
                }

                //ダイアログを表示する
                FrameInfoDialog dlg = new FrameInfoDialog();
                dlg.FrameInfoText = builder.ToString();

                dlg.ShowDialog(ApplicationForm);
            }
        }

        /// <summary>
        /// プラグイン破棄処理
        /// </summary>
        public void Dispose()
        {

        }
    }
}
