//  TestCapture		Capture Screen Images
//  Copyright 2003-2004 Excellence Foundation		Don@xfnd.org
//
//  This product is free software. You can redistribute it and/or modify it under the
//  terms of the Public Source Distribution License Agreement (PSDLA) as published by
//  Excellence Foundation.
//
//  This product is distributed in the hope it will be instructional and useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the PSDLA for more details. You should have
//  received a copy of the PSDLA with this product. If not, browse
//  http://www.xfnd.org/licenses/PSDLA.doc.

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.IO;
using System.Threading;
using Emxx;

namespace ImageCapture
{
	/// <summary>
	/// Structure for image type information. Image type information
	/// is available for each supported image format.
	/// </summary>
	internal struct ImageType
	{
		static ImageType[]		fmt;					// array of image types
		static int				fmtIndex;				// image type index
		internal static string	fmtFilter;				// filter for array

		internal ImageFormat	format;					// image type format
		internal string			extension;				// image type extension
		internal string			filter;					// image type filter
		
		/// <summary>
		/// Creates and initializes and ImageType. An ImageType structure
		/// contains the ImageFormat associated file extension and filter.
		/// </summary>
		/// <param name="fmt"></param>
		/// <param name="ext"></param>
		/// <param name="description"></param>
		internal	ImageType(ImageFormat fmt,string ext,string description)
		{
			format = fmt;
			extension = ext;
			filter = description + " (*." + extension + ")|*." + extension;
		}

		/// <summary>
		/// Static method to create and initialize an array of ImageType structures.
		/// The resulting array contains an ImageType for each supported image format. 
		/// </summary>
		internal static void	InitFormats()
		{
			fmt = new ImageType[10];				// create image type formats
			fmt[0] = new ImageType(ImageFormat.Bmp,       "bmp",  "Bitmap");
			fmt[1] = new ImageType(ImageFormat.Emf,       "emf",  "Enhanced Metafile Format");
			fmt[2] = new ImageType(ImageFormat.Exif,      "exif", "Exchange Image Format");
			fmt[3] = new ImageType(ImageFormat.Gif,       "gif",  "Graphics Interchange Format");
			fmt[4] = new ImageType(ImageFormat.Icon,      "ico",  "Windows ICON Format");
			fmt[5] = new ImageType(ImageFormat.Jpeg,      "jpg",  "Joint Photographic Experts Group");
			fmt[6] = new ImageType(ImageFormat.MemoryBmp, "mbmp", "Memory Bitmap");
			fmt[7] = new ImageType(ImageFormat.Png,       "png",  "Portable Network Graphics");
			fmt[8] = new ImageType(ImageFormat.Tiff,      "tif",  "Tag Image File Format");
			fmt[9] = new ImageType(ImageFormat.Wmf,       "wmf",  "Windows Metafile Format");

			//	create an overall filter for all image types
			fmtFilter = fmt[0].filter;
			for (int i=1; i<fmt.Length; i++)
				fmtFilter += "|" + fmt[i].filter;
			fmtIndex = 0;								// set default format
		}

		/// <summary>
		/// Sets the default image format. The default format is specified
		/// as an index into the image format array fmt.
		/// </summary>
		/// <param name="i">index into image format array fmt</param>
		/// <returns></returns>
		internal static ImageType	SetImageType(int i)
		{
			if (fmt == null)							// if no format array
				InitFormats();							//   create one

			if (i < 0)	i = 0;							// make sure new index not too small
			if (i >= fmt.Length)						// make sure new index not too large
				i = fmt.Length - 1;
			fmtIndex = i;								// set new format index
			return	fmt[fmtIndex];						// return selected format
		}
	}													// end struct ImageType

	/// <summary>
	/// Test Image Capture
	/// </summary>
	public class TestCapture : System.Windows.Forms.Form
	{
		private Color	oldTKey;						// old transparency key
		private bool	viewportOpen = false;			// viewport is closed
		private Bitmap	capBM = null;					// captured bitmap
		private bool	capBMSaved = false;				// capture bitmap not yet saved
		private Control	curCtl = null;					// control containing captured image
		
		//	options
		private bool		crossHairs;					// crosshairs in viewport
		private bool		mute = false;				// mute flag
		private string		soundDir;					// directory of sounds
		private string		imageDir;					// directory of images
		private bool		aspect = false;				// aspect ratio preservation

		private const int	ICON_W = 96;				// max size for 256 color screen
		private const int	ICON_H = 96;				// max size for 256 color screen

		/// <summary>
		/// TestCapture form controls and components
		/// </summary>
		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.MenuItem miAction;
		private System.Windows.Forms.MenuItem miOpenViewport;
		private System.Windows.Forms.MenuItem miCaptureView;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem miCaptureDesktop;
		private System.Windows.Forms.MenuItem miFile;
		private System.Windows.Forms.MenuItem miHelp;
		private System.Windows.Forms.MenuItem miSave;
		private System.Windows.Forms.MenuItem menuItem4;
		private System.Windows.Forms.MenuItem miExit;
		private System.Windows.Forms.MenuItem miAbout;
		private System.Windows.Forms.MenuItem miCloseViewport;
		private System.Windows.Forms.MenuItem miReleaseView;
		private System.Windows.Forms.MenuItem miEdit;
		private System.Windows.Forms.MenuItem menuItem3;
		private System.Windows.Forms.SaveFileDialog sfd;
		private System.Windows.Forms.OpenFileDialog ofd;
		private System.Windows.Forms.MenuItem miLoad;
		private System.Windows.Forms.MenuItem miCrossHairs;
		private System.Windows.Forms.MenuItem miMute;
		private System.Windows.Forms.MenuItem miAspect;
		private System.Windows.Forms.Panel panPic;
		private System.Windows.Forms.MenuItem miCaptureDesktopWA;
		private System.Windows.Forms.MenuItem miCaptureTI;
		private System.ComponentModel.IContainer components = null;

		/// <summary>
		/// TestCapture demonstrates image capture.
		/// </summary>
		internal TestCapture()
		{
			InitializeComponent();						// Required for Windows Form Designer support

			soundDir = Environment.CurrentDirectory		// initialize sound directory
				+ @"\Sounds\";
			imageDir = Environment.GetFolderPath(		// set initial image directory
				Environment.SpecialFolder.MyPictures);
			curCtl = this.panPic;						// control containing image
			oldTKey = TransparencyKey;					// save old transparency key
			ImageType.InitFormats();					// initialize formats
			ofd = InitOFD(null);						// initialize open dialog
			sfd = InitSFD(null);						// initialize save dialog
			miCrossHairs_Click(null,null);				// select crosshair display
			miAspect_Click(null,null);					// select aspect preservation
			Play("Signon.wav");							// play signon sound
		}												// end constructor TestCapture

		/// <summary>
		/// Checks if image has been saved
		/// </summary>
		/// <returns>true if image unsaved</returns>
		private bool	UnsavedBM()
		{
			if ((capBM != null) && !capBMSaved)			// if unsaved image exists
			{
				Play("Error.wav");						//   play unsaved image sound
				MessageBox.Show("Please save or release your captured view or image." );
				return true;
			}
			return false;
		}

		/// <summary>
		/// Plays specified .wav file in default sounds directory
		/// if the muting option is not set.
		/// </summary>
		/// <param name="s">name of .wav file</param>
		private void	Play(string s)
		{
			if (mute)
				return;
			Wav.Play(soundDir+s);
		}

		/// <summary>
		/// Disposes of an image and returns null.
		/// </summary>
		/// <param name="img">image to dispose of</param>
		/// <returns>null</returns>
		private Image	Dispose(Image img)
		{
			if (img == null)
				return null;
			img.Dispose();
			return null;
		}

		/// <summary>
		/// Disposes of a bitmap and returns null.
		/// </summary>
		/// <param name="bm">bitmap to dispose of</param>
		/// <returns>null</returns>
		private Bitmap	Dispose(Bitmap bm)
		{
			if (bm == null)
				return null;
			bm.Dispose();
			return null;
		}

		/// <summary>
		/// Disposes of an icon and returns null.
		/// </summary>
		/// <param name="icon">icon to dispose of</param>
		/// <returns>null</returns>
		private Icon	Dispose(Icon icon)
		{
			if (icon == null)
				return null;
			icon.Dispose();
			return null;
		}

		/// <summary>
		/// Disposes of the curCtl's background image as well
		/// as the current bitmap.
		/// </summary>
		private void	DisposeImages()
		{
			curCtl.BackgroundImage = Dispose(curCtl.BackgroundImage);	// no background image
			capBM = Dispose(capBM);										// no captured bitmap
		}

		/// <summary>
		/// Opens the viewport for image capture. The viewport is the client
		/// area of the form. To open the viewport, components on top of the
		/// client area are hidden and the transparency key is set to the background
		/// color of the form.
		/// </summary>
		private void	OpenViewport()
		{
			if (viewportOpen)							// if viewport already open
				return;									//   return

			if (UnsavedBM())							// if unsaved image exists
				return;									//   return

			TransparencyKey = BackColor;				// make client background transparent
			DisposeImages();							// set background & capture bm to null
			panPic.Visible = false;						// disappear picture panel
			viewportOpen = true;						// viewport is now open
			Refresh();									// refresh client area
			Play("OpenViewport.wav");					// play viewport open sound
		}

		/// <summary>
		/// Closes the image capture viewport. The close the viewport, the
		/// transparency key is restored so the background is once again shown,
		/// and hidden components are made visible.
		/// </summary>
		private void	CloseViewport()
		{
			viewportOpen = false;						// viewport is closed
			TransparencyKey = oldTKey;					// restore old transparency key
			panPic.Visible = true;						// show picture panel
			Refresh();									// refresh client area
			Play("CloseViewport.wav");					// play viewport closed sound
		}

		/// <summary>
		/// If the viewport is open, the image in the viewport is captured. The
		/// image will be shown in control curCtl.
		/// </summary>
		private void	CaptureView()
		{
			if (!viewportOpen)							// if viewport closed
			{
				Play("Error.wav");						//   play capture error sound
				MessageBox.Show(this,
					"You must open the viewport prior to capturing a view!");
				return;
			}
			CloseViewport();							// close viewport
			capBM = Emxx.Capture.Control(this,true,true);	// capture bitmap under client area
			PicInCtl(curCtl,false);						// place new image in current control
			capBMSaved = false;							// captured bitmap not yet saved
			Play("CaptureView.wav");					// play image captured sound
		}

		/// <summary>
		/// Deletes the image shown in control curCtl.
		/// </summary>
		private void	ReleaseView()
		{
			CloseViewport();							// close viewport
			DisposeImages();							// nullify bitmap and background image
			Play("ReleaseView.wav");					// play image released sound
		}

		/// <summary>
		/// Capture image of either entire desktop or desktop work area
		/// </summary>
		/// <param name="entire">If true captures entire desktop else only desktop work area</param>
		private void	CaptureDesktop(bool entire)
		{
			if (UnsavedBM())							// if unsaved image exists
				return;									//   return
			CloseViewport();							// close viewport
			DisposeImages();							// dispose of prior images
			capBM = entire? Emxx.Capture.Desktop()		// capture desktop image
				: Emxx.Capture.DesktopWA(this);			// capture desktop work area image
			PicInCtl(curCtl,false);						// place new image in current control
			capBMSaved = false;							// captured bitmap not yet saved
			Play("CaptureDesktop.wav");					// play desktop captured sound
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(TestCapture));
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.miFile = new System.Windows.Forms.MenuItem();
			this.miLoad = new System.Windows.Forms.MenuItem();
			this.miSave = new System.Windows.Forms.MenuItem();
			this.menuItem4 = new System.Windows.Forms.MenuItem();
			this.miExit = new System.Windows.Forms.MenuItem();
			this.miEdit = new System.Windows.Forms.MenuItem();
			this.menuItem3 = new System.Windows.Forms.MenuItem();
			this.miAspect = new System.Windows.Forms.MenuItem();
			this.miCrossHairs = new System.Windows.Forms.MenuItem();
			this.miMute = new System.Windows.Forms.MenuItem();
			this.miAction = new System.Windows.Forms.MenuItem();
			this.miOpenViewport = new System.Windows.Forms.MenuItem();
			this.miCaptureView = new System.Windows.Forms.MenuItem();
			this.miReleaseView = new System.Windows.Forms.MenuItem();
			this.miCloseViewport = new System.Windows.Forms.MenuItem();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.miCaptureDesktop = new System.Windows.Forms.MenuItem();
			this.miCaptureDesktopWA = new System.Windows.Forms.MenuItem();
			this.miHelp = new System.Windows.Forms.MenuItem();
			this.miAbout = new System.Windows.Forms.MenuItem();
			this.sfd = new System.Windows.Forms.SaveFileDialog();
			this.ofd = new System.Windows.Forms.OpenFileDialog();
			this.panPic = new System.Windows.Forms.Panel();
			this.miCaptureTI = new System.Windows.Forms.MenuItem();
			this.SuspendLayout();
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.miFile,
																					  this.miEdit,
																					  this.miAction,
																					  this.miHelp});
			// 
			// miFile
			// 
			this.miFile.Index = 0;
			this.miFile.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																				   this.miLoad,
																				   this.miSave,
																				   this.menuItem4,
																				   this.miExit});
			this.miFile.Text = "File";
			// 
			// miLoad
			// 
			this.miLoad.Index = 0;
			this.miLoad.Text = "Load Image";
			this.miLoad.Click += new System.EventHandler(this.miLoad_Click);
			// 
			// miSave
			// 
			this.miSave.Index = 1;
			this.miSave.Text = "Save Image";
			this.miSave.Click += new System.EventHandler(this.miSave_Click);
			// 
			// menuItem4
			// 
			this.menuItem4.Index = 2;
			this.menuItem4.Text = "-";
			// 
			// miExit
			// 
			this.miExit.Index = 3;
			this.miExit.Text = "Exit";
			this.miExit.Click += new System.EventHandler(this.miExit_Click);
			// 
			// miEdit
			// 
			this.miEdit.Index = 1;
			this.miEdit.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																				   this.menuItem3,
																				   this.miAspect,
																				   this.miCrossHairs,
																				   this.miMute});
			this.miEdit.Text = "Edit";
			// 
			// menuItem3
			// 
			this.menuItem3.Index = 0;
			this.menuItem3.Text = "-";
			// 
			// miAspect
			// 
			this.miAspect.Index = 1;
			this.miAspect.Text = "Aspect Preserved";
			this.miAspect.Click += new System.EventHandler(this.miAspect_Click);
			// 
			// miCrossHairs
			// 
			this.miCrossHairs.Index = 2;
			this.miCrossHairs.Text = "Cross Hairs";
			this.miCrossHairs.Click += new System.EventHandler(this.miCrossHairs_Click);
			// 
			// miMute
			// 
			this.miMute.Index = 3;
			this.miMute.Text = "Mute";
			this.miMute.Click += new System.EventHandler(this.miMute_Click);
			// 
			// miAction
			// 
			this.miAction.Index = 2;
			this.miAction.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.miOpenViewport,
																					 this.miCaptureView,
																					 this.miReleaseView,
																					 this.miCloseViewport,
																					 this.menuItem1,
																					 this.miCaptureDesktop,
																					 this.miCaptureDesktopWA,
																					 this.miCaptureTI});
			this.miAction.Text = "Action";
			// 
			// miOpenViewport
			// 
			this.miOpenViewport.Index = 0;
			this.miOpenViewport.Text = "Open Viewport";
			this.miOpenViewport.Click += new System.EventHandler(this.miOpenViewport_Click);
			// 
			// miCaptureView
			// 
			this.miCaptureView.Index = 1;
			this.miCaptureView.Text = "Capture View";
			this.miCaptureView.Click += new System.EventHandler(this.miCaptureView_Click);
			// 
			// miReleaseView
			// 
			this.miReleaseView.Index = 2;
			this.miReleaseView.Text = "Release View";
			this.miReleaseView.Click += new System.EventHandler(this.miReleaseView_Click);
			// 
			// miCloseViewport
			// 
			this.miCloseViewport.Index = 3;
			this.miCloseViewport.Text = "Close Viewport";
			this.miCloseViewport.Click += new System.EventHandler(this.miCloseViewport_Click);
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 4;
			this.menuItem1.Text = "-";
			// 
			// miCaptureDesktop
			// 
			this.miCaptureDesktop.Index = 5;
			this.miCaptureDesktop.Text = "Capture Desktop";
			this.miCaptureDesktop.Click += new System.EventHandler(this.miCaptureDesktop_Click);
			// 
			// miCaptureDesktopWA
			// 
			this.miCaptureDesktopWA.Index = 6;
			this.miCaptureDesktopWA.Text = "Capture Desktop Work Area";
			this.miCaptureDesktopWA.Click += new System.EventHandler(this.miCaptureDesktopWA_Click);
			// 
			// miHelp
			// 
			this.miHelp.Index = 3;
			this.miHelp.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																				   this.miAbout});
			this.miHelp.Text = "Help";
			// 
			// miAbout
			// 
			this.miAbout.Index = 0;
			this.miAbout.Text = "About";
			this.miAbout.Click += new System.EventHandler(this.miAbout_Click);
			// 
			// panPic
			// 
			this.panPic.BackColor = System.Drawing.Color.PowderBlue;
			this.panPic.Location = new System.Drawing.Point(0, 0);
			this.panPic.Name = "panPic";
			this.panPic.Size = new System.Drawing.Size(184, 96);
			this.panPic.TabIndex = 0;
			// 
			// miCaptureTI
			// 
			this.miCaptureTI.Index = 7;
			this.miCaptureTI.Text = "Capture TestImage";
			this.miCaptureTI.Click += new System.EventHandler(this.miCaptureTI_Click);
			// 
			// TestCapture
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.BackColor = System.Drawing.Color.PowderBlue;
			this.ClientSize = new System.Drawing.Size(224, 177);
			this.Controls.Add(this.panPic);
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.Menu = this.mainMenu1;
			this.Name = "TestCapture";
			this.Text = "Test Image Capture";
			this.TopMost = true;
			this.Closing += new System.ComponentModel.CancelEventHandler(this.TestForm_Closing);
			this.SizeChanged += new System.EventHandler(this.TestCapture_SizeChanged);
			this.Paint += new System.Windows.Forms.PaintEventHandler(this.TestForm_Paint);
			this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// The main entry point for application TestCapture
		/// </summary>
		[STAThread]
		static void Main() 
		{
			Application.Run(new TestCapture());
		}												// end startup method Main

		/// <summary>
		/// Menu selection to open the image capture viewport.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miOpenViewport_Click(object sender, System.EventArgs e)
		{
			OpenViewport();								// open client area viewport
		}												// end action miOpenViewport_Click

		/// <summary>
		/// Menu selection to close the image capture viewport.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miCloseViewport_Click(object sender, System.EventArgs e)
		{
			CloseViewport();							// close client area viewport		
		}												// end action miCloseViewport_Click

		private void miCaptureTI_Click(object sender, System.EventArgs e)
		{
			capBM = Emxx.Capture.Control(this,false,false);	// capture bitmap for control
			PicInCtl(curCtl,false);						// place new image in current control
			capBMSaved = false;							// captured bitmap not yet saved
			Play("CaptureView.wav");					// play image captured sound
		}

		/// <summary>
		/// Menu selection to capture the image in the capture viewport.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miCaptureView_Click(object sender, System.EventArgs e)
		{
			CaptureView();								// capture view in client area viewport
		}												// end action miCaptureView_Click

		/// <summary>
		/// Menu selection to release/delete the loaded/captured image.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miReleaseView_Click(object sender, System.EventArgs e)
		{
			ReleaseView();								// release captured view or image		
		}												// end action miReleaseView_Click

		/// <summary>
		/// Menu selection to capture the entire desktop as an image.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miCaptureDesktop_Click(object sender, System.EventArgs e)
		{
			CaptureDesktop(true);						// capture view of entire desktop
		}												// end action miCaptureDesktop_Click

		/// <summary>
		/// Menu selection to capture the desktop work area as an image
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miCaptureDesktopWA_Click(object sender, System.EventArgs e)
		{
			CaptureDesktop(false);						// capture view of desktop work area
		}												// end action TestCapture_SizeChanged

		/// <summary>
		/// Menu selection to exit the applictaion. To exit, this action
		/// method requests form closure.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miExit_Click(object sender, System.EventArgs e)
		{
			this.Close();								// close the form
		}												// end action miExit_Click

		/// <summary>
		/// Exit the application when the form is closing.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TestForm_Closing(object sender, System.ComponentModel.CancelEventArgs e)
		{
			if ((capBM != null) && !capBMSaved)			// if unsaved image exists
				miSave_Click(null,null);				//   allow it to be saved
			Play("Signoff.wav");						// play signoff sound
			Thread.Sleep(1000);							// allow signoff to play for a sec
			Application.Exit();							// exit the application
		}												// end action TestForm_Closing

		/// <summary>
		/// Initialize save file dialog.
		/// </summary>
		/// <param name="s">save file dialog</param>
		/// <returns>save file dialog</returns>
		private SaveFileDialog	InitSFD(SaveFileDialog s)
		{
			if (s != null)								// if save dialog exists
				return s;								//   return
			s = new SaveFileDialog();					// create save file dialog
			s.InitialDirectory = imageDir;				// initial directory
			s.Title = "Save Captured Image";			// title
			s.CheckPathExists = true;					// make sure path exists

			//	create filter
			s.Filter = ImageType.fmtFilter;				// use filter for all formats
			return s;
		}												// end method InitSFD

		/// <summary>
		/// Menu selection to save the image to a file.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miSave_Click(object sender, System.EventArgs e)
		{
			if (capBM == null)							// if no bitmap
				return;									//   return
			DialogResult dr = sfd.ShowDialog(this);		// show the dialog
			if (dr != DialogResult.OK)					// if not ok
				return;									//   forget it

			string	fn = sfd.FileName;					// pick up file path
			if (fn.Equals(""))							// if no filename
			{
				Play("Error.wav");						//   play nothing saved sound
				MessageBox.Show("No filename specified, nothing saved");
				return;
			}

			ImageType it = ImageType.SetImageType(sfd.FilterIndex-1);	// set default image type to selected filter format
			if (it.format == ImageFormat.Icon)			// if saving an icon
			{
				Icon = BitmapToIcon(capBM,aspect);		//   convert bitmap to icon
				Stream	s = sfd.OpenFile();				//   open file
				Icon.Save(s);							//   save the icon
				s.Close();								//   close file
			}
			else										// if saving other format
				capBM.Save(fn,it.format);				//   use generic image save
			capBMSaved = true;							// image has been saved
			Play("SaveView.wav");						// play image saved sound
		}												// end action miSave_Click

		/// <summary>
		/// Convert bitmap to icon preserving aspect if requested
		/// </summary>
		/// <param name="obm">original bitmap</param>
		/// <param name="preserve">true if preserving aspect</param>
		/// <returns>icon size bitmap</returns>
		private Icon	BitmapToIcon(Bitmap obm,bool preserve)
		{
			Bitmap		bm;
			//	if not preserving aspect
			if (!preserve)								// if not preserving aspect
				bm = new Bitmap(obm,ICON_W,ICON_H);		//   rescale from original bitmap

			//	if preserving aspect drop excess significance in least significant direction
			else										// if preserving aspect
			{
				Rectangle	rc = new Rectangle(0,0,ICON_W,ICON_H);
				if (obm.Width >= obm.Height)			//   if width least significant
				{										//     rescale with width based on max icon height
					bm = new Bitmap(obm,(ICON_H*obm.Width)/obm.Height,ICON_H);
					rc.X = (bm.Width - ICON_W) / 2;		//     chop off excess width significance
					if (rc.X < 0)	rc.X = 0;
				}
				else									//   if height least significant
				{										//     rescale with height based on max icon width
					bm = new Bitmap(obm,ICON_W,(ICON_W*obm.Height)/obm.Width);	
					rc.Y = (bm.Height - ICON_H) / 2;	//     chop off excess height significance
					if (rc.Y < 0)	rc.Y = 0;
				}
				bm = bm.Clone(rc,bm.PixelFormat);		//   bitmap for icon rectangle
			}

			//	create icon from bitmap
			Icon	icon = Icon.FromHandle(bm.GetHicon());	// create icon from bitmap
			bm.Dispose();								// dispose of bitmap
			return icon;								// return icon
		}												// end method BitmapToIcon

		/// <summary>
		/// Initialize open file dialog
		/// </summary>
		/// <param name="o">open file dialog</param>
		/// <returns>open file dialog</returns>
		private OpenFileDialog	InitOFD(OpenFileDialog o)
		{
			if (o != null)								// if open dialog exists
				return o;								//   return

			o = new OpenFileDialog();					// create open file dialog
			o.InitialDirectory = imageDir;				// initial directory
			o.Title = "Load Image File";				// title
			o.CheckPathExists = true;					// make sure path exists
			o.CheckFileExists = true;					// make sure file exists

			//	create filter
			o.Filter =	"Most Images(*.BMP;*.JPG;*.GIF;*.TIF;*.PNG)"
				+				"|*.BMP;*.JPG;*.GIF;*.TIF;*.PNG"
				+		"|Some Images(*.EMF;*.WMF;*.EXIF;*.MBMP)"
				+				"|*.EMF;*.WMF;*.EXIF;*.MBMP"
				+		"|Icon Files(*.ICO)"
				+				"|*.ICO"
				+		"|All files (*.*)"
				+				"|*.*";

			return o;
		}												// end method InitOFD

		/// <summary>
		/// Menu selection to load an image from a file.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miLoad_Click(object sender, System.EventArgs e)
		{
			if (UnsavedBM())							// if unsaved image exists
				return;									//   return
			CloseViewport();							// close viewport

			DialogResult dr = ofd.ShowDialog(this);		// show the dialog
			if (dr != DialogResult.OK)					// if not ok
				return;									//   forget it
	
			string	fn = ofd.FileName;					// pick up file path
			int	idx = fn.LastIndexOf("\\");				// get last backslash
			sfd.InitialDirectory = (idx <= 0)? "" : fn.Substring(0,idx);
			sfd.FileName = fn;							// make default for save
			Dispose(capBM);								// dispose of previous bitmap (if any)
			
			idx = fn.LastIndexOf(".") + 1;				// find start of extension
			string	ext = (idx > 0) && (idx < fn.Length)? fn.Substring(idx) : "";
			if (ext.ToLower().Equals("ico"))			// if file is an icon
			{
				this.Icon = new Icon(fn);				//   read new form icon from file
				capBM = this.Icon.ToBitmap();			//   convert to bitmap
			}
			else										// if file not an icon
				capBM = new Bitmap(Image.FromFile(fn));	//   read bitmap from file

			PicInCtl(curCtl,false);						// place new image in current control
			capBMSaved = true;							// image has been saved, it came from a file
			Play("LoadView.wav");						// play load view sound
		}												// end action miLoad_Click

		/// <summary>
		/// Menu seleciton to toggle the crosshairs option. When set,
		/// this option causes crosshairs to be drawn in the viewport
		/// during when seeking an image to capture.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miCrossHairs_Click(object sender, System.EventArgs e)
		{
			miCrossHairs.Checked = !miCrossHairs.Checked;	// toggle crosshairs option
			crossHairs = miCrossHairs.Checked;			// update crosshairs flag
			Refresh();									// refresh client area
		}												// end action miCrossHairs_Click

		/// <summary>
		/// Paints the crosshairs when the viewport is open and the crosshairs
		/// option is set.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TestForm_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			if (crossHairs && viewportOpen)								// if viewport open and doing crosshairs
			{
				int			rad = 20;									//   set radius
				Rectangle	r = ClientRectangle;						//   get client area rectangle
				Point		c = new Point(r.X + r.Width / 2,			//   find its center point
					r.Y + r.Height / 2);
				Rectangle	cr = new Rectangle(c.X - rad, c.Y - rad		//   create rectange about center
					, 2 * rad, 2 * rad);								//   with width & height twice radius
				Pen			p = new Pen(Color.Black);
				e.Graphics.DrawEllipse(p,cr);							//   draw a circular scope
				e.Graphics.DrawLine(p, c.X-rad, c.Y, c.X+rad, c.Y);		//   draw horizontal crosshair
				e.Graphics.DrawLine(p, c.X, c.Y-rad, c.X, c.Y+rad);		//   draw vertical crosshair
			}
		}												// end action TestForm_Paint

		/// <summary>
		/// Menu selection to toggle the mute option. Sounds are not
		/// played when the mute option is set.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miMute_Click(object sender, System.EventArgs e)
		{
			miMute.Checked = !miMute.Checked;			// toggle mute option
			mute = miMute.Checked;						// update mute flag
		}												// end action miMute_Click

		/// <summary>
		/// Menu selection to toggle the aspect option. The aspect ration of
		/// the image is retained during resize operations if the aspect option
		/// is set.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miAspect_Click(object sender, System.EventArgs e)
		{
			miAspect.Checked = !miAspect.Checked;		// toggle aspect option
			aspect = miAspect.Checked;					// update aspect flag
			PicInCtl(curCtl,false);						// force repaint of picture in control
		}												// end action miAspect_Click

		/// <summary>
		/// Menu selection to show product information.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void miAbout_Click(object sender, System.EventArgs e)
		{
			MessageBox.Show(this 
				,	"Hi " + Environment.UserName + "!"
				+	"\n\nVersion: " + ProductVersion 
				+	"\nCopyright 2003-2004 " + CompanyName  
				+	"\n\nSysDir: " + Environment.SystemDirectory
				+	"\n" + System.AppDomain.CurrentDomain.ToString()
				,	"About " + ProductName
				,	MessageBoxButtons.OK
				,	MessageBoxIcon.Information
				);
		}												// end action miAbout_Click

		/// <summary>
		/// Place the current image/picture in the specified control.
		/// </summary>
		/// <param name="c">control</param>
		/// <param name="oldImage">is it an old image flag</param>
		private void	PicInCtl(Control c,bool oldImage)
		{
			if (viewportOpen || capBM == null)			// if viewport open or no bitmap
				return;									//   return

			Control	pc = c.Parent;						// get parent container control
			if (pc == null)	return;						//   return if none

			//	update size of control's client area
			Size	cs = pc.ClientSize;					// assume parent's entire client area
			if ((cs.Width==0) || (cs.Height==0))		// if zero width or height
				return;									//   return

			//	calc control's client area to retain image aspect
			if (aspect)									// if retaining aspect
			{
				int		mxW = cs.Width;					//   get max width
				int		mxH = cs.Height;				//   get max height
				int		asW =							//   calc aspect width from max height
					(capBM.Width * mxH) / capBM.Height;
				int		asH =							//   calc aspect height from max width
					(capBM.Height * mxW) / capBM.Width;
				
				if (asW <= mxW)							//   if max height yields in range width
				{
					cs.Width = asW;						//     use aspect width
					cs.Height = mxH;					//     use max height
				}
				else									//   if max width yields in range height
				{
					cs.Width = mxW;						//     use max width
					cs.Height = asH;					//     use aspect height
				}
			}

			//	check existing image size
			c.ClientSize = cs;							// set client area for control
			if (oldImage && (c.BackgroundImage != null))		// if old background image exists
			{
				if (	(c.BackgroundImage.Width == cs.Width)	//   if image already client area size
					&&	(c.BackgroundImage.Height == cs.Height))
				{
					return;										//     return
				}
			}

			//	update image size for control's new client area
			Dispose(c.BackgroundImage);					// dispose of previous background image
			c.BackgroundImage = new Bitmap(capBM,cs);	// resize background image
		}												// end method PicInCtl

		/// <summary>
		/// Update image size when TestCapture form size changes.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TestCapture_SizeChanged(object sender, System.EventArgs e)
		{
			PicInCtl(curCtl,true);						// place picture in control
		}

	}													// end class TestCapture
}														// end namespace ImageCapture
