//  Wav Module		Play Wav Sounds
//  Copyright 2003-2004 Excellence Foundation		Don@xfnd.org
//
//  This product is free software. You can redistribute it and/or modify it under the
//  terms of the Public Source Distribution License Agreement (PSDLA) as published by
//  Excellence Foundation.
//
//  This product is distributed in the hope it will be instructional and useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the PSDLA for more details. You should have
//  received a copy of the PSDLA with this product. If not, browse
//  http://www.xfnd.org/licenses/PSDLA.doc.

using System;
using System.IO;
using Dll;

namespace Emxx
{	
	/// <summary>
	/// Play .WAV files
	/// </summary>
	public class Wav
	{

		/// <summary>
		/// Parameter Flags for dll PlaySound
		/// </summary>
		public const uint	SND_SYNC		= 0x0000;		// play synchronously (default)
		public const uint	SND_ASYNC		= 0x0001;		// play asynchronously
		public const uint	SND_NODEFAULT	= 0x0002;		// silence (!default) if sound not found
		public const uint	SND_MEMORY		= 0x0004;		// pszSound points to a memory file
		public const uint	SND_LOOP		= 0x0008;		// loop the sound until next sndPlaySound
		public const uint	SND_NOSTOP		= 0x0010;		// don't stop any currently playing sound
		public const uint	SND_NOWAIT		= 0x00002000;	// don't wait if the driver is busy
		public const uint	SND_ALIAS		= 0x00010000;	// name is a registry alias
		public const uint	SND_ALIAS_ID	= 0x00110000;	// alias is a pre d ID
		public const uint	SND_FILENAME	= 0x00020000;	// name is file name
		public const uint	SND_RESOURCE	= 0x00040004;	// name is resource name or atom
		public const uint	SND_PURGE		= 0x0040;		// purge non-static events for task
		public const uint	SND_APPLICATION	= 0x0080;		// look for application specific association
	
		/// <summary>
		/// Play specified wav file
		/// </summary>
		/// <param name="path">wav file path</param>
		/// <param name="hmod"></param>
		/// <param name="parameters">parameters (or of desired paramters in enum WavParm</param>
		/// <returns>true if sound is played</returns>
		public static bool	Play(String path,int hmod,uint parameters)
		{
			if (path != null)
			{
				bool	playFile = (parameters & SND_FILENAME) == SND_FILENAME;
				if (playFile && !File.Exists(path))			// if playing non-existent file
					return false;						//   ignore it
			}
			int ok = WINMM.PlaySound(path,hmod,(int)parameters);
			return ok != 0;
		}												// end method Play

		/// <summary>
		/// Play specified wav file
		/// </summary>
		/// <param name="path">wav file path</param>
		/// <param name="parameters">parameters (or of desired paramters in enum WavParm </param>
		/// <returns>true if sound is played</returns>
		public static bool	Play(String path,uint parameters)
		{
			int	hmod = 0;
			return Play(path,hmod,parameters);
		}												// end method Play

		/// <summary>
		/// Plays specified wav file
		/// </summary>
		/// <param name="path">wav file path</param>
		/// <returns>true if sound is played</returns>
		public static bool	Play(String path)
		{
			uint	parameters = 						// create parameters
					( SND_ASYNC							// play asynchronously
					| SND_FILENAME						// play a file
					| SND_NOWAIT						// play immediately
					);
			return Play(path,parameters);				// play wav file
		}												// end method Play

		/// <summary>
		/// Plays specified wav file (synchronously
		/// </summary>
		/// <param name="path">wav file path</param>
		/// <returns>true if sound is played</returns>
		public static bool	PlaySync(String path)
		{
			uint	parameters = 						// create parameters
				( SND_SYNC								// play synchronously
				| SND_FILENAME							// play a file
				| SND_NOWAIT							// play immediately
				);
			return Play(path,parameters);				// play wav file
		}												// end method Play

		/// <summary>
		/// Play System Sound Event, A registry alias stored in directory
		/// HKEY_CURRENT_USER\AppEvents\Schemes\Apps\.Default
		/// </summary>
		/// <param name="path">System Event Alias</param>
		/// <returns>true if sound is played</returns>
		public static bool	PlaySoundEvent(String path)
		{	
			uint	parameters =					// create parameters
				( SND_ASYNC							// play asynchronously
				| SND_ALIAS							// play registry alias
				| SND_NOWAIT						// play immediately
				);
			return Play(path,parameters);			// play sound
		}											// end method PlaySoundEvent

		/// <summary>
		/// Stops the currently playing asynchronous sound
		/// </summary>
		public static void	Stop()
		{
			Play(null,SND_ASYNC);					// stop playing asynchronous sound
		}

	}												// end class Wav
}													// end namespace Emxx

