/*
 * Copyright (c) 2015 Adobe Systems.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, regexp: true, indent: 4, maxerr: 50 */
/*global require, exports, process */
"use strict";

var USBMUXBinaryProtocol  = require("./USBMUXBinaryProtocol").USBMUXBinaryProtocol,
    USBMUXPlistProtocol   = require("./USBMUXPlistProtocol").USBMUXPlistProtocol,
    CONSTANTS             = require("./USBMUXConstants");

// Helper
/**
 * Create a instance of the protocol specified by usbmuxProtocolVersion.
 *
 * @param {!Number} usbmuxProtocolVersion either 0 (BinaryProtocol) or 1 (PlistProtocol)
 * @param {!USBMUXSocket} usbmuxSocket  the socket that will be used in conjunction with the protocol.
 * @returns {(USBMUXBinaryProtocol|USBMUXPlistProtocol)} instance of the protocol
 *
 * @throws {Error} if an unknown protocol version is specified
 */
function USBMUXProtocolFactory(usbmuxProtocolVersion, usbmuxSocket) {
    switch (usbmuxProtocolVersion) {
        case CONSTANTS.USBMUX_PROTOCOL_VERSION_BINARY: {
            return new USBMUXBinaryProtocol(usbmuxSocket);
        }

        case CONSTANTS.USBMUX_PROTOCOL_VERSION_PLIST: {
            return new USBMUXPlistProtocol(usbmuxSocket);
        }

        default:
            throw new Error("Undefined Protocol version:", usbmuxProtocolVersion);
    }
}

/**
 * Create an socketOptions object for the specified platform or for the current platform.
 *
 * @param {string=} platform    darwin or windows
 * @returns {{address}} the unix domain socket or ip address to use to connect to usbmuxd
 */
function getUSBMUXSocketOptions(platform) {
    var socketOptions = {},
        _platform = platform || process.platform;

    if (_platform === "darwin") {
        socketOptions.address = CONSTANTS.DOMAIN_SOCKET_PATH;
    } else {
        socketOptions.address = CONSTANTS.USBMUXD_SOCKET_PORT;
    }

    return socketOptions;
}

// API
exports.USBMUXProtocolFactory   = USBMUXProtocolFactory;
exports.getUSBMUXSocketOptions  = getUSBMUXSocketOptions;
