//==============================================================================
// Scn28p234: AI Scenario Script for scenario 28 players 2, 3 and 4
//==============================================================================
/*
   AI owner:  Mike Kidd
   Scenario owner: Joe Gillum

   Overview:
   The player faces three computer opponents, each in a fortified town.  The
   player's goal is to defeat each town.  The CP towns will convert to the human
   player when the hero for each town is defeated.

   The hook is that the player must use the flag bearer to infuriate enemy armies 
   and lure them out of the cities.  

   Each CP is constructed as a hero in the center of the town, and six armies
   at various locations within the town.  When the flag bearer gets within 
   attackRadius meters of any of the small armies, that little army goes out and
   attacks.  If the flag bearer gets within townRadius meters of the center, the 
   whole town attacks via resetting the defend plan to "defend" the enemy flag bearer.

   The player's best strategy is to get close enough to lure individual armies out,
   but not close enough to trigger the swarm until he has defeated several individual armies.

   Casting healing spring will help.  Joe will add periodic reinforcements to the armies,
   so that the player must move quickly to capture the towns before they get too large.

   Difficulty will be controlled by army sizes.

   This code runs for all three CPs.  Main() checks the player's ID number, and then uses
   the corresponding cinematic blocks.  Two rules are used - a low-speed rule when the flag-bearer
   is not on this player's map section, and a high-speed rule when the flag bearer is near.  
   This rule checks for proximity to the center, or to any not-yet-angered armies.

*/
//==============================================================================


include "scn lib.xs";

vector   townCenter = cInvalidVector;
vector   army1 = cInvalidVector;
vector   army2 = cInvalidVector;
vector   army3 = cInvalidVector;
vector   army4 = cInvalidVector;
vector   army5 = cInvalidVector;
vector   army6 = cInvalidVector;
float    townRadius = 39.0;               // How close does the flagbearer need to get to lure hero?
float    attackRadius = 30.0;             // How close to each attack group to draw them out?
float    attackGroupGatherRadius = 7.0;   // Used to determine who gets pulled into each attack group.

string   flagBearerString = "3732";
int      flagBearer = -1;                 // Object ID of flag bearer
int      defendPlan = -1;
bool     flagBearerIsNear = true;         // Flipped to true when flag bearer is on my part of the map, used
                                          // to limit high-frequency rule firing

bool     army1Active = false;
bool     army2Active = false;
bool     army3Active = false;
bool     army4Active = false;
bool     army5Active = false;
bool     army6Active = false;
bool     defendPlanEngaged = false;


void attack(vector center=cInvalidVector)
{
   int   attackID=aiPlanCreate("Attack at "+timeString()+"  ", cPlanAttack);
   if (attackID < 0)
   {
      return;
   }

   if (aiPlanSetVariableInt(attackID, cAttackPlanPlayerID, 0, 1) == false)
   {
      return;
   }

   if (aiPlanSetNumberVariableValues(attackID, cAttackPlanTargetTypeID, 2, true) == false)
   {
      return;
   }


   // add "unit" and "building" to attack list
   aiPlanSetVariableInt(attackID, cAttackPlanTargetTypeID, 0, cUnitTypeUnit);
   aiPlanSetVariableInt(attackID, cAttackPlanTargetTypeID, 1, cUnitTypeBuilding);

   aiPlanSetVariableInt(attackID, cAttackPlanRefreshFrequency, 0, 30);

   aiPlanSetVariableVector(attackID, cAttackPlanGatherPoint, 0, center);
   aiPlanSetVariableFloat(attackID, cAttackPlanGatherDistance, 0, 100.0);     // Gather instantly...essentially skips gather step

   // Figure out how many units we have here
   int query = kbUnitQueryCreate("Query at "+timeString());
   configQuery(query, cUnitTypeMilitary, -1, cUnitStateAlive, cMyID, center, false, attackGroupGatherRadius);
   int qty = 0;
   kbUnitQueryResetResults(query);
   qty = kbUnitQueryExecute(query);

   // And recruit them
   aiPlanAddUnitType(attackID, cUnitTypeMilitary, 0, qty, qty);

//   aiPlanSetInitialPosition(attackID, center);
   aiPlanSetInitialPosition(attackID, kbUnitGetPosition(flagBearer));      // Have it form near the flag bearer
   aiPlanSetUnitStance(attackID, cUnitStanceAggressive);
   aiPlanSetRequiresAllNeedUnits(attackID, true);
   aiPlanSetActive(attackID);
   aiEcho("Activating attack plan "+attackID);

   // And pop those initital units into the plan, and move them to attack the flagbearer while the attack plan is figuring stuff out
   // This is an ugly, evil, revolting hack to get around the 10-20 seconds delay that attack plans need to before taking action.
   int i=0;
   int unit=0;
   for (i=0; < qty)
   {
      unit = kbUnitQueryGetResult(query, i);
      aiPlanAddUnit(attackID, unit);
      aiTaskUnitWork(unit, flagBearer);   // Attack the flag bearer
      aiEcho("Tasking unit "+unit+" to attack the flagbearer.");
   }


}



float getDistanceSquared(vector v1=cInvalidVector, vector v2=cInvalidVector)
{
   float dx= 0.0;
   float dz= 0.0;
   
   dx = xsVectorGetX(v1) - xsVectorGetX(v2);
   if (dx < 0)
      dx = -1 * dx;

   dz = xsVectorGetZ(v1) - xsVectorGetZ(v2);
   if (dz < 0)
      dz = -1 * dz;

   return((dx*dx)+(dz*dz));
}




void wakeup(int ignore = -1)
{
   aiEcho("Wakeup running at "+timeString());
   // Init defend plan for all units, to not move initially, and to not be lured out.
   defendPlan =aiPlanCreate("Defend Plan", cPlanDefend);
   if (defendPlan >= 0)
   {
      aiPlanAddUnitType(defendPlan, cUnitTypeMilitary, 0, 200, 200);    // All unassigned mil units
      aiPlanSetDesiredPriority(defendPlan, 10);                       // Way low, below scouting and attack
      aiPlanSetVariableVector(defendPlan, cDefendPlanDefendPoint, 0, townCenter);
      aiPlanSetVariableFloat(defendPlan, cDefendPlanEngageRange, 0, 20.0);
      aiPlanSetVariableBool(defendPlan, cDefendPlanPatrol, 0, false);
      aiPlanSetVariableFloat(defendPlan, cDefendPlanGatherDistance, 0, 50.0);
      aiPlanSetInitialPosition(defendPlan, townCenter);
      aiPlanSetUnitStance(defendPlan, cUnitStanceDefensive);

      aiPlanSetVariableInt(defendPlan, cDefendPlanRefreshFrequency, 0, 5);
      aiPlanSetNumberVariableValues(defendPlan, cDefendPlanAttackTypeID, 2, true);
      aiPlanSetVariableInt(defendPlan, cDefendPlanAttackTypeID, 0, cUnitTypeUnit);
      aiPlanSetVariableInt(defendPlan, cDefendPlanAttackTypeID, 1, cUnitTypeBuilding);    // Just in case...

      aiPlanSetActive(defendPlan); 
      aiEcho("Creating defend plan");
   }
}



void main()
{
   aiEcho("Starting Scn07p2.xs");
   aiRandSetSeed();

   //Calculate some areas.
   kbAreaCalculate(1200.0);

   aiSetAttackResponseDistance(25.0);

   // Kill escrows
   kbEscrowSetPercentage( cEconomyEscrowID, cAllResources, 0.0);
   kbEscrowSetPercentage( cMilitaryEscrowID, cAllResources, 0.0);
   kbEscrowAllocateCurrentResources();


   flagBearer = kbGetBlockID(flagBearerString);

   aiEcho("Difficulty = "+aiGetWorldDifficulty());

   switch(cMyID)
   {
   case 2:
      {
         aiEcho("I am player #"+cMyID+".");
         townCenter = kbGetBlockPosition("3807");
         army1 = kbGetBlockPosition("3808");
         army2 = kbGetBlockPosition("3809");
         army3 = kbGetBlockPosition("3810");
         army4 = kbGetBlockPosition("3811");
         army5 = kbGetBlockPosition("3812");
         army6 = kbGetBlockPosition("3813");
         townRadius = 35.0;               // How close does the flagbearer need to get to lure hero?
         attackRadius = 30.0;             // How close to each attack group to draw them out?
         attackGroupGatherRadius = 10.0;   // Used to determine who gets pulled into each attack group.
         break;
      }
   case 3:
      {
         aiEcho("I am player #"+cMyID+".");
         townCenter = kbGetBlockPosition("3814");
         army1 = kbGetBlockPosition("3815");
         army2 = kbGetBlockPosition("3816");
         army3 = kbGetBlockPosition("3817");
         army4 = kbGetBlockPosition("3818");
         army5 = kbGetBlockPosition("3819");
         army6 = kbGetBlockPosition("3820");
         townRadius = 45.0;               // How close does the flagbearer need to get to lure hero?
         attackRadius = 35.0;             // How close to each attack group to draw them out?
         attackGroupGatherRadius = 15.0;   // Used to determine who gets pulled into each attack group.
         break;
      }
   case 4:
      {
         aiEcho("I am player #"+cMyID+".");
         townCenter = kbGetBlockPosition("3821");
         army1 = kbGetBlockPosition("3822");
         army2 = kbGetBlockPosition("3823");
         army3 = kbGetBlockPosition("3824");
         army4 = kbGetBlockPosition("3825");
         army5 = kbGetBlockPosition("3826");
         army6 = kbGetBlockPosition("3827");
         townRadius = 40.0;               // How close does the flagbearer need to get to lure hero?
         attackRadius = 25.0;             // How close to each attack group to draw them out?
         attackGroupGatherRadius = 10.0;   // Used to determine who gets pulled into each attack group.
         break;
      }
   }
   kbSetTownLocation(townCenter);
}



rule isFlagBearerNear         // Is active when flag bearer isn't near, it looks for his approach.
   inactive
   minInterval 5
{
   vector flagPosition = kbUnitGetPosition(flagBearer);
   if (getDistanceSquared(flagPosition, townCenter) < (70*70))
   {           // Argh!  He approaches!
      xsDisableSelf();
      xsEnableRule("watchFlagBearer");
      aiEcho("Elvis has entered the building.");
   }
}



rule watchFlagBearer          // Is active when the flag bearer IS near the town.
   active
   // no mininterval, must run fast
{
   vector flagPosition = kbUnitGetPosition(flagBearer);
   if (getDistanceSquared(flagPosition, townCenter) > (70*70))    // He's gone, calm down
   {
      aiEcho("Elvis has left the building.");
      xsDisableSelf();
      xsEnableRule("isFlagBearerNear");
   }
   else                                                  // He's here, check everything
   {
      if (getDistanceSquared(flagPosition, townCenter) < (townRadius * townRadius))    // Hero should respond, reset defend parameters
      {
         if (defendPlanEngaged == false)
         {
            aiPlanSetVariableInt(defendPlan, cDefendPlanDefendTargetID, 0, flagBearer);         // "Defend" the flag bearer. ;-)
            aiPlanSetVariableInt(defendPlan, cDefendPlanEngageRange, 0, 20.0);
            aiPlanSetUnitStance(defendPlan, cUnitStanceAggressive);
            aiEcho("Kill the flag bearer!");
            defendPlanEngaged = true;
            army1Active = true;
            army2Active = true;
            army3Active = true;
            army4Active = true;
            army5Active = true;
            army6Active = true;
			if (aiGetWorldDifficulty() == 3)
				aiCastGodPowerAtPosition(cTechFlamingWeapons, townCenter);
         }

      }
      else     // Hero not lured, check individual armies
      {
         if (army1Active == false)
         {
            if (getDistanceSquared(flagPosition, army1) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 1");
               attack(army1);
               army1Active = true;
            }
         }
         if (army2Active == false)
         {
            if (getDistanceSquared(flagPosition, army2) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 2");
               attack(army2);
               army2Active = true;
            }
         }
         if (army3Active == false)
         {
            if (getDistanceSquared(flagPosition, army3) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 3");
               attack(army3);
               army3Active = true;
            }
         }
         if (army4Active == false)
         {
            if (getDistanceSquared(flagPosition, army4) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 4");
               attack(army4);
               army4Active = true;
            }
         }
         if (army5Active == false)
         {
            if (getDistanceSquared(flagPosition, army5) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 5");
               attack(army5);
               army5Active = true;
            }
         }
         if (army6Active == false)
         {
            if (getDistanceSquared(flagPosition, army6) < (attackRadius * attackRadius))
            {
               aiEcho("Activating Army 6");
               attack(army6);
               army6Active = true;
            }
         }
      }
   }
}
