//==============================================================================
// Scn08p2: AI Scenario Script for scenario 8 player 2
//==============================================================================
/*
   AI owner:  Mike Kidd
   Scenario owner: Jeff Brown

   Overview:
   The human player rescues Ajax from the Trojan army, then joins forces at the 
   allied TC.  His goal is to build up and take out the red town to the north, without
   the availability of siege units.

   The CP sends one trivial attack early, which dies hopelessly to the HP's huge 
   army left over from the start.  Several of the following attack rules are currently 
   implementedby triggers which attempt to build military buildings near
   the HP's town.
   
   The CP attacks down any one of three channels, with the west channel reserved for 
   higher difficulty levels.

   No attempt is made to train units out of the forward buildings, even if the 
   feeble attempt to build succeeds.  If the forward building concept stays in, we should
   add rules to make a local army out of each building and use them for extra attacks.   

   CP starts as age 2 with Animal Magnet (not used) and Pestilence.
   Age 3: Aphrodite: Curse
   Age 4: Hephaestus: Plenty
     
   Difficulty:  First try on 7/9/2002.  Deactivated the entire attack rule chain,
   jumped straight to attack generator.

   Defend plan added 7/19/2002.

   Increased Titan difficulty on 9/10/2002.

   9/17/2002:  Tweaked titan back down a touch.
*/
//==============================================================================


include "scn lib.xs";

// Globals
int         lastAttackPlanID = -1;        // Updated with plan ID as each attack is launched
int         defendPlan = -1;
int         reservePlan = -1;             // Extra defend plan, higher priority, for final resistance

int         startTime = -1;               // Updated with start time when cinematic is done.  In seconds.
int         age3Time = 1500;               // Time (in seconds) to go to age 3
int         age4Time = 10000;              

int         siegeMaintainPlan = -1;
int         mythMaintainPlan = -1;
int         hopliteMaintainPlan = -1;
int         toxotesMaintainPlan = -1;
int         hippikonMaintainPlan = -1;
int         villagerMaintainPlan = -1;

int         routeWestPass = -1;
int         routeNorthPass = -1;
int         routeEastPass = -1;

// Initial settings are for the easiest difficulty level, others are set in main().
int         nextAttackTime = 480000;     // 8 minutes
float       attackSize = 4.0; 
float       maxAttackSize = 9.0;
float       attackSizeMultiplier = 1.2; 
int         attackInterval = 300000;      // 5 minutes
float       reserveSize = 15.0;           // Army to keep in reserve
int         attackCount = 0;
int         trainDelay = 30;
int         cavTrainDelay = 30;
int         archerTrainDelay = 15;
int         infantryTrainDelay = 20;

int builderQty = 1;     // For forward building
int defendQty = 0;      // For escort group

// Cinematic block markers
const string cbVillagerGather = "1872";
const string cbSiegeGather = "1873";
const string cbArcherGather = "1878";
const string cbCavalryGather = "1874";
const string cbInfantryGather = "1875";
const string cbP2TC = "1876";
const string cbGateEast = "1826";
const string cbGateWest = "1827";
const string cbWestPass1 = "1833";
const string cbWestPass2 = "1832";
const string cbNorthWestPlateau = "1834";
const string cbNorthEastPlateau = "1835";
const string cbP1Gold = "1831";
const string cbP1Granary = "1830";
const string cbSouthField = "1877";
const string cbEastPass1 = "1828";
const string cbEastPass2 = "1829";
const string cbP1TC = "1836";
const string cbNorthPass = "2272";
const string cbAttack2Stable = "1882";
const string cbAttack2Army = "1883";
const string cbAttack5Tower = "1880";
const string cbAttack3Army = "1881";

// Econ 
int   maxVills = 20;        // Will scale with difficulty
int   maxFishBoats = 0;    // Including one to scout
float goldPercent = 0.25;
float woodPercent = 0.15;
float foodPercent = 0.60;
int   gathererTypeID = -1;
int   fishGatherer = -1;
int   mainBase = -1;
float mainRadius = 100.0;


// *****************************************************************************
//
// Building plan support for forward building
//
// *****************************************************************************
int buildPlan = -1;
int buildDefend = -1;
// Set the following four vars, then just enable testBuildPlan.
int buildUnit = cUnitTypeTower;
vector buildVec = cInvalidVector;
vector startPoint = cInvalidVector;

rule buildPlanRule
   inactive
{

   buildPlan = aiPlanCreate("Build Plan", cPlanBuild);
   if(buildPlan >= 0)
   {
      aiPlanSetVariableInt(buildPlan, cBuildPlanBuildingTypeID, 0, buildUnit);
      aiPlanSetDesiredPriority(buildPlan, 40);
      aiPlanSetVariableVector(buildPlan, cBuildPlanCenterPosition, 0, buildVec);
      aiPlanSetInitialPosition(buildPlan, startPoint);
		aiPlanSetVariableFloat(buildPlan, cBuildPlanCenterPositionDistance, 0, 12);
      aiPlanAddUnitType(buildPlan, kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionBuilder, 0), 1, builderQty, builderQty);
      aiPlanSetEscrowID(buildPlan, cRootEscrowID);
      aiPlanSetInitialPosition(buildPlan, buildVec);
      aiPlanSetActive(buildPlan);
   }

   buildDefend =aiPlanCreate("Build Defend Plan", cPlanDefend);
   if (buildDefend >= 0)
   {
      aiPlanAddUnitType(buildDefend, cUnitTypeMilitary, 0, defendQty, defendQty);    
      aiPlanSetDesiredPriority(buildDefend, 60);                       // Above other defend plans
      aiPlanSetVariableVector(buildDefend, cDefendPlanDefendPoint, 0, buildVec);
      aiPlanSetVariableFloat(buildDefend, cDefendPlanEngageRange, 0, 15);
      aiPlanSetVariableBool(buildDefend, cDefendPlanPatrol, 0, false);
      aiPlanSetVariableFloat(buildDefend, cDefendPlanGatherDistance, 0, 8.0);
      aiPlanSetInitialPosition(buildDefend, kbGetBlockPosition(cbInfantryGather));
      aiPlanSetUnitStance(buildDefend, cUnitStanceDefensive);

      aiPlanSetVariableInt(buildDefend, cDefendPlanRefreshFrequency, 0, 5);
      aiPlanSetNumberVariableValues(buildDefend, cDefendPlanAttackTypeID, 2, true);
      aiPlanSetVariableInt(buildDefend, cDefendPlanAttackTypeID, 0, cUnitTypeUnit);
      aiPlanSetVariableInt(buildDefend, cDefendPlanAttackTypeID, 1, cUnitTypeBuilding);
      
      aiPlanSetActive(buildDefend); 
      aiEcho("Creating defend plan");
   }
   xsEnableRule("endBuildDefend");
   xsDisableSelf();
}

rule endBuildDefend
   inactive
   minInterval 7
{
   aiEcho("Build Plan state is "+aiPlanGetState(buildPlan));
   if (aiPlanGetState(buildPlan) == -1)
   {
      aiPlanDestroy(buildDefend);
      aiEcho("Destroying build defend plan");
      xsDisableSelf();
   }
}





// *****************************************************************************
//
//                                FUNCTIONS
//
// *****************************************************************************


void age2EventHandler(int bogus=-1)
{
   xsEnableRule("usePestilence");
   xsEnableRule("goToAge3");
   xsEnableRule("getAge2UnitUpgrades");
   xsEnableRule("getAge2ArmoryUpgrades");
}



void age3EventHandler(int bogus=-1)
{
   xsEnableRule("useCurse");
   siegeMaintainPlan = maintainUnit(cUnitTypePetrobolos, reserveSize/8, kbGetBlockPosition(cbSiegeGather), 60);
   xsEnableRule("getAge3UnitUpgrades");
   xsEnableRule("getAge3ArmoryUpgrades");
}


void age4EventHandler(int bogus=-1)
{
   xsEnableRule("usePlenty");
   xsEnableRule("getAge4UnitUpgrades");
   xsEnableRule("getAge4ArmoryUpgrades");
   xsEnableRule("usePlenty");
}


// Called by a trigger, to let AI know that the game has started
void wakeup(int parm=-1)
{
   startTime = xsGetTime()/1000;
   aiEcho("Wakeup running at "+startTime);

   if (kbGetAge() >= cAge2)
      age2EventHandler(0);
   if (kbGetAge() >= cAge3)
      age3EventHandler(0);
   if (kbGetAge() >= cAge4)
      age4EventHandler(0);


//   xsEnableRule("attack1");
   xsEnableRule("attackGenerator");
   xsEnableRule("scout");
//   villagerMaintainPlan = maintainUnit(cUnitTypeVillagerGreek, 25, kbGetBlockPosition(cbVillagerGather), 20);
   hippikonMaintainPlan = maintainUnit(cUnitTypeHippikon, reserveSize/3, kbGetBlockPosition(cbCavalryGather), cavTrainDelay);
   toxotesMaintainPlan = maintainUnit(cUnitTypeToxotes, reserveSize/3, kbGetBlockPosition(cbArcherGather), archerTrainDelay);
   hopliteMaintainPlan = maintainUnit(cUnitTypeHoplite, reserveSize/3, kbGetBlockPosition(cbInfantryGather), infantryTrainDelay);

   // Init low-priority defend plan to manage spare units
   defendPlan =aiPlanCreate("Defend Plan", cPlanDefend);
   if (defendPlan >= 0)
   {
      aiPlanAddUnitType(defendPlan, cUnitTypeMilitary, 0, 200, 200);    // All unassigned mil units
      aiPlanSetDesiredPriority(defendPlan, 10);                       // Way low, below scouting and attack
      aiPlanSetVariableVector(defendPlan, cDefendPlanDefendPoint, 0, kbGetBlockPosition(cbInfantryGather));
      aiPlanSetVariableFloat(defendPlan, cDefendPlanEngageRange, 0, 30);

      aiPlanSetVariableFloat(defendPlan, cDefendPlanGatherDistance, 0, 15.0);
      aiPlanSetInitialPosition(defendPlan, kbGetBlockPosition(cbInfantryGather));
      aiPlanSetUnitStance(defendPlan, cUnitStanceDefensive);

      aiPlanSetNumberVariableValues(defendPlan, cDefendPlanPatrolWaypoint, 3, true);
      aiPlanSetVariableVector(defendPlan, cDefendPlanPatrolWaypoint, 0, kbGetBlockPosition(cbGateEast));
      aiPlanSetVariableVector(defendPlan, cDefendPlanPatrolWaypoint, 1, kbGetBlockPosition(cbEastPass1));
      aiPlanSetVariableVector(defendPlan, cDefendPlanPatrolWaypoint, 2, kbGetBlockPosition(cbGateWest));
      aiPlanSetVariableBool(defendPlan, cDefendPlanPatrol, 0, true);

      aiPlanSetVariableInt(defendPlan, cDefendPlanRefreshFrequency, 0, 5);
      aiPlanSetNumberVariableValues(defendPlan, cDefendPlanAttackTypeID, 2, true);
      aiPlanSetVariableInt(defendPlan, cDefendPlanAttackTypeID, 0, cUnitTypeUnit);
      aiPlanSetVariableInt(defendPlan, cDefendPlanAttackTypeID, 1, cUnitTypeBuilding);
      
      aiPlanSetActive(defendPlan); 
      aiEcho("Creating defend plan");
   }

   // Higher-priority defend plan to reserve units for final push
   reservePlan =aiPlanCreate("Reserve Plan", cPlanDefend);
   if (reservePlan >= 0)
   {
      aiPlanAddUnitType(reservePlan, cUnitTypeMilitary, 0, reserveSize/2, reserveSize/2);    // All unassigned mil units
      aiPlanSetDesiredPriority(reservePlan, 30);                       // Higher than other defend plans
      aiPlanSetVariableVector(reservePlan, cDefendPlanDefendPoint, 0, kbGetBlockPosition(cbP2TC));
      aiPlanSetVariableFloat(reservePlan, cDefendPlanEngageRange, 0, 15);

      aiPlanSetVariableFloat(reservePlan, cDefendPlanGatherDistance, 0, 15.0);
      aiPlanSetInitialPosition(reservePlan, kbGetBlockPosition(cbInfantryGather));
      aiPlanSetUnitStance(reservePlan, cUnitStanceDefensive);

      aiPlanSetVariableBool(reservePlan, cDefendPlanPatrol, 0, false);

      aiPlanSetVariableInt(reservePlan, cDefendPlanRefreshFrequency, 0, 5);
      aiPlanSetNumberVariableValues(reservePlan, cDefendPlanAttackTypeID, 2, true);
      aiPlanSetVariableInt(reservePlan, cDefendPlanAttackTypeID, 0, cUnitTypeUnit);
      aiPlanSetVariableInt(reservePlan, cDefendPlanAttackTypeID, 1, cUnitTypeBuilding);
      
      aiPlanSetActive(reservePlan); 
      aiEcho("Creating defend plan");
   }

      // Create villager maintain plan
   createSimpleMaintainPlan(gathererTypeID, maxVills, true, mainBase);
}



void initMainBase()
{
   // Nuke bases, add one base to rule them all
   kbBaseDestroyAll(cMyID);

   mainBase = kbBaseCreate(cMyID, "Base "+kbBaseGetNextID(), kbGetBlockPosition(cbP2TC), mainRadius);
   if (mainBase < 0)
      aiEcho("***** Main base creation failed. *****");

   vector baseFront=xsVectorNormalize(kbGetMapCenter()-kbGetBlockPosition(cbP2TC));     // Set front
   kbBaseSetFrontVector(cMyID, mainBase, baseFront);                 
   kbBaseSetMaximumResourceDistance(cMyID, mainBase, mainRadius+20.0);                    // Gather up to 20m beyond base perimeter
   kbBaseSetMain(cMyID, mainBase, true);     // Make this the main base

   // Add the buildings
   int buildingQuery = -1;
   int count = 0;
   buildingQuery = kbUnitQueryCreate("Building Query");     // All buildings in the base
   configQuery(buildingQuery, cUnitTypeBuilding, -1, cUnitStateAliveOrBuilding, cMyID, kbGetBlockPosition(cbP2TC), false, mainRadius);
   kbUnitQueryResetResults(buildingQuery);
   count = kbUnitQueryExecute(buildingQuery);

   int i = 0;
   int buildingID = -1;
   for (i=0; < count)
   {
      buildingID = kbUnitQueryGetResult(buildingQuery, i);
      // Add it to the base
      kbBaseAddUnit( cMyID, mainBase, buildingID );
   }
}


void initEcon()
{



   aiSetAutoGatherEscrowID(cRootEscrowID);
   aiSetAutoFarmEscrowID(cRootEscrowID);
   gathererTypeID = kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionGatherer,0);

   
   int herdPlanID=aiPlanCreate("GatherHerdable Plan", cPlanHerd);
   if (herdPlanID >= 0)
   {
      aiPlanAddUnitType(herdPlanID, cUnitTypeHerdable, 0, 100, 100);
      aiPlanSetVariableInt(herdPlanID, cHerdPlanBuildingTypeID, 0, cUnitTypeSettlementLevel1);
      aiPlanSetActive(herdPlanID);
   }

   aiSetResourceGathererPercentageWeight(cRGPScript, 1);
   aiSetResourceGathererPercentageWeight(cRGPCost, 0);

   kbSetAICostWeight(cResourceFood, 1.0);
   kbSetAICostWeight(cResourceWood, 0.7);
   kbSetAICostWeight(cResourceGold, 0.8);
   kbSetAICostWeight(cResourceFavor, 7.0);

   aiSetResourceGathererPercentage(cResourceFood, foodPercent, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceWood, woodPercent, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceGold, goldPercent, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceFavor, 0.0, false, cRGPScript);
   aiNormalizeResourceGathererPercentages(cRGPScript);

   //bool aiSetResourceBreakdown( int resourceTypeID, int resourceSubTypeID, int numberPlans, int planPriority, float percentage, int baseID )
//	aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeEasy, numFoodEasyPlans, 50, 1.0, gMainBaseID);
//   aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeHuntAggressive, numFoodHuntAggressivePlans, 100, 1.0, gMainBaseID);
   aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeFish, 1, 50, 1.0, mainBase);
   aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeFarm, 1, 50, 1.0, mainBase);
   aiSetResourceBreakdown(cResourceWood, cAIResourceSubTypeEasy, 1, 50, 1.0, mainBase);
	aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, 1, 50, 1.0, mainBase);
//   aiSetResourceBreakdown(cResourceFavor, cAIResourceSubTypeEasy, numFavorPlans, 50, 1.0, gMainBaseID);
}





void main()
{
   aiEcho("Starting Scn08p2.xs");
   kbSetTownLocation(kbGetBlockPosition(cbP2TC));
   aiRandSetSeed();

   //Calculate some areas.
   kbAreaCalculate(1200.0);
   aiSetAttackResponseDistance(10.0);

   kbEscrowSetPercentage( cEconomyEscrowID, cAllResources, 0.0);
   kbEscrowSetPercentage( cMilitaryEscrowID, cAllResources, 0.0);
   kbEscrowAllocateCurrentResources();


   initMainBase();    // Destroy all auto-bases, make one manual base for everything
   initEcon();

   aiSetAgeEventHandler(cAge2, "age2EventHandler");
   aiSetAgeEventHandler(cAge3, "age3EventHandler");
   aiSetAgeEventHandler(cAge4, "age4EventHandler");

   // Create attack routes
   routeWestPass = attackRoute("West Pass",cbGateWest, cbWestPass1, cbWestPass2);
   routeEastPass = attackRoute("East Pass",cbGateEast, cbEastPass1, cbEastPass2);
   routeNorthPass = attackRoute("North Pass", cbGateWest, cbNorthPass, cbP1TC);

   switch(aiGetWorldDifficulty())
   {     // ignores 0 (easiest), uses initial values in that case.
   case 1:     // medium
      {
         nextAttackTime = 300000;      // 5 minutes
         attackSize = 6.0; 
         maxAttackSize = 12.0;
         attackSizeMultiplier = 1.2; 
         attackInterval = 240000;      // 4 minutes   
         reserveSize = 20;
         age3Time = 1200;               // Time (in seconds) to go to age 3
         age4Time = 1800; 
         defendQty = 2;
         builderQty = 2;
         trainDelay = 25;
         break;
      }
   case 2:     // difficult/hard
      {
         nextAttackTime = 180000;      // 3 minutes
         attackSize = 8.0; 
         maxAttackSize = 30.0;
         attackSizeMultiplier = 1.2; 
         attackInterval = 180000;      // 3 minutes    
         reserveSize = 35;
         age3Time = 900;               // Time (in seconds) to go to age 3
         age4Time = 1500;                   
         defendQty = 3;
         builderQty = 5;
         trainDelay = 15;
         break;
      }
   case 3:     // hardest/nightmare
      {
         nextAttackTime = 60000;      // 60 sec
         attackSize = 10.0; 
         maxAttackSize = 40.0;
         attackSizeMultiplier = 1.2; 
         attackInterval = 150000;      // 2.5 minutes     
         reserveSize = 45;
         age3Time = 500;               // Time (in seconds) to go to age 3
         age4Time = 900;                   
         defendQty = 6;
         builderQty = 10;
         trainDelay = 10;
         break;
      }
   }

   cavTrainDelay = trainDelay;
   archerTrainDelay = trainDelay/2;
   infantryTrainDelay = (2*trainDelay)/3;


}


void siegeUnitAttack(vector location=vector(-1,-1,-1), int qty=1)
{

   static int attackQuery = -1;
   static int attackQuery2 = -1;
   int   attackID=aiPlanCreate("Mixed Unit Attack at "+timeString(),cPlanAttack);
   if (attackID < 0)
   {
      return;
   }

   if (aiPlanSetVariableInt(attackID, cAttackPlanPlayerID, 0, 1) == false)
   {
      return;
   }

   if (aiPlanSetNumberVariableValues(attackID, cAttackPlanTargetTypeID, 3, true) == false)
      return;

   aiPlanSetVariableInt(attackID, cAttackPlanTargetTypeID, 0, cUnitTypeUnit);
   aiPlanSetVariableInt(attackID, cAttackPlanTargetTypeID, 1, cUnitTypeBuilding);
   aiPlanSetVariableInt(attackID, cAttackPlanTargetTypeID, 2, cUnitTypeAbstractWall);

   // Set up the attack queries with the appropriate vectors
   if (aiPlanSetNumberVariableValues(attackID, cAttackPlanQueryID, 2, true) == false)
   {
      return;
   }
   if (attackQuery < 0)
   {
      attackQuery = kbUnitQueryCreate("Attack Query");
      configQueryRelation(attackQuery, cUnitTypeUnit, -1, cUnitStateAlive, cPlayerRelationEnemy, location, false, 50);
   }
   else
   {
      kbUnitQuerySetPosition(attackQuery, location);
   }

   if (attackQuery2 < 0)
   {
      attackQuery2 = kbUnitQueryCreate("Attack Query #2");
      configQuery(attackQuery2, cUnitTypeBuilding, -1, cUnitStateAliveOrBuilding, cPlayerRelationEnemy, location, false, 50);
   }
   else
   {
      kbUnitQuerySetPosition(attackQuery2, location);
   }
   aiPlanSetVariableInt(attackID, cAttackPlanQueryID, 0, attackQuery);
   aiPlanSetVariableInt(attackID, cAttackPlanQueryID, 1, attackQuery2);


   if (aiRandInt(2) == 0)
      aiPlanSetVariableInt(attackID, cAttackPlanAttackRouteID, 0, routeNorthPass);
   else
      aiPlanSetVariableInt(attackID, cAttackPlanAttackRouteID, 0, routeEastPass);


   aiPlanSetVariableVector(attackID, cAttackPlanGatherPoint, 0, kbGetBlockPosition(cbInfantryGather));
   aiPlanSetVariableFloat(attackID, cAttackPlanGatherDistance, 0, 25.0);

   aiPlanAddUnitType(attackID, cUnitTypeHoplite, 1, 3*qty/10, 3*qty/10);
   aiPlanAddUnitType(attackID, cUnitTypeToxotes, 1, 4*qty/10, 4*qty/10);
   aiPlanAddUnitType(attackID, cUnitTypeHippikon, 1, 3*qty/10, 3*qty/10);
   // Add siege if this is attack 4 or higher.
   if (attackCount > 4)
      aiPlanAddUnitType(attackID, cUnitTypePetrobolos, 1, (qty+5)/10, (qty+5)/10);

   aiPlanSetInitialPosition(attackID, kbGetBlockPosition(cbInfantryGather));
   aiPlanSetRequiresAllNeedUnits(attackID, false);
   aiPlanSetActive(attackID);
   aiEcho("Activating attack plan "+attackID);
//   if (lastAttackPlanID >= 0)
//      aiPlanDestroy(lastAttackPlanID);   // free up last set of units?
   lastAttackPlanID = attackID; // update the global var
}





// *****************************************************************************
//
// RULES
//
// *****************************************************************************


rule goToAge3
   inactive
   mininterval 20
{
   if ( (xsGetTime()/1000) < age3Time )
      return;
   researchTech(cTechAge3Aphrodite);
   xsEnableRule("goToAge4");
   xsEnableRule("getAge3UnitUpgrades");
   xsEnableRule("getAge3ArmoryUpgrades");
   xsDisableSelf();
}

rule goToAge4
   inactive
   mininterval 20
{
   return;  // disabled for now.

   if ( (xsGetTime()/1000) < age4Time )
      return;
   researchTech(cTechAge4Hephaestus);
   xsEnableRule("getAge4UnitUpgrades");
   xsEnableRule("getAge4ArmoryUpgrades");
   xsDisableSelf();
}



rule getAge2UnitUpgrades
   inactive
   minInterval 20
{
   if ( (xsGetTime()/1000) < (age3Time/3) )
      return;     
   researchTech(cTechMediumInfantry);
   researchTech(cTechMediumArchers);
//   researchTech(cTechMediumCavalry);
   xsDisableSelf();
}

rule getAge2ArmoryUpgrades
   inactive
   minInterval 20
{
   if ( (xsGetTime()/1000) < (2*age3Time)/3 )
      return;     
   aiEcho("Getting age 2 armory upgrades");
   researchTech(cTechCopperWeapons);
   researchTech(cTechCopperMail);
   researchTech(cTechCopperShields);
   xsDisableSelf();
}

rule getAge3UnitUpgrades
   inactive
   minInterval 20
{
   researchTech(cTechHeavyInfantry);
   researchTech(cTechHeavyArchers);
   researchTech(cTechHeavyCavalry);
   xsDisableSelf();
}

rule getAge3ArmoryUpgrades
   inactive
   minInterval 20
{
   researchTech(cTechBronzeWeapons);
   researchTech(cTechBronzeMail);
   researchTech(cTechBronzeShields);
   xsDisableSelf();
}

rule getAge4UnitUpgrades
   inactive
   minInterval 20
{
   researchTech(cTechChampionInfantry);
   researchTech(cTechChampionArchers);
   researchTech(cTechChampionCavalry);
   xsDisableSelf();
}

rule getAge4ArmoryUpgrades
   inactive
   minInterval 20
{
   researchTech(cTechIronWeapons);
   researchTech(cTechIronMail);
   researchTech(cTechIronShields);
   xsDisableSelf();
}



rule scout
   inactive
{
   // just set up an explore plan
   int exploreID = aiPlanCreate("Explore", cPlanExplore);
   if(exploreID >= 0)
   {
      //aiPlanAddVariableFloat( exploreID, cExplorePlanLOSMultiplier, "LOS Multiplier", 1);
      aiPlanSetVariableFloat( exploreID, cExplorePlanLOSMultiplier,  0, 4.0 );
      aiPlanAddUnitType(exploreID, cUnitTypeHippikon, 1, 1, 1);
      aiPlanSetInitialPosition( exploreID, kbGetBlockPosition(cbNorthWestPlateau));
      aiPlanSetActive(exploreID);
   }
   xsDisableSelf();
}





rule attackGenerator
   minInterval 10
   inactive
{
   static int buildingQuery = -1;

    //aiEcho("attack check running, next time is "+nextAttackTime);
   if ( xsGetTime() < nextAttackTime )
      return;

   if (buildingQuery < 0)
   {
      buildingQuery = kbUnitQueryCreate("Building Query");
      configQuery(buildingQuery, cUnitTypeBuilding, -1, cUnitStateAliveOrBuilding, 1, kbGetBlockPosition(cbAttack5Tower), false, 20);
   }


   attackCount = attackCount + 1;

   // Add forward building coincident with some attacks
   if (attackCount == 3)   // Make tower
   {
      buildUnit = cUnitTypeTower;
      buildVec = kbGetBlockPosition(cbAttack5Tower);    // Gold mine
      startPoint = kbGetBlockPosition(cbWestPass1);
      kbUnitQueryResetResults(buildingQuery);
      if (kbUnitQueryExecute(buildingQuery) == 0)
         xsEnableRule("buildPlanRule");
      else
         aiEcho("Skipping tower build because of enemy buildings.");
   }
   if (attackCount == 5)   // Make stable
   {
      buildUnit = cUnitTypeStable;
      buildVec = kbGetBlockPosition(cbEastPass2);
      startPoint = kbGetBlockPosition(cbEastPass1);
      kbUnitQuerySetPosition(buildingQuery, startPoint);
      kbUnitQueryResetResults(buildingQuery);
      if (kbUnitQueryExecute(buildingQuery) == 0)
         xsEnableRule("buildPlanRule");
      else
         aiEcho("Skipping stable build because of enemy buildings.");   }
   if (attackCount == 7)   // Make fortress
   {
      buildUnit = cUnitTypeFortress;
      buildVec = kbGetBlockPosition(cbNorthEastPlateau);   // Elevated area north of town
      startPoint = kbGetBlockPosition(cbWestPass1);
      kbUnitQuerySetPosition(buildingQuery, startPoint);
      kbUnitQueryResetResults(buildingQuery);
      if (kbUnitQueryExecute(buildingQuery) == 0)
         xsEnableRule("buildPlanRule");
      else
         aiEcho("Skipping fortress build because of enemy buildings.");   }
   if (aiRandInt(3) > 0)
   {
      aiEcho("Normal attack");
      siegeUnitAttack(kbGetBlockPosition(cbP1TC), attackSize);
   }
   else
   {
      siegeUnitAttack(kbGetBlockPosition("2259"), attackSize); // 1 in 3 chance of aiming for green TC
      aiEcho("Deep attack");
   }
   nextAttackTime = xsGetTime() + attackInterval;
   attackSize = attackSize * attackSizeMultiplier;
   if (attackSize > maxAttackSize)
      attackSize = maxAttackSize;
   aiEcho("Next attack size will be "+attackSize+".");
}



rule useCurse 
   minInterval 5
   inactive
{

   // look for a group of 6 enemy military units, at my TC location or at my main army's location
   int targetUnit = -1;

   
   static int tempQuery = -1;
   if (tempQuery < 0)
   {  // Doesn't exist, set it up
      tempQuery = kbUnitQueryCreate("useCurseHome");
      if ( configQuery(tempQuery, cUnitTypeUnit, -1, cUnitStateAlive, 1, kbGetBlockPosition(cbP2TC), true, 50) == false)
         return;
   }
   kbUnitQueryResetResults(tempQuery);
   int targetCount = kbUnitQueryExecute(tempQuery);  

   if ( targetCount < 6 )
   {
      vector pVec = aiPlanGetLocation(lastAttackPlanID);
      if (xsVectorGetX(pVec)>=0)
      {
         static int tempQuery2 = -1;
         if (tempQuery2 < 0)
         {  // Doesn't exist, set it up
            tempQuery2 = kbUnitQueryCreate("useCurseArmy");
            if ( configQuery(tempQuery2, cUnitTypeUnit, -1, cUnitStateAlive, 1, pVec, true, 50) == false)
               return;
         }
         else
            kbUnitQuerySetPosition(tempQuery2, pVec); // Because pVec changes as army moves
         kbUnitQueryResetResults(tempQuery2);
         targetCount = kbUnitQueryExecute(tempQuery2);  
         if (targetCount < 6)
            return;
         else
            targetUnit = kbUnitQueryGetResult(tempQuery2, targetCount/2);  // grab middle unit
      }
      else
         return;
   } 
   else
      targetUnit = kbUnitQueryGetResult(tempQuery, targetCount/2);  // grab middle unit


   aiEcho("Using Curse on unit "+targetUnit+" at "+kbUnitGetPosition(targetUnit));
   if ( aiCastGodPowerAtUnit(cTechCurse, targetUnit) == true)
      xsDisableSelf();
}



rule usePestilence
   minInterval 5
   inactive
{
   // look for a group of 2 enemy buildings near my army.

   if (lastAttackPlanID < 0)
      return;
   

   vector pVec = aiPlanGetLocation(lastAttackPlanID);
   if (xsVectorGetX(pVec)<0)
      return;

   static int tempQuery = -1;
   if (tempQuery < 0)
   {  // Doesn't exist, set it up
      tempQuery = kbUnitQueryCreate("useBronze");

      if ( configQuery(tempQuery, cUnitTypeLogicalTypeBuildingsThatTrainMilitary, -1, cUnitStateAlive, 1, pVec, true, 50) == false)
         return;
   }
   else
      kbUnitQuerySetPosition(tempQuery, pVec); // Because pVec changes as army moves
   kbUnitQueryResetResults(tempQuery);
   int targetCount = kbUnitQueryExecute(tempQuery);  

   if (targetCount < 2)
      return;

   aiEcho("Using Pestilence at "+kbUnitGetPosition(kbUnitQueryGetResult(tempQuery, 0)));
   if ( aiCastGodPowerAtPosition(cTechPestilence, kbUnitGetPosition(kbUnitQueryGetResult(tempQuery, 0))) == true)
   {
      aiEcho("Deactivating UsePestilence");
      xsDisableSelf();
   }
}


rule usePlenty
   minInterval 5
   inactive
{
   vector aimHere = kbGetBlockPosition(cbP2TC);
   float dx = aiRandInt(40)-20.0;
   float dz = aiRandInt(40)-20.0;
   float newX = xsVectorGetX(kbGetBlockPosition(cbP2TC));
   float newZ = xsVectorGetZ(kbGetBlockPosition(cbP2TC));
   newX = newX + dx;
   newZ = newZ + dz;

   aimHere = xsVectorSetX(aimHere, newX);    // TC vector +/-20 in x and z
   aimHere = xsVectorSetZ(aimHere, newZ);

   if ( aiCastGodPowerAtPosition(cTechPlenty, aimHere) == true)
   {
      aiEcho("Plenty worked at "+aimHere);
      xsDisableSelf();
   }
   else
      aiEcho("Plenty failed at "+aimHere);
}



