//==============================================================================
// aomdefaultEcon.xs
//
// Handles common economy functions.
//==============================================================================

//Farming vars.
bool gFarming=false;


//==============================================================================
// findUnit
//
// Will find the first unit of the given playerID
//==============================================================================
int findUnit(int playerID=0, int state=cUnitStateAny, int unitTypeID=-1)
{
   int count=-1;
   static int unitQueryID=-1;

   //If we don't have the query yet, create one.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("miscFindUnitQuery");
   
	//Define a query to get all matching units
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, playerID);
      kbUnitQuerySetUnitType(unitQueryID, unitTypeID);
      kbUnitQuerySetState(unitQueryID, state);
	}
	else
   	return(-1);

   kbUnitQueryResetResults(unitQueryID);
	int numberFound=kbUnitQueryExecute(unitQueryID);
   for (i=0; < numberFound)
      return(kbUnitQueryGetResult(unitQueryID, i));
   return(-1);
}

//==============================================================================
// findNumberOfUnitsInBase
//
//==============================================================================
int findNumberOfUnitsInBase(int playerID=0, int baseID=-1, int unitTypeID=-1)
{
   int count=-1;
   static int unitQueryID=-1;

   //Create the query if we don't have it.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("getUnitsInBaseQuery");
   
	//Define a query to get all matching units
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, playerID);
      kbUnitQuerySetBaseID(unitQueryID, baseID);
      kbUnitQuerySetUnitType(unitQueryID, unitTypeID);
      kbUnitQuerySetState(unitQueryID, cUnitStateAny);
	}
	else
   	return(-1);

   kbUnitQueryResetResults(unitQueryID);
	return(kbUnitQueryExecute(unitQueryID));
}

//==============================================================================
// findBestSettlement
//
// Will find the closet settlement of the given playerID
//==============================================================================
vector findBestSettlement(int playerID=0)
{
   vector townLocation=kbGetTownLocation();
   vector best=townLocation;

   int count=-1;
   static int unitQueryID=-1;

   //Create the query if we don't have it yet.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("getUnClaimedSettlements");
   
	//Define a query to get all matching units.
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, playerID);
      kbUnitQuerySetUnitType(unitQueryID, cUnitTypeSettlement);
      kbUnitQuerySetState(unitQueryID, cUnitStateAny);
	}
	else
   	return(best);

   //Find the best one.
	float bestDistSqr=100000000.0;
   kbUnitQueryResetResults(unitQueryID);
	int numberFound=kbUnitQueryExecute(unitQueryID);
   for (i=0; < numberFound)
   {
      vector position=kbUnitGetPosition(kbUnitQueryGetResult(unitQueryID, i));
      float dx=xsVectorGetX(townLocation)-xsVectorGetX(position);
      float dz=xsVectorGetZ(townLocation)-xsVectorGetZ(position);
      
      float curDistSqr=((dx*dx) + (dz*dz));
      if(curDistSqr < bestDistSqr)
      {
         best=position;
         bestDistSqr=curDistSqr;
      }
   }
   return(best);
}

//==============================================================================
// findASettlement
//
// Will find an unclaimed settlement
//==============================================================================
bool findASettlement()
{
   int count=-1;
   static int unitQueryID=-1;

   //Create the query if we don't have it yet.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("getAnUnClaimedSettlements");
   
	//Define a query to get all matching units.
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, 0);
      kbUnitQuerySetUnitType(unitQueryID, cUnitTypeSettlement);
      kbUnitQuerySetState(unitQueryID, cUnitStateAny);
	}
	else
   	return(false);

   kbUnitQueryResetResults(unitQueryID);
	int numberFound=kbUnitQueryExecute(unitQueryID);
   if (numberFound > 0)
      return(true);
   return(false);
}

//==============================================================================
// getNumberUnits
//==============================================================================
int getNumberUnits(int unitType=-1, int playerID=-1, int state=cUnitStateAlive)
{
	int count=-1;
   static int unitQueryID=-1;

   //Create the query if we don't have it yet.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("GetNumberOfUnitsQuery");
   
	//Define a query to get all matching units.
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, playerID);
      kbUnitQuerySetUnitType(unitQueryID, unitType);
      kbUnitQuerySetState(unitQueryID, state);
	}
	else
   	return(0);

	kbUnitQueryResetResults(unitQueryID);
	return(kbUnitQueryExecute(unitQueryID));
}

//==============================================================================
// getUnit
//==============================================================================
int getUnit(int unitType=-1)
{
  	int retVal=-1;
   int count=-1;
	static int unitQueryID=-1;
   
   //Create the query if we don't have it yet.
   if (unitQueryID < 0)
      unitQueryID=kbUnitQueryCreate("getUnitQuery");

	//Define a query to get all matching units.
	if (unitQueryID != -1)
	{
		kbUnitQuerySetPlayerID(unitQueryID, cMyID);
      kbUnitQuerySetUnitType(unitQueryID, unitType);
      kbUnitQuerySetState(unitQueryID, cUnitStateAlive);
	}
	else
   	return(-1);

   kbUnitQueryResetResults(unitQueryID);
	count=kbUnitQueryExecute(unitQueryID);

	//Pick a unit and return its ID, or return -1.
	if (count > 0)
      retVal=kbUnitQueryGetResult(unitQueryID, 0);
	return(retVal);
}

//==============================================================================
// getNextGathererUpgrade
//
// sets up a progression plan to research the next upgrade that benefits the given
// resource.
//==============================================================================
rule getNextGathererUpgrade
   minInterval 30
   inactive
   runImmediately
{
   if (kbSetupForResource(kbBaseGetMainID(cMyID), cResourceWood, 25.0, 600) == false)
      return;

   static int id=0;

	int gathererTypeID=kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionGatherer,0);
	if (gathererTypeID < 0)
      return();
	
	for (i=0; < 3)
   {
      int affectedUnitType=-1;
      if (i == cResourceGold)
         affectedUnitType=cUnitTypeGold;
      else if (i == cResourceWood)
         affectedUnitType=cUnitTypeWood;
      else //(i == cResourceFood)
      {
         //If we're not farming yet, don't get anything.
         if (gFarming != true)
            continue;
         if (kbUnitCount(cMyID, cUnitTypeFarm, cUnitStateAlive) >= 5)
            affectedUnitType=cUnitTypeFarm;
      }

      //Get the building that we drop this resource off at.
	   int dropSiteFilterID=kbTechTreeGetDropsiteUnitIDByResource(i, 0);
	   if (dropSiteFilterID < 0)
		   continue;

      //Don't do anything until you have a dropsite.
      if (getUnit(dropSiteFilterID) == -1)
         continue;

      //Get the cheapest thing.
	   int upgradeTechID=kbTechTreeGetCheapestUnitUpgrade(gathererTypeID, cUpgradeTypeWorkRate, -1, dropSiteFilterID, false, affectedUnitType);
	   if (upgradeTechID < 0)
		   continue;
	   //Dont make another plan if we already have one.
      if (aiPlanGetIDByTypeAndVariableType(cPlanProgression, cProgressionPlanGoalTechID, upgradeTechID) != -1)
         continue;

      //Make plan to get this upgrade.
	   int planID=aiPlanCreate("nextGathererUpgrade - "+id, cPlanProgression);
	   if (planID < 0)
		   continue;

	   aiPlanSetVariableInt(planID, cProgressionPlanGoalTechID, 0, upgradeTechID);
	   aiPlanSetDesiredPriority(planID, 25);
	   aiPlanSetEscrowID(planID, cEconomyEscrowID);
	   aiPlanSetActive(planID);
      aiEcho("getNextGathererUpgrade: successful in creating a progression to "+kbGetTechName(upgradeTechID));
	   id++;
   }
}


//==============================================================================
// findBiggestBorderArea
//
// given an areaid, find the biggest border area in tiles.
//==============================================================================
int findBiggestBorderArea(int areaID=-1)
{
	if(areaID == -1)
		return(-1);

	int numBorders=kbAreaGetNumberBorderAreas(areaID);
	int borderArea=-1;
	int numTiles=-1;
	int bestTiles=-1;
	int bestArea=-1;

	for (i=0; < numBorders)
	{
		borderArea=kbAreaGetBorderAreaID(areaID, i);
		numTiles=kbAreaGetNumberTiles(borderArea);
		if (numTiles > bestTiles)
		{
			bestTiles=numTiles;
			bestArea=borderArea;
		}
	}

	return(bestArea);
}

//==============================================================================
// RULE: UpdateGoldBreakdown
//==============================================================================
rule updateGoldBreakdown
   minInterval 10
   inactive
{
   int goldPriority=50;
   if (cMyCulture == cCultureEgyptian)
      goldPriority=55;

   //If we're not post-1st age Egyptian, we exhaust our main base gold first.
   //2nd age+ Egyptians always try to maintain an external gold base, too.
   if ((cMyCulture != cCultureEgyptian) || (kbGetAge() == 0))
   {
      //Count our main base gold sites.
      int numberGoldSites=kbGetNumberValidResources(kbBaseGetMainID(cMyID), cResourceGold, cAIResourceSubTypeEasy);
      //Get the count of plans we currently have going.
      int numGoldPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0);
      //If we have more than 1 gold site at our main base, don't do anything (unless
      //we've turned off gold; if that's the case, start it back up).
      if (numberGoldSites > 0)
      {
         if (numGoldPlans <= 0)
         {
	         aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, 1, goldPriority, 1.0, kbBaseGetMainID(cMyID));
            aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0, 1);
         }
         return;
      }
      //Else, we're out of main gold.  Ensure that we don't have a RB for it.
      else if (numGoldPlans > 0)
      {
         aiRemoveResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, kbBaseGetMainID(cMyID));
         aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0, 0);
      }
   }

   //If we have a gold base already, see if that's still valid.
   if (gGoldBaseID >= 0)
   {
      numberGoldSites=kbGetNumberValidResources(gGoldBaseID, cResourceGold, cAIResourceSubTypeEasy);
      //If we have gold there, ensure that we have a plan setup for it.
      if (numberGoldSites > 0)
      {
         aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0, 1);
	      aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, 1, goldPriority, 1.0, gGoldBaseID);
         return;
      }
      //Else, nuke the RB.
      aiRemoveResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, gGoldBaseID);
      //Whack the gold plan goal.
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0, 0);
      //Whack the gold base.
      gGoldBaseID=-1;
   }

   //Try to find a new gold base.
   gGoldBaseID=kbBaseFindCreateResourceBase(cResourceGold, cAIResourceSubTypeEasy, kbBaseGetMainID(cMyID));
   if (gGoldBaseID >= 0)
   {
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0, 1);
	   aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, 1, goldPriority, 1.0, gGoldBaseID);
   }
}

//==============================================================================
// updateFoodBreakdown
//==============================================================================
rule updateFoodBreakdown
   minInterval 31
   active
{
   //Get the number of valid resources spots.
   float numberAggressiveResourceSpots=kbGetNumberValidResources(kbBaseGetMainID(cMyID),
      cResourceFood, cAIResourceSubTypeHuntAggressive);
   float distance=-1.0;
   if (kbGetAge() <= 1)
      distance=60.0;
   if (kbGetAge() >= 2)
      distance=45.0;
   float numberEasyResourceSpots=kbGetNumberValidResources(kbBaseGetMainID(cMyID), cResourceFood, cAIResourceSubTypeEasy, distance);
   float totalNumberResourceSpots=numberAggressiveResourceSpots + numberEasyResourceSpots;

   //If we don't have "enough", whack the total to 0 so that we start farming.
   float aggressiveAmount=kbGetAmountValidResources(kbBaseGetMainID(cMyID), cResourceFood, cAIResourceSubTypeHuntAggressive);
   float easyAmount=kbGetAmountValidResources(kbBaseGetMainID(cMyID), cResourceFood, cAIResourceSubTypeEasy);
   float totalAmount=aggressiveAmount+easyAmount;
   if (totalAmount < 10.0)
   {
      aiEcho("UpdateEcon: Total Food Sites: "+totalNumberResourceSpots+" agg: "+numberAggressiveResourceSpots+" easy: "+numberEasyResourceSpots+".");
      aiEcho("  BUT, aggAmt: "+aggressiveAmount+" easyAmt: "+easyAmount+", so whacking total to -1.0.");
      totalNumberResourceSpots=-1.0;
   }
   else
      aiEcho("UpdateEcon: Total Food Sites: "+totalNumberResourceSpots+" agg: "+numberAggressiveResourceSpots+" easy: "+numberEasyResourceSpots+".");

   if (totalNumberResourceSpots > 0.0)
   {
      float aggPercent=numberAggressiveResourceSpots / totalNumberResourceSpots;
      float easyPercent=numberEasyResourceSpots / totalNumberResourceSpots;

      //Ghange the goal.
      int numHuntAggPlans=1;
      int numFoodEasyPlans=1;
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeHuntAggressive, numHuntAggPlans);
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeEasy, numFoodEasyPlans);
      //Set the RB.
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeHuntAggressive, numHuntAggPlans, 65, aggPercent, kbBaseGetMainID(cMyID));
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeEasy, numFoodEasyPlans, 65, easyPercent, kbBaseGetMainID(cMyID));
   }

   //Egyptians can farm in the first age.
   if (((kbGetAge() > 0) || (cMyCulture == cCultureEgyptian)) &&
      (numberEasyResourceSpots <= 0) && (gFarmBaseID != -1))
   {
      //Change the goal.
      int numFarmPlans=2;
      numHuntAggPlans=0;
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeFarm, numFarmPlans);
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeHuntAggressive, numHuntAggPlans);
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeEasy, numFoodEasyPlans);

      //Get the overall max farms.  Modulate that by the current age.
      int maxFarms=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanMaxFarmLimit, 0);
      //If we're not on KOTH, limit the farms.
      if (cRandomMapName != "king of the hill")
      {
         //Second age is half the max.
         if (kbGetAge() == 1)
            maxFarms=maxFarms/2;
         //Third age is 3/4 of the max.
         else if (kbGetAge() == 2)
            maxFarms=maxFarms*3/4;
      }
      int perPlanFarmLimit=maxFarms/numFarmPlans;
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanFarmLimitPerPlan, 0, perPlanFarmLimit);
      aiSetFarmLimit(perPlanFarmLimit);

      //Set breakdown based on goals.
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeFarm, numFarmPlans, 100, 1.0, gFarmBaseID);
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeHuntAggressive, numHuntAggPlans, 0, 0.0, gFarmBaseID);
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeEasy, numFoodEasyPlans, 0, 0.1, gFarmBaseID);
      gFarming=true;
   }
}

//==============================================================================
// updateResourceHandler
//==============================================================================
void updateResourceHandler(int parm=0)
{
   aiEcho("UpdateResourceHandler: parm="+parm+".");

   //Handle food.
   if (parm == cResourceFood)
      updateFoodBreakdown();
   //Handle Gold.
   else if (parm == cResourceGold)
   {
      updateGoldBreakdown();
      xsEnableRule("updateGoldBreakdown");
   }
}

//==============================================================================
// RULE: relocateFarming
//==============================================================================
rule relocateFarming
   minInterval 30
   inactive
{
   //Not farming yet, don't do anything.
   if (gFarming == false)
      return;

   //Fixup the old RB for farming.
   if (gFarmBaseID != -1)
   {
      //Check the current farm base for a settlement.
      if (findNumberOfUnitsInBase(cMyID, gFarmBaseID, cUnitTypeAbstractSettlement) > 0)
         return;
      //Remove the old breakdown.
      aiRemoveResourceBreakdown(cResourceFood, cAIResourceSubTypeFarm, gFarmBaseID);
   }

   //If no settlement, then move the farming to another base that has a settlement.
   int unit=findUnit(cMyID, cUnitStateAlive, cUnitTypeAbstractSettlement);
   if (unit != -1)
   {
      //Get new base ID.
      gFarmBaseID=kbUnitGetBaseID(unit);
      //Make a new breakdown.
      int numFarmPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeFarm);
      aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeFarm, numFarmPlans, 100, 1.0, gFarmBaseID);
   }
   else
   {
      //If there are no other bases without settlements... stop farming.
      gFarmBaseID=-1;
      aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeFarm, 0);
   }
}

//==============================================================================
// RULE: startLandScouting
//
// grabs the first scout in the scout list and starts scouting with it.
//==============================================================================
rule startLandScouting
   minInterval 1
   active
{
   //If no scout, go away.
   if (gLandScout == -1)
   {
      xsDisableSelf();
      return;
   }

   //Land based Scouting.
	gLandExplorePlanID=aiPlanCreate("Explore_Land", cPlanExplore);
   if (gLandExplorePlanID >= 0)
   {
      aiPlanAddUnitType(gLandExplorePlanID, gLandScout, 1, 1, 1);
      aiPlanSetActive(gLandExplorePlanID);
      aiPlanSetEscrowID(gLandExplorePlanID, cEconomyEscrowID);
      
      //Don't loop as egyptian.
      if (cMyCulture == cCultureEgyptian)
         aiPlanSetVariableBool(gLandExplorePlanID, cExplorePlanDoLoops, 0, false);
   }

   //Go away now.
   xsDisableSelf();
}

//==============================================================================
// RULE: autoBuildOutpost
//
// Restrict Egyptians from building outposts until they have a temple.
//==============================================================================
rule autoBuildOutpost
   minInterval 10
   active
{
   if ((gLandScout == -1) || (cMyCulture != cCultureEgyptian))
   {
      xsDisableSelf();
      return;
   }
   if (getUnit(cUnitTypeTemple) == -1)
      return;

   aiPlanSetVariableBool(gLandExplorePlanID, cExplorePlanCanBuildLOSProto, 0, true);
   xsDisableSelf();
}

//==============================================================================
// RULE: airScouting
//
// scout with a flying scout.
//==============================================================================
rule airScouting
   minInterval 1
   inactive
{
   //Stop this if there are no flying scout.
   if (gAirScout == -1)
   {
      aiEcho("No Air scout specified.  Turning off air scout rule");
      xsDisableSelf();
      return;
   }

   //Maintain 1 air scout.
   createSimpleMaintainPlan(gAirScout, gMaintainNumberAirScouts, true, -1);

   //Create a progression to the air scout.
   int pid=aiPlanCreate("AirScoutProgression", cPlanProgression);
	if (pid >= 0)
	{ 
      aiEcho("Creating air scout progression.");
      aiPlanSetVariableInt(pid, cProgressionPlanGoalUnitID, 0, gAirScout);
		aiPlanSetDesiredPriority(pid, 100);
		aiPlanSetEscrowID(pid, cEconomyEscrowID);
		aiPlanSetActive(pid);
	}
   else
      aiEcho("Could not create train air scout plan.");

   //Once we have unit to scout with, set it in motion.
   int exploreID=aiPlanCreate("Explore_Air", cPlanExplore);
	if (exploreID >= 0)
	{
		aiEcho("Setting up air explore plan.");
      aiPlanAddUnitType(exploreID, gAirScout, 1, 1, 1);
      aiPlanSetVariableBool(exploreID, cExplorePlanDoLoops, 0, false);
		aiPlanSetActive(exploreID);
      aiPlanSetEscrowID(exploreID, cEconomyEscrowID);
	}
   else
      aiEcho("Could not create air explore plan.");

   //Go away.
   xsDisableSelf();
}


//==============================================================================
// econAge2Handler
//==============================================================================
void econAge2Handler(int age=1)
{
   aiEcho("Economy Age "+age+".");

   //Start up air scouting.
   airScouting();
   //Re-enable buildHouse.
   xsEnableRule("buildHouse");
   //Fire up opportunities.
   xsEnableRule("opportunities");

   //Make plan to get husbandry.
	int husbandryPlanID=aiPlanCreate("getHusbandry", cPlanProgression);
	if (husbandryPlanID != 0)
   {
      aiPlanSetVariableInt(husbandryPlanID, cProgressionPlanGoalTechID, 0, cTechHusbandry);
	   aiPlanSetDesiredPriority(husbandryPlanID, 25);
	   aiPlanSetEscrowID(husbandryPlanID, cEconomyEscrowID);
	   aiPlanSetActive(husbandryPlanID);
   }
   //Hunting dogs.
   int huntingDogsPlanID=aiPlanCreate("getHuntingDogs", cPlanProgression);
	if (huntingDogsPlanID != 0)
   {
      aiPlanSetVariableInt(huntingDogsPlanID, cProgressionPlanGoalTechID, 0, cTechHuntingDogs);
	   aiPlanSetDesiredPriority(huntingDogsPlanID, 25);
	   aiPlanSetEscrowID(huntingDogsPlanID, cEconomyEscrowID);
	   aiPlanSetActive(huntingDogsPlanID);
   }
}

//==============================================================================
// econAge3Handler
//==============================================================================
void econAge3Handler(int age=0)
{
   aiEcho("Economy Age "+age+".");

   //Enable misc rules.
   xsEnableRule("buildHouse");
   xsEnableRule("buildSettlements");
   xsEnableRule("relocateFarming");

   //Get FTC.
   int planID=aiPlanCreate("GetFTCUpgrade", cPlanProgression);
	if (planID >= 0)
	{ 
      aiPlanSetVariableInt(planID, cProgressionPlanGoalTechID, 0, cTechFortifyTownCenter);
		aiPlanSetDesiredPriority(planID, 75);
		aiPlanSetEscrowID(planID, cEconomyEscrowID);
		aiPlanSetActive(planID);
	}
   
   //Enable gatherer upgrades.
   xsEnableRule("getNextGathererUpgrade");
}

//==============================================================================
// econAge4Handler
//==============================================================================
void econAge4Handler(int age=0)
{
   aiEcho("Economy Age "+age+".");
   xsEnableRule("buildHouse");
   xsEnableRule("randomUpgrader");
}

//==============================================================================
// initEcon
//
// setup the initial Econ stuff.
//==============================================================================
void initEcon()
{
   aiEcho("Economy Init.");

   //Set our update resource handler.
   aiSetUpdateResourceEventHandler("updateResourceHandler");

   //Set up auto-gather escrows.
   aiSetAutoGatherEscrowID(cEconomyEscrowID);
   aiSetAutoFarmEscrowID(cEconomyEscrowID);
	
   //Distribute the resources we have.
   kbEscrowAllocateCurrentResources();

   //Set our farm base.
   gFarmBaseID=kbBaseGetMainID(cMyID);
	
   //Make a plan to manage the villager population.
   gCivPopPlanID=aiPlanCreate("civPop", cPlanTrain);
   if (gCivPopPlanID >= 0)
   {
      //Get our mainline villager PUID.
      int gathererPUID=kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionGatherer,0);
      aiPlanSetVariableInt(gCivPopPlanID, cTrainPlanUnitType, 0, gathererPUID);
      //Train off of economy escrow.
      aiPlanSetEscrowID(gCivPopPlanID, cEconomyEscrowID);
      //Default to 10.
      aiPlanSetVariableInt(gCivPopPlanID, cTrainPlanNumberToMaintain, 0, 10);
		aiPlanSetDesiredPriority(gCivPopPlanID, 100);
      aiPlanSetActive(gCivPopPlanID);
   }

	//Create a herd plan to gather all herdables that we ecounter.
   int herdPlanID=aiPlanCreate("GatherHerdable Plan", cPlanHerd);
   if (herdPlanID >= 0)
   {
      aiPlanAddUnitType(herdPlanID, cUnitTypeHerdable, 0, 100, 100);
      if ((cRandomMapName != "vinlandsaga") && (cRandomMapName != "team migration"))
         aiPlanSetBaseID(herdPlanID, kbBaseGetMainID(cMyID));
      else
      {
         if (cMyCulture != cCultureNorse)
            aiPlanSetVariableInt(herdPlanID, cHerdPlanBuildingTypeID, 0, cUnitTypeGranary);
         else
            aiPlanSetVariableInt(herdPlanID, cHerdPlanBuildingTypeID, 0, cUnitTypeOxCart);
      }
      aiPlanSetActive(herdPlanID);
   }
}

//==============================================================================
// postInitEcon
//==============================================================================
void postInitEcon()
{
   aiEcho("Post Economy Init.");

   //Set the RGP weights.  Script in charge.
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanScriptRPGPct, 0, 0.6);
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanCostRPGPct, 0, 0.4);
   aiSetResourceGathererPercentageWeight(cRGPScript, 0.6);
   aiSetResourceGathererPercentageWeight(cRGPCost, 0.4);

   //Setup AI Cost weights.
   kbSetAICostWeight(cResourceFood, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFood));
   kbSetAICostWeight(cResourceWood, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceWood));
   kbSetAICostWeight(cResourceGold, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceGold));
   kbSetAICostWeight(cResourceFavor, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFavor));

   //Set initial gatherer percentages.
   float foodGPct=aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanGathererPct, cResourceFood);
   float woodGPct=aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanGathererPct, cResourceWood);
   float goldGPct=aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanGathererPct, cResourceGold);
   float favorGPct=aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanGathererPct, cResourceFavor);
   aiSetResourceGathererPercentage(cResourceFood, foodGPct, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceWood, woodGPct, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceGold, goldGPct, false, cRGPScript);
   aiSetResourceGathererPercentage(cResourceFavor, favorGPct, false, cRGPScript);
   aiNormalizeResourceGathererPercentages(cRGPScript);

   //Set up the initial resource break downs.
   int numFoodEasyPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeEasy);
   int numFoodHuntAggressivePlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeHuntAggressive);
   int numFishPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeFish);
   int numWoodPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumWoodPlans, 0);
   int numGoldPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumGoldPlans, 0);
   int numFavorPlans=aiPlanGetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFavorPlans, 0);
	aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeEasy, numFoodEasyPlans, 100, 1.0, kbBaseGetMainID(cMyID));
   aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeHuntAggressive, numFoodHuntAggressivePlans, 90, 1.0, kbBaseGetMainID(cMyID));
   aiSetResourceBreakdown(cResourceFood, cAIResourceSubTypeFish, numFishPlans, 100, 1.0, kbBaseGetMainID(cMyID));
   if (cMyCulture == cCultureEgyptian)
   {
      aiSetResourceBreakdown(cResourceWood, cAIResourceSubTypeEasy, numWoodPlans, 50, 1.0, kbBaseGetMainID(cMyID));
	   aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, numGoldPlans, 55, 1.0, kbBaseGetMainID(cMyID));
   }
   else
   {
      aiSetResourceBreakdown(cResourceWood, cAIResourceSubTypeEasy, numWoodPlans, 55, 1.0, kbBaseGetMainID(cMyID));
	   aiSetResourceBreakdown(cResourceGold, cAIResourceSubTypeEasy, numGoldPlans, 50, 1.0, kbBaseGetMainID(cMyID));
   }
   aiSetResourceBreakdown(cResourceFavor, cAIResourceSubTypeEasy, numFavorPlans, 40, 1.0, kbBaseGetMainID(cMyID));
}


//==============================================================================
// RULE: fishing
//==============================================================================
rule fishing
   minInterval 30
   active
{
   //If we're on a non-water map or RS, go away.
   if ((gWaterMap == false) || (cRandomMapName == "river styx"))
   {
      aiEcho("Not going to explore water or fish on this map.");
		xsDisableSelf();
      return;
   }

   //Don't do anything until we have choosen an age 2 god.
	if (gAge2MinorGod == -1)
		return;
	//Don't do anything until we can get the age 2 god (i.e. we have our temple up).
	int techStatus=kbGetTechStatus(gAge2MinorGod);
	if ((techStatus == cTechStatusUnobtainable))
		return;

	//Get the closest water area.  if there isn't one, we can't fish.
	static int areaID=-1;
	if (areaID == -1)
		areaID=kbAreaGetClosetArea(kbBaseGetLocation(cMyID, kbBaseGetMainID(cMyID)), cAreaTypeWater);
	if (areaID == -1)
	{
		aiEcho("Can't fish on this map, no water.");
		xsDisableSelf();
		return;
	}
	
   //Don't fish until we're setup for wood.
	if (kbSetupForResource(kbBaseGetMainID(cMyID), cResourceWood, 25.0, 600) == false)
      return;
   //Get our fish gatherer.
	int fishGatherer=kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionFish,0);

	//Create the fish plan.
	int fishPlanID=aiPlanCreate("FishPlan", cPlanFish);
	if (fishPlanID >= 0)
	{
		aiEcho("Starting up the fishing plan.  Will fish when I find fish.");
      aiPlanSetDesiredPriority(fishPlanID, 95);
		aiPlanSetVariableVector(fishPlanID, cFishPlanLandPoint, 0, kbBaseGetLocation(cMyID, kbBaseGetMainID(cMyID)));
		//If you don't explicitly set the water point, the plan will find one for you.
		aiPlanSetVariableBool(fishPlanID, cFishPlanAutoTrainBoats, 0, true);
		aiPlanSetEscrowID(fishPlanID, cEconomyEscrowID);
		aiPlanAddUnitType(fishPlanID, fishGatherer, 2, gNumBoatsToMaintain, gNumBoatsToMaintain);
		aiPlanSetActive(fishPlanID);
	}
   aiPlanSetVariableInt(gGatherGoalPlanID, cGatherGoalPlanNumFoodPlans, cAIResourceSubTypeFish, 1);

   //Build a dock in water, so we can scout.
   int buildDock=aiPlanCreate("BuildDock", cPlanBuild);
   if (buildDock >= 0)
   {
      aiEcho("Building dock for scouting.");
      //BP Type and Priority.
      aiPlanSetVariableInt(buildDock, cBuildPlanBuildingTypeID, 0, cUnitTypeDock);
      aiPlanSetDesiredPriority(buildDock, 100);
      aiPlanSetVariableVector(buildDock, cBuildPlanDockPlacementPoint, 0, kbBaseGetLocation(cMyID, kbBaseGetMainID(cMyID)));
      aiPlanSetVariableVector(buildDock, cBuildPlanDockPlacementPoint, 1, kbAreaGetCenter(areaID));
      aiPlanAddUnitType(buildDock, kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionBuilder, 0), 1, 1, 1);
      aiPlanSetEscrowID(buildDock, cEconomyEscrowID);
      aiPlanSetActive(buildDock);
   }

   //Make a plan to explore with the water scout.
	int waterExploreID=aiPlanCreate("Explore_Water", cPlanExplore);
	if (waterExploreID >= 0)
	{
		aiEcho("Creating water explore plan.");
      aiPlanAddUnitType(waterExploreID, gWaterScout, 1, 1, 1);
		aiPlanSetDesiredPriority(waterExploreID, 100);
      aiPlanSetVariableBool(waterExploreID, cExplorePlanDoLoops, 0, false);
      aiPlanSetActive(waterExploreID);
      aiPlanSetEscrowID(cEconomyEscrowID);
	}
	xsDisableSelf();
}

//==============================================================================
// RULE: buildHouse
//==============================================================================
rule buildHouse
   minInterval 11
   active
{
	//Don't build another house if we've got at least gHouseAvailablePopRebuild open pop slots.
   if (kbGetPop()+gHouseAvailablePopRebuild < kbGetPopCap())
      return;
   //If we have any houses that are building, skip.
   if (kbUnitCount(cMyID, cUnitTypeHouse, cUnitStateBuilding) > 0)
      return;
   
	//If we already have gHouseBuildLimit houses, we shouldn't build anymore.
   if (gHouseBuildLimit != -1)
   {
      int numberOfHouses=kbUnitCount(cMyID, cUnitTypeHouse, cUnitStateAliveOrBuilding);
      if (numberOfHouses >= gHouseBuildLimit)
         return;
   }

	//Get the current Age.
	int age=kbGetAge();
	//Limit the number of houses we build in each age.
	if (gAgeCapHouses == true)
   {
      if (age == 0)
	   {
		   if (numberOfHouses >= 2)
		   {
			   xsDisableSelf();
			   return;
		   }
	   }
   }

   //If we already have a house plan active, skip.
   if (aiPlanGetIDByTypeAndVariableType(cPlanBuild, cBuildPlanBuildingTypeID, cUnitTypeHouse) > -1)
      return;

   //Over time, we will find out what areas are good and bad to build in.  Use that info here, because we want to protect houses.
	int planID=aiPlanCreate("BuildHouse", cPlanBuild);
   if (planID >= 0)
   {
      aiPlanSetVariableInt(planID, cBuildPlanBuildingTypeID, 0, cUnitTypeHouse);
      aiPlanSetVariableBool(planID, cBuildPlanInfluenceAtBuilderPosition, 0, true);
      aiPlanSetVariableFloat(planID, cBuildPlanInfluenceBuilderPositionValue, 0, 100.0);
      aiPlanSetVariableFloat(planID, cBuildPlanInfluenceBuilderPositionDistance, 0, 5.0);
      aiPlanSetVariableFloat(planID, cBuildPlanRandomBPValue, 0, 0.99);
      aiPlanSetBaseID(planID, kbBaseGetMainID(cMyID));
      aiPlanSetDesiredPriority(planID, 100);

		int builderTypeID = kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionBuilder,0);

		aiPlanAddUnitType(planID, builderTypeID, gBuildersPerHouse, gBuildersPerHouse, gBuildersPerHouse);
      aiPlanSetEscrowID(planID, cEconomyEscrowID);

      vector backVector = kbBaseGetBackVector(cMyID, kbBaseGetMainID(cMyID));

      float x = xsVectorGetX(backVector);
      float z = xsVectorGetZ(backVector);
      x = x * 30.0;
      z = z * 30.0;

      backVector = xsVectorSetX(backVector, x);
      backVector = xsVectorSetZ(backVector, z);
      backVector = xsVectorSetY(backVector, 0.0);
      vector location = kbBaseGetLocation(cMyID, kbBaseGetMainID(cMyID));
      location = location + backVector;
      aiPlanSetVariableVector(planID, cBuildPlanInfluencePosition, 0, location);
      aiPlanSetVariableFloat(planID, cBuildPlanInfluencePositionDistance, 0, 20.0);
      aiPlanSetVariableFloat(planID, cBuildPlanInfluencePositionValue, 0, 1.0);

      aiPlanSetActive(planID);
   }
}

//==============================================================================
// RULE: buildSettlements
//==============================================================================
rule buildSettlements
   minInterval 10
   inactive
{
   //Figure out if we have any active BuildSettlements.
   int numberBuildSettlementGoals=aiGoalGetNumber(cGoalPlanGoalTypeBuildSettlement, cPlanStateWorking, true);
   if (numberBuildSettlementGoals > 0)
      return;
   if (findASettlement() == false)
      return;
   
   int popCapBuffer=10;
   int currentPopNeeds=aiGetPopNeeds();
   int adjustedPopCap=kbGetPopCap()-popCapBuffer;

   //Dont do this unless we need the pop
   if (currentPopNeeds < adjustedPopCap)
      return;
   //Dont get more than other players.
   int numberSettlements=getNumberUnits(cUnitTypeAbstractSettlement, cMyID, cUnitStateAliveOrBuilding);

   //If we're on Easy and we have 3 settlements, go away.
   if ((aiGetWorldDifficulty() == cDifficultyEasy) && (numberSettlements >= 3))
   {
      xsDisableSelf();
      return;
   }

   //Don't get too many more than our allies.
   int largestAllyCount=0;
   for (i=1; < cNumberPlayers)
   {
      if (i == cMyID)
         continue;
      if(kbIsPlayerAlly(i) == false)
         continue;
      int count=getNumberUnits(cUnitTypeAbstractSettlement, i, cUnitStateAliveOrBuilding);
      if(count > largestAllyCount)
         largestAllyCount=count;
   }
   //Count the one I want to build on.
   numberSettlements=numberSettlements+1;
   //Never have more than 2 more settlements than any ally.
   int difference=numberSettlements-largestAllyCount;
   if ((difference > 2) && (largestAllyCount!=0))
      return;

   //See if there is another human on my team.
   bool haveHumanTeammate=false;
   for (i=1; < cNumberPlayers)
   {
      if(i == cMyID)
         continue;
      //Find the human player
      if (kbIsPlayerHuman(i) != true)
         continue;

      //This player is a human ally and not resigned.
      if ((kbIsPlayerAlly(i) == true) && (kbIsPlayerResigned(i) == false))
      {
         haveHumanTeammate=true;
         break;
      }
   }
   if(haveHumanTeammate == true)
   {
      if (kbGetAge() == 2)
      {
         if (numberSettlements > 4)
            return;
      }
      else if (kbGetAge() == 3)
      {
         if (numberSettlements > 5)
            return;
      }
   }

   //Else, do it.
   createBuildSettlementGoal("BuildSettlement", kbGetAge(), -1, kbBaseGetMainID(cMyID), 3, kbTechTreeGetUnitIDTypeByFunctionIndex(cUnitFunctionBuilder,0), true, 100);
}

//==============================================================================
// RULE: opportunities
//==============================================================================
rule opportunities
   minInterval 31
   inactive
   runImmediately
{
   float currentFood=kbResourceGet(cResourceFood);
   //float currentWood=kbResourceGet(cResourceWood);
   float currentGold=kbResourceGet(cResourceGold);
   //float currentFavor=kbResourceGet(cResourceFavor);
   if (currentFood > 500 && currentGold > 300)
      getNextGathererUpgrade();
}

//==============================================================================
// RULE: randomUpgrader
//
//==============================================================================
rule randomUpgrader
   minInterval 30
   active
   runImmediately
{
   static int id=0;
   //Don't do anything until we have some pop.
   int maxPop=kbGetPopCap();
   if (maxPop < 130)
      return;
   //If we still have some pop slots to fill, quit.
   int currentPop=kbGetPop();
   if ((maxPop-currentPop) > 20)
      return;

   //If we have lots of resources, get a random upgrade.
   float currentFood=kbResourceGet(cResourceFood);
   float currentWood=kbResourceGet(cResourceWood);
   float currentGold=kbResourceGet(cResourceGold);
   float currentFavor=kbResourceGet(cResourceFavor);
   if ((currentFood > 1000) && (currentWood > 1000) && (currentGold > 1000))
   {
      int upgradeTechID=kbTechTreeGetRandomUnitUpgrade();
      //Dont make another plan if we already have one.
      if (aiPlanGetIDByTypeAndVariableType(cPlanProgression, cProgressionPlanGoalTechID, upgradeTechID) != -1)
         return;

      //Make plan to get this upgrade.
	   int planID=aiPlanCreate("nextRandomUpgrade - "+id, cPlanProgression);
	   if (planID < 0)
         return;
      
	   aiPlanSetVariableInt(planID, cProgressionPlanGoalTechID, 0, upgradeTechID);
	   aiPlanSetDesiredPriority(planID, 25);
	   aiPlanSetEscrowID(planID, cEconomyEscrowID);
	   aiPlanSetActive(planID);
      aiEcho("randomUpgrader: successful in creating a progression to "+kbGetTechName(upgradeTechID));
	   id++;
   }
}

//==============================================================================
// RULE: updateAICost
//==============================================================================
rule updateAICost
   minInterval 60
   active
{
   //Get our current resources.
   float currentFood=kbResourceGet(cResourceFood);
   float currentWood=kbResourceGet(cResourceWood);
   float currentGold=kbResourceGet(cResourceGold);
   float currentFavor=kbResourceGet(cResourceFavor);
   //Find the highest.
   float highestResource=-1.0;
   if (currentFood > highestResource)
      highestResource=currentFood;
   if (currentWood > highestResource)
      highestResource=currentWood;
   if (currentGold > highestResource)
      highestResource=currentGold;
   if (currentFavor > highestResource)
      highestResource=currentFavor;

   //Min and max costs.
   float minCost=0.1;
   float maxCost=10.0;
   //Now, go back through and figure out the relative costs.  The most plentiful resource is 1.0.
   float foodCost=maxCost;
   if (currentFood > 1.0)
   {
      foodCost=highestResource/currentFood;
      if (foodCost < minCost)
         foodCost=minCost;
      if (foodCost > maxCost)
         foodCost=maxCost;
   }
   float woodCost=maxCost;
   if (currentWood > 1.0)
   {
      woodCost=highestResource/currentWood;
      if (woodCost < minCost)
         woodCost=minCost;
      if (woodCost > maxCost)
         woodCost=maxCost;
   }
   float goldCost=maxCost;
   if (currentGold > 1.0)
   {
      goldCost=highestResource/currentGold;
      if (goldCost < minCost)
         goldCost=minCost;
      if (goldCost > maxCost)
         goldCost=maxCost;
   }
   float favorCost=maxCost;
   if (currentFavor > 1.0)
   {
      favorCost=highestResource/currentFavor;
      if (favorCost < minCost)
         favorCost=minCost;
      if (favorCost > maxCost)
         favorCost=maxCost;
   }

   //Update the goal.
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceGold, goldCost);
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceWood, woodCost);
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFood, foodCost);
   aiPlanSetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFavor, favorCost);
   //Setup AI Cost weights.
   kbSetAICostWeight(cResourceFood, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFood));
   kbSetAICostWeight(cResourceWood, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceWood));
   kbSetAICostWeight(cResourceGold, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceGold));
   kbSetAICostWeight(cResourceFavor, aiPlanGetVariableFloat(gGatherGoalPlanID, cGatherGoalPlanResourceCostWeight, cResourceFavor));
}

