--' ************************************ Fix Event ************************************ '--
StayAlertEvent = deriveLuaClass("StayAlertEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="StayAlertEvent",
	entrancePositions = levelDesc.suitcaseEntrancePositions,
	exitPositions = levelDesc.suitcaseExitPositions,
	generateFindObjectFunc = levelDesc.generateFindObject,
	initializeFindObjectFunc = levelDesc.initializeFindObject,
	objectsToFind = {	{searchingObject="yup2female", objects={{overhead="lost_suitcase_1:room2_icons", objectName="suitcase_1:room2_icons"},}, }, 
						{searchingObject="biggie2male", objects={{overhead="lost_suitcase_2:room2_icons", objectName="suitcase_2:room2_icons"},}, }, 
						{searchingObject="yup2male", objects={{overhead="lost_suitcase_3:room2_icons", objectName="suitcase_3:room2_icons"},}, },
						{searchingObject="child2male", objects={{overhead="lost_suitcase_4:room2_icons", objectName="suitcase_4:room2_icons"},}, },
					},
	minNextFindObjectDuration = 2000,
	maxNextFindObjectDuration = 5000,
}
--]]

function StayAlertEvent:init(desc)
	StayAlertEvent.superInit(self, desc)

	--' entrancePositions
	assert(self.entrancePositions)
	--' exitPositions
	assert(self.exitPositions)
	--' generateFindObjectFunc = levelDesc.generateFindObject,
	assert(self.generateFindObjectFunc)
	--' initializeFindObjectFunc = levelDesc.initializeFindObject,
	assert(self.initializeFindObjectFunc)

	--' objectsToFind example: 
	--'	objectsToFind = { {searchingObject="yup2female", objects={{overhead="lost_suitcase_1", objectName="suitcase_1"},}, }, 
	--'					{searchingObject="biggie2male", objects={{overhead="lost_suitcase_2", objectName="suitcase_2"},}, }, 
	--'					{searchingObject="yup2male", objects={{overhead="lost_suitcase_3", objectName="suitcase_3"},}, },
	--'					{searchingObject="child2male", objects={{overhead="lost_suitcase_4", objectName="suitcase_4"},}, },
	--'				},
	assert(self.objectsToFind)

	self.minNextFindObjectDuration = desc.minNextFindObjectDuration or 2000
	self.maxNextFindObjectDuration = desc.maxNextFindObjectDuration  or 5000

	self.curSearchingObjectId = 1
	self.curObjectToFindId = 0 --' will be incremented by SetupNextSearchObject
end


function StayAlertEvent:onInitialized()
	StayAlertEvent.super.onInitialized(self)
	self.nextFindObjectDuration = math.Random(self.minNextFindObjectDuration, self.maxNextFindObjectDuration)
end

function StayAlertEvent:getNumItemsTodo()
	local numItems = 0
	for _o,oitem in ipairs(self.objectsToFind) do
		for _i,iitem in ipairs(oitem.objects) do
			numItems = numItems + 1
		end
	end
	return numItems
end

function StayAlertEvent:onStart()
	StayAlertEvent.super.onStart(self)
	self:SetupNextSearchObject()
end

function StayAlertEvent:SetupNextSearchObject()
	self.curObjectToFindId = self.curObjectToFindId + 1
	if (not self:getCurFindObject()) then
		self.curObjectToFindId = 1
		self.curSearchingObjectId = self.curSearchingObjectId + 1
		if (not self:getCurFindObject()) then
			self:eventWon()
			return
		else
			self:UpdatePrevSearchingObject()
		end
	end
	self:UpdateCurSearchingObject()
end

function StayAlertEvent:UpdatePrevSearchingObject()
	local searchingName = self.objectsToFind[self.curSearchingObjectId-1] and self.objectsToFind[self.curSearchingObjectId-1].searchingObject
	if (searchingName) then
		level:getObject(searchingName):removeOverhead()
	end
end

function StayAlertEvent:UpdateCurSearchingObject()
	local obj = self:getCurSearchingObject()
	obj:removeOverhead()
	local findObject = self:getCurFindObject()
	if (findObject.linkUnderParent == true) then
		obj.overheadParent = obj
	end
	obj:createOverhead(findObject.overhead)
end

--' will be called from generateFindObjectFunc in the shiftfile
function StayAlertEvent:initializeFindObject(object)

	object.oldOnClick = object.onClick
	object.oldClickable = object.clickable
	
	self.initializeFindObjectFunc(level, self, object)
end

function StayAlertEvent:getCurFindObject()
	return self.objectsToFind[self.curSearchingObjectId] and self.objectsToFind[self.curSearchingObjectId].objects[self.curObjectToFindId]
end

function StayAlertEvent:getCurSearchingObject()
	local searchingName = self.objectsToFind[self.curSearchingObjectId] and self.objectsToFind[self.curSearchingObjectId].searchingObject
	if (searchingName) then
		return level:getObject(searchingName)
	end
	return nil
end

function StayAlertEvent:onTick(time)
	if (not self.started or self.completed) then
		return
	end
	
	self.nextFindObjectDuration = self.nextFindObjectDuration - time
	if (self.nextFindObjectDuration <= 0) then
		local objectName = 	self:getCurFindObject().objectName
		self.generateFindObjectFunc(level, self, objectName)

		self.nextFindObjectDuration = math.Random(self.minNextFindObjectDuration, self.maxNextFindObjectDuration)			
	end
	
end