--' ************************************ Shout and Make Event ************************************ '--
ShoutAndMakeEvent = getClass("ShoutAndMakeEvent")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="ShoutAndMakeEvent",
	bar = "progressbar_big",
	productPool = {"#sock", "#bread#cheese"},
	numProductsPerOrder = 2,
	numOrders = 2,
	timeUntilNextOrder = 0,
	timeUntilFirstOrder = 0,
	maxGroupCount = 2,
	limitToMaxGroupCount = true,
	items = {
		{ obj="cleaner", barPos = {x=10, y=-100}, workPos={x=300, y=683}, workAnimation="clean", workDuration=2000, timeUntilEnd=120000, },
	},
	maxNumTimesProductOrdered = 1,
	matchProductPoolProductOrder = true
}

function levelDesc:onDeliverProducts(item)
	...
end
--]]

function ShoutAndMakeEvent:init(desc)
	ShoutAndMakeEvent.superInit(self, desc)

	--' Configuration variables
	self.timeUntilFirstOrder = self.timeUntilFirstOrder or 0
	self.timeUntilNextOrder = self.timeUntilNextOrder or 0		--' Interval at which the orders come
	self.numProductsPerOrder = self.numProductsPerOrder or 0	--' Number of products ordered each order
	self.numOrders = self.numOrders or 0						--' Total number of orders
	self.productPool = desc.productPool or {}					--' Product pool
	self.limitToMaxGroupCount = desc.limitToMaxGroupCount or false
	self.eventWonDelay = desc.eventWonDelay or 0

	self.maxNumTimesProductOrdered = desc.maxNumTimesProductOrdered or -1
	self.matchProductPoolProductOrder = desc.matchProductPoolProductOrder or false
	
	--' Status variables
	self.numObjectsOutOfTime = 0
	self.numObjectsDone = 0

	assert(self.maxNumTimesProductOrdered < 0 or
		(self.numProductsPerOrder * self.numOrders <= self.maxNumTimesProductOrdered * #self.productPool),
		"More orders requested than there are products * maxNumTimesProductOrdered!")
end

function shoutAndMakeWorkStart(self, task)
	event:hideHintArrow()
	if (not event.started or event.completed or self.item.order.currentStep.numProducts <= 0) then
		return false
	end

	local duration = self:getWorkDurationWith(task.actor)
	task.duration = duration
	if (duration > 0) then
		if (self.workDirection) then
			task.actor.direction = self.workDirection
		end
		task.actor:playAnimation(self.workAnimation or "work", self.workCycles or -1, self.workFrameTime or 150, self.workWhenFinished or "default")
	end
end

function shoutAndMakeWorkFinish(obj, task)
	--' Check the products
	assert(obj.item.order)
	--' Check if all the products for this object are delivered
	local numProducts = obj.item.order.currentStep.numProducts
	obj.item.order:deliver(obj.item)
	local numProductsLeft = obj.item.order.currentStep.numProducts
	if (numProductsLeft == 0 and obj.item.numOrders <= 0) then
		event:objectAllProductsDone(obj.item)
	else
		event:updateOrderOverhead(obj.item)
	end

	task.actor:playDefaultAnimation()
	if (numProductsLeft < numProducts) then
		local pos = obj.floaterPosition
		local score = event.scoreOneDone * (numProducts - numProductsLeft)
		local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), score)
		level:handleScore(score, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, score, 0)
		if (level.BM_eventScore) then level:BM_eventScore(score) end

		if (level.onDeliverProducts) then
			level:onDeliverProducts(obj.item)
		end
	end
	
end

function ShoutAndMakeEvent:onStart()
	ShoutAndMakeEvent.super.onStart(self)
	for _,item in ipairs(self.items) do
		self:preserveItem(item)
		if (isTable(item.obj)) then
			item.obj.onWorkStart = shoutAndMakeWorkStart
			item.obj.onWorkUpdate = nil
			item.obj.onWorkFinish = shoutAndMakeWorkFinish

			item.productPool = item.productPool or self.productPool

			item.numProductsPerOrder = item.numProductsPerOrder or self.numProductsPerOrder
			item.numOrders = item.numOrders or self.numOrders
			item.maxGroupCount = item.maxGroupCount or self.maxGroupCount
			item.maxHorizontalSlotCount = item.maxHorizontalSlotCount or self.maxHorizontalSlotCount
			item.timeUntilNextOrder = item.timeUntilNextOrder or self.timeUntilNextOrder
			item.nextOrderTimer = self.timeUntilFirstOrder

			if (self.maxNumTimesProductOrdered > 0) then
				item.timesProductOrdered = {}
			end

			item.order = self:addOrder()
			item.duration = item.timeUntilEnd
			item.active = false

			self:createProgressBar(item, 1)
		end
	end

	eventBar.progress = 1
	eventBar.startValue = 1
	eventBar.targetValue = 0
	eventBar.showValue = false
	eventBar.showTarget = false
	eventBar.draining = true
end

function ShoutAndMakeEvent:showItem(item)
	ShoutAndMakeEvent.super.showItem(self, item)

	self:showProgressBar(item)
	self:showHint(item.obj)
	self:onProgress()
end

function ShoutAndMakeEvent:objectOutOfTime(item)
	self.numObjectsOutOfTime = self.numObjectsOutOfTime + 1
	self:removeDecals(item)
	item.done = true
	item.obj.clickable = false

	self.won = false
	if (level.onObjectOutOfTime) then
		level:onObjectOutOfTime(self, item)
	end
end

function ShoutAndMakeEvent:objectAllProductsDone(item)
	self.numObjectsDone = self.numObjectsDone + 1
	self:removeDecals(item)
	item.done = true
	item.obj.clickable = false
end

function ShoutAndMakeEvent:onTick(time)
	if (self.completed or not self.started) then
		return
	end

	for _,item in ipairs(self.items) do
		if (item.obj and not item.done) then
			item.duration = item.duration - time
			if (self.blinkStartFraction and
				item.duration / item.timeUntilEnd < self.blinkStartFraction)
			then
				self:blinkBar(item)
			end
			if (item.duration <= 0 and item.obj.pendingWorkTaskCount == 0) then
				self:objectOutOfTime(item)
				break
			end
			--'if (item.numOrders > 0 and item.order.currentStep.numProducts < self.numProductsPerOrder) then
			if (item.numOrders > 0) then
				item.nextOrderTimer = item.nextOrderTimer - time
				if (item.nextOrderTimer <= 0) then
					if (not self.limitToMaxGroupCount or item.order.currentStep.numProducts < item.maxGroupCount) then
						if self.orderFirstTime then
							self:orderFirstTime(item)
						end
						local validProductPool = {}
						for _,v in ipairs(item.productPool) do
							if (table.contains(level.possibleProducts, v)) then
								if (self.maxNumTimesProductOrdered < 0 or
										not item.timesProductOrdered[v] or
										item.timesProductOrdered[v] < self.maxNumTimesProductOrdered) then
									table.insert(validProductPool, v)
								end
							end
						end
						assert(#validProductPool > 0, "No valid products to choose from found!")

						local missingProducts = nil
						if (isBeta and self.matchProductPoolProductOrder and #validProductPool ~= #item.productPool) then
							for _,v in ipairs(item.productPool) do
								if (not table.contains(level.possibleProducts, v)) then
									if (missingProducts) then
										missingProducts = missingProducts..", "
									end
									missingProducts = (missingProducts or "").."'"..v.."'"
								end
							end
							if (not missingProducts) then
								print("All products were ordered the maximum nr of times")
							else
								print("Missing: "..missingProducts)
							end
						end
						assert(not missingProducts, "Could not find all the products: "..ts(missingProducts))
						--'assert(not self.matchProductPoolProductOrder or #validProductPool == #item.productPool, "some ("..ts(#item.productPool - #validProductPool)..") were not found in the level!")

						item.numOrders = item.numOrders - 1
						if (#validProductPool > 0) then
							for i=1, self.numProductsPerOrder do
								local product = nil
								local orderedTooOften = false
								local maxTries = 1000
								repeat
									if (self.matchProductPoolProductOrder) then
										local productId = (item.nextProductId or 0) + 1
										item.nextProductId = productId
										productId = ((productId - 1)% #item.productPool) + 1
										if (table.contains(level.possibleProducts, item.productPool[productId])) then
											product = item.productPool[productId]
										end
									else
										product = validProductPool[math.Random(#validProductPool)]
									end

									if (product and self.maxNumTimesProductOrdered > 0) then
										local nr = item.timesProductOrdered[product] or 0
										nr = nr + 1
										if (nr > self.maxNumTimesProductOrdered) then
											orderedTooOften = true
										elseif (product) then
											item.timesProductOrdered[product] = nr
										end
									end
									maxTries = maxTries - 1
									assert(maxTries > 0, "Inifinte loop fetching product: No valid product found"..(orderedTooOften and "(everything was ordered too often)"or""))
								until (product and (self.maxNumTimesProductOrdered < 0 or not orderedTooOften) or maxTries <= 0)
								if (product) then
									item.order.currentStep:addProduct(product)
								end
							end

							self:updateOrderOverhead(item)
							item.nextOrderTimer = self.timeUntilNextOrder

							if (level.onProductAdded) then
								level:onProductAdded()
							end
						end
						self.nextOrderTimerDoneAlreadyCalled = false
					else
						if (not self.nextOrderTimerDoneAlreadyCalled) then
							self.nextOrderTimerDoneAlreadyCalled = true
							if (level.onNextOrderTimerDone) then
								level:onNextOrderTimerDone()
							end
						end
					end
				end
			end
		end
	end

	if (self.numObjectsOutOfTime + self.numObjectsDone >= self:getNumItemsTodo()) then
		if (self.numObjectsOutOfTime > 0 and not self.alreadyWon) then
			self:eventLost()
		else
			self.alreadyWon = true
			--' Give customers a chance to finish their last meal before firing eventWon/End animation
			if (self.eventWonDelay <= 0 or level.hasShiftFinished) then
				self:eventWon()
			else
				self.eventWonDelay = self.eventWonDelay - time
			end
		end
	end
	self:onProgress()
end

function ShoutAndMakeEvent:onProgress()
	ShoutAndMakeEvent.super.onProgress(self)

	for _,item in ipairs(self.items) do
		if (item.obj and not item.done and item.bar) then
			item.bar:setProgress(item.duration / item.timeUntilEnd)
		end
	end
end

function ShoutAndMakeEvent:getProgress()
	local lowest
	for _,item in ipairs(self.items) do
		if (not item.done and isNumber(item.duration)) then
			local p = item.duration / item.timeUntilEnd
			if (not lowest or p < lowest) then
				lowest = p
			end
		end
	end
	return lowest or 0
end

function ShoutAndMakeEvent:removeDecals(item)
	ShoutAndMakeEvent.super.removeDecals(self, item)
	if (item and item.obj and item.order) then
		item.obj:removeOverhead(true)
		item.order = nil
	end
end
