--' ************************************ Fix Event ************************************ '--
RetreiveCharacterEvent = deriveLuaClass("RetreiveCharacterEvent", "Event")
RetreiveCharacter = deriveLuaClass("RetreiveCharacter", "DelCharacter")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="RetreiveCharacterEvent",
	matchingItemType = "RetreiveCharacter",
	escapeDurationMin = 5000,
	escapeDurationMax = 10000,	
	self.escapeSound = "dflknsd:sound"
}
--]]

function RetreiveCharacterEvent:init(desc)
	RetreiveCharacterEvent.superInit(self, desc)
	
	self.leaveDuration = desc.leaveDuration or 10000
	self.escapeSound = self.escapeSound
end

function RetreiveCharacterEvent:onInitialized()
	self.super.onInitialized(self)	
	
	eventBar.showValue = false
	eventBar.showTarget = false
	eventBar.showBar = false
	self.won = true
	
	for _,item in ipairs(self.items) do
		assert(item.obj)
		self:preserveItem(item)
		item.obj:initCharacter()
	end	
end

function RetreiveCharacterEvent:onStart()
	RetreiveCharacterEvent.super.onStart(self)

	for _,item in ipairs(self.items) do
		assert(item.obj)
		item.obj.clickable = false
		item.obj:reinitializeCharacter()
	end	
end

function RetreiveCharacterEvent:onShiftFinish()
	local anim = RetreiveCharacterEvent.super.onShiftFinish(self)
	
	for _,item in ipairs(self.items) do
		assert(item.obj)
		taskSystem:cancelAllTasksFor(item.obj)
		item.obj:stopLoopSample()
	end
	return anim
end

function RetreiveCharacterEvent:eventWon()
	RetreiveCharacterEvent.super.eventWon(self)
	for _,item in ipairs(self.items) do
		assert(item.obj)
		taskSystem:cancelAllTasksFor(item.obj)
		item.obj:stopLoopSample()
	end
end

RetreiveCharacter = deriveLuaClass("RetreiveCharacter", "Object")

function RetreiveCharacter:initCharacter()

	self.escapeDurationMin = self.escapeDurationMin or 5000
	self.escapeDurationMax = self.escapeDurationMax or 10000

	self.startLocation = self.location
	self.workPosOffset = {x=self.startLocation.x - self.workPos.x, y=self.startLocation.y - self.workPos.y}
	self.walkTimer = math.Random(self.escapeDurationMin, self.escapeDurationMax)	
	
	self.clickable = false
	self.onClick = self.retreiveEventOnClick
	self.onWorkStart = self.retreiveEventOnWorkStart
	self.onWorkUpdate = self.retreiveEventOnWorkUpdate
	self.onWorkFinish = self.retreiveEventOnWorkFinish
	self.onTick = self.retreiveEventOnTick
	self.oldWalkSpeed = self.curWalkSpeed	
end


function RetreiveCharacter:reinitializeCharacter()
	self.workPos = {x=self.location.x - self.workPosOffset.x, y=self.location.y - self.workPosOffset.y}
end

function RetreiveCharacter:retreiveEventOnClick()
	taskSystem:cancelAllTasksFor(self)

	if (level.event_onClick) then
		local handled = level:event_onClick(self)
		if (handled ~= nil) then return handled end
	end

	self.super.onClick(self)
	self.clickable = false

	return true
end

function RetreiveCharacter:retreiveEventOnWorkStart(task)
	if (level.event_onWorkStart) then
		local handled = level:event_onWorkStart(self, task)
		if (handled ~= nil) then return handled end
	end
	
	self:stopLoopSample()
	
	self.retreiving = true
	if (event.retreiveObjectSound) then
		self:loopSample(event.retreiveObjectSound)
	end
	
	local duration = (math.distance(self.startLocation, self.location) / hero.walkSpeed) * 1000
	self:addTask({class="PositionTask", duration=duration, target=self.startLocation})

	local actor = task.actor
	actor:addTaskAtFront({class="CallTask", func=function() self:retreiveEventDone(actor) end})
	actor:addTaskAtFront({class="WalkToTask", target={x=self.startLocation.x - self.workPosOffset.x, y=self.startLocation.y - self.workPosOffset.y}, playDefaultOnFinish=true })

	return true
end

function RetreiveCharacter:retreiveEventOnWorkUpdate(task, time)
	return true
end

function RetreiveCharacter:retreiveEventOnWorkFinish(task)		
end

function RetreiveCharacter:retreiveEventDone(actor)
	if (level.event_retreiveEventDone) then
		level:event_retreiveEventDone(self, actor)
	end

	self.retreiving = false
	self.walking = false
	self.clickable = false
	self:stopLoopSample()

--[[	
	eventBar:incValue(1)
	event.numRetreivesDone = event.numRetreivesDone + 1
	if (event.numRetreivesDone >= event:getNumItemsTodo()) then
		event:eventWon()
	else
]]
		if (event.scoreOneDone and event.scoreOneDone > 0) then
			local pos = self.floaterPosition
			local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), event.scoreOneDone)
			level:handleScore(event.scoreOneDone, pos.x, pos.y, false, 0);
			effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreOneDone, 0)
			if (level.BM_eventScore) then level:BM_eventScore(event.scoreOneDone) end
		end
		
		self.walkTimer = math.Random(event.escapeDurationMin, event.escapeDurationMax)
--'	end
end

function RetreiveCharacter:retreiveEventOnTick(time)
	self.super.tick(self, time)

	if (not event.started or event.completed) then
		return
	end
	
	if (level.event_retreiveEventTick) then
		level:event_retreiveEventTick(self, time)
	end
	
	if (self.walking) then
		self:reinitializeCharacter()
	elseif (self.walkTimer > 0) then
		self.walkTimer = self.walkTimer - time
		if (self.walkTimer <= 0) then
			self.clickable = true
			self.walking = true

			if (event.escapeSound) then
				print("event.escapeSound: "..event.escapeSound)
				self:addTask({class="LoopSoundTask", sound=event.escapeSound})
			end
			self:addTask({class="PositionTask", duration=event.leaveDuration, target=self.escapePosition})
			self:addTask({class="CallTask", func=function(self) 
					print("LOST!!!!!!!!!!!!!")
					self.clickable = false 
					self.walking = false
					self.lost = true
					self:stopLoopSample()
					event:eventLost() 
				end })
		end
	end
end
