--' ************************************ Fix Event ************************************ '--
PushAwayEvent = deriveLuaClass("PushAwayEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="PushAwayEvent",
	scoreOneDone=100,
	scoreAllDone=200,
	numChases=4,
	walktoWaitPosition={x=100, y=100},
	waitPosition={x=100, y=100},
	waitAnimation="sit",
	waitDirection="right",
	annoyAnimation="",
	waitFrameTime = 100,
	chaseCharacter={base="cat", x=100, y=100},
	minFindSeatDuration=2000,
	maxFindSeatDuration=6000,
	doWalktoWaitPosition = false,
}
]]--

function PushAwayEvent:init(desc)
	PushAwayEvent.superInit(self, desc)

	--' Configuration variables
	self.scoreOneDone = self.scoreOneDone or 100
	self.scoreAllDone = self.scoreAllDone or 300
	self.numChases = self.numChases or 5
	self.walktoWaitPosition = self.walktoWaitPosition or self.waitPosition or assert(self.waitPosition, "need walktoWaitPosition")
	self.waitPosition = self.waitPosition or self.walktoWaitPosition or assert(self.waitPosition, "need waitPosition")
	self.waitAnimation = self.waitAnimation or "sit"
	self.annoyAnimation = self.annoyAnimation  or "sit"
	self.annoySound = self.annoySound
	self.waitFrameTime = self.waitFrameTime  or "sit"
	self.minFindSeatDuration = self.minFindSeatDuration or 2000
	self.maxFindSeatDuration = self.maxFindSeatDuration or 6000
	self.doWalktoWaitPosition = self.doWalktoWaitPosition or true
	self.waitAnimation = self.waitAnimation or "sit"
	self.waitDirection = self.waitDirection

	--' Status variables
	self.hintFired = false
	self.activeTable = nil
	self.nextSeatDuration = 0

	--' Override
	self.matchingItemType = "Table"
end

function PushAwayEvent:getNumItemsTodo()
	return self.numChases
end

function PushAwayEvent:onInitialized()
	PushAwayEvent.super.onInitialized(self)

	if (self.trackProgress) then
		eventBar.soundOnProgress = false
	end
end

function PushAwayEvent:onStart()
	PushAwayEvent.super.onStart(self)

	if (isTable(self.chaseCharacterDesc)) then
		if (self.chaseCharacterDesc.name) then
			self.chaseCharacter = level:getCharacter(self.chaseCharacterDesc.name)
		end
		if (not self.chaseCharacter) then
			self.chaseCharacter = level:createCharacter(self.chaseCharacterDesc)
		end
	elseif (isString(self.chaseCharacterDesc)) then
		self.chaseCharacter = level:getCharacter(self.chaseCharacterDesc)
	end

	assert(isTable(self.chaseCharacter), "Need chaseCharacter or chaseCharacterDesc")
	self.chaseCharacter.clickable = false

	if (self.doWalktoWaitPosition) then
		self:walkToWaitPosition()
	end
end

function PushAwayEvent:onTick(time)
	if (not self.started or self.completed) then
		return
	end

	if (self.nextSeatDuration > 0) then
		self.nextSeatDuration = self.nextSeatDuration - time
--'print("self.nextSeatDuration: " .. self.nextSeatDuration)
		if (self.nextSeatDuration <= 0) then
			if (not self:walkToRandomTable()) then
				self.nextSeatDuration = 1000
			end
		end
	end
end

function PushAwayEvent:walkToWaitPosition()
	if (self.activeTable) then
		self.activeTable.usedInGameplay = true
		self.activeTable = nil
	end
	self.chaseCharacter.clickable = false
	self.chaseCharacter:addTask({class="WalkToTask", target=self.walktoWaitPosition, playDefaultOnFinish=true})
	self.chaseCharacter:addTask({class="PositionTask", position=self.waitPosition})
	if (self.waitDirection) then
		self.chaseCharacter:addTask({class="DirectionTask", direction=self.waitDirection})	
	end
	self.chaseCharacter:addTask({class="PlayAnimationTask", animation=self.waitAnimation, cycles=-1, frameTime=self.waitFrameTime or self.chaseCharacter.defaultFrameTime})
	self.chaseCharacter:addTask({class="CallTask", 
		func=function()
			self.nextSeatDuration = math.Random(self.minFindSeatDuration, self.maxFindSeatDuration)
		end})
end

function PushAwayEvent:walkToRandomTable()
	local tables = {}
	for _,item in ipairs(self.items) do
		local tmpTable = item.obj
		if (tmpTable.usedInGameplay) then
			if (not tmpTable.isOccupied) then
				table.push_back(tables, tmpTable)
			end
		end
	end

	if (#tables > 0) then
		self.activeTable = tables[math.Random(#tables)]
		self.activeTable.usedInGameplay = false

		local chair = self.activeTable:getChair(math.Random(self.activeTable.getNumChairs)-1)
		self.chaseCharacter:addTask({class="PositionTask", position=self.walktoWaitPosition})
		self.chaseCharacter:addTask({class="WalkToTask", target=chair, playDefaultOnFinish=true})
		self.chaseCharacter:addTask({class="SitTask", chair=chair.name})
		self.chaseCharacter:addTask({class="PlayAnimationTask", animation=self.annoyAnimation, cycles=-1, frameTime=self.waitFrameTime or self.chaseCharacter.defaultFrameTime})
		self.chaseCharacter:addTask({class="CallTask", func=function()
			if (self.annoySound) then
				self.chaseCharacter:loopSample(self.annoySound)
			end
			self.chaseCharacter.clickable = true				
			self.chaseCharacter.workPos = chair:getTable().workPos
			chair:getTable().oldOnclick = chair:getTable().onClick
			chair:getTable().oldClickable = chair:getTable().clickable
			chair:getTable().clickable = true
			chair:getTable().onClick = function() 
				self.chaseCharacter:onClick()
				chair:getTable().clickable = false
			end

			print("hintManager:placeArrow({")
			hintManager:placeArrow({
				hintId="PushAwayEvent",
				parent=level.dragLayer,
				x = self.chaseCharacter.screenX,
				y = self.chaseCharacter.screenY - self.chaseCharacter.height,
			})
			switchHintArrowClick(self.chaseCharacter, "PushAwayEvent")
			
		end})

		return true
	end

	return false
end

--' ************************************ PushAwayCharacter ************************************ '--
PushAwayCharacter = deriveLuaClass("PushAwayCharacter", "DelCharacter")

function PushAwayCharacter:init(desc)
	PushAwayCharacter.superInit(self, desc)
end

function PushAwayCharacter:onClick()
	PushAwayCharacter.super.onClick(self)
	self.clickable = false
	return true
end

function PushAwayCharacter:onWorkStart(task)
	PushAwayCharacter.super.onWorkStart(self, task)
	print("PushAwayCharacter:onWorkStart(task)")
	self:stopLoopSample()
	return true
end

function PushAwayCharacter:onWorkFinish(task)
	PushAwayCharacter.super.onWorkFinish(self, task)
	print("PushAwayCharacter:onWorkFinish(task)")

	local chair = event.activeTable:getChair(math.Random(event.activeTable.getNumChairs)-1)

	--' add score
	eventBar:incValue(1)
	if (event.scoreOneDone and event.scoreOneDone > 0) then
		local pos = event.chaseCharacter.floaterPosition
		local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), event.scoreOneDone)
		level:handleScore(event.scoreOneDone, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreOneDone, 0)
		if (level.BM_eventScore) then level:BM_eventScore(event.scoreOneDone) end
	end
	
	if (eventBar.value >= eventBar.targetValue) then
		if (event.scoreAllDone and event.scoreAllDone > 0) then
			local pos = event.chaseCharacter.floaterPosition
			local text = string.format(ghf.getText("EVENT_SCORE_ALL_DONE"), event.scoreAllDone)
			level:handleScore(event.scoreAllDone, pos.x, pos.y, false, 0);
			effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreAllDone, 0)
			if (level.BM_eventScore) then level:BM_eventScore(event.scoreAllDone) end
			event.scoreAllDone = nil
		end

		event:eventWon()
	end

	chair:getTable().onClick = chair:getTable().oldOnclick
	chair:getTable().clickable = chair:getTable().oldClickable

	event:walkToWaitPosition()
end

function PushAwayCharacter.fixEventOnWorkCancel(self, task)
	if (level.event_onWorkCancel) then
		local handled = level:event_onWorkCancel(self, task)
		if (handled ~= nil) then return handled end
	end
	self:stopLoopSample()
end
