--' ************************************ Popup Event ************************************ '--
PopupEvent = deriveLuaClass("PopupEvent", "Event")
PopupObject = deriveLuaClass("PopupObject", "Object")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="PopupEvent",
	canAppearAtTheSameTime = true
}
--]]

function PopupEvent:init(desc)
	PopupEvent.superInit(self, desc)

	--' Configuration variables
	self.canAppearAtTheSameTime = self.canAppearAtTheSameTime or false

	--' Status variables
	self.itemsActive = 0

	--' Override
	self.matchingItemType = "PopupObject"
	self.initObjectsHidden = true
end

function PopupEvent:onInitialized()
	PopupEvent.super.onInitialized(self)
--'print("PopupEvent:onInitialized()")
	for _,item in ipairs(self.items) do
--'print("item name: "..item.obj.name)
		item.obj.drawOutline = false
		item.popupActive = false
	end
end

function PopupEvent:showItem(item)
	PopupEvent.super.showItem(self, item)

	--' Item config
	item.popupAnimation = item.popupAnimation or item.obj.popupAnimation or ""
	item.popupFrameTime = item.popupFrameTime or item.obj.popupFrameTime or 150
	item.popbackAnimation = item.popbackAnimation or item.obj.popbackAnimation or ""
	item.popbackFrameTime = item.popbackFrameTime or item.obj.popbackFrameTime or 150
	item.stayDuration = item.stayDuration or item.obj.stayDuration or 4000
	item.appearDurationMin = item.appearDurationMin or item.obj.appearDurationMin or 5000
	item.appearDurationMax = item.appearDurationMax or item.obj.appearDurationMax or 15000
	if (item.walkToObject == nil) then
		item.walkToObject = item.obj.walkToObject ~= false --' Default true
	end

	--' Object config
	item.obj.workAnimation = item.obj.workAnimation or "WORK"
	item.obj.workCycles = item.obj.workCycles or -1
	item.obj.workWhenFinished = item.obj.workWhenFinished or "default"
	item.obj.workFrameTime = item.obj.workFrameTime or 150
	item.obj.workDirection = item.obj.workDirection or "up"
	item.obj.workDuration = item.obj.workDuration or 0

	--' Object status
	item.obj.visible = false
	item.obj.highlightSound = "" --' Mute the highlight

	--' Functionality
	item.obj.onClick = popupEventOnClick
	item.obj.onWorkStart = popupEventOnWorkStart
	item.obj.onWorkFinish = popupEventOnWorkFinish

	--' Item ststus
	item.popupActive = true
	item.stayTimer = item.stayDuration
	item.appearTimer = math.Random(item.appearDurationMin, item.appearDurationMax)
end

function PopupEvent:onTick(time)
	if (level.inStoryMode or not self.started) then
		return
	end

	for _,item in ipairs(self.items) do
		if (not item.done) then
			if (not item.obj.visible) then
				if (self.canAppearAtTheSameTime or self.itemsActive <= 0) then
					item.appearTimer = item.appearTimer - time
					if (item.appearTimer <= 0) then
						item.popupActive = true
						item.stayTimer = item.stayDuration
						item.obj.clickable = true
						if (item.popupAnimation) then
							item.obj:addTask({class="PlayAnimationTask", animation=item.popupAnimation, cycles=1, whenFinished="stay", frameTime=item.popupFrameTime})
						end
						item.obj.visible = true
						self.itemsActive = self.itemsActive + 1

						if (not self.canAppearAtTheSameTime) then
							self:resetAppearTimers()
						end
					end
				end
			elseif (item.popupActive) then
				item.stayTimer = item.stayTimer - time
				if (item.stayTimer <= 0) then
					if (item.popbackAnimation ~= "") then
						item.obj:addTask({class="PlayAnimationTask", animation=item.popbackAnimation, cycles=1, whenFinished="stay", frameTime=item.popbackFrameTime, wait=true})
					elseif (item.popupAnimation) then
						item.obj:addTask({class="PlayAnimationTask", animation=item.popupAnimation, cycles=1, reversed=true, whenFinished="stay", frameTime=item.popupFrameTime, wait=true})
					end

					item.obj:addTask({class="CallTask", func=function()
						item.obj.visible = false
						self.itemsActive = self.itemsActive - 1
						item.appearTimer = math.Random(item.appearDurationMin, item.appearDurationMax)
						item.obj.clickable = false
					end})
					item.popupActive = false
				end
			end
		end
	end
end

function PopupEvent:resetAppearTimers()
	for _,item in ipairs(self.items) do
		item.appearTimer = math.Random(item.appearDurationMin, item.appearDurationMax)
	end
end

function PopupEvent:onItemDone(item)
	return PopupEvent.super.onItemDone(self, item)
end

function popupEventOnClick(self)
	--' First check if we should even be able to click on this object:
	if (self.visible) then
		--' Disappearing is not by default since some stuff
		--' needs to float to a destination, so fade it out yourself
		self.clickable = false

		self.item.popupActive = false
		event.itemsActive = event.itemsActive - 1

		if (self.item.walkToObject) then
			self.item.done = true
			self.class.onClick(self)
		else
			event:itemDone(self.item)
		end
	end
	return true
end

function popupEventOnWorkStart(self, task)
	--' TODO: remove? self.item.working = true

	task.duration = self.workCycles * self.workFrameTime
	if (task.duration <= 0) then
		task.duration = self.workDuration
	end

	if (task.duration > 0) then
		task.actor.direction = self.workDirection
		task.actor:playAnimation(self.workAnimation, self.workCycles, self.workFrameTime, self.workWhenFinished)
	end
end

function popupEventOnWorkFinish(self, task)
	task.actor:playDefaultAnimation()
	--' TODO: remove? self.item.working = false
	self.item.done = false
	event:itemDone(self.item)
end
