--' ************************************ Delivery Event ************************************ '--
PackageDeliveryEvent = deriveLuaClass("PackageDeliveryEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="PackageDeliveryEvent",
	deliveryItem = "foodBox",
	deliveryCharacter = {base="tashi"},
	--'deliveryCharacter = {"antonio", "tashi"},
	deliveryIcon = "icon#foodbox:room1_icons",
	--'deliveryIcon = {"icon#foodbox:room1_icons", "icon#refreshing_beverage:room1_icons"},

--'	deliverySound = "",
	numDeliveriesNeeded = 5,
	deliveryStartPosition = {x=-65, y=145},
	deliveryPosition = {x=365, y=285},
	storagePosition = {x=1080, y=720},
	--'timeUntilFirstDelivery = 5000,
	timeBetweenDeliveriesMin = 10000,
	timeBetweenDeliveriesMax = 12000,
	deliveryWaitTime = 3000,
}
]]--

function PackageDeliveryEvent:init(desc)
	PackageDeliveryEvent.superInit(self, desc)

	self.numDeliveriesLeft = self.numDeliveriesNeeded
	self.deliveryStartPosition = desc.deliveryStartPosition or {}
	self.deliveryPosition = desc.deliveryPosition or {}
	self.storagePosition = desc.storagePosition or {}
	self.deliverySound = desc.deliverySound or "SOUND_PREPARE_POPUP:sound"
	self.deliveryAppearance = desc.deliveryAppearance or "carry"
	if (desc.deliveryIcon) then
		self.deliveryIcons = self.deliveryIcons or {}
		if (isTable(desc.deliveryIcon)) then
			for _,v in ipairs(desc.deliveryIcon) do
				assert(isString(v))
				table.insert(self.deliveryIcons, v)
			end
		elseif (isString(desc.deliveryIcon)) then
			table.insert(self.deliveryIcons, desc.deliveryIcon)
		end
	end
	if (desc.deliveryEmotion) then
		self.deliveryEmotions = self.deliveryEmotions or {}
		if (isTable(desc.deliveryEmotion)) then
			for _,v in ipairs(desc.deliveryEmotion) do
				assert(isString(v))
				table.insert(self.deliveryEmotions, v)
			end
		elseif (isString(desc.deliveryEmotion)) then
			table.insert(self.deliveryEmotions, desc.deliveryEmotion)
		end
	end

	self.eventState = "waitForNextDelivery"
	self.nextDeliveryTimer = self.timeForFirstDelivery or self.timeBetweenDeliveriesMin + math.Random(self.timeBetweenDeliveriesMax - self.timeBetweenDeliveriesMin)
	self.pickupTimer = self.deliveryWaitTime
	self.numDeliveriesPickedup = 0
end

function PackageDeliveryEvent:onInitialized()
	PackageDeliveryEvent.super.onInitialized(self)
end

function PackageDeliveryEvent:onStart()
	PackageDeliveryEvent.super.onStart(self)

	self.deliveryCharacters = self.deliveryCharacters or {}
	if (isTable(self.deliveryCharacter)) then
		for _,v in ipairs(self.deliveryCharacter) do
			local char = nil
			if (isTable(v)) then
				char = level:createCharacter(v)
				char.location = self.deliveryStartPosition
			elseif (isString(v)) then
				char = level:getCharacter(v)
			end
			if (char) then
				table.insert(self.deliveryCharacters, char)
			end
		end
	end
	if (#self.deliveryCharacters == 0) then
		if (isTable(self.deliveryCharacter)) then
			self.deliveryCharacter = level:createCharacter(self.deliveryCharacter)
			self.deliveryCharacter.location = self.deliveryStartPosition
		elseif (isString(self.deliveryCharacter)) then
			self.deliveryCharacter = level:getCharacter(self.deliveryCharacter)
		end
		if (isTable(self.deliveryCharacter)) then
			table.insert(self.deliveryCharacters, self.deliveryCharacter)
		end
	end
	assert(self.deliveryCharacters, "Problem fetching delivery character(s)");

	for _,v in ipairs(self.deliveryCharacters) do
		v.onClick = self.onDeliveryCharacterClick
		v.workPosition = v.workPosition or {0,20}
		v.curWalkSpeed = v.curWalkSpeed or 400
	end

	local deliveryChar = self:getDeliveryCharacter()
	deliveryChar:addTask({class="WalkToTask", target=self.deliveryStartPosition, playDefaultOnFinish=true})
	self:carryDelivery(deliveryChar, true)
end

function PackageDeliveryEvent:onTick(time)
	if (self.allItemsDone or self.completed or not self.started) then
		return
	end

	--'if (self.numDeliveriesLeft == 0) then
	--'	return
	--'end

	if (self.eventState == "waitForNextDelivery") then
		--' Timer until next delivery
		self.nextDeliveryTimer = self.nextDeliveryTimer - time

		if (self.nextDeliveryTimer <= 0) then
			--' Start delivery routine
			self.eventState = "makingDelivery"

			local deliveryChar = self:getDeliveryCharacter()
			self:carryDelivery(deliveryChar, true)
			deliveryChar:addTask({class="WalkToTask", target=self.deliveryPosition, playDefaultOnFinish=true, directionOnFinish="down"})
			deliveryChar:addTask({class="OverheadTask", image=self.deliveryIcons[(self:getDeliveryIdx() % #self.deliveryIcons) + 1], soundId=self.deliverySound})
			deliveryChar:addTask({class="CallTask",	
				func=function()
					self.eventState = "waitingForPickup"
					deliveryChar.clickable = true
					self.pickupPosition = {self.deliveryPosition.x, self.deliveryPosition.y + 30}
					if (self.onWaitingForPickup) then
						self:onWaitingForPickup()
					end
				end})
		end
	elseif (self.eventState == "waitingForPickup") then
		self.pickupTimer = self.pickupTimer - time
		if (not self.deliveryDone and self.pickupTimer <= 0) then
			self:returnFromDelivery(self:getDeliveryCharacter(), false)
		end	
	end
end

function PackageDeliveryEvent:getNumItemsTodo()
	return self.numDeliveriesNeeded
end

function PackageDeliveryEvent:getDeliveryIdx()
	return (self.numDeliveriesNeeded - self.numDeliveriesLeft)
end

function PackageDeliveryEvent:getDeliveryCharacter()
	local charId = (self:getDeliveryIdx() % #self.deliveryCharacters) + 1
	return self.deliveryCharacters[charId]
end

function PackageDeliveryEvent:itemDone()
	self.itemsDone = self.itemsDone + 1

	if (self.scoreOneDone and self.scoreOneDone > 0) then
		local pos = self:getFloaterPosition()
		local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), self.scoreOneDone)
		level:handleScore(self.scoreOneDone, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, self.scoreOneDone, 0)
		if (level.BM_eventScore) then level:BM_eventScore(self.scoreOneDone) end
	end

	if (level.onEventStepDone) then
		level:onEventStepDone(self, item)
	end

	self:onProgress()

	SpriteExt.playSample(nil, "SOUND_EVENT_ITEM_COMPLETE")
end

function PackageDeliveryEvent:getFloaterPosition()
	if (isTable(self.floaterPos)) then
		return self.floaterPos
	elseif (self.deliveryCharacters) then
		return self:getDeliveryCharacter().floaterPosition
	end
end

function PackageDeliveryEvent.onDeliveryCharacterClick(obj)
	if (table.contains(event.deliveryCharacters, obj)) then
		obj.clickable = false
		event.deliveryDone = true
		event:hideHintArrow()

		--'hero:addTask({class="WorkTask", target=event.pickupPosition, directionOnFinish="up"})
		hero:addTask({class="WalkToTask", target=event.pickupPosition, playDefaultOnFinish=true, directionOnFinish="up"})
		hero:addTask({class="PlayAnimationTask", animation="work", cycles=1, duration=500})
		hero:addTask({class="CallTask", func=function() --' Move to onWorkFinish
			event:itemDone({})
			event:returnFromDelivery(obj, true) --' Let delivery char walk away
		end})
		event:carryDelivery(hero, true)

		hero:addTask({class="WalkToTask", target=event.storagePosition, playDefaultOnFinish=true})
		event:carryDelivery(hero, false)
		hero:addTask({class="Task", duration=500})
		hero:addTask({class="WalkToTask", target=level.heroPosition, playDefaultOnFinish=true, directionOnFinish="down"})
		hero:addTask({class="CallTask", func=function() event:onItemDone() end})
	end
end

function PackageDeliveryEvent:returnFromDelivery(deliveryCharacter, withoutObject)
	deliveryCharacter.clickable = false
	self.eventState = "waitForNextDelivery"
	local dif = self.timeBetweenDeliveriesMax - self.timeBetweenDeliveriesMin
	self.nextDeliveryTimer = self.timeBetweenDeliveriesMin + math.Random(dif)
	self.pickupTimer = self.deliveryWaitTime
	self.deliveryDone = false

	deliveryCharacter:addTask({class="RemoveOverheadTask"})

--'	if (self.numDeliveriesLeft == 0 and self.itemsDone < self:getNumItemsTodo()) then --' Event done, and lost
--'		hero:addTask({class="CallTask", func=function() self:eventLost() end})
--'	end

	self.numDeliveriesLeft = self.numDeliveriesLeft - 1

	if (withoutObject) then self:carryDelivery(deliveryCharacter, false) end
	deliveryCharacter:addTask({class="Task", duration=200})
	deliveryCharacter:addTask({class="WalkToTask", target=self.deliveryStartPosition, playDefaultOnFinish=true, directionOnFinish="down"})

	if (level.hasShiftFinished and self.itemsDone < self:getNumItemsTodo()) then
		deliveryCharacter:addTask({class="CallTask", func=function() self:eventLost() end})
	end
end

function PackageDeliveryEvent:carryDelivery(character, carry)
	if (carry) then
		if (self.deliveryAppearance)  then
			character:addTask({class="SetAppearanceTask", appearanceId=self.deliveryAppearance, appearanceValue=self.deliveryAppearance})
		end
		if (self.deliveryEmotions) then
			local emote = self.deliveryEmotions[(self:getDeliveryIdx() % #self.deliveryEmotions) + 1]
			character:addTask({class="EmotionTask", emotion=emote})
		end
	else
		if (self.deliveryAppearance)  then
			character:addTask({class="RemoveAppearanceTask", appearanceId=self.deliveryAppearance})
		end
		if (self.deliveryEmotions) then
			character:addTask({class="EmotionTask"})
		end
	end
end