--' ************************************ Fix Event ************************************ '--
FixEvent = deriveLuaClass("FixEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="FixEvent",
	items = {
		{ obj="scarecrow_upgrade_defect", swapObj="scarecrow_upgrade" },
	},
}
--]]

function FixEvent:init(desc)
	FixEvent.superInit(self, desc)

	--' Configuration variables
	self.fixAmount = self.fixAmount or 10000
	self.fixEventOnWorkStart = desc.fixEventOnWorkStart-- or FixEvent.initfixEventOnWorkStart
	self.addExclamationMarks = desc.addExclamationMarks == true
	self.allowCancel = desc.allowCancel ~= false

	--' Status variables
	self.hintFired = false

	--' Override
	self.matchingItemType = "FixableObject"
end

function FixEvent:onInitialized()
	FixEvent.super.onInitialized(self)

	if (self.trackProgress) then
		eventBar.soundOnProgress = false
	end
	eventBar.showValue = not self.trackProgress
	eventBar.showTarget = not self.trackProgress
end

function FixEvent:showItem(item)
	FixEvent.super.showItem(self, item)

	if (item.base_animations and #item.base_animations > 0) then
		item.obj.baseAnimation = item.base_animations[1]
	end

	item.workAnimation = item.obj.workAnimation or self.workAnimation or "work"
	item.workAnimationFrameTime = item.obj.workFrameTime or self.workAnimationFrameTime or 200
	item.workDirection = item.obj.workDirection or self.workDirection or "up"

	item.fixAmount = item.fixAmount or item.obj.fixAmount or self.fixAmount
	if (item.obj and item.obj.numSteps) then
		item.totalFixAmount = item.fixAmount * item.obj.numSteps
	end
	
	item.bar = item.bar or self.bar or "progressbar_big"

	item.progress = 0
	item.totalProgress = 0
	--' TODO: remove? item.working = false

	item.obj.onClick = self.fixEventOnClick
	item.obj.onWorkStart = self.fixEventOnWorkStart
	item.obj.onWorkUpdate = self.fixEventOnWorkUpdate
	item.obj.onWorkFinish = self.fixEventOnWorkFinish
	item.obj.onWorkCancel = self.fixEventOnWorkCancel
	
	if (self.addExclamationMarks) then
		item.obj:createOverhead("ICON_EXCLAMATION")
	end
	
	item.onRestoreItem = item.onRestoreItem or function()
		if (item.obj:getWorkCharacter()) then
			item.obj:getWorkCharacter().direction = "down"
			item.obj:getWorkCharacter():playDefaultAnimation()
			taskSystem:cancelAllTasksFor(item.obj:getWorkCharacter())
		end
	end

	self:createProgressBar(item)
	if (item.bar) then
		item.bar.visible = self.allowCancel
	end
	if (not self.showHintOnStart) then
		self:showHint(item.obj)
	end
	
end

function FixEvent:onItemDone(item)
	FixEvent.super.onItemDone(self, item)

	if (item.base_animations and #item.base_animations > 0) then
		item.obj.baseAnimation = item.base_animations[#item.base_animations]
	else
		item.obj:showLastFrame()
	end
	item.obj:removeOverhead(true)
end

function FixEvent:getProgress()
	if (self.trackProgress) then
		local fixAmount, progress = 0, 0
		
		for i=1, self:getNumStepsTodo() do
			fixAmount = fixAmount + self.items[i].fixAmount
			progress = progress + self.items[i].progress
		end

		return (fixAmount > 0 and progress / fixAmount) or 1
	end
	return FixEvent.super.getProgress(self)
end

--' ************************************ Fix Object ************************************ '--
FixableObject = deriveLuaClass("FixableObject", "EventObject")

function FixEvent.fixEventOnClick(self)
	if (level.event_onClick) then
		local handled = level:event_onClick(self)
		if (handled ~= nil) then return handled end
	end
--'print("FixEvent.fixEventOnClick(self)")
	if (not self.item.done) then
		self.clickable = false
		hero:addTask({class="SetTask", positionLock=self})
		hero:addTask({class="WorkTask", target=self})
		self.item.obj:removeOverhead(true)		
	end
	return true
end

function FixEvent.fixEventOnWorkStart(self, task)
	if (level.event_onWorkStart) then
		local handled = level:event_onWorkStart(self, task)
		if (handled ~= nil) then return handled end
	end
	event:hideHintArrow()
	if (not self.item.done) then
		task.actor.direction = self.item.workDirection
		task.actor:playAnimation(self.item.workAnimation, -1, self.item.workAnimationFrameTime)
		task.duration = self.item.fixAmount
		if (event.allowCancel) then
			event:showProgressBar(self.item)
		end
	end

	if (self.workSoundId and self.workSoundId ~= "") then
		self:loopSample(self.workSoundId)
	end

	return true
end

function FixEvent.fixEventOnWorkUpdate(self, task, time)
	if (level.event_onWorkUpdate) then
		local handled = level:event_onWorkUpdate(self, task, time)
		if (handled ~= nil) then return handled end
	end
	self.item.progress = self.item.progress + time --' We cannot use the task's timer!
	self.item.totalProgress = self.item.totalProgress + time
	
	local isDone = (self.item.progress >= self.item.fixAmount)
	local progress = math.min(1, self.item.progress / self.item.fixAmount)

	if (self.item.matchEventAppearance ~= false and event.matchItemAppearance ~= false) then
		if (self.item.base_animations and #self.item.base_animations > 0) then
			local c = #self.item.base_animations-1
			local idx = math.floor(math.min(progress * c, c)+1)
			if (self.baseAnimation ~= self.item.base_animations[idx]) then
				self.baseAnimation = self.item.base_animations[idx]
				if (self.stateChange) then
					self:stateChange()
				end
			end
		else
			local c = self.frameCount-1
			local frame = math.min(progress * c, c)
			if (self.frame ~= frame) then
				self.frame = frame
				if (self.stateChange) then
					self:stateChange()
				end
			end
		end
	end

	if (self.item.bar) then
		self.item.bar:setProgress(progress)
	end
	event:onProgress()
	return isDone or (taskSystem:hasTaskFor(task.actor) and event.allowCancel)  or event.completed or level.hasShiftFinished
end

function FixEvent.fixEventOnWorkFinish(self, task)
	if (level.event_onWorkFinish) then
		local handled = level:event_onWorkFinish(self, task)
		if (handled ~= nil) then return handled end
	end

	self:stopLoopSample()
	task.actor.direction = "down"
	task.actor:playDefaultAnimation()
	
	local isDone = (self.item.totalProgress >= self.item.totalFixAmount)
	if (isDone) then
		print("if (isDone) then")
		event:stepDone(self.item)
		if (self.stateChange) then
			self:stateChange()
		end
	else
		if (not event.allowCancel) then
			event:stepDone(self.item)
		end
		self.clickable = true
	end
	
	if (not event.allowCancel) then
		self.item.progress = 0
	end
end

function FixEvent.fixEventOnWorkCancel(self, task)
	if (level.event_onWorkCancel) then
		local handled = level:event_onWorkCancel(self, task)
		if (handled ~= nil) then return handled end
	end
	self:stopLoopSample()
	self.clickable = not self.item.done
end
