--' ************************************ Deliver Event ************************************ '--
DeliverEvent = deriveLuaClass("DeliverEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="DeliverEvent",
	bar = "progressbar_big",
	productPool = {"#sock", "#bread#cheese"},
	numProductsPerOrder = 2,
	numOrders = 2,
	timeUntilNextOrder = 0,
	timeUntilFirstOrder = 0,
	maxGroupCount = 2,
	limitToMaxGroupCount = true,
	items = {
		{ obj="cleaner", barPos = {x=10, y=-100}, workPos={x=300, y=683}, workAnimation="clean", workDuration=2000, timeUntilEnd=120000, },
	},
}

function levelDesc:onDeliverProducts(item)
	...
end
--]]

--' ************************************ Deliver Button (object to grab an bring to bin) ************************************ '--
DeliverObject = deriveLuaClass("DeliverButton", "IngredientButton")
--' ************************************ Deliver Object (bin) ************************************ '--
DeliverObject = deriveLuaClass("DeliverObject", "ObjectWithOrder")

function DeliverEvent:init(desc)
	DeliverEvent.superInit(self, desc)

	--' Configuration variables

	--' Status variables

	--' Override
	self.matchingItemType = "DeliverObject"
end

function deliverWorkStart(self, task)
	event:hideHintArrow()
	if (not event.started or event.completed or self.order.currentStep.numProducts <= 0 or not self.order.hasDeliverableProducts) then
		return false
	end

	local duration = self:getWorkDurationWith(task.actor)
	task.duration = duration
	if (duration > 0) then
		task.actor.direction = self.workDirection or "up"
		task.actor:playAnimation(self.workAnimation or "work", self.workCycles or -1, self.workFrameTime or 150, self.workWhenFinished or "default")
	end
end

function deliverWorkFinish(obj, task)
	--' Check the products
	assert(obj.order)
	--' Check if all the products for this object are delivered
	local numProductsAtStart = obj.order.currentStep.numProducts
	obj.order:deliver(obj.item)
	obj:updateOrderOverhead()
	local numProductsLeft = obj.order.currentStep.numProducts

	if (numProductsLeft < numProductsAtStart) then
		local pos = obj.floaterPosition
		local numDelivered = numProductsAtStart - numProductsLeft
		local score = event.scoreOneDone * numDelivered
		local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), score)
		level:handleScore(score, pos.x, pos.y, false, 0)
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, score, 0)
		if (level.BM_eventScore) then level:BM_eventScore(score) end

		if (level.onDeliverProducts) then
			level:onDeliverProducts(obj.item)
		end

		--' Inc items done
		event.itemsDone = event.itemsDone + numDelivered
		event:onProgress()

	elseif (obj.showMessageTimer <= 0) then
		obj.showMessageTimer = 5000
		if (#(tray.products) > 0) then
			local msgId = event:getMessageId(obj)
			if (msgId) then
				task.actor:addTaskAtFront({class="MessageTask", textId=msgId, duration=2000, needsClick=false,})
			end
		end
	end

	if (numProductsLeft == 0) then
		obj.clickable = false
		event:itemDone(obj.item)
	end

	task.actor:playDefaultAnimation()
end

function DeliverEvent:onInitialized()
	DeliverEvent.super.onInitialized(self)

	for _,item in ipairs(self.items) do
		if (item.obj) then
			item.obj.onWorkStart = deliverWorkStart
			item.obj.onWorkUpdate = nil
			item.obj.onWorkFinish = deliverWorkFinish

			item.obj.showMessageTimer = 0

			for _,v in ipairs(item.obj.acceptedProducts) do
				if (v and v ~= "") then
					local ingButton = level:getObject(v)
					assert(ingButton, "Object '"..v.."' not found in the level!")
					if (ingButton) then
						local ingName = ingButton:getIngredientNameForLogic()
						item.obj.order.currentStep:addProduct(ingName)
					end
				end
			end
		end
	end

	self.numItemsTotal = 0
	for _,item in ipairs(self.items) do
		if (item.obj and item.obj.order) then
			self.numItemsTotal = self.numItemsTotal + item.obj.order.currentStep.numProducts
		end
	end

	eventBar.targetValue = self.numItemsTotal
	eventBar.showValue = true
	eventBar.showTarget = true
	self:onProgress()
end

function DeliverEvent:incItemDone(item)
end

function DeliverEvent:scoreItemDone(item)
end

function DeliverEvent:onStart()
	DeliverEvent.super.onStart(self)
	for _,item in ipairs(self.items) do
		if (item.obj) then
			item.obj:updateOrderOverhead()
		end
	end
end

function DeliverEvent:getNumItemsTodo()
	return self.numItemsTotal or 1
end

function DeliverEvent:getMessageId(obj)
	return obj.badDeliveryMsgId
end

function DeliverEvent:onTick(time)
	for _,item in ipairs(self.items) do
		if (item.obj) then
			if (item.obj.showMessageTimer > 0) then
				item.obj.showMessageTimer = item.obj.showMessageTimer - time
			end
		end
	end
end

function DeliverEvent:getProductNames()
	local products = {}
	
	for _,item in ipairs(self.items) do
		if (item.obj and item.obj.order) then
			for _p,prodName in ipairs(item.obj.order.currentStep.productNames) do
				table.push_back(products, prodName)
--'				print("prodName: "..prodName)
			end
		end
	end

	return products
end