--' ************************************ Fix Event ************************************ '--
CuriousCustomerEvent = deriveLuaClass("CuriousCustomerEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="CuriousCustomerEvent",
	scoreAllDone = 100,
	scoreOneDone = 25,
	numCustomers = 6,
	numCustomersNeeded = 4,
	minNextCustomerDuration = 20000,
	maxNextCustomerDuration = 25000,
	talkDuration = 800,
	talkDirection = "down",
	talkAnimation = "talk",
	talkPos={x=230, y=530},
	waitPos={x=230, y=500},
	customers={"big", "child", "elderly", "locals", "normal", "yup"},
	talkToIcon = "talk_to_emily_icon:room5_icons",
}
]]--

function CuriousCustomerEvent:init(desc)
	CuriousCustomerEvent.superInit(self, desc)

	--' Configuration variables
	self.scoreOneDone = self.scoreOneDone or 100
	self.scoreAllDone = self.scoreAllDone or 300
	self.numCustomers = self.numCustomers or 6
	self.numCustomersNeeded = self.numCustomersNeeded or 4
	self.talkDuration = self.talkDuration or 500
	self.talkDirection = self.talkDirection or "up"
	self.talkAnimation = self.talkAnimation or ""
	self.waitPos = self.waitPos or assert(self.waitPos, "need waitPos")
	self.talkPos = self.talkPos or assert(self.talkPos, "need talkPos")
	self.customers = self.customers or assert(self.customers, "need customers")
	self.talkToIcon = self.talkToIcon or assert(self.talkToIcon, "need talkToIcon")
	
	--' Status variables
	self.nextCustomerDuration = 0
	self.numCustomersDone = 0
	self.customerActive = false

	--' Override
end

function CuriousCustomerEvent:getNumItemsTodo()
	return self.numCustomersNeeded
end

function CuriousCustomerEvent:onInitialized()
	CuriousCustomerEvent.super.onInitialized(self)

	if (self.trackProgress) then
		eventBar.soundOnProgress = false
	end
end

function CuriousCustomerEvent:onStart()
	CuriousCustomerEvent.super.onStart(self)
	
	self.nextCustomerDuration = self.minNextCustomerDuration
end

function CuriousCustomerEvent:onTick(time)
	if (not self.started or self.completed or self.numCustomersDone >= self.numCustomers) then
		return
	end

	self.nextCustomerDuration = self.nextCustomerDuration - time
	if (not self.customerActive and self.nextCustomerDuration <= 0) then
		self:createCuriousCustomer()
		self.nextCustomerDuration = math.Random(self.minNextCustomerDuration, self.maxNextCustomerDuration)
	end
end

function CuriousCustomerEvent:createCuriousCustomer()
	--' TODO: Get a random customer
	local entrance = level.currentRoom.entrances[math.Random(#level.currentRoom.entrances)]
	local curiousCustomerDesc = levelDesc.characterTypes[self.customers[math.Random(#self.customers)]]
	curiousCustomerDesc.class = "CuriousCharacter"
	curiousCustomerDesc.name = "curiousCharacter"
	curiousCustomerDesc.x = entrance.x
	curiousCustomerDesc.y = entrance.y
	curiousCustomerDesc.workPos = self.talkPos
	curiousCustomerDesc.gender = (math.Random(1000) < 500 and "male") or "female"
	
	assert(curiousCustomerDesc)
	self.curiousCustomer = level:createCharacter(curiousCustomerDesc)

	assert(isTable(self.curiousCustomer), "Need chaseCharacter or chaseCharacterDesc")
	self.curiousCustomer.clickable = false

	self:walkToWaitPosition()
	
	self.numCustomersDone = self.numCustomersDone + 1
	self.customerActive = true
end

function CuriousCustomerEvent:walkToWaitPosition()
	self.curiousCustomer.clickable = false
	self.curiousCustomer:addTask({class="WalkToTask", target=self.waitPos, playDefaultOnFinish=true})
	self.curiousCustomer:addTask({class="Task", duration=300})
	self.curiousCustomer:addTask({class="OverheadTask", image=self.talkToIcon, sound=self.talkToSound})
	self.curiousCustomer:addTask({class="CallTask",
		func=function()
			self.curiousCustomer.clickable = true
			if (self.onCustomerArrived) then
				self:onCustomerArrived(self.curiousCustomer)
			end
			--' show text
		end})
end

function CuriousCustomerEvent:curiousCustomerLeave()
	self.curiousCustomer.clickable = false
	if (self.onCustomerLeave) then
		self:onCustomerLeave(self.curiousCustomer)
	end
	self.curiousCustomer:addTask({class="RemoveOverheadTask"})
	self.curiousCustomer:addTask({class="ExitTask", math.Random(#level.currentRoom.exits)-1})
	self.customerActive = false

	if (eventBar.value >= eventBar.targetValue) then
		self:eventWon()
	elseif (self.numCustomersDone >= self.numCustomers) then
		self:eventLost()
	end	
end

--' ************************************ PushAwayCharacter ************************************ '--
CuriousCharacter = deriveLuaClass("CuriousCharacter", "DelCharacter")

function CuriousCharacter:init(desc)
	CuriousCharacter.superInit(self, desc)
	self.moodDropDuration = 0
	self.mood = 3
end

function CuriousCharacter:onCreated()
	self.moodDropDuration  = self:calcValue("moodDropTime")
end

function CuriousCharacter:onClick()
	CuriousCharacter.super.onClick(self)
	self.clickable = false
	return true
end

function CuriousCharacter:onWorkStart(task)
	CuriousCharacter.super.onWorkStart(self, task)
	print("CuriousCharacter:onWorkStart(task)")
	task.duration = event.talkDuration
	task.actor.direction = event.talkDirection
	task.actor:playAnimation(event.talkAnimation, -1)
	return true
end

function CuriousCharacter:onWorkFinish(task)
	CuriousCharacter.super.onWorkFinish(self, task)
	print("CuriousCharacter:onWorkFinish(task)")

	--' add score
	eventBar:incValue(1)
	if (event.scoreOneDone and event.scoreOneDone > 0) then
		local pos = event.curiousCustomer.floaterPosition
		local text = string.format(ghf.getText("EVENT_SCORE_ONE_DONE"), event.scoreOneDone)
		level:handleScore(event.scoreOneDone, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreOneDone, 0)
		if (level.BM_eventScore) then level:BM_eventScore(event.scoreOneDone) end
	end
	
	if (eventBar.value >= eventBar.targetValue) then
		if (event.scoreAllDone and event.scoreAllDone > 0) then
			local pos = event.curiousCustomer.floaterPosition
			local text = string.format(ghf.getText("EVENT_SCORE_ALL_DONE"), event.scoreAllDone)
			level:handleScore(event.scoreAllDone, pos.x, pos.y, false, 0);
			effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreAllDone, 0)
			if (level.BM_eventScore) then level:BM_eventScore(event.scoreAllDone) end
			event.scoreAllDone = nil
		end
	end

	task.actor:playAnimation("")

	event:curiousCustomerLeave()
end

function CuriousCharacter:onTick(time)
	self:tick(time)

	if (not level or not event or not event.started or event.completed) then
		return
	end

	if (self.moodDropDuration > 0) then
		self.moodDropDuration = self.moodDropDuration - time
		if (self.moodDropDuration <= 0) then
			self.moodDropDuration = self:calcValue("moodDropTime")
			if (self.mood == 3) then
				self:setAppearance("mood", "happy", "high")
			elseif (self.mood == 2) then
				self:removeAppearance("mood")			
			elseif (self.mood == 1) then
				self:setAppearance("mood", "angry", "high")		
			elseif (self.mood == 0) then
				event:curiousCustomerLeave()
			end
			
			self.mood = self.mood - 1
		end
	end
end
