--' ************************************ Chase Event ************************************ '--
ChaseEvent = deriveLuaClass("ChaseEvent", "Event")
PathingObject = deriveLuaClass("PathingObject", "Object")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="ChaseEvent",
	chaseCharacter = "antonio",
	alternatePathDirection = false,

	nrTimesToCatch = 4,
	maxNrTries = 5,
	timeBetweenRunsMin = 3500,
	timeBetweenRunsMax = 4500,
	tauntTimes = {1500, 2000,},
	tauntTextIds =
	{
		"TEXT_Taunt1",
		"TEXT_Taunt2",
	},
}
--]]


function ChaseEvent:init(desc)
	ChaseEvent.superInit(self, desc)
	self.chaseCharacter = desc.chaseCharacter or ""
	self.timeBetweeRuns = desc.timeBetweeRuns or 0
	self.waitTime = desc.timeUntilFirstRun or 0
	self.nrTimesToCatch = desc.nrTimesToCatch or 1
	self.nrTries = desc.maxNrTries or self.nrTimesToCatch
	self.timeBetweenRunsMin = desc.timeBetweenRunsMin or desc.timeBetweenRunsMax or 0
	self.timeBetweenRunsMax = desc.timeBetweenRunsMax or desc.timeBetweenRunsMin or 0
	self.tauntTimes = desc.tauntTimes or {1000}

	self.matchingItemType = "PathingObject"
end

function ChaseEvent:onInitialized()
	ChaseEvent.super.onInitialized(self)

	assert(self.chaseCharacter, "No chase character specified!")

	assert(self.items and #self.items > 0, "No 'PathingObject' found, did you forget to add it to the level?")
	if (self.items) then
		local item = nil
		if (isString(self.pathingObject)) then
			local desiredItem = level:getObject(self.pathingObject)
			if (table.contains(self.items, item)) then
				item = desiredItem
			end
		elseif (#self.items >= 1) then
			assert(#self.items == 1, "Multiple PathingObjects found; specify a specific object by setting 'pathingObject = \"<name of object>\"'")
			item = self.items[1]
		end
		if (item) then
			self.paths = item.obj.queueDesc.positions
			assert(#self.paths, "No paths found!")
			assert(#self.paths[1] >= 2, "Not enough path positions!")
		end
	end
end

function ChaseEvent:onStart(anim)
	ChaseEvent.super.onStart(self)

	self.chaseCharacter = level:getChild(self.chaseCharacter)
	assert(isTable(self.chaseCharacter))
	self.chaseCharacter.old_walkSpeed = self.chaseCharacter.curWalkSpeed

	self.state = "waiting"
	self.nrRunsDone = 0
end

function ChaseEvent:onTick(time)
	if (level.inStoryMode or not self.started or self.completed) then
		return
	end

	if (self.nrTries > 0 and self.state == "waiting") then
		self.waitTime = self.waitTime - time
		if (self.waitTime <= 0) then
			self.state = "running"
			self:startNewRun()
		end
	elseif (self.state == "running") then
		if (not taskSystem:hasTaskFor(self.chaseCharacter)) then
			self.nrRunsDone = self.nrRunsDone + 1
			self.waitTime = math.Random(self.timeBetweenRunsMin, self.timeBetweenRunsMax)
			self.state = "waiting"
			return
		end
	end
end

function ChaseEvent:startNewRun(path)
	if (not self.chaseCharacter.old_onClick) then
		self.chaseCharacter.old_onClick = self.chaseCharacter.onClick
		self.chaseCharacter.onClick = function()
			self.chaseCharacter.clickable = false
			taskSystem:cancelAllTasksFor(self.chaseCharacter)

			self:incItemDone()
			self:scoreItemDone()
			self:onProgress()
			local itemDoneAnim = self:onItemDone()
			if (not itemDoneAnim and self.itemsDone < self:getNumItemsTodo()) then
				local tauntMsgId = self.tauntTextIds[math.Random(#self.tauntTextIds)]
				local tauntMsgTime = 1000
				self.chaseCharacter:addTask({class="MessageTask", textId=tauntMsgId, duration=tauntMsgTime, needsClick=false, })
				self.chaseCharacter:addTask({class="RemoveMessageTask", textId=tauntMsgId})
				self:walkAway(self.chaseCharacter)
			end
		end
	end

	local tauntTime = self.getClampedTableValue(self.tauntTimes, self.nrRunsDone+1)

	if (not path) then
		if (self.setPath and self.getClampedTableValue(self.setPath, self.itemsDone +1, #self.setPath) > 0) then
			path = self.paths[self.getClampedTableValue(self.setPath, self.itemsDone +1, #self.setPath)]
		else
			path = self.paths[math.Random(#self.paths)]
		end
	end
	self.chaseCharacter.path = path
	self.chaseCharacter.curWalkSpeed = self.chaseCharacterWalkSpeed or self.chaseCharacter.animationWalkSpeed
	self.nrTries = self.nrTries - 1

	self.chaseCharacter:addTask({class="PositionTask", position=path[1]})
	for i = 2, #path do
		self.chaseCharacter:addTask({class="WalkToTask", target=path[i], playDefaultOnFinish = true, directionOnFinish="down"})
	end
	self.chaseCharacter:addTask({class="CallTask",
		func=function()
			self.chaseCharacter.clickable = true
			if (event.onArriveAtTauntLocation) then
				event.onArriveAtTauntLocation(self, self.chaseCharacter)
			end
		end})
	self.chaseCharacter:addTask({class="Task", duration=tauntTime})
	self.chaseCharacter:addTask({class="CallTask",
		func=function()
			self.chaseCharacter.clickable = false
			self:walkAway(self.chaseCharacter)
			if (event.onLeaveTauntLocation) then
				event.onLeaveTauntLocation(self, self.chaseCharacter)
			end
		end})
end

function ChaseEvent:eventEnd()
	ChaseEvent.super.eventEnd()

	self.chaseCharacter.curWalkSpeed = self.chaseCharacter.old_walkSpeed
	self.chaseCharacter.onClick = self.chaseCharacter.old_onClick
	self.chaseCharacter.clickable = false
	taskSystem:cancelAllTasksFor(self.chaseCharacter)
	self.chaseCharacter:playDefaultAnimation()
--'	self.chaseCharacter.emotion = ((math.Random(1, 2) < 2) and "shrug" or "confused")
end

function ChaseEvent:walkAway(character)
	assert(character.path)
	for i = 1, #character.path -1 do
		character:addTask({class="WalkToTask", target=character.path[#character.path-i], playDefaultOnFinish = true})
	end
	if (self.nrTries <= 0) then
		character:addTask({class="CallTask", func=function() self:eventLost() end })
	end
end

function ChaseEvent.getClampedTableValue(t, id, max, min)
	if (isTable(t)) then
		min = min or 1
		max = max or #t
		if (id < min) then
			id = min
		elseif (id > max) then
			id = max
		end
		return t[id]
	end
	return 0
end

function ChaseEvent:getFloaterPosition()
	return self.chaseCharacter.floaterPosition
end

function ChaseEvent:getNumItemsTodo()
	return self.nrTimesToCatch
end
