--' ************************************ Ingredient Button ************************************ '--
IngredientButton = getClass("IngredientButton")

IngredientButton.__getters = IngredientButton.__getters or {}

function IngredientButton:init(desc)
	IngredientButton.superInit(self, desc)

	self.statusObjects = self.statusObjects or {}

	self.workAnimation = desc.workAnimation or "WORK"
	self.workDirection = desc.workDirection or "up"
	self.workCycles = desc.workCycles or -1
	self.workWhenFinished = desc.workWhenFinished or "default"
	self.workFrameTime = desc.workFrameTime or 100
	self.prepareFrameTime = self.prepareFrameTime or -1
	self.prepareDuration = self.prepareDuration or 0
	self.decayDuration = self.decayDuration or 1000
	self.decayFrameTime = self.decayFrameTime or -1
	self.inactiveFrameTime = self.inactiveFrameTime or 100
	self.worth = desc.worth or 1
	self.compoundPosition = self.compoundPosition or {x=self.x, y=self.y}
	self.availableBlink = self.availableBlink or {startAlpha=1,endAlpha=0,fadeIn=75,fadeOut=75,loops=4}
	self.blinkWhenAvailable = self.blinkWhenAvailable ~= false -- Default true
	self.allowRestockOverload = self.allowRestockOverload ~= false -- Default true

	if (not self.ingredientName) then
		self.ingredientName = self:getIngredientNameFromResource()
	end

	self:onDisplayStockChange()
end

function IngredientButton:onLevelInitialized()
	IngredientButton.super.onLevelInitialized(self)

	if (self.notifyObjectId) then
		self.notifyObject = level:getSpriteExt(self.notifyObjectId)
	end
	for i=1,2 do
		local so = self["statusObject"..i]
		if (isString(so)) then
			so = level:getSpriteExt(so)
			if (so) then
				table.push_back(self.statusObjects, so)
			end
		end
	end
end

function IngredientButton:getProductCenter()
	return self.screenCenter
end

function IngredientButton:setIngredientAndResource(ingredientName)
	--''pb("["..self.name.."] Set ingredient name: "..ingredientName)
	self.ingredientName = ingredientName
	self.baseAnimation = "BUTTON"..ingredientName
end

function IngredientButton:getIngredientNameFromResource(res)
	res = res or self.baseAnimation
	--''pi("["..self.name.."] Get ingredient name from: "..res)
	for _,v in ipairs({"BUTTON#", "PRODUCT#", "ICON#"}) do
		if (string.starts_with(res, v)) then
			res = string.sub(res, #v)
		end
	end
	for _,v in ipairs({"_PREPARE", "_DECAY"}) do
		if (string.ends_with(res, v)) then
			res = string.sub(res, 1, #res - #v)
		end
	end
	--''pd("-> "..res)
	return res
end

function IngredientButton:listPossibleIngredients()
	return {self.ingredientName or ""}
end

function IngredientButton:getIngredientNameForLogic()
	local r
	if (self.notifyObject) then
		if (self.notifyObject.notify_getIngredientNameForLogic) then
			r = self.notifyObject:notify_getIngredientNameForLogic(self)
		end
	end
	return r or self.ingredientName or ""
end

function IngredientButton:notify_getIngredientNameForLogic(ing)
	--''pb("["..self.name.."] IngredientButton Notify get ingredient name for logic (from '"..ing.name.."')")
	return ing.ingredientName
end

function IngredientButton:getIngredientNameForTrayIcon()
	local r
	if (self.notifyObject) then
		if (self.notifyObject.notify_getIngredientNameForTrayIcon) then
			r = self.notifyObject:notify_getIngredientNameForTrayIcon(self)
		end
	end
	return r or self.ingredientName or ""
end

function IngredientButton:notify_getIngredientNameForTrayIcon(ing)
	--''pb("["..self.name.."] IngredientButton Notify get ingredient name for tray icon (from '"..ing.name.."')")
	return ""
end

function IngredientButton:onStartPreparing()
	--''pi("["..self.name.."] Start prepare: "..ts(self.ingredientName))
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onStartPreparing) then
			--''pr("("..v.name..")")
			local handled = v:status_onStartPreparing(self)
			if (handled ~= nil) then
				--''pd("Handled")
				return
			end
		end
	end

	self:playAnimation("PREPARE", -1, self.prepareFrameTime)
	if (not self.prepareFrameTime or self.prepareFrameTime < 0) then
		self:matchFrameTimeTo("prepareDuration")
	end
	pd()
end

function IngredientButton:onBecomeAvailable()
	pi()
	if (self.name ~= "") then
		pr("["..self.name.."] Become available: "..ts(self.ingredientName))
	end
	if (self.statusObjects) then
		for _,v in ipairs(self.statusObjects) do
			if (v.status_onBecomeAvailable) then
				pr("("..v.name..")")
				local handled = v:status_onBecomeAvailable(self)
				if (handled ~= nil) then pd("Handled") return end
			end
		end
	end

	self:playAnimation(self.defaultAnimation, 0, self.productFrameTime)
	if (self.blinkWhenAvailable and self.availableBlink) then
		if (player:mayShowHint("EggReady") and self.ingredientName == "#egg.ready") then
			hintManager:openHint({hintId="EggReady", target=self, subject=self, showButton=false})
		end
		if (player:mayShowHint("PieReady") and string.find(self.ingredientName, "#pie")) then
			hintManager:openHint({hintId="PieReady", target=self, subject=self})
		end
		self:animate({blink=self.availableBlink})
	end
	pd()
end

function IngredientButton:onDecay()
	pi("["..self.name.."] Decays: "..ts(self.ingredientName))
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onDecay) then
			pr("("..v.name..")")
			local handled = v:status_onDecay(self)
			if (handled ~= nil) then pd("Handled") return end
		end
	end

	self:playAnimation("DECAY", 1, self.decayFrameTime, "stay")
	if (not self.decayFrameTime or self.decayFrameTime < 0) then
		self:matchFrameTimeTo("decayDuration")
	end
	--''pd()
end

function IngredientButton:onDeactivate()
	--''pi("["..self.name.."] Becomes deactivated: "..ts(self.ingredientName))
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onDeactivate) then
			--''pr("("..v.name..")")
			local handled = v:status_onDeactivate(self)
			if (handled ~= nil) then
				--''pd("Handled")
				return
			end
		end
	end

	if (self.doInactiveAnimation) then
		self:playAnimation("INACTIVE", 0, self.inactiveFrameTime)
		self.visible = true
	end
	--''pd()
end

function IngredientButton:onClick()
	--''pi("["..self.name.."] On click")
	if (self.notifyObject) then
		--''pr("-> notifyObject")
		if (self.notifyObject.notify_onClick) then
			--''pr("("..self.notifyObject.name..")")
			local handled = self.notifyObject:notify_onClick(self)
			if (handled ~= nil) then
				--''pd()
				return true
			end
		end
	end
	--''pd()
	IngredientButton.super.onClick(self)
	return true
end

function IngredientButton:allowWork(actor)
	--''pi("["..self.name.."] Allow work?")
	if (self.notifyObject) then
		--''pr("-> notifyObject")
		if (self.notifyObject.notify_allowWork) then
			--''pr("("..self.notifyObject.name..")")
			local allow = self.notifyObject:notify_allowWork(self, actor)
			if (allow ~= nil) then
				--''pd()
				return allow
			end
		end
	end

	if (self.deactivateOnWorkFinish and self.pendingWorkTaskCount > 0) then
		--''pd("Cannot click twice on a single-fire button")
		return false
	end

	--''pd("-> true")
	return true
end

function IngredientButton:allowIngredientWork(actor)
	--''pi("["..self.name.."] IngredientButton allowIngredientWork")
	if (not self.ingredientName) then
		--''pd("No ingredient name!")
		return false;
	end
	if (not tray:makesValidProductOrStart(self.ingredientName)) then
		--''pd("Not a valid product or start")
		return false
	end
	--''pd("-> true")
	return true
end

function IngredientButton:notify_allowWork(ing, actor)
	return true
end

function IngredientButton:onWorkStart(task)
	--''pi("["..self.name.."] IngredientButton onWorkStart")
	IngredientButton.super.onWorkStart(self, task)

	self:stopLoopSample()
	if (not self:checkStock()) then
		--''pd("-> no stock")
		self:playSample("SOUND_NO_STOCK:sound")
		return false --' Do not start
	end

	if (not self:checkTrayAvailable(task)) then
		--''pd("-> tray full")
		return false --' Do not start
	end

	if (self.notifyObject) then
		--''pr("-> notifyObject")
		if (self.notifyObject.notify_onWorkStart) then
			local success = self.notifyObject:notify_onWorkStart(self, task)
			if (success ~= nil) then
				--''pd("success = "..ts(success))
				return success
			end
		end
		--''pr("-> unhandled...")
	end

	if (self.state ~= "available") then
		--''pd("-> not available yet")
		return false --' Do not start
	end

	if (not self:allowIngredientWork()) then
		--''pd("-> !allow ingredient work")
		return false --' Do not start
	end

	self:defaultOnWorkStart(task)
	tray:startIngredient(task.duration, self.compoundPosition, self.compoundTipType, false)
	--''pd("started ingredient '"..ts(self.ingredientName).."'")
	return true
end

function IngredientButton:checkTrayAvailable(task)
	if (self.notifyObject) then
		if (self.notifyObject.notify_checkTrayAvailable) then
			local available = self.notifyObject:notify_checkTrayAvailable(self, task)
			if (available ~= nil) then return available end
		end
	end

	return not tray.full or not tray:makesValidProduct(self.ingredientName)
end

function IngredientButton:defaultOnWorkStart(task)
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onWorkStart) then
			v:status_onWorkStart()
		end
	end

	--''pi("["..self.name.."] Default work start: "..ts(self.ingredientName))
	task.duration = self:getWorkDurationWith(task.actor)
	--''pr("duration: "..tostring(task.duration))
	if (task.duration > 0) then
		--''pr("-> playing work anim '"..ts(self.workAnimation).."'")
		task.actor.direction = self.workDirection
		task.actor:playAnimation(self.workAnimation, self.workCycles, self.workFrameTime, self.workWhenFinished)
		if (self.workFrameTime and self.workFrameTime < 0) then
			task.actor:matchFrameTimeTo(task.duration)
		end
		if (self.workSound and self.workSound ~= "") then
			self:playSample(self.workSound)		
		elseif (self.workLoop and self.workLoop ~= "") then
			self:loopSample(self.workLoop)
		end
	end
	--''pd()
end

function IngredientButton:notify_onWorkStart(ing, task)
	if (ing.state ~= "available") then
		--''pd("-> not available yet")
		return false --' Do not start
	end
	
	if (not self:allowIngredientWork()) then
		return false --' Do not start
	end

	--''pi("["..self.name.."] IngredientButton notify_allowWork (from '"..ing.name.."')"..
	--''	" allowRestockOverload: "..ts(self.allowRestockOverload).." usesStock: "..ts(self.usesStock)..
	--''	" stock: "..ts(self.stock).." maxStock: "..ts(self.maxStock))
	if (not self.allowRestockOverload and self.usesStock and
		self.stock >= self.maxStock) then
		--''pd("-> No filling beyond the maximum stock")
		return false
	end
	--''pd()

	--''pb("["..self.name.."] IngredientButton notify_onWorkStart (from '"..ing.name.."')")
	ing:defaultOnWorkStart(task)
	return true
end

function IngredientButton:onWorkUpdate(task, time)
	if (self.notifyObject) then
		if (self.notifyObject.notify_onWorkUpdate) then
			return self.notifyObject:notify_onWorkUpdate(self, task, time)
		end
	end
end

function IngredientButton:onWorkFinish(task)
	IngredientButton.super.onWorkFinish(self, task)
	self:stopLoopSample()
	--''pi("["..self.name.."] Work finish")
	if (self.notifyObject) then
		--''pr("-> notifyObject")
		if (self.notifyObject.notify_onWorkFinish) then
			local handled = self.notifyObject:notify_onWorkFinish(self, task)
			if (handled ~= nil) then
				--''pd("Handled!")
				return
			end
		end
		--''pr("-> unhandled...")
	end

	local success = tray:commitIngredient(self:getProductCenter(), self)
	self:defaultOnWorkFinish(task, success)
	--''pd("success = "..ts(success))

	if (not self:checkStock()) then
		if (string.find(self.name, "melon")) then
			if (player:mayShowHint("MelonPatch")) then
				local tree = level:getSpriteExt("melon_1")
				if (tree) then
					hintManager:placeArrow({hintId="MelonPatch", x=500, y=163, parent=level.dragLayer})
					switchHintArrowClick(tree, "MelonPatch")
				end
			end
		elseif string.find(self.name, "_crate") then
			if (player:mayShowHint("FruitTree")) then
				local tree = level:getObject("tree_"..string.sub(self.ingredientName, 2))
				if (tree) then
					hintManager:placeArrow({hintId="FruitTree", parent=tree, type=tree.hintArrowDirection or "down"})
					switchHintArrowClick(tree, "FruitTree")
				end
			end
		elseif string.find(self.name, "_soup") then
			if (player:mayShowHint("SoupIngs")) then
				local obj = level:getObject("cutting_board")
				if (obj) then
					hintManager:placeArrow({hintId="SoupIngs",x=obj.x + obj.width/2, y=obj.y, parent=level.dragLayer})
					switchHintArrowClick(obj, "SoupIngs")
				end
			end
		elseif string.find(self.name, "bottle_") then
			if (player:mayShowHint("WineBox")) then
				local obj = level:getObject("juice_box")
				if (obj) then
					hintManager:placeArrow({hintId="WineBox",x=obj.x + obj.width/2, y=obj.y, parent=level.dragLayer})
					switchHintArrowClick(obj, "WineBox")
				end
			end
		elseif string.find(self.name, "pancakes") then
			if (player:mayShowHint("PancakesArrow")) then
				local obj = level:getCharacter("antonio")
				if (obj) then
					hintManager:placeArrow({hintId="PancakesArrow",x=obj.x, y=obj.y - obj.height, parent=level.dragLayer})
					switchHintArrowClick(obj, "PancakesArrow")
				end
			end			
		else		
			if (player:mayShowHint("TentSupplies")) then
				local obj = level:getObject("tent_suplies_box")
				if (obj) then
					hintManager:placeArrow({hintId="TentSupplies",x=obj.x + obj.width/2, y=obj.y, parent=level.dragLayer})
					switchHintArrowClick(obj, "TentSupplies")
				end
			end
		end
	end

	if (self.notifyObject and self.ingredientName == "#pancakes" and player.room == 5) then
		if (self.notifyObject.stock == 1 and not player:mayShowHint("PancakeAgain")) then
			hintManager:openHint({hintId="PancakeAgain", target=self, subject=self})
		end
		if (self.notifyObject.stock == 3 and not player:mayShowHint("PancakeReady")) then
			hintManager:openHint({hintId="PancakeReady", target=self.notifyObject, subject=self.notifyObject})
		else
			player:setNeverShowHint("PancakeReady")
			player:setNeverShowHint("PancakeAgain")
		end
	end
end

function IngredientButton:defaultOnWorkFinish(task, success)
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onWorkFinish) then
			v:status_onWorkFinish()
		end
	end

	self:stopLoopSample()

	--''pb("["..self.name.."] Default work finish: "..ts(self.ingredientName).." success = "..tostring(success))
	task.actor:playDefaultAnimation()
	task.actor.direction = "down"
end

function IngredientButton:notify_onWorkFinish(ing, task)
	--''pb("["..self.name.."] Notify work finish: (from '"..ing.name.."')")
end

function IngredientButton:onCommit(productName, x, y)
	--''pi("["..self.name.."] On commit: "..productName)
	if (self.notifyObject) then
		--''pr("-> notifyObject")
		if (self.notifyObject.notify_onCommit) then
			local success = self.notifyObject:notify_onCommit(self, productName, x, y)
			if (success ~= nil) then
				--''pd("success = "..ts(success))
				return success
			end
		end
		--''pr("-> unhandled...")
	end

	if (level:isValidProduct(productName)) then
		--''pr("-> valid product!")
		self:defaultOnCommit(productName, x, y)
	end
	--''pd()
end

function IngredientButton:defaultOnCommit(productName, x, y)
	--''pi("["..self.name.."] Default commit: "..productName)
	if (self.deactivateOnWorkFinish) then
		--''pr("-> deactivating!")
		self.state = "inactive"
	end
	if (self.usesStock) then
		--''pr("-> decreased stock!")
		self:decStock(true)
	end
	--''pd()
end

function IngredientButton:notify_onCommit(ing, productName, x, y)
	--''pi("["..self.name.."] Notify on commit: "..productName.." (from '"..ing.name.."')")
	local c = self.screenCenter
	local mod = ing:throwCopy(c.x, c.y, level.topLayer)
	self:incStock(ing.worth)
	self:animateIncDisplayStock(ing.worth, mod)
	ing:defaultOnCommit(productName, x, y)
	if (self.usesStock and self.stock == self.maxStock and player.room == 2 and player.shift >= 6) then
		player.trophyManager:checkTrophyGoalReached(11)
	end
	--''pd()
	return true
end

function IngredientButton:onDisplayStockChange()
	for _,v in ipairs(self.statusObjects) do
		if (v.status_onDisplayStockChange) then
			local handled = v:status_onDisplayStockChange(self)
			if (handled) then return end
		end
	end
	IngredientButton.super.onDisplayStockChange(self)
end
