--' ************************************ Where Is My Event ************************************ '--
WhereIsMyEvent = deriveLuaClass("WhereIsMyEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="WhereIsMyEvent",
	scoreOneDone = 40,
	scoreAllDone = 100,
	showLostItemPos = {x=57, y=507},
	exitPos = {x=-100, y=507},
	numItemsToLose = 4,
	askItemSoundId=""
	triesPerItem = 2,
	doNextObjectDuration = 10000,
	waitWhileAskingDuration = 10000,
	loseObjectCharacter = "daughter",
	retrieveObjectCharacter = "mother",
}
--]]

function WhereIsMyEvent:init(desc)
	WhereIsMyEvent.superInit(self, desc)

	--' Configuration variables
	self.doNextObjectDuration = desc.doNextObjectDuration or 10000
	self.waitWhileAskingDuration = desc.waitWhileAskingDuration or 10000

	--' Status variables
	self.doNextObjectTimer = self.doNextObjectDuration
	self.waitWhileAskingTimer = self.waitWhileAskingDuration
	self.eventState = "offScreenItemLost"
	self.activeItem = nil
	self.numTries = 0
	self.numItemsShown = 0
	self.allItemsDone = false
	
	--' Override
	self.matchingItemType = "WhereIsMyObject"
	self.initObjectsHidden = true
	self.showItemsOnStart = false
	self.hideItemOnDone = true
end

function WhereIsMyEvent:onInitialized()
	WhereIsMyEvent.super.onInitialized(self)

	for _,item in ipairs(self.items) do
		item.obj.drawOutline = false
	end
	
	table.sort(self.items, function(a,b) return a.obj.walkIndex < b.obj.walkIndex end)
	
	eventBar.targetValue = math.max(1, self.numItemsToLose)	
	assert(#self.items >= self.numItemsToLose)	
end

function WhereIsMyEvent:onStart()
	WhereIsMyEvent.super.onStart(self)

	local characterName = self.loseObjectCharacter
	self.loseObjectCharacter = level:getCharacter(characterName)
	assert(self.loseObjectCharacter, "wrong charactername, loseObjectCharacter: " .. characterName)
	
	characterName = self.retrieveObjectCharacter
	self.retrieveObjectCharacter = level:getCharacter(characterName)
	assert(self.retrieveObjectCharacter, "wrong charactername, retrieveObjectCharacter: " .. characterName)	
end

function WhereIsMyEvent:onShiftStart()
	WhereIsMyEvent.super.onShiftStart(self)
end

function WhereIsMyEvent:getRandomItem()
	local itemsNotDone = {}
	for _,item in ipairs(self.items) do
		if (item.obj and not item.handled) then
			table.push_back(itemsNotDone, item)
		end
	end	
	return itemsNotDone[math.Random(#itemsNotDone)] or self.items[1]
end

function WhereIsMyEvent:loseItem()
	self.eventState = "losingItem"

	for _,item in ipairs(self.items) do
		if (item.obj) then
			self.loseObjectCharacter:addTask({class="WalkToTask", target=item.obj.walkPos})
		end
	end
	
	self.loseObjectCharacter:addTask({class="WalkToTask", target=self.exitPos, playDefaultOnFinish=true})
	self.loseObjectCharacter:addTask({class="CallTask", func=function() self.numTries = 0 self.eventState = "offScreenItemLost"  end})
	assert(not self.activeItem, "not self.activeItem")
	self.activeItem = self:getRandomItem()
	assert(self.activeItem, "self.activeItem")
end

function WhereIsMyEvent:onTick(time)
	if (self.allItemsDone or self.completed or not self.started) then
		return
	end
	
	if (self.numItemsShown >= self:getNumItemsTodo()) then
		self.allItemsDone = true
	end

	if (self.eventState == "offScreenNoItemLost") then
		self.doNextObjectTimer = self.doNextObjectTimer - time
		if (self.doNextObjectTimer <= 0) then
			self:loseItem()
			self.doNextObjectTimer = self.doNextObjectDuration			
		end
	elseif (self.eventState == "offScreenItemLost") then
		self.doNextObjectTimer = self.doNextObjectTimer - time
		if (self.doNextObjectTimer <= 0) then
			self.retrieveObjectCharacter:addTask({class="WalkToTask", target=self.showLostItemPos, playDefaultOnFinish=true})
			self.retrieveObjectCharacter:addTask({class="CallTask", func=function()
				if (not self.activeItem) then
					self.activeItem = self:getRandomItem()
					if (not self.activeItem) then
						self.completed = true
						return
					end
				end
				self.activeItem.obj.clickable = true
				self.numTries = self.numTries+1 
				self.retrieveObjectCharacter:createOverhead(self.activeItem.obj.overheadImage..":icons") 
				self.eventState = "showLostItem" 
				if (self.askItemSoundId) then
					self.activeItem.obj:playSample(self.askItemSoundId)
				end
				self:showItem(self.activeItem)
				SpriteExt.playSample(nil, "SOUND_EVENT_ATTENTION")
			end})
			self.doNextObjectTimer = self.doNextObjectDuration		
			self.eventState = "walkToShowLostItem" 
		end	
	elseif (self.eventState == "showLostItem" and self.activeItem and not self.activeItem.working) then
		self.waitWhileAskingTimer = self.waitWhileAskingTimer - time
		if (self.waitWhileAskingTimer <= 0) then
			assert(self.activeItem)
			self.activeItem.obj.clickable = false
			self.retrieveObjectCharacter:removeOverhead(true)
			self.eventState = "walkToExitPos" 
			self.waitWhileAskingTimer = self.waitWhileAskingDuration
			self.retrieveObjectCharacter:addTask({class="WalkToTask", target=self.exitPos, playDefaultOnFinish=true})
			self.retrieveObjectCharacter:addTask({class="CallTask", func=function()
				if (self.numTries < self.triesPerItem) then
					self.eventState = "offScreenItemLost"
				else
					self:hideItem(self.activeItem)
					self.activeItem.handled = true
					self.activeItem = nil
					self.numTries = 0
					self.numItemsShown = self.numItemsShown + 1
--					self.eventState = "offScreenNoItemLost"  
					self.eventState = "offScreenItemLost"
				end
			end})		
		end
	end
end

function WhereIsMyEvent:getNumItemsTodo()
	return self.numItemsToLose
end

function WhereIsMyEvent:itemFound(item)
	item.handled = true
	self:itemDone(item)
	self.activeItem = nil
	self.eventState = "walkToExitPos" 
	self.numItemsShown = self.numItemsShown + 1	
	self.numTries = 0
	self.waitWhileAskingTimer = self.waitWhileAskingDuration
	self.retrieveObjectCharacter:removeOverhead(true)
	self.retrieveObjectCharacter:addTask({class="Task", duration=800})
	self.retrieveObjectCharacter:addTask({class="WalkToTask", target=self.exitPos, playDefaultOnFinish=true})
	self.retrieveObjectCharacter:addTask({class="CallTask", func=function()	self.eventState = "offScreenItemLost" end})		
end

--' ************************************ Where Is My Object ************************************ '--
WhereIsMyObject = deriveLuaClass("WhereIsMyObject", "Object")

function WhereIsMyObject:init(desc)
	WhereIsMyObject.superInit(self, desc)

	self.highlightSound = "" --' Mute the highlight
	self.drawOutline = false

	self.workAnimation = desc.workAnimation or "WORK"
	self.workCycles = desc.workCycles or -1
	self.workWhenFinished = desc.workWhenFinished or "default"
	self.workFrameTime = desc.workFrameTime or 150
	self.workDirection = desc.workDirection or "up"
	self.workDuration = desc.workDuration or 1000
	
	self.walkIndex = desc.walkIndex or 1
	assert(self.walkPos, "need to set walkPosition")
end

function WhereIsMyObject:onWorkStart(task)
	self.item.working = true

	task.duration = self.workDuration
	if (task.duration > 0) then
		task.actor.direction = self.workDirection
		task.actor:playAnimation(self.workAnimation, self.workCycles, self.workFrameTime, self.workWhenFinished)
	end

	--' Remove the hint arrow
	event:hideHintArrow()
end

function WhereIsMyObject:onWorkFinish(task)
	self.item.working = false
	
	task.actor:playAnimation("IDLE")
	task.actor.direction = "down"
	
	event:itemFound(self.item)
end
