--' ************************************ Shout and Make Event ************************************ '--
ShoutAndMakeEvent = getClass("ShoutAndMakeEvent")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="ShoutAndMakeEvent",
	bar = "progressbar_big",
	productPool = {"#sock", "#bread#cheese"},
	numProductsPerOrder = 2,
	numOrders = 2,
	timeUntilNextOrder = 0,
	timeUntilFirstOrder = 0,
	maxGroupCount = 2,
	items = {
		{ obj="cleaner", barPos = {x=10, y=-100}, workPos={x=300, y=683}, workAnimation="CLEAN", workDuration=2000, timeUntilEnd=120000, },
	},
}

function levelDesc:onDeliverProducts(item)
	...
end
--]]

function ShoutAndMakeEvent:init(desc)
	ShoutAndMakeEvent.superInit(self, desc)

	--' Configuration variables
	self.timeUntilFirstOrder = self.timeUntilFirstOrder or 0
	self.timeUntilNextOrder = self.timeUntilNextOrder or 0		--' Interval at which the orders come
	self.numProductsPerOrder = self.numProductsPerOrder or 0	--' Number of products ordered each order
	self.numOrders = self.numOrders or 0						--' Total number of orders
	self.productPool = desc.productPool or {}					--' Product pool

	--' Status variables
	self.numObjectsOutOfTime = 0
	self.numObjectsDone = 0

	--' Override	
end

function shoutAndMakeWorkStart(self, task)
	event:hideHintArrow()
	if (not event.started or event.completed or self.item.order.currentStep.numProducts <= 0) then
		return false
	end

	local duration = self:getWorkDurationWith(task.actor)
	task.duration = duration
	if (duration > 0) then
		if (self.workDirection) then
			task.actor.direction = self.workDirection
		end
		task.actor:playAnimation(self.workAnimation or "WORK", self.workCycles or -1, self.workFrameTime or 150, self.workWhenFinished or "default")
	end
end

function shoutAndMakeWorkFinish(obj, task)
	--' Check the products
	assert(obj.item.order)
	--' Check if all the products for this object are delivered
	local numProducts = obj.item.order.currentStep.numProducts
	local newNumProducts = obj.item.order:deliver(obj.item)
	if (newNumProducts == 0 and obj.item.numOrders <= 0) then
		event:objectAllProductsDone(obj.item)
	else
		event:updateOrderOverhead(obj.item)
	end

	task.actor:playDefaultAnimation()
	if (newNumProducts < numProducts) then

		local pos = obj.floaterPosition
		local score = event.scoreOneDone * (numProducts - newNumProducts)
		local text = string.format(ghf.getString("EVENT_SCORE_ONE_DONE"), score)
		level:handleScore(score, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, score, 0)
		
		if (level.onDeliverProducts) then
			level:onDeliverProducts(obj.item)
		end
	end
end

function ShoutAndMakeEvent:onInitialized()
	ShoutAndMakeEvent.super.onInitialized(self)

	for _,item in ipairs(self.items) do
		if (item.obj) then
			item.obj.onWorkStart = shoutAndMakeWorkStart
			item.obj.onWorkUpdate = nil
			item.obj.onWorkFinish = shoutAndMakeWorkFinish
			
			item.productPool = item.productPool or self.productPool

			item.numProductsPerOrder = item.numProductsPerOrder or self.numProductsPerOrder
			item.numOrders = item.numOrders or self.numOrders
			item.maxGroupCount = item.maxGroupCount or self.maxGroupCount
			item.maxHorizontalSlotCount = item.maxHorizontalSlotCount or self.maxHorizontalSlotCount
			item.timeUntilNextOrder = item.timeUntilNextOrder or self.timeUntilNextOrder
			item.nextOrderTimer = self.timeUntilFirstOrder

			item.order = self:addOrder()
			item.duration = item.timeUntilEnd
			item.active = false

			self:createProgressBar(item, 1)		
		end
	end

	eventBar.progress = 1
	eventBar.startValue = 1
	eventBar.targetValue = 0
	eventBar.showValue = false
	eventBar.showTarget = false
	
end

function ShoutAndMakeEvent:showItem(item)
	ShoutAndMakeEvent.super.showItem(self, item)

	self:showProgressBar(item)
	self:showHint(item.obj)
	self:onProgress()
end

function ShoutAndMakeEvent:objectOutOfTime(item)
	self.numObjectsOutOfTime = self.numObjectsOutOfTime + 1
	self:removeDecals(item)
	item.done = true
	item.obj.clickable = false

	self.won = false
	if (level.onObjectOutOfTime) then
		level:onObjectOutOfTime(self, item)
	end
end

function ShoutAndMakeEvent:objectAllProductsDone(item)	
	self.numObjectsDone = self.numObjectsDone + 1
	self:removeDecals(item)
	item.done = true
	item.obj.clickable = false
end

function ShoutAndMakeEvent:onTick(time)
	if (self.completed or not self.started) then
		return
	end

	for _,item in ipairs(self.items) do
		if (item.obj and not item.done) then
			item.duration = item.duration - time
			if (self.blinkStartFraction and
				item.duration / item.timeUntilEnd < self.blinkStartFraction)
			then
				self:blinkBar(item)
			end
			if (item.duration <= 0) then
				self:objectOutOfTime(item)
				break
			end

			if (item.numOrders > 0) then
				item.nextOrderTimer = item.nextOrderTimer - time
				if (item.nextOrderTimer <= 0) then
					if self.orderFirstTime then
						self:orderFirstTime(item)
					end
					item.numOrders = item.numOrders - 1					
					for i=1, self.numProductsPerOrder do
						item.order.currentStep:addProduct(item.productPool[math.Random(#item.productPool)])
					end
					self:updateOrderOverhead(item)
					item.nextOrderTimer = self.timeUntilNextOrder
				end
			end
		end
	end

	if (self.numObjectsOutOfTime + self.numObjectsDone >= self:getNumItemsTodo()) then
		if (self.numObjectsOutOfTime > 0) then
			self:eventLost()
		else
			self:eventWon()
		end
	end
	self:onProgress()
end

function ShoutAndMakeEvent:onProgress()
	ShoutAndMakeEvent.super.onProgress(self)

	for _,item in ipairs(self.items) do
		if (item.obj and not item.done) then
			item.bar:setProgress(item.duration / item.timeUntilEnd)
		end
	end
end

function ShoutAndMakeEvent:getProgress()
	local lowest
	for _,item in ipairs(self.items) do
		if (not item.done and isNumber(item.duration)) then
			local p = item.duration / item.timeUntilEnd
			if (not lowest or p < lowest) then
				lowest = p
			end
		end
	end
	return lowest or 0
end

function ShoutAndMakeEvent:removeDecals(item)
	ShoutAndMakeEvent.super.removeDecals(self, item)	
	if (item and item.obj and item.order) then
		item.obj:removeOverhead(true)
		item.order = nil
	end
end
