--' ************************************ Search Spots Event ************************************ '--
SearchSpotsEvent = deriveLuaClass("SearchSpotsEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="SearchSpotsEvent",
	treasureImageIds = { "PAY_ICON_0:icons", "PAY_ICON_1:icons", "PAY_ICON_2:icons"},
	customCursor = {image="PAY_ICON_0:icons", x=10, y=10}
}
--]]

function SearchSpotsEvent:init(desc)
	SearchSpotsEvent.superInit(self, desc)

	--' Configuration variables
	self.noTreasureOverheadImageId = self.noTreasureOverheadImageId or "ICON_THINK:icons"
	self.canAppearAtTheSameTime = self.canAppearAtTheSameTime ~= false

	--' Status variables
	self.objectsFound = 0
	self.hintTimer = 5000

	--' Override
	self.matchingItemType = "SearchSpotsObject"
	self.initObjectsHidden = true
end

function SearchSpotsEvent:onInitialized()
	SearchSpotsEvent.super.onInitialized(self)

	for _,item in ipairs(self.items) do
		item.obj.drawOutline = false
		item.active = false
		item.hiddenItem = (item.obj and item.obj.hiddenItem) or item.hiddenItem or ""
	end

	assert(#self.treasureImageIds < #self.items, "number of treasures ids ("..#self.treasureImageIds..") needs to be smaller than number of objects in the level ("..#self.items..")")

	for i=1,100 do
		local i1, i2 = math.Random(#self.items), math.Random(#self.items)
		self.items[i1], self.items[i2] = self.items[i2], self.items[i1]
	end

	--' Divide the treasures
	local idx = 1
	self.numTreasuresToFind = 0
	for k, v in ipairs(self.items) do
		if (v.hiddenItem == "") then
			v.hiddenItem = self.treasureImageIds[idx] or ""
			idx = idx + 1
		end
		if (v.hiddenItem ~= "") then
			self.numTreasuresToFind = self.numTreasuresToFind + 1
		end
	end

	eventBar.targetValue = math.max(1, self:getNumItemsTodo())
	eventBar.showTarget = (eventBar.targetValue ~= 1)
	eventBar.showValue = (eventBar.targetValue ~= 1)
end

function SearchSpotsEvent:getNumItemsTodo()
	return self.numTreasuresToFind or 0
end

function SearchSpotsEvent:showItem(item)
	SearchSpotsEvent.super.showItem(self, item)
	item.active = true

	item.obj.doRender = false
	item.obj.highlightSound = "" --' Mute the highlight

	item.obj.workAnimation = item.workAnimation or item.obj.workAnimation or "WORK"
	item.obj.workCycles = item.workCycles or item.obj.workCycles or -1
	item.obj.workWhenFinished = item.workWhenFinished or item.obj.workWhenFinished or "default"
	item.obj.workFrameTime = item.workFrameTime or item.obj.workFrameTime or 150
	item.obj.workDirection = item.workDirection or item.obj.workDirection or "up"
	item.obj.workDuration = item.workDuration or item.obj.workDuration or 1000

	item.obj.onClick = searchSpotOnClick
	item.obj.onWorkStart = searchSpotOnWorkStart
	item.obj.onWorkUpdate = searchSpotOnWorkUpdate
	item.obj.onWorkFinish = searchSpotOnWorkFinish
	item.obj.timeForOverhead = item.obj.workDuration/2
end

function SearchSpotsEvent:itemDone(item)
	if (item.hiddenItem ~= "") then
		SearchSpotsEvent.super.itemDone(self, item)
	end
end

function SearchSpotsEvent:onItemDone(item)
	SearchSpotsEvent.super.onItemDone(self, item)
end

function SearchSpotsEvent:hideItem(item)
	SearchSpotsEvent.super.hideItem(self, item)
	item.obj.visible = false
end

function SearchSpotsEvent:onTick(time)
	if (not self.started or self.completed or self.showHintOnStart) then
		return
	end
	
	if (self.hintTimer > 0) then
		self.hintTimer = self.hintTimer - time
		if (self.hintTimer <= 0) then
			if (self.objectsFound < self:getNumItemsTodo()) then
				self:showHint()
			end
		end
	end
end

--' ************************************ Search Spots Object ************************************ '--
SearchSpotsObject = deriveLuaClass("SearchSpotsObject", "EventObject")

function searchSpotOnClick(self)
	if (self.old_onClick(self)) then
		local workCharacter = self:getWorkCharacter()
		if (workCharacter) then
			workCharacter.tmp_stationaryActive = workCharacter.stationaryActive
			workCharacter.stationaryActive = false
			workCharacter.tmp_blockWork = workCharacter.blockWork
			workCharacter.blockWork = true
			self.tmp_clickable = self.clickable
			self.clickable = false
		end
	end
	return true
end
	
function searchSpotOnWorkStart(self, task)
	local duration = self:getWorkDurationWith(task.actor)
	task.duration = duration
	if (duration > 0) then
		task.actor.direction = self.workDirection
		task.actor:playAnimation(self.workAnimation, self.workCycles, self.workFrameTime, self.workWhenFinished)
	end
	
	if (self.item.hiddenItem ~= "") then
		event.objectsFound = event.objectsFound + 1
	end
	
	--' Remove the hint arrow
	event:hideHintArrow()
end

function searchSpotOnWorkUpdate(self, task, time)
	self.timeForOverhead = self.timeForOverhead - time
	if (self.timeForOverhead <= 0) then
		--' Create the 'found item' overhead
		if (self.item.hiddenItem == "") then
			task.actor:createOverhead(event.noTreasureOverheadImageId)
		else		
			task.actor:createOverhead(self.item.hiddenItem)	
			if (event.foundItemSoundId) then
				task.actor:playSample(event.foundItemSoundId)
				event.foundItemSoundId = ""				
			end
		end
	end
end

function searchSpotOnWorkFinish(self, task)
	local actor = task.actor
	task.actor:playAnimation("IDLE")
	actor.direction = "down"

	actor:removeOverhead()	

	self.item.actor = actor
	event:itemDone(self.item)
	if (self.walkPosOnFind) then
		actor:addTask({class="WalkToTask", target=self.walkPosOnFind, playDefaultOnFinish=true, directionOnFinish="down"})
	end
	actor:addTask({class="CallTask", func=function()
		actor.stationaryActive = actor.tmp_stationaryActive
		actor.blockWork = actor.tmp_blockWork
		self.clickable = self.tmp_clickable
	end})

	if (player.room == 1) then
		hintManager:openHint({hintId="PupNotHere", showButton=true, closeOnClickAnywhere=true})
	end
end
