--' ************************************ Fix Event ************************************ '--
LearnToRideEvent = deriveLuaClass("LearnToRideEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="LearnToRideEvent",
	scoreOneDone = 10,
	scoreAllDone = 100,
	newPushDuration = 10000,
	waitForRideDuration = 6000
}
--]]

function LearnToRideEvent:init(desc)
	LearnToRideEvent.superInit(self, desc)

	--' Configuration variables
	self.newPushDuration = desc.newPushDuration or 10000
	self.waitForRideDuration = desc.waitForRideDuration or 6000

	--' Status variables
	self.hintFired = false
	self.rideObject = nil
	self.rideCharacter = nil
	self.rideActive = false
	self.numRidesDone = 0
	
	--' Override
	self.newPushTimer = self.newPushDuration
	self.waitForRideTimer = self.waitForRideDuration
end

function LearnToRideEvent:onInitialized()
	local objects = level:getSpritesByExactType("LearnToRideObject")
	self.rideObject = objects[1]
	assert(self.rideObject)
	self.rideObject.workPosRel = {}
	self.rideObject.workPosRel.x = self.rideObject.location.x - self.rideObject.workPos.x
	self.rideObject.workPosRel.y = self.rideObject.location.y - self.rideObject.workPos.y
	
	self.rideCharacter = level:getCharacter("emily")
	self.rideCharacter.workPosRel = {}
	self.rideCharacter.workPosRel.x = self.rideCharacter.location.x - self.rideCharacter.workPos.x
	self.rideCharacter.workPosRel.y = self.rideCharacter.location.y - self.rideCharacter.workPos.y
	
	LearnToRideEvent.super.onInitialized(self)
end

function LearnToRideEvent:getNumItemsTodo()
	if (self.rideObject) then
		return #self.rideObject.ridePositions + 1
	end
	return 0
end

function LearnToRideEvent:onTick(time)
	if (not self.started or self.completed or level.inStoryMode or self.rideActive) then
		return
	end

	if (self.newPushTimer > 0) then
		self.newPushTimer = self.newPushTimer - time
		if (self.newPushTimer <= 0) then
			self.rideCharacter.stationaryActive = false
			self.rideCharacter.blockWork = true
			
			self.rideCharacter:addTask({class="WalkToTask", target=self.rideObject})
			self.rideCharacter:addTask({class="CallTask", func=function() 
				self:setupRideCharacter()
				self.rideCharacter:playAnimation("WAVE", -1, 150)
				self.rideCharacter:playSample("SOUND_BIKE_BELL:sound")
				self.rideObject.visible = false				
				self.waitForRideTimer = self.waitForRideDuration
				hintManager:placeArrow({hintId="LearnToRideEvent", x=self.rideCharacter.x, y=self.rideCharacter.y-self.rideCharacter.height+20})		
			end})			
		end
	elseif (self.waitForRideTimer > 0) then
		self.waitForRideTimer = self.waitForRideTimer - time
		if (self.waitForRideTimer <= 0) then
			self:restoreRideCharacter()
		end
	end
end

function LearnToRideEvent:setupRideCharacter()
	if (self.rideCharacter.baseAnimation == "EMILY-BIKE") then
		return
	end

	self.rideCharacter.baseAnimation="EMILY-BIKE" 
	self.rideCharacter.direction = "right"
	self.rideCharacter:playAnimation("IDLE")
	self.rideCharacter.clickable = true
	self.rideCharacter.workPos = {x=self.rideCharacter.location.x - self.rideCharacter.workPosRel.x, y=self.rideCharacter.location.y - self.rideCharacter.workPosRel.y}
	
	-- keep old functions
	self.rideCharacter.old_onClick = self.rideCharacter.onClick
	self.rideCharacter.old_onWorkStart = self.rideCharacter.onWorkStart
	self.rideCharacter.old_onWorkUpdate = self.rideCharacter.onWorkUpdate
	self.rideCharacter.old_onWorkFinish = self.rideCharacter.onWorkFinish
	self.rideCharacter.old_onWorkCancel = self.rideCharacter.onWorkCancel
	
	self.rideCharacter.onWorkStart = LearnToRideWorkStart
	self.rideCharacter.onWorkFinish = LearnToRideWorkFinish
	self.rideCharacter.onClick = function()	
		self.rideCharacter:old_onClick()
		self.rideActive = true
		self.rideCharacter.clickable = false
		hintManager:removeArrows("LearnToRideEvent")
		return true
	end		
end

function LearnToRideEvent:restoreRideCharacter()
	if (self.rideCharacter.baseAnimation == "EMILY") then
		return
	end

	hintManager:removeArrows("LearnToRideEvent")

	self.rideCharacter.baseAnimation="EMILY" 
	self.rideCharacter:playAnimation("IDLE")
	self.rideCharacter.stationaryActive = true
	self.rideCharacter.blockWork = false
	self.rideCharacter.clickable = false
	self.rideCharacter.onClick = self.rideCharacter.old_onClick
	self.rideCharacter.onWorkStart = self.rideCharacter.old_onWorkStart
	self.rideCharacter.onWorkUpdate = self.rideCharacter.old_onWorkUpdate
	self.rideCharacter.onWorkFinish = self.rideCharacter.old_onWorkFinish
	self.rideCharacter.onWorkCancel = self.rideCharacter.old_onWorkCancel

	self.rideObject.visible = true	
	self.rideObject.location = {x=self.rideCharacter.location.x + self.rideObject.workPosRel.x, y=self.rideCharacter.location.y + self.rideObject.workPosRel.y}
	self.rideObject.workPos = {x=self.rideObject.location.x - self.rideObject.workPosRel.x, y=self.rideObject.location.y - self.rideObject.workPosRel.y}
	
	self.rideActive = false
	self.newPushTimer = self.newPushDuration
end

function LearnToRideEvent:getFloaterPosition()
	return self.rideCharacter.floaterPosition
end

function LearnToRideWorkStart(rideCharacter, task)
	if (not event.started or event.completed) then
		return false
	end	
	
	hero.direction = "right"
	hero:playAnimation("PUSH", 1, 60)
	task.duration = hero.frameCount*60
	event.numRidesDone = event.numRidesDone + 1
	
	--' do the ride animation
	local anim = "FALL"
	if (event.numRidesDone >= event:getNumItemsTodo()) then
		anim = "IDLE"
	end

	eventBar.value = event.numRidesDone
	
	if (event.numRidesDone < event:getNumItemsTodo()) then
		rideCharacter:addTask({class="Task", duration=200})
		rideCharacter:addTask({class="WalkToTask", target=event.rideObject.ridePositions[event.numRidesDone]})
		rideCharacter:addTask({class="PlayAnimationTask", animation=anim, cycles=1, frameTime=120, wait=true})
		rideCharacter:addTask({class="CallTask", func=function() 
			event:restoreRideCharacter()
			hero:addTask({class="MessageTask", duration=2000, textId="R1_D03_EVENT1", needsClick=false, keepOpen=false, })		
			hero:addTask({class="CallTask", func=function() 
				rideCharacter:addTask({class="PlayAnimationTask", animation="NODYES", cycles=3})
				rideCharacter:resetTimer()
			end})
		end})
	end
end

function LearnToRideWorkFinish(rideCharacter, task)
	hero:playDefaultAnimation()	
	
	--' handle score
	if (event.scoreOneDone and event.scoreOneDone > 0) then
		local pos = event:getFloaterPosition()
		local text = string.format(ghf.getString("EVENT_SCORE_ONE_DONE"), event.scoreOneDone)
		level:handleScore(event.scoreOneDone, pos.x, pos.y, false, 0);
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, event.scoreOneDone, 0)
	end

	if (event.numRidesDone >= event:getNumItemsTodo()) then
		event:restoreRideCharacter()
		event:eventWon()
	end	
end

--' LearnToRideObject
LearnToRideObject = deriveLuaClass("LearnToRideObject", "Object")

function LearnToRideObject:init(desc)
	self.ridePositions = desc.ridePositions or {}
end
