--' ************************************ Fix Event ************************************ '--
KeepItUpEvent = deriveLuaClass("KeepItUpEvent", "Event")

--' Sample usage '--
--[[
levelDesc.eventDesc =
{
	class="KeepItUpEvent",
	duration = 20000,
	bar = "progressbar_big",
	items = {
		{ obj="cleaner", barPos = {x=10, y=-100}, workPos={x=300, y=683}, workAnimation="CLEAN", workDuration=2000, drainDuration=10000},
	},
}

function levelDesc:onFillBarStart(item)
end

function levelDesc:onFillBarEnd(item)
	...
end

function levelDesc:onObjectDrained(item)
	...
end
--]]

function KeepItUpEvent:init(desc)
	KeepItUpEvent.superInit(self, desc)

	--' Configuration variables
	self.numDrainedObjects = self.numDrainedObjects or 0
	self.initialProgress = self.initialProgress or 1
	self.duration = self.duration or 40000

	--' Status variables
	self.won = true --' Assume we win

	--' Override
end

local function keepItUpOnClick(self)
	if (self.item and self.item.old_onClick) then
		self.item.old_onClick(self)
		self.clickable = false
		self.item.draining = false
	end
	event:hideHintArrow()
	return true
end


local function keepItUpWorkStart(self, task)
	if (self.item.drained) then
		return true
	end

	if (self.event_onWorkStart) then
		local handled = self:event_onWorkStart(task)
		if (handled ~= nil) then return handled end
	end

	event:stopBlinkBar(self.item)
	self.item.draining = false
	local duration = self:getWorkDurationWith(task.actor)
	task.duration = duration
	if (duration > 0) then
		if (self.workDirection) then
			task.actor.direction = self.workDirection
		end
		task.actor:playAnimation(self.workAnimation or "WORK", self.workCycles or -1, self.workFrameTime or 150, self.workWhenFinished or "default")
	end
	
	if (level.onFillBarStart) then
		level:onFillBarStart(self.item)
	else
		task.stayActive = true
	end	
end

local function keepItUpWorkFinish(self, task)
	if (self.event_onWorkFinish) then
		local handled = self:event_onWorkFinish(task)
		if (handled ~= nil) then return handled end
	end

	self.item.draining = true
	self.item.duration = self.item.drainDuration
	task.actor:playAnimation("IDLE")
	self.clickable = true
	event:stopBlinkBar(self.item)

	if (level.onFillBarEnd) then
		level:onFillBarEnd(self.item)
	end

	hintManager:openHint({hintId="KeepCaring", showButton=false, closeOnClickAnywhere=true})
end

function KeepItUpEvent:onInitialized()
	KeepItUpEvent.super.onInitialized(self)

	eventBar.progress = 1
	eventBar.showValue = false
	eventBar.showTarget = (#self.items ~= 1)
end

function KeepItUpEvent:showItem(item)
	KeepItUpEvent.super.showItem(self, item)

	if (item.obj) then
		item.obj.onWorkStart = keepItUpWorkStart
		item.obj.onWorkUpdate = nil
		item.obj.onWorkFinish = keepItUpWorkFinish
		item.obj.onClick = keepItUpOnClick
	end

	item.draining = true
	item.duration = item.drainDuration * self.initialProgress
	self:createProgressBar(item, 1)
	self:showProgressBar(item)
	item.bar:setProgress(item.duration / item.drainDuration)

	self:onProgress()
end

function KeepItUpEvent:objectDrained(item)
	item.drained = true
	item.obj.clickable = false
	self.numDrainedObjects = self.numDrainedObjects + 1
	self:removeDecals(item)
	self.won = false
	if (level.onObjectDrained) then
		level:onObjectDrained(self, item)
	end
end

function KeepItUpEvent:onTick(time)
	if (not self.started or self.completed) then
		return
	end

	self.duration = self.duration - time
	if (self.duration <= 0) then
		self:eventWon()	
		return
	end

	for _,item in ipairs(self.items) do
		if (not item.drained and item.draining) then
			item.duration = item.duration - time
			if (self.blinkStartFraction and
				item.duration / item.drainDuration < self.blinkStartFraction)
			then
				self:blinkBar(item)
			end
			if (item.duration < 0) then
				self:objectDrained(item)
			end
		end
	end

	if (self.numDrainedObjects >= #self.items) then
		self:eventLost()
	end

	self:onProgress()
end

function KeepItUpEvent:onProgress()
	KeepItUpEvent.super.onProgress(self)

	for _,item in ipairs(self.items) do
		if (item.bar) then
			item.bar:setProgress(item.duration / item.drainDuration)
		end
	end
end

function KeepItUpEvent:getProgress()
	local lowest = 1
	for _,item in ipairs(self.items) do
		if (item.drained) then
			return 0
		else
			local p = item.duration / item.drainDuration
			if (p < lowest) then
				lowest = p
			end
		end
	end
	return lowest
end
