--' ************************************ Event ************************************ '--
Event = getClass("Event")

function Event:init(desc)
	Event.superInit(self, desc)

	--' Configuration variables
	self.autoStart = self.autoStart or false --' Default FALSE
	self.initObjectsHidden = self.initObjectsHidden or false --' Default FALSE
	self.showHintOnStart = self.showHintOnStart ~= false --' Default true
	self.showItemsOnStart = self.showItemsOnStart ~= false --' Default true
	self.hideItemsAtEnd = self.hideItemsAtEnd or false --' Default FALSE
	self.hideItemsOnWon = self.hideItemsOnWon or false --' Default FALSE
	self.hideItemsOnLose = self.hideItemsOnLose or false --' Default FALSE
	self.hideItemOnDone = self.hideItemOnDone or false --' Default FALSE
	self.hintArrowDesc = self.hintArrowDesc or {}
	self.showFadeInDuration = self.showFadeInDuration or 1000
	self.foundFadeOutDuration = self.foundFadeOutDuration or 200
	self.barFadeDuration = self.barFadeDuration or 600
	self.floatersFromHero = self.floatersFromHero or false --' Default FALSE
	self.blinkStartFraction = self.blinkStartFraction or 0.16

	--' Status variables
	self.completed = false
	self.started = false
	self.won = false
	self.items = self.items or {}
	self.itemsDone = 0
end

function Event:onInitialized()
	eventBar.startValue = 0
	eventBar.progress = 0
	eventBar.showBar = self.showBar ~= false --' Default true

	if (self.eventBarText) then
		eventBar.description = self.eventBarText
	elseif (player) then
		local eventBarText = "TEXT_EVENT"..player.room.."_"..player.shift
		if (ghf.hasString(eventBarText)) then
			eventBar.description = eventBarText
		end
	end

	local upgraded = ""
	if (self.eventBarIconObject) then
		if (player:isObjectBought(self.eventBarIconObject, player.room)) then
			upgraded = "_UPGRADED"
		end
	end

	if (self.eventBarImage) then
		eventBar.icon = self.eventBarImage
	elseif (self.baseAnimation) then
		local eventBarImage = self.baseAnimation.."_BAR"
		if (self.defaultSection) then
			eventBarImage = eventBarImage..upgraded..":"..self.calcSection
		end
		eventBar.icon = eventBarImage
	elseif (player) then
		local eventBarImage = "EVENT_ICON"..player.shift..upgraded..":"..level.iconSection
		if (level:hasResource(eventBarImage)) then
			eventBar.icon = eventBarImage
		end
	end

	if (self.matchingItemType) then
		local objects = level:getSpritesByExactType(self.matchingItemType)
		for _,obj in ipairs(objects) do
			if (not self:getItemForObject(obj)) then
				table.insert(self.items, {obj=obj})
			end
		end
	end
	
	for _,item in ipairs(self.items) do
		if (isString(item.obj)) then
			item.obj = level:getChild(item.obj)
		elseif (isTable(item.obj) and not item.obj.__cpp and isString(item.obj.base)) then
			item.obj = level:createCharacter(item.obj)
		end
		
		if (self.initObjectsHidden and item.obj) then
			item.obj.visible = false
		end
	end

	if (self.trackProgress == nil) then
		self.trackProgress = (self:getNumItemsTodo() == 1)
	end

	eventBar.targetValue = math.max(1, self:getNumItemsTodo())
end

function Event:getNumItemsTodo()
	return #self.items
end

function Event:getItemForObject(obj)
	for _,item in ipairs(self.items) do
		if (obj and item.obj == obj) then
			return item
		end
	end
end

function Event:onShiftStart()
	if (self.autoStart) then
		level:fireTrigger("startEvent")
	end
end

function Event:onStart()
	eventBar:activate()
	self.started = true
	if (self.showItemsOnStart) then
		self:showItems()
	end
	if (self.showHintOnStart) then
		self:showHint()
	end
	if (self.playStartAnimation) then
		self:doStartAnimation()
	end
end

function Event:doStartAnimation()
	if (self.startAnimation) then
		for _,item in ipairs(self.items) do
			if (item.obj) then
				item.obj:playAnimation(self.startAnimation, self.startAnimationCycles, self.startAnimationFrameTime, self.startAnimationWhenFinished)
			end
		end
	end
end

function Event:preserveBool(item, k)
	local v = item.obj[k]
	local old_k = "old_"..k
	if (v == false) then
		item[old_k] = false
	elseif (v == true) then
		item[old_k] = true
	else
		item[old_k] = nil
	end
end

function Event:preserveItem(item)
	if (item.obj) then
		if (isTable(item.obj)) then
			item.done = false
			item.obj.item = item
			item.old_onWorkStart = item.old_onWorkStart or item.obj.onWorkStart
			item.old_onWorkUpdate = item.old_onWorkUpdate or item.obj.onWorkUpdate
			item.old_onWorkFinish = item.old_onWorkFinish or item.obj.onWorkFinish
			item.old_onWorkCancel = item.old_onWorkCancel or item.obj.onWorkCancel
			item.old_onClick = item.old_onClick or item.obj.onClick
			item.old_workAnimation = item.obj.workAnimation
			item.old_workDuration = item.obj.workDuration
			item.old_workPos = item.obj.workPos

			self:preserveBool(item, "clickable")
			self:preserveBool(item, "doHighlight")
			self:preserveBool(item, "stationaryActive")
			--' Copy all the keys in the event item table
			for k,v in pairs(item) do
				if (k ~= "obj") then
					if (isTable(v)) then
						item.obj[k] = table.copy(v, true, false, true)
					else
						item.obj[k] = v
					end
				end
			end
		end
	end
end

function Event:showItems()
	for _,item in ipairs(self.items) do
		if (item.obj) then
			self:showItem(item)
		end
	end
end

function Event:showItem(item)
	if (item.shownForEvent) then
		return
	end

	item.shownForEvent = true
	self:preserveItem(item)
	item.obj.clickable = true
	item.obj.doHighlight = true

	if (item.swapObj == nil) then
		item.swapObj = item.obj.swapObj
	end
	if (isString(item.swapObj)) then
		item.swapObj = level:getChild(item.swapObj)
	end
	if (isTable(item.swapObj)) then
		item.swapObj.visible = false
		item.obj.visible = true
	end
	if (not item.obj.visible) then
		item.obj.visible = true
		item.obj.alpha = 0
	end
	if (item.obj.alpha < 1) then
		item.obj:animate({alpha=1, duration=self.showFadeInDuration})
	end
	if (item.obj.onEventShow) then
		item.obj:onEventShow()
	end
end

function Event:hideItems()
	for _,item in ipairs(self.items) do
		if (item.obj) then
			self:hideItem(item)
		end
	end
end

function Event:hideItem(item)
	item.obj.clickable = false
	if (item.swapObj) then
		item.obj.visible = false
		--if (item.bar) then
			--item.bar.alpha = 0
		--end
		item.swapObj.visible = true
	else
		item.obj:animate({alpha=0, duration=self.foundFadeOutDuration})
		--if (item.bar) then
			--item.bar:animate({alpha=0, duration=self.foundFadeOutDuration})
		--end
	end
	if (item.obj.onEventHide) then
		item.obj:onEventHide()
	end
end

function Event:getProgress()
	local numItemsTodo = self:getNumItemsTodo()
	return (numItemsTodo > 0 and event.itemsDone / numItemsTodo) or 1
end

function Event:onProgress()
	local progress = self:getProgress()
	eventBar.progress = progress
	if (level.onEventProgress) then
		level:onEventProgress(self, progress)
	end
end

function Event:onItemDone(item)
	if (self.itemsDone >= self:getNumItemsTodo()) then
		return self:eventWon()
	else
		local animName = "eventThanks"..event.itemsDone
		if (isTable(taskSystem:getAnimationDesc(animName))) then
			return taskSystem:createAnimation(animName, sequenceThanksParams)
		else
			return taskSystem:createAnimation("eventThanks", self.sequenceThanksParams)
		end
	end
end

function Event:itemDone(item)
	if (not item.done) then
		self.lastItemDone = item
		item.done = true
		self.itemsDone = self.itemsDone + 1

		if (item.obj.useCustomCursor ~= nil) then
			item.obj.useCustomCursor = false
		end
		if (self.hideItemOnDone) then
			self:hideItem(item)
		end
		if (item.obj.onEventItemDone) then
			item.obj:onEventItemDone()
		end

		if (self.scoreOneDone and self.scoreOneDone > 0) then
			local pos = self:getFloaterPosition()
			local text = string.format(ghf.getString("EVENT_SCORE_ONE_DONE"), self.scoreOneDone)
			level:handleScore(self.scoreOneDone,pos.x, pos.y, false, 0);
			effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, self.scoreOneDone, 0)
		end

		if (level.onEventStepDone) then
			level:onEventStepDone(self, item)
		end

		if (self.playDoneAnimation and self.doneAnimation) then
			if (item.obj) then
				item.obj:playAnimation(self.doneAnimation, self.doneAnimationCycles, self.doneAnimationFrameTime, self.doneAnimationWhenFinished)
			end
		end

		self:hideHintArrow()
		self:restoreItem(item)
		self:onProgress()

		SpriteExt.playSample(nil, "SOUND_EVENT_ITEM_COMPLETE")
		return self:onItemDone(item)
	end
end

function Event:restoreItem(item)
	if (item) then
		if (item.onRestoreItem) then
			item:onRestoreItem()
		end
		if (item.obj) then		
			item.obj.onWorkStart = item.old_onWorkStart
			item.obj.onWorkUpdate = item.old_onWorkUpdate
			item.obj.onWorkFinish = item.old_onWorkFinish
			item.obj.onWorkCancel = item.old_onWorkCancel
			item.obj.onClick = item.old_onClick
			item.obj.clickable = item.old_clickable
			item.obj.doHighlight = item.old_doHighlight
			item.obj.stationaryActive = item.old_stationaryActive
			item.obj.workAnimation = item.old_workAnimation
			item.obj.workDuration = item.old_workDuration
			item.obj.workPos = item.old_workPos				
		end
	end
	self:removeDecals(item)
end

function Event:restoreItems()
    for _,item in ipairs(self.items) do
		self:restoreItem(item)
	end
end

function Event:createProgressBar(item, initialProgress)
	if (not isTable(item.bar) and item.obj) then
		item.bar = level:add(blueprints[item.bar or self.bar or "progressbar_big"])
		
		if (item.obj.absBarPos) then
			item.bar.location = item.obj.absBarPos
		else
			item.bar.location = item.barPos or item.obj.barPos or self.barPos or {x=0,y=0}
			local pos = {x=item.bar.location.x + item.obj.x, y=item.bar.location.y + item.obj.y}
			item.bar.location = pos
		end	
		
		item.bar.floor = item.obj.barFloor or item.obj.worldFloor-item.bar.worldY+1
		item.bar.alpha = 0
		item.bar:setProgress(initialProgress or 0)
	end
end

function Event:showProgressBar(item)
	if (item.bar) then
		item.bar:animate({alpha=1, duration=self.barFadeDuration})
	end
end

function Event:stopBlinkBar(item)
	if (item.bar and item.bar.blink) then
		ghf.cancelAnimation(item.bar.blink)
		item.bar.blink = nil
		item.bar.alpha = 1
	end
end

function Event:blinkBar(item, fadeTime, forceNew)
	if (item.bar) then
		if (item.bar.blink and not forceNew) then
			return --' Already blinking
		end
		self:stopBlinkBar(item)
		fadeTime = fadeTime or 200
		item.bar.blink = item.bar:animate({blink={startAlpha=1,endAlpha=0.3,fadeIn=fadeTime,fadeOut=fadeTime}})
	end
end

function Event:removeDecals(item, fadeTime)
	if (item and item.bar) then
		self:stopBlinkBar(item)
		item.bar:animate({alpha=0, duration=fadeTime or self.barFadeDuration}, {remove=true})
		item.bar = nil
	end
end

function Event:determineItem(v)
	if (isNumber(v)) then
		return self.items[v]
	end

	if (isString(v)) then
		for _,item in ipairs(self.items) do
			if (item.name == v or (item.obj and item.obj.name == v)) then
				return item
			end
		end
		v = level:getSpriteExt(v) --' Try table...
	end

	if (isTable(v)) then
		for _,item in ipairs(self.items) do
			if (t == item or (item.obj and item.obj == v)) then
				return item
			end
		end

		return
			self:determineItem(v.params) or
			self:determineItem(v.meta) or
			self:determineItem(v.name)
	end
end

function Event:handleTrigger(trigger, data)
	if (trigger == "startEvent") then
		local anim = taskSystem:createAnimation("eventStart", self.eventStartParams or data)
		anim = anim or ghf.anim.animation()
		anim:atEnd({class="CallTask", func=function() self:onStart() end})

	elseif (trigger == "eventStep") then
		self:onStep()

	elseif (trigger == "endEvent") then
		self:onShiftFinish()

	elseif (trigger == "showEventItems") then
		self:showItems()

	elseif (trigger == "showEventItem") then
		self:showItem(self:determineItem(data))

	elseif (trigger == "showHint") then
		self:showHint(data)
	end
end

function Event:showHint(obj)
	if (isTable(self.hintDesc) or self.hintArrowDesc.hintId) then
		self:onShowHint(obj)
	end
end

function Event:onShowHint(obj)
	if (self.hintDesc) then
		self.hintDesc.subject = self.hintDesc.subject or obj
		hintManager:openHint(self.hintDesc)
	end
	if (self.hintArrowDesc.hintId) then
		self.hintArrowDesc.parent = self.hintArrowDesc.parent or obj
		hintManager:placeArrow(self.hintArrowDesc)
	end
end

function Event:hideHintArrow()
	if (self.hintArrowDesc.hintId) then
		hintManager:removeArrows(self.hintArrowDesc.hintId)
	end
end

function Event:getFloaterPosition()
	if (isTable(self.floaterPos)) then
		return self.floaterPos
	elseif (self.floatersFromHero and hero) then	
		return {x=hero.x + hero.floaterPosition.x, y=hero.y + hero.floaterPosition.y}
	elseif (self.lastItemDone and self.lastItemDone.obj) then
		local src = self.lastItemDone.obj
		if (src.floaterPosition) then
			return src.floaterPosition
		end
		return {x=src.left + src.width/2, y=src.top}			
	end
	error("Could not figure out a floater position.")
end

function Event:eventWon()
	if (self.scoreAllDone and self.scoreAllDone > 0) then
		local delay = (self.scoreOneDone and self.scoreOneDone > 0 and 400) or 0
		local pos = self:getFloaterPosition()
		local text = string.format(ghf.getString("EVENT_SCORE_ALL_DONE"), self.scoreAllDone)
		level:handleScore(self.scoreAllDone, pos.x, pos.y, false, 0);
		level.topLayer:animate({delay=200},{particle="EVENT_DONE_BLAST", ex=pos.x, ey=pos.y})
		effects.createScoreFloater(level.floaterLayer, text, pos.x, pos.y, self.scoreAllDone, delay)
	end

	self:hideHintArrow()
	self:restoreItems()
	self.won = true
	self.completed = true
	eventBar:deactivate()
	if (self.hideItemsOnWin or self.hideItemsAtEnd) then
		self:hideItems()
	end

	SpriteExt.playSample(nil, "SOUND_EVENT_COMPLETE")

	return taskSystem:createAnimation("eventWon")
end

function Event:eventLost()
	self:hideHintArrow()
	self:restoreItems()
	self.won = false
	self.completed = true
	eventBar:deactivate()
	if (self.hideItemsOnLose or self.hideItemsAtEnd) then
		self:hideItems()
	end
	return taskSystem:createAnimation("eventLost")
end

function Event:onShiftFinish()
	self:hideHintArrow()
	self:restoreItems()
	eventBar:deactivate()

	local anim
	if (not self.completed) then
		if (self.won) then
			anim = self:eventWon()
		else
			anim = self:eventLost()
		end
	end
	if (self.won) then
		anim = taskSystem:createAnimation("eventWonAtEnd") or anim
	else
		anim = taskSystem:createAnimation("eventLostAtEnd") or anim
	end
	return anim
end

function Event:onMouseDown(x, y)
end

function Event:onMouseUp(x, y)
end

function Event:onMouseMoved(x, y, lButtonDown)
end

function Event:onMessageClick(command, id, object)
end
