--' ************************************ Start Story Task ************************************ '--
StartStoryTask = deriveLuaClass("StartStoryTask", "LuaTask")
function StartStoryTask:start()
	self.duration = self.duration or storyBarsDuration
	level:startStory()
	--' Let the duration determine the time.
	return self.wait
end

--' ************************************ End Story Task ************************************ '--
EndStoryTask = deriveLuaClass("EndStoryTask", "LuaTask")
function EndStoryTask:init(desc)
	EndStoryTask.superInit(self, desc)
	self.barsAnimated = self.barsAnimated ~= false
end

function EndStoryTask:start()
	self.duration = self.duration or storyBarsDuration
	self.barsAnimated = self.barsAnimated ~= false
	level:endStory(self.barsAnimated)
	--' Let the duration determine the time.
	return self.wait
end

--' ************************************ Remove Task ************************************ '--
RemoveTask = deriveLuaClass("RemoveTask", "LuaTask")
function RemoveTask:start()
	if (self.actor and not isString(self.actor.__cpp)) then
		if (self.actor.removeFromParent) then
			self.actor:removeFromParent()
		elseif (self.actor.remove) then
			self.actor:remove()
		end
	end
	return false
end

--' ************************************ Trigger Task ************************************ '--
TriggerTask = deriveLuaClass("TriggerTask", "LuaTask")
function TriggerTask:start()
	self.root = self
	level:fireTrigger(self.trigger or "", self)
	return false
end

--' ************************************ Set Task ************************************ '--
SetTask = deriveLuaClass("SetTask", "LuaTask")
function SetTask:start()
	assert(isTable(self.actor), "Must provide an actor")
	for k,v in pairs(self) do
		if (not table.contains({"class", "actor", "__cpp", "values"}, k)) then
			--'print(">>>> SetTask: "..tostring(k).." = "..tostring(v).."\n")
			self.actor[k] = v
		end
	end
	if (isString(self.values)) then
		local r = tryParse(self.values)
		if (isTable(r)) then
			self.values = r
		end
	end
	if (isTable(self.values)) then
		for k,v in pairs(self.values) do
			if (not table.contains({"class", "actor", "__cpp"}, k)) then
				--'print(">>>> SetTask: "..tostring(k).." = "..tostring(v).."\n")
				self.actor[k] = v
			end
		end
	end
	return false
end

--' ************************************ Call Task ************************************ '--
CallTask = deriveLuaClass("CallTask", "LuaTask")
function CallTask:start()
	local object = self.object or self.actor
	local func = self.func or self.call
	if (not isCallable(func)) then
		if (isTable(object)) then
			func = object[func]
		end
		if (not isCallable(func)) then
			func = self[func]
			if (not isCallable(func)) then
				func = _G[func]
			end
		end
	end

	self.params = self.params or {}
	assert(isCallable(func), "Uncallable in CallTask: "..tostring(func))
	if (object) then
		func(object, unpack(self.params))
	else
		func(unpack(self.params))
	end
	return false
end

--' ************************************ Play Sample Task ************************************ '--
PlaySoundTask = deriveLuaClass("PlaySoundTask", "LuaTask")
function PlaySoundTask:start()
	assert(self.sound, "Need to add sound")
	self.actor:playSample(self)
	return false
end

--' ************************************ Loop Sample Task ************************************ '--
LoopSoundTask = deriveLuaClass("LoopSoundTask", "LuaTask")
function LoopSoundTask:start()
	assert(self.sound, "Need to add sound")
	assert(self.loop, "Need to add loop")
	self.actor:loopSample(self.sound)
	return false
end

--' ************************************ Stop Loop Sample Task ************************************ '--
StopLoopSoundTask = deriveLuaClass("StopLoopSoundTask", "LuaTask")
function StopLoopSoundTask:start()
	assert(self.sound, "Need to add sound")
	assert(self.loop, "Need to add loop")
	self.actor:stopLoopSample()
	return false
end

--' ************************************ Set Position Task ************************************ '--
PositionTask = deriveLuaClass("PositionTask", "LuaTask")
function PositionTask:init(desc)
	PositionTask.superInit(self, desc)
	self.target = desc.target or desc.position or {x=500,y=500}

	if (not self.ease) then
		self.ease = math.linear
	elseif (not isCallable(self.ease)) then
		local options = {easeIn=math.jcurve, easeOut=math.rcurve, easeInOut=math.scurve}
		self.ease = options[self.ease] or math.linear
	end
end

function PositionTask:start()
	assert(self.target)--', "Target not found for this task. (actor='"..self.actor.name.."')")
	if (self.timer.duration <= 0) then
		self.actor.locationByTarget = self.target
		return false
	end
	self.startLocation = {x=self.actor.x, y=self.actor.y}
end

function PositionTask:update(time)
	local finished = PositionTask.super.update(self, time)
	local f = self.ease(self.timer.fraction)
	local endLocation = Level.targetToWorldPos(self.target)
	self.actor.location = {
		x=self.startLocation.x * (1-f) + endLocation.x * f,
		y=self.startLocation.y * (1-f) + endLocation.y * f,
	}
	return finished
end

function PositionTask:finish()
	self.actor.locationByTarget = self.target
end

--' ************************************ Emotion Task ************************************ '--
EmotionTask = deriveLuaClass("EmotionTask", "LuaTask")
function EmotionTask:init(desc)
	EmotionTask.superInit(self, desc)
	self.emotion = desc.emotion or ""
end

function EmotionTask:start()
	assert(self.emotion)
	self.actor.emotion = self.emotion
	return false
end

--' ************************************ Direction Task ************************************ '--
DirectionTask = deriveLuaClass("DirectionTask", "LuaTask")
function DirectionTask:init(desc)
	DirectionTask.superInit(self, desc)
	self.direction = desc.direction or "down"
end

function DirectionTask:start()
	assert(self.direction)
	self.actor.direction = self.direction
	return false
end

--' ************************************ Play Animation Task ************************************ '--
PlayAnimationTask = deriveLuaClass("PlayAnimationTask", "LuaTask")
function PlayAnimationTask:init(desc)
	PlayAnimationTask.superInit(self, desc)
	self.cycles = desc.cycles or 0
	self.whenFinished = desc.whenFinished or "stay"
	self.frameTime = desc.frameTime or 150
	self.wait = desc.wait or false
	if (self.playDuringStory == "default") then
		self.playDuringStory = nil
	else
		self.playDuringStory = self.playDuringStory or true
	end
end

function PlayAnimationTask:start()
--'print("["..tostring(self.actor.name).."] base: "..tostring(self.actor.baseAnimation).." ani: "..tostring(self.animation).." cyc: "..tostring(self.cycles).." fin: "..tostring(self.whenFinished).." time: "..tostring(self.frameTime))

	self.actor:playAnimation(self.animation, self.cycles, self.frameTime, self.whenFinished, self.reversed, self.playDuringStory)

	--' Here, we calculate the frame time based on the duration and the number of cycles
	if (self.frameTime < 0 and
		isNumber(self.duration) and
		self.duration > 0 and
		self.cycles > 0)
	then
		self.actor:matchFrameTimeTo(self.duration)
	end

	if (self.loopSound) then
		self.actor:loopSample(self.loopSound, self.volume, self.pan, self.pitch)
	end
end

function PlayAnimationTask:update(time)
	PlayAnimationTask.super.update(self, time)

	--' Wait until the actor has finished animating.
	if (not self.timer.infinite and not self.timer.expired) then
		return false
	end
	if (self.wait) then
		return not self.actor.isCyclingAnimation
	end
	return true
end

function PlayAnimationTask:skip()
	self.actor:finishCycleAnimation()
end

function PlayAnimationTask:finish()
	if (self.loopSound) then
		self.actor:stopLoopSample()
	end
end

--' ************************************ Message Task ************************************ '--
MessageTask = deriveLuaClass("MessageTask", "LuaTask")
function MessageTask:init(desc)
	MessageTask.superInit(self, desc)
	self.msgType = self.msgType or "message"
	self.tipType = self.tipType or "point"
	self.tipSide = self.tipSide or "center"
	self.maxLineCount = self.maxLineCount or 4
	self.keepOpen = self.keepOpen or false
	self.needsClick = self.needsClick ~= false
	self.labelWidth = self.labelWidth or 200
	self.task = self

	if (table.contains({ "ok", "yesno", "choiceList", "input" }, self.msgType)) then
		self:requireUserInput()
	end

	if (self.keepOpen) then
		self.skippable = true
	end
end

function MessageTask:start()

	if (self.msgType == "ok") then
		self.dlg = self.actor:showOkMessage(self)

	elseif (self.msgType == "yesno") then
		self.dlg = self.actor:showYesNoMessage(self)

	elseif (self.msgType == "choiceList") then
		self.dlg = self.actor:showChoiceMessage(self)

	elseif (self.msgType == "input") then
		self.dlg = self.actor:showInputMessage(self)

	else
		self.dlg = self.actor:showMessageDialog(self)
	end

	if (self.needsClick) then
		level:showClickToContinue(2000)
	end

	if (self.keepOpen) then
		return true
	end
end

function MessageTask:requireUserInput()
	self.needsUserInput = true
	self.needsClick = false
	self.skippable = false
	self.duration = -1
end

function MessageTask:update(time)
	local durationEnded = MessageTask.super.update(self, time)
	local shouldCloseDialog = durationEnded
	local dialogOpen = self.actor:findMessageDialog(self.dlg or self.textId) ~= nil

	if (self.needsUserInput or self.needsClick) then
		shouldCloseDialog = false--'not dialogOpen
	end

	if (shouldCloseDialog and not self.keepOpen) then
		self.actor:hideMessageDialog(self.dlg or self.textId)
	end

	local ret = not dialogOpen or self.keepOpen
	if (ret or shouldCloseDialog) then
		level:hideClickToContinue()
	end

	return ret
end

function MessageTask:skip()
	if (not self.keepOpen) then
		local dlg = self.actor:findMessageDialog(self.dlg or self.textId)
		if (dlg and dlg.onTaskSkip) then
			dlg:onTaskSkip()
		end
		self.actor:removeMessageDialog(self.dlg or self.textId)
		level:hideClickToContinue()
	end
	return true
end

function MessageTask:click()
	if (self.needsClick) then
		self.actor:hideMessageDialog(self.textId)
		return true
	end
	return false
end

--' ************************************ Remove Message Task ************************************ '--
RemoveMessageTask = deriveLuaClass("RemoveMessageTask", "LuaTask")
function RemoveMessageTask:start()
	assert(isString(self.textId), "A RemoveMessageTask needs a text ID to remove that message")
	if (self.actor.hideMessageDialog) then
		self.actor:hideMessageDialog(self.textId)
	end
	return self.wait ~= nil
end

function RemoveMessageTask:update(time)
	if (self.textId == nil) then
		return true --' Guard against infinite loop
	end
	local dlg
	if (self.actor.hideMessageDialog) then
		self.actor:hideMessageDialog(self.textId)
	end
	if (self.actor.findMessageDialog) then
		dlg = self.actor:findMessageDialog(self.textId)
	end
	return dlg == nil
end

function RemoveMessageTask:skip()
	if (self.actor.removeMessageDialog) then
		self.actor:removeMessageDialog(self.textId)
	end
end

--' ************************************ Overhead Task ************************************ '--
OverheadTask = deriveLuaClass("OverheadTask", "LuaTask")
function OverheadTask:start()
	local img = self.image
	if (self.section and self.section ~= "") then
		img = img..":"..self.section
	end
	self.actor:createOverhead(img, self.soundId, self.doPop, self.showDuringStory)
	self.duration = self.duration or 0
	return self.remove or self.duration > 0
end

function OverheadTask:finish()
	OverheadTask.super.finish(self)
	if (self.remove and self.actor.removeOverhead) then
		self.actor:removeOverhead(self.animated ~= false)
	end
end

--' ************************************ Remove Overhead Task ************************************ '--
RemoveOverheadTask = deriveLuaClass("RemoveOverheadTask", "LuaTask")
function RemoveOverheadTask:start()
	self.actor:removeOverhead(self.animated ~= false)
	return false
end

--' ************************************ Leave Station Task ************************************ '--
LeaveStationTask = deriveLuaClass("LeaveStationTask", "LuaTask")
function LeaveStationTask:start()
	self.actor:setState("walking to station")
	local station = self.station or self.actor.station
	if (station) then
		station:leave(self.actor)
	end
	return false
end

--' ************************************ Place Customer Task ************************************ '--
PlaceCustomerTask = deriveLuaClass("PlaceCustomerTask", "LuaTask")
function PlaceCustomerTask:start()
	self.target:onObjectDrop(self.actor)
	return false
end

--' ************************************ Fade Screen Task ************************************ '--
FadeScreenTask = deriveLuaClass("FadeScreenTask", "LuaTask")
function FadeScreenTask:start()
	self.fadeIn = self.fadeIn or 0
	self.fadeOut = self.fadeOut or 0
	self.fadeDelay = self.fadeDelay or 0
	self.width = self.width or 1024
	self.height = self.height or 768
	self.draw = self.draw or (self.color and "color") or "image"
	self.section = self.section or ""
	self.color = self.color or 0
	if (isString(self.color)) then
		self.color = tryParse(self.color)
	end

	local anims = {}
	local startAlpha = 1
	if (self.fadeIn > 0) then
		table.insert(anims, {alpha=1, duration=self.fadeIn})
		startAlpha = 0
	end
	if (self.fadeDelay > 0) then
		table.insert(anims, {delay=self.fadeDelay})
	end
	if (self.fadeOut > 0) then
		table.insert(anims, {alpha=0, duration=self.fadeOut})
	end

	local image, section
	if (isString(self.image) and self.image ~= "") then
		image = self.image
	end
	if (isString(self.section) and self.section ~= "") then
		section = self.section
	end

	self.cover = level.topLayer:newChild({class="Sprite",
		name="COVER_"..(image or ""),
		x=self.x, y=self.y, width=self.width, height=self.height,
		alpha=startAlpha,
		image=image, section=section,
		color=self.color, draw=self.draw,
	})

	if (self.remove ~= false) then
		table.insert(anims, {remove=true})
	end

	self.cover:animate(anims)

	self.duration = self.fadeIn + self.fadeDelay + self.fadeOut
	return self.wait
end

function FadeScreenTask:skip()
	if (self.remove ~= false) then
		if (isTable(self.cover) and self.cover.remove) then
			self.cover:remove()
		end
	end
end

--' ************************************ Arrow Task ************************************ '--
ArrowTask = deriveLuaClass("ArrowTask", "LuaTask")
function ArrowTask:start()
	assert(self.position)
	self.arrow = hintManager:placeArrow({type=self.arrowType, hintId=self.id,
		parent=self.actor, x=self.position.x, y=self.position.y})
	return self.wait or false
end

--' ************************************ Remove Arrow Task ************************************ '--
RemoveArrowTask = deriveLuaClass("RemoveArrowTask", "LuaTask")
function RemoveArrowTask:start()
	hintManager:removeArrows(self.id or "", self.animated ~= false)
end

function RemoveArrowTask:skip()
	if (isTable(self.arrow) and self.arrow.remove) then
		hintManager:removeArrows(self.arrow, false)
	end
end

--' ************************************ Play Music Task ************************************ '--
PlayMusicTask = deriveLuaClass("PlayMusicTask", "LuaTask")
function PlayMusicTask:start()
	--' Avalable music: "INTRO" "SELECTION" "GAME" "DELICIOUS"
	--' Avalable music moods: "HAPPY" "SAD" "ANGRY"
	self.quickFade = (self.quickFade ~= false) --' Guard against nil
	level:setMusicAndMood(
		self.music or level.music,
		self.mood or level.musicMood,
		self.quickFade)
	return false
end

--' ************************************ Fade Music Task ************************************ '--
FadeMusicTask = deriveLuaClass("FadeMusicTask", "LuaTask")
function FadeMusicTask:start()
	level:fadeMusic(self.volume, self.duration)
	return self.wait ~= true
end

--' ************************************ Sit Task ************************************ '--
SitTask = deriveLuaClass("SitTask", "LuaTask")
function SitTask:init(desc)
	SitTask.superInit(self, desc)
	assert(self.chair)
end

function SitTask:start()
	assert(self.actor)
	self.chair = level:getSpriteExt(self.chair)
	assert(self.chair)
	self.chair:place(self.actor)
	return false
end

--' ************************************ Stand Up Task ************************************ '--
StandUpTask = deriveLuaClass("StandUpTask", "LuaTask")
function StandUpTask:init(desc)
	SitTask.superInit(self, desc)
end

function StandUpTask:start()
	assert(self.actor)
	if (self.actor.station) then
		self.actor.station:standUp()
	end
	return false
end

--' ************************************ Base Animation Task ************************************ '--
BaseAnimationTask = deriveLuaClass("BaseAnimationTask", "LuaTask")
function BaseAnimationTask:init(desc)
	BaseAnimationTask.superInit(self, desc)
end

function BaseAnimationTask:start()
	self.actor.baseAnimation = self.baseAnimation or ""
	return false
end

--' ************************************ Play Default Animation Task ************************************ '--
PlayDefaultAnimationTask = deriveLuaClass("PlayDefaultAnimationTask", "LuaTask")
function PlayDefaultAnimationTask:start()
	self.actor:playDefaultAnimation()
	return false
end

--' ************************************ Walk To Stationary Position Task ************************************ '--
WalkToStationaryPositionTask = deriveLuaClass("WalkToStationaryPositionTask", "WalkToTask")
function WalkToStationaryPositionTask:init(desc)
	WalkToStationaryPositionTask.superInit(self, desc)
	local actor = level:getCharacter(desc.actor)
	if (actor and actor.waitPosition) then
		self.target = actor.waitPosition
		self.playDefaultOnFinish = true
	else
		assert(false, desc.actor.." : is not a stationary character")
	end
end

--' ************************************ Upgrades Walk To Task ************************************ '--
UpgradesWalkToTask = deriveLuaClass("UpgradesWalkToTask", "WalkToTask")
function UpgradesWalkToTask:init(desc)
	UpgradesWalkToTask.superInit(self, desc)
	self.exitPosition = desc.exitPosition or {x=0,y=0}
	local actor = level:getCharacter(desc.actor)
	if (actor.createdInAnimation) then
		self.target = desc.exitPosition
		actor:addTask({class="RemoveTask"})
	end
end
