--' ************************************ Level ************************************ '--
Level = getClass("Level")

function Level:fireTrigger(trigger, data)
	local handled = false
	if (not handled and self.event) then
		handled = self.event:handleTrigger(trigger, data)
	end
	if (not handled) then
		handled = taskSystem:handleTrigger(trigger, data)
	end
	if (not handled) then
		handled = self:handleTrigger(trigger, data)
	end
	return handled
end

function Level:onRemoveProductFromTray(product)
	if (self.event and self.event.onRemoveProductFromTray) then
		return self.event:onRemoveProductFromTray(product)
	end
end

function Level:onWorkStart(task)
	return false
end

function Level:onWorkFinish(task)
end

function Level:onAnimationStarted(anim)
	local musicMoodStart = musicMoodString(anim.musicMoodStart)
	anim.musicMoodEnd_original = self.musicMood
	if (isCallable(self.setMusicAndMood) and musicMoodStart) then
		self:setMusicAndMood({mood=musicMoodStart, quickFade=true})
	end

	if (self.selectedObject) then
		self.wasSelected = self.selectedObject.selected
		self.selectedObject.selected = false
	end
end

function Level:onAnimationFinished(anim)
	local musicMoodEnd = musicMoodString(anim.musicMoodEnd)
	if (not musicMoodEnd) then
		musicMoodEnd = anim.musicMoodEnd_original
	end
	if (isCallable(self.setMusicAndMood)) then
		self:setMusicAndMood({mood=musicMoodEnd, quickFade=true})
	end

	if (self.selectedObject) then
		self.selectedObject.selected = self.wasSelected
	end

	anim.resetEmotionsOnFinish = anim.resetEmotionsOnFinish or "default"
	if (anim.resetEmotionsOnFinish ~= "none") then
		local actors = { hero, cleaner, entertainer }
		if (anim.resetEmotionsOnFinish == "all") then
			actors = anim:getActors()
		elseif (anim.resetEmotionsOnFinish ~= "default") then
			actors = {level:getSpriteExt(anim.resetEmotionsOnFinish)}
		end
		for _,v in ipairs(actors) do
			if (v) then
				local emo = v.emotion
				if (isString(emo) and emo ~= "") then
					v.emotion = ""
				end
			end
		end
	end
end

function Level:resolveValue(v)
	if (v == "hero") then return hero end
	return nil
end

function Level:getDifficultyValue(t)
	return getInterpolatedEmbedValue(t, player.difficultyFrac, player.metaDifficultyFrac)
end

function getInterpolatedEmbedValue(t, frac, ...)
	if (isNumber(t)) then
		return t
	end
	assert(isTable(t))
	local min, max
	for k,_ in pairs(t) do
		assert(isNumber(k))
		if (min == nil or k < min) then min = k end
		if (max == nil or k > max) then max = k end
	end
	assert(isNumber(min))
	assert(isNumber(max))
	frac = (isNumber(frac) and frac) or 0
	local index = min + (max - min) * frac
	local v = t[index]
	if (v) then
		return getInterpolatedEmbedValue(v, unpack(arg))
	end
	local lo, hi
	for k,v in pairs(t) do
		if (lo == nil or (k < lo and k > index)) then lo = k end
		if (hi == nil or (k < hi and k > index)) then hi = k end
	end
	local vLo = getInterpolatedEmbedValue(t[lo], unpack(arg))
	local vHi = getInterpolatedEmbedValue(t[hi], unpack(arg))

	print("index: " .. index)
	print("vLo: " .. vLo)
	print("vHi: " .. vHi)

	return math.lerp(math.inv_lerp(index, lo, hi), vLo, vHi)
end

function getInterpolatedValue(t, frac)
	if (isNumber(t)) then
		return t
	end
	assert(isTable(t))
	-- for now only support 2 value's
	assert(#t == 2)
	return math.lerp(frac, t[1], t[2])
end

--' ************************************ Game Level ************************************ '--
DelLevel = getClass("DelLevel")

function DelLevel:init(desc)
	DelLevel.superInit(self, desc)

	self.closeDelay = self.closeDelay or 4000
	self.maxTableGroupSize = self.maxTableGroupSize or 0
	self:mergeProducts()
 	self:startGame(desc)

	self.placeChances = self.placeChances or { table = 50, counter = 50 }
end

function DelLevel:onLevelInitialized()
	self:traverse(function(_, node)
		if (node.onLevelInitialized and node ~= self) then
			node:onLevelInitialized()
		end
	end)
    self:onInitialized()

    if (hero) then
		hero.doWalkToLock = true
		hero.positionLock = hero.location
	end
    if (entertainer) then
		entertainer.doWalkToLock = true
		entertainer.positionLock = entertainer.location
	end
    if (cleaner) then
		cleaner.doWalkToLock = true
		cleaner.positionLock = cleaner.location
	end

	-- Hint stuff
	local obj = level:getSpriteExt("egg_basket")
	if (obj) then
		switchNeverShowHintClick(obj, "EggOrdered")
	end

	 MemoryEgg.initSource("rooster")
	 MemoryEgg.initSource("rake")
	 MemoryEgg.initSource("plant_disco")
	 MemoryEgg.initSource("wall_decorations_upgrade")
	 MemoryEgg.initSource("rocking_chair")
end

function DelLevel:onArrowRemoved(arrowId, object)
	if (arrowId == "Entertainer") then
		assert(object)
		assert(object.hintTable)
		hintManager:placeArrow({hintId="EntertainTable", parent=object.hintTable})
		switchHintArrowClick(object.hintTable, "EntertainTable")
	elseif (arrowId == "CompoundOrdered") then
		if (object.nextObject) then
			doCompoundArrow("compound_ordered_2", object.nextObject)
		end
	elseif (arrowId == "compound_ordered_2") then
		if (object.nextObject) then
			doCompoundArrow("compound_ordered_3", object.nextObject)
		end
	elseif (arrowId == "JamEmpty") then
		if (object.nextObject) then
			doCompoundArrow("jam_empty_2", object.nextObject)
		end
	elseif (arrowId == "PieReady") then
		if (object.nextObject) then
			doCompoundArrow("pie_2", object.nextObject)
		end
	elseif (arrowId == "Pizza") then
		if (object.nextObject) then
			doCompoundArrow("pizza_2", object.nextObject)
		end
	elseif (arrowId == "HoneyOrdered" and player:mayShowHint("HoneyReady")) then
		hintManager:openHint({hintId="HoneyReady", x=object.x+22, y=object.y-22})
	end
end

function doCompoundArrow(arrowId, object)
	assert(object)
	hintManager:placeArrow({hintId=arrowId, x=object.x + object.width/2, y=object.y + object.height/2, parent=level.dragLayer})
	switchHintArrowClick(object, arrowId)
end

function DelLevel:onHintClosed(hintId, object)
	if (hintId == "SendEntertainer") then
		if (entertainer) then
			hintManager:placeArrow({
				hintId="Entertainer",
				parent=level.dragLayer,
				x = entertainer.x,
				y = entertainer.y - entertainer.height,
			})
			entertainer.hintTable = object
			switchHintArrowClick(entertainer, "Entertainer")
		end

	elseif (hintId == "EggReady") then
		local object = level:getSpriteExt("mushroom")
		if (object) then
			hintManager:placeArrow({
				hintId="Mushroom",
				parent=level.dragLayer,
				x = object.x + (object.width / 2.0),
				y = object.y,
			})
			switchHintArrowClick(object, "Mushroom")
		end

	elseif (hintId == "PupNotHere") then
		hintManager:openHint({hintId="KeepSearching", showButton=false, closeOnClickAnywhere=true})

	elseif (hintId == "AllDecorations") then
		for i=1,4 do
			hintManager:removeArrows("AllDecorationsArrow"..i)
		end

	elseif (hintId == "AngelaSadBar") then
		local evelyn = level:getCharacter("evelyn")
		hintManager:openHint({hintId="DoLittleDance", showButton=false, subject=evelyn, target=evelyn})

	end
end

function DelLevel:getIngredientButton(ingredientName)
	return self.gameLayer:traverse(function(_, node)
		if (node.ingredientName == ingredientName) then
			return node
		end
	end)
end

function DelLevel:mergeProducts()
	if (self.productSettings) then
		for k, v in pairs(self.productSettings) do
			local productInfo = table.copy(self.defaultPoductSettings, true)
			table.copy_into(productInfo, v, true)
			self.productSettings[k] = productInfo
		end
	end
end

function DelLevel:onDeliver(order, owner)
	if (event and event.onDeliver) then
		local n = event:onDeliver(order, owner)
		if (n) then return n end
	end
end

function DelLevel:resolveValue(v)
	if (v == "cleaner") then return cleaner end
	if (v == "entertainer") then return entertainer end
	return DelLevel.super.resolveValue(self, v)
end

function DelLevel:createCharacter(desc)
	local classDesc = table.copy(self.characterTypes.default, true)
	local base = self.characterTypes[desc.base]
	table.copy_into(classDesc, base, true)
	table.copy_into(classDesc, desc, true)
	if (classDesc.location) then
		print(classDesc.location)
		print("location x:" .. classDesc.location.x)
		print("location y:" .. classDesc.location.y)
	end
	return self:add(classDesc)
end

function DelLevel:spawnCustomerGroup(meta, playSound)
	--' Build a class to spawn a customer
	--' The properties of the character can be defined in 3 places with increasing priority
	--' * classDesc (defaults)
	--' * self.customerList (overwrites defaults)
	--' * trigger metaData (overwrites defaults and self)

	--' Determine room
	local room = nil
	if (isNumber(self.areaId)) then
		room = self:getRoom(self.areaId)
	end

	--' Determine entrance
	local entrance = {x=512, y=300}
	local entrances = room and room.entrances
	if (entrances) then
		assert(#entrances > 0, "The room has no entrances!")
		entrance = entrances[math.Random(#entrances)]
		assert(entrance.x and entrance.y)
	end

	--' Build class description
	--' These properties might be overwritten below
	local classDesc = {}
	classDesc.class = "CustomerGroup"

	--' Get customer info
	--' If the trigger predefines a type, take that one...
	--' ... otherwise choose one at random,
	--' ... otherwise no info
	classDesc.customers = {}
	local customerInfo = meta and meta.customers
	if (isTable(customerInfo)) then
		for _,v in pairs(customerInfo) do
			local ci = (isString(v) and self.customerList and self.customerList[v]) or v
			if (isTable(ci)) then
				table.push_back(classDesc.customers, table.copy(ci))
			else
				print("Error! Customer info table could not be found: "..tostring(v).." ("..tostring(ci)..")")
			end
		end
	end
	meta.customers = nil

	--' Determine the place
	meta.place = meta.place or randomFromWeightedTable(self.placeChances)

	meta.size = meta.size or 0
	if (isNumber(meta.male)) then
		meta.size = meta.size + meta.male
	end
	if (isNumber(meta.female)) then
		meta.size = meta.size + meta.female
	end

	--' Create customer desctiptions
	if (self.customerList) then
		if (table.empty(classDesc.customers)) then
			local customerCount = 1
			if (meta.place == "table") then
				assert(self.maxTableGroupSize > 0, "No tables (or chairs) have been added to the level!")
				customerCount = meta.size ~= 0 and meta.size or math.Random(self.maxTableGroupSize-1) + 1
				customerCount = math.min(customerCount, self.maxTableGroupSize)
			elseif (meta.place == "disco") then
				assert(self.maxDiscoGroupSize and self.maxDiscoGroupSize > 0, "maxDiscoGroupSize needs to be set in the rest default file!")
				customerCount = meta.size ~= 0 and meta.size or math.Random(self.maxDiscoGroupSize-1) + 1
				customerCount = math.min(customerCount, self.maxDiscoGroupSize)
			end

			local tempCharacter = randomFromWeightedTable(self.customerList)

			assert(customerCount > 0)
			for i=1, customerCount do
				table.push_back(classDesc.customers, table.copy(tempCharacter))
			end
		else
			local tempCharacter = randomFromWeightedTable(self.customerList)
			for _,v in ipairs(classDesc.customers) do
				if (not isTable(v) or not v.base) then
					table.copy_into(v, tempCharacter, true, false)
print(tostring(v.easy))
				end
			end
		end
	end

	--' Determine the number customers can be used for generating orders
	local numCustomers = table.count(classDesc.customers)

	--' Set a unique id
	self.customerGroupId = (self.customerGroupId or 0) + 1
	if (classDesc.name == nil or classDesc.name == "") then
		classDesc.name = "CustomerGroup "..tostring(self.customerGroupId)
	end

	meta.male = meta.male or 0
	meta.female = meta.female or 0

	if (isTable(meta.customer)) then
		for _,v in ipairs(classDesc.customers) do
			table.copy_into(v, meta.customer)
print("dfsd"..tostring(v.easy))
		end
	end

	for k,v in pairs(classDesc.customers) do
		assert(v and v.base)
		assert(not v.__used)
		v.__used = true

		v.class = "Customer"
		v.x = entrance.x
		v.y = entrance.y

		assert(meta.male == nil or isNumber(meta.male), "'male' was not number (probably a string from XML)!")
		assert(meta.female == nil or isNumber(meta.female), "'female' was not number (probably a string from XML)!")
		assert(meta.size == nil or isNumber(meta.size), "'size' was not number (probably a string from XML)!")

		local chosenGender
		if (k <= meta.male) then
			chosenGender = "MALE"
		elseif (k <= meta.male + meta.female) then
			chosenGender = "FEMALE"
		else
			chosenGender = (math.Random(1000) < 500 and "MALE") or "FEMALE"
		end

		v.gender = v.gender or chosenGender
		if (meta.place == "table") then
			v.edible = true --' Table customers only order edible products
		end

		v.name = v.name or ("Customer "..self.customerGroupId.."-"..k)
	end

	--' Set the initial mood
	classDesc.startMood = self:calcValueI("startMood")

	--' TODO: create an new advanced random order generation system
	if (not isTable(meta.order)) then
		if (meta.place == "table") then
			meta.orderSize = meta.orderSize or (math.min(self.tableOrderSize, numCustomers))
			meta.orderSteps = meta.orderSteps or self.orderSteps
		elseif (meta.place == "disco") then
			if (self.getDiscoStep) then
				local step = self:getDiscoStep()
				meta.order = {class="Order"}
				meta.order.steps = step
			end
		else --' counter
			meta.orderSize = meta.orderSize or self.counterOrderSize
			meta.orderSteps = 1
		end
	end

	--' Merge with trigger
	--' This overwrites existing properties
	if (meta) then
		for k, v in pairs(meta) do
			local copy = true
			copy = copy and (k ~= "trigger")
			copy = copy and (k ~= "delay")
			copy = copy and (k ~= "type")
			copy = copy and (k ~= "customer")
			if (copy) then
				if (isTable(v)) then
					classDesc[k] = {}
					classDesc[k] = table.copy_into(classDesc[k], v, true, false, true)
				else
					classDesc[k] = v
				end
			end
		end
	end

	--' Add CustomerGroup
	local customerGroup = self.systemLayer:newChild(classDesc)
	customerGroup:onInitialized()
	return customerGroup
end

function DelLevel:createOrder(orderSteps, orderSize, customer)
	local products = self.possibleProducts
	local order = { class="Order", }
	order.steps = {}
	for stepsIdx=1, orderSteps do
		customer.hasHardProduct = false
		local step = table.push_back(order.steps, {})
		for ingIdx=1, orderSize do
			table.push_back(step, self:getNewProductFor(customer))
		end
	end
	return order
end

--' Get a product using the settings in the customer and the productInfo
function DelLevel:getNewProductFor(customer)
	local posibleProducts = self.possibleProducts
	local products = {}
	for _,p in ipairs(posibleProducts) do
		self.productSettings = self.productSettings or {}
		local product = self.productSettings[p]
		if (not product) then
			product = table.copy(level.defaultPoductSettings)
			self.productSettings[p] = product
		end

		local weight = 1
		if (not product.orderable) then
			weight = 0
		elseif (customer.hasHardProduct and product.hard) then
			weight = 0
		else
			for k, v in pairs(product) do
				if (isBoolean(customer[k])) then
					if (isBoolean(v) and customer[k] ~= v) then
						weight = 0
						break
					elseif (isNumber(v) and (k == "chance" or customer[k] == true)) then
						weight = weight * (v / 100) --' Percentage
					end
				end
			end
		end

		if (weight > 0) then
			products[p] = weight * 1000
		end
	end

	if (not table.empty(products)) then
		local resultProduct = randomFromWeightedTable(products)
		local settings = self.productSettings[resultProduct]
		if (settings and settings.hard) then
			customer.hasHardProduct = true
		end
		return resultProduct
	end

	--' Fallback
	if (not table.empty(self.possibleProducts)) then
		local orderableProducts = {}
		for _,v in ipairs(self.possibleProducts) do
			local settings = self.productSettings[v]
			if (settings and settings.orderable and not settings.hard) then
				table.push_back(orderableProducts, v)
			end
		end
		return orderableProducts[math.Random(#orderableProducts)]
	end

	return nil --' Error!
end

function DelLevel:handleTrigger(trigger, meta)
	if (trigger == "spawn" or trigger == "spawnCustomer") then
		self:spawnCustomerGroup(meta, meta.playSound)

	elseif (trigger == "object") then
		if (not meta.name) then
			DelLevel.super.handleTrigger(self, trigger, meta)

		else
			local object = level:getSpriteExt(meta.name)
			assert(object)
			if (isCallable(object.handleTrigger)) then
				object:handleTrigger(meta.type, meta)
			end
		end

	elseif (trigger == "startShift") then
		self:shiftStart()

	elseif (trigger == "finishShift") then
		self:shiftFinish()
		
	elseif (trigger == "finishGame") then
		self:gameFinished()
	
	else
		DelLevel.super.handleTrigger(self, trigger, meta)
	end
end

function DelLevel:createPostShiftAnimation()
	local doPostShiftAnimation = true
	if (self.isGamePlayLevel) then
		doPostShiftAnimation =
			(level.score >= level.targetScores[player.room][player.shift].target)
	end

	local anim
	if (event) then
		anim = event:onShiftFinish()
	end
	anim = anim or ghf.anim.animation()
	anim:atEnd({class="CallTask", func=function()
		if (doPostShiftAnimation) then
			local anim2 = taskSystem:createAnimation("postShift")
			if (anim2) then
				anim2:atEnd({class="CallTask", func=function() self:onShiftFinishEnd() end})
				return anim2
			end
		end
		self:onShiftFinish()
	end})
	return anim
end

function DelLevel:onCancelChain()
end

function DelLevel:debugHeroWalkTo(x, y)
	hero:addTask({class="WalkToTask", target={x=x, y=y}, playDefaultOnFinish=true})
end
