--
-- this is a batterbowl product
-- it moves to an appliance to get backed and when it's ready
-- it can be clicked again to move to a product stock
--
-- it's the first Lua implemented product,
-- so it contains extensive comments

class 'grillProduct' (Product)

--
-- additional variables are declared here
-- when you add an extra variable, don't forget it in the
-- copy constructor!
grillProduct.timer = 0
grillProduct.timerSuspended = true
grillProduct.busy = false
grillProduct.appliance = "rest0_grill"
grillProduct.workAnimation = "WORK_LEFT"
grillProduct.burnOffsetY = 19 -- use this to tweak how much the burn animation moves up/down
grillProduct.playing = false
grillProduct.grillSound = "SOUND_SKILLETING_LOOP"

function grillProduct:__init(root) super(root)

	-- C++ product has two constructor:
	-- a const std::string& which is the baseName of the product
	-- a const Product& which is the copy constructor
	-- 
	-- in this Lua constructor, we need to do explicit type checking
	-- on the 'root' parameter (which, as said, can be either a 
	-- string or a source product), so we know when the copy 
	-- constructor gets called. if the copy constructor is called
	-- all lua variables need to be copied over as well

	if (type(root) == "userdata") then
		-- called constructor (const Product&)
		-- this is the copy constructor
		-- copy over all lua values
		self.timer = root.timer
		self.timerSuspended = root.timerSuspended
		self.busy = root.busy
		self.appliance = root.appliance
		self.burnOffsetY = root.burnOffsetY
		self.playing = root.playing
		self.grillSound = root.grillSound
	end	
end

--
-- called when the product is first created
-- note that it's _not_ called when the copy constructor is used.
function grillProduct:onCreate()
	self:setState("RAW")
end

--
-- handle the balloon
function grillProduct:onClick()	
    if (self:getState() == "RAW") then
        self.timerSuspended = true
		Product.onClick(self)	
    elseif (self:getState() == "READY") then
		self.timerSuspended = true		
		if (self:isQueuableToTray()) then
			if (not self.busy) then		
				self:queue()	
				self.busy = true;			
				Product.onClick(self)					
			end
		end
	end		
end

--
-- called when the product enters a new state
function grillProduct:onStateEnter(state)
	if (self:getState() == "RAW") then
		self:setWorkAnimation(self.workAnimation)
		self:animate("PROD_BUT_" .. self:getBase() .. "_RAW", 0)
	
	elseif (self:getState() == "BAKING") then

		self:setClickable(false)
		--self:animate("PROD_BUT_BATTER", 0)
		self:doGrillSound(true)
		self.timer = 1500
		self.timerSuspended = false
	
	elseif (self:getState() == "READY") then
		self:blink()
		self:setClickable(true)
		self:animate("PROD_BUT_" .. self:getBase(), 0)
		self:playSound("SOUND_FRYER_READY")		
		self:setWorkTime(0)
		self:setWorkAnimation(self.workAnimation)
		self.timer = 6000
		self.timerSuspended = false
		
		hm:openHint(getHintIndex("HINT_KEBAB_FINISHED"), self:getScreenX() + (self:getWidth() / 2), self:getScreenY())
		
	elseif (self:getState() == "BURNING") then
		self:setClickable(false)
		self:clearOverhead()
		self:animate("PROD_BUT_" .. self:getBase(), 0) -- Temp
		self:doGrillSound(false)
		self:playSound("SOUND_BURGER_EGG_BURNS")
		self:animate("PROD_ANIM_" .. self:getBase() .. "_BURN", 1, 18)
		self:setY(self:getY() - self.burnOffsetY)
		self.timer = 1000
		self.timerSuspended = false
		
	end
end

--
-- called when the product is completely received by an iProductReceiver
function grillProduct:onReceiveComplete()
	if (self:getState() == "RAW") then
		self:setState("BAKING")
	end
end

--
-- called when the worker starts to work on this product
function grillProduct:onWorkStart(task)

	if (self:isDestroyed()) then
		warn("grillProduct:onWorkStart - is destroyed")
		return false
		
	elseif (self:getState() == "RAW") then
        self.timerSuspended = false
		if (getAppliance(self.appliance):hasSlotsOpen()) then
			return Product.onWorkStart(self, task)
		else
			return false
		end
	elseif (self:getState() == "READY") then
	
		if (self:isQueuedOnTray()) then
			Product.onWorkStart(self, task)		
			return true		
		else
			self.busy = false
			tray:clearInvisible()
			self.timerSuspended = false
			return false
		end

	end

	warn("grillProduct:onWorkStart - invalid state: " .. self:getState())
	return false	
	
end

-- called when a person completed work on a product
function grillProduct:onWorkComplete(task)
	if (self:getState() == "RAW") then
		self:moveToAppliance(getAppliance(self.appliance))
		
	elseif (self:getState() == "READY") then
		Product.onWorkComplete(self, task)
		self:moveToTray()
		self:setDestroyed(true)
		self:doGrillSound(false)		
		-- check trophy '500 grill procucts made'
		tm:checkGrillProductsMade(1)
	end
	
	self.busy = false
end

-- called each tick
-- useful for timers
function grillProduct:onTick(time)

	if (not scene:isStartingMovie() and not scene:isPlayingMovie()) then
		if (not self.timerSuspended) then
			if (self:getState() == "BAKING") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("READY")				
				end
					
			elseif (self:getState() == "READY") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("BURNING")					
				end	
			end	
		end
	end
	
	if (not self.timerSuspended) then
		if (self:getState() == "BURNING") then
			self.timer = self.timer - time
			if (self.timer <= 0) then
				self:setY(self:getY() + self.burnOffsetY)
				self:setDestroyed(true)
				self:setVisible(false)
				
			end	
		end
	end

end

function grillProduct:onSlotEnter(slot)

	if (slot:getName() == "slot1") then
		--self:setWorkOffsetX(-1)
	
	elseif (slot:getName() == "slot2") then
		--self:setWorkOffsetX(2)
	
	else
		warn("grillProduct:onSlotEnter - invalid slot name")
		
	end

end

function grillProduct:doGrillSound(play)

	if (play) then
		if (not self.playing) then
			self:playSound(self.grillSound, true)			
			self.playing = true
		end
	else
		if (self.playing) then
			self:stopSound(self.grillSound)
			self.playing = false
		end
	end

end