//                  Copyright (c) 2008, Lexmark International, Inc.
//      THIS IS AN UNPUBLISHED WORK CONTAINING CONFIDENTIAL AND PROPRIETARY
//         INFORMATION WHICH IS THE PROPERTY OF LEXMARK INTERNATIONAL, INC.
//    ANY DISCLOSURE, USE, OR REPRODUCTION, WITHOUT WRITTEN AUTHORIZATION FROM
//               LEXMARK INTERNATIONAL, INC., IS STRICTLY PROHIBITED


////////////////////////////////////////////////////////////////////////////////
// Control behaviors
////////////////////////////////////////////////////////////////////////////////

/**
* Mouseover handler for 'really big button's. Changes the button's CSS class to 
* add the 'rbb-hover' class.
*
* @param   id  The id of the button that was moused over.
*/
function onRbbMouseOver(id)
{
    var elem = document.getElementById(id);
    if( !window.event.srcElement.contains(window.event.fromElement) )
    {
        addClassName(elem, 'rbb-hover');
    }
}

/**
* Mouseout handler for 'really big button's. Changes the button's CSS class to 
* remove the 'rbb-hover' class.
*
* @param   id  The id of the button that was moused out of.
*/
function onRbbMouseOut(id)
{
    var elem = document.getElementById(id);
    if( !window.event.toElement || !elem.contains(window.event.toElement) )
    {
        removeClassName(elem, 'rbb-hover');
    }
}

/**
* Mouseover handler for tab headers. Changes the tab header's CSS class to add 
* the 'tab-hover' class, unless the tab is already the active tab.
*
* @param   id  The id of the tab that was moused over.
*/
function onTabMouseOver(id)
{
    var elem = document.getElementById(id);
    if( window.event.srcElement === elem && 
        !window.event.srcElement.contains(window.event.fromElement) )
    {
        // The active tab does not have a mouseover effect.
        if(!hasClassName(elem, 'tab-active'))
        {
            addClassName(elem, 'tab-hover');
        }
    }
}

/**
* Mouseout handler for tab headers. Changes the tab header's CSS class to remove 
* the 'tab-hover' class.
*
* @param   id  The id of the tab that was moused out of.
*/
function onTabMouseOut(id)
{
    var elem = document.getElementById(id);
    if( !window.event.toElement || !elem.contains(window.event.toElement) )
    {
        removeClassName(elem, 'tab-hover');
    }
}

/**
* Click handler for tabs. Activates the clicked tab.
*
* @param   id  The id of the tab that was clicked.
*/
function onTabClick(id)
{
    activateTab(id);
}

/**
* Shows a menu as a dropdown from a really big button.
*
* @param   id  The id of the button whose menu will be shown.
*/
function showMenu(id)
{
    var menu = document.getElementById('menu' + id);      
    var btn = document.getElementById(id);

    // Strip the hover effect from the button
    removeClassName(btn, 'rbb-hover');

    // Find the button's exact position and use it is a starting point to 
    // calculate the proper position for the menu.
    var offset = findAbsolutePos(btn);
    menu.style.left = offset.X;
    menu.style.top = btn.offsetHeight + offset.Y;

    // Make the menu visible
    menu.style.display = 'block';

    // Capture the mouse input, while allowing events to continue occurring on 
    // the child elements of the menu.
    if(!isIe5())
    {
        menu.setCapture(false);
    }
    else
    {
        menu.setCapture();
        
        // Ensure we only attach these events once. The first time through, 
        // 'menu' will not have an eventsInitialized property. For the purposes
        // of this test, not having the property is equivalent to the property
        // being false
        if(!menu.eventsInitialized)
        {
            menu.eventsInitialized = true;
            
            // Create a simple function to dispatch the event within the menu.
            // This will be added as a handler for any events that need to be
            // dispatched.
            var doDispatch = function()
            {
                // Due to the nature of javascript closures, 'menu' will 
                // continue to exist and contain the proper value after the
                // 'showMenu' method completes.
                dispatchEvent(menu);
            };
            
            // IE5's setCapture() behaves differently than newer versions. Work 
            // around this by dispatching the events manually.
            menu.attachEvent('onmouseover', doDispatch);
            menu.attachEvent('onmouseout', doDispatch);
            menu.attachEvent('onclick', doDispatch);
        }
    }
}

/**
* Hides a menu that is currently being shown as a dropdown from a really big 
* button.
*
* @param   id  The id of the menu that will be hidden.
*/
function hideMenu(id)
{
    var menu = document.getElementById(id);
    menu.style.display = 'none';
}

/**
* Click handler for the menu (NOT the menu items). Listens to all clicks that 
* occur while the menu is open. Releases the mouse capture.
*
* @param   id  The id of the menu.
*/
function onMenuClick(id)
{
    var menu = document.getElementById(id);
    menu.releaseCapture();
}

/**
* Handler for the 'losecapture' event. This event indicates that the menu has
* lost mouse capture. Closes the menu.
*
* @param   id  The id of the menu.
*/
function onMenuCaptureLost(id)
{
    hideMenu(id);
}

/**
* Mouseover handler for menu items. Adds a highlighted effect.
*
* @param   id  The id of the menu item that was moused over.
*/
function onMenuItemMouseOver(id)
{
    var elem = document.getElementById(id);
    addClassName(elem, 'menu-item-highlight');
    removeClassName(elem.childNodes[0], 'menu-item-icon-inactive');
}

/**
* Mouseout handler for menu items. Removes the highlighted effect.
*
* @param   id  The id of the menu item that was moused out of.
*/
function onMenuItemMouseOut(id)
{
    var elem = document.getElementById(id);
    removeClassName(elem, 'menu-item-highlight');
    addClassName(elem.childNodes[0], 'menu-item-icon-inactive');
}

////////////////////////////////////////////////////////////////////////////////
// App behaviors.
////////////////////////////////////////////////////////////////////////////////

/**
* Click handler for all tasks, regardless of presentation. 
*
* @param   actionID     The ID of the task being clicked.
*/
function onTaskClick(actionID)
{
    // Use try/catch block in case there's a Dispatch issue
    try
    {
        if(actionID == "CheckForFwUpdates")
            window.external.doActionAsync(actionID);
        else
            window.external.doAction(actionID);
    }
    catch(e) {}
}

/**
* Handler for changing the selected printer. 
*
* @param   printerID    The ID of the printer being selected.
*/
function onPrinterSelected(printerID)
{
    // Use try/catch block in case there's a Dispatch issue
    try
    {
        window.external.setSelectedPrinter(printerID);
    }
    catch(e) {}
}

/**
* Activates a tab. Makes the tab the active tab, and shows the associated tab 
* body. It does this by changing the tab's CSS class to add the 'tab-active' 
* class, removing the 'tab-active' and 'tab-hover' class from all tabs in the 
* same group, hiding the all tab bodies, and showing the active tab body.
*
* @param   id  The id of the tab to activate.
*/
function activateTab(id)
{
    var tab = document.getElementById(id);
    var tabBody = document.getElementById('tab' + id);

    // Make all tabs inactive / not hovered
    for(var i=0; i < tab.parentNode.childNodes.length; i++)
    {
        var child = tab.parentNode.childNodes[i];
        removeClassName(child, 'tab-active');
        removeClassName(child, 'tab-hover');
    }
    // Activate the clicked tab
    addClassName(tab, 'tab-active');

    // Hide all the tab bodies
    for(var j=0; j < tabBody.parentNode.childNodes.length; j++)
    {
        var child = tabBody.parentNode.childNodes[j];
        child.style.display = 'none';
    }
    // Show the selected tab body
    tabBody.style.display = 'block';
}

////////////////////////////////////////////////////////////////////////////////
// Utilities
////////////////////////////////////////////////////////////////////////////////

/**
* Check for a CSS class assigned to an element
*
* @param    element     The element to test.
*
* @param    className   The name of the class to look for.
*
* @return   true, if the element's CSS class includes the specified className.
*/
function hasClassName(element, className)
{
    if(element.className.length > 0)
    {
        if(element.className === className)
        {
            return true;
        }

        var re = new RegExp("(^|\\s)" + className + "(\\s|$)");

        if(re.test(element.className))
        {
            return true;
        }
    }

    return false;
}

/**
* Adds a CSS class to an element
*
* @param    element     The element to modify.
*
* @param    className   The name of the class to add.
*
* @return   Returns the supplied element.
*/
function addClassName(element, className) 
{
    if(!hasClassName(element, className))
    {
        if(element.className.length > 0)
        {
            element.className += " ";
        }

        element.className += className;
    }

    return element;
}

/**
* Removes a CSS class from an element
*
* @param    element     The element to modify.
*
* @param    className   The name of the class to remove.
*
* @return   Returns the supplied element.
*/
function removeClassName(element, className) 
{
    var re = new RegExp("(^|\\s+)" + className + "(\\s+|$)");

    element.className = strip(element.className.replace(re, " "));

    return element;
}

/**
* Removes blanks from both ends of a string
*
* @param    src     The string to remove whitespace from.
*
* @return   Returns the supplied string, without any whitespace on either end.
*/
function strip(src)
{
    var s = src;

    s = s.replace(/^\s+/, "");
    s = s.replace(/\s+$/, "");

    return s;
}

/**
* Determines the absolute position of an element.
*
* @param    elem    The element whose position is to be determined
*
* @return   A new object, with the absolute position stored as X (horizontal) 
*           and Y (vertical).
*/
function findAbsolutePos(elem)
{
    var pos = 
        {
            X: elem.offsetLeft,
            Y: elem.offsetTop
        };
    
    while(elem.offsetParent)
    {
        elem = elem.offsetParent;
        pos.X += elem.offsetLeft;
        pos.Y += elem.offsetTop;
    }
    
    return pos; 
}

/**
* Determines if the browser is IE 5.0x 
*
* @return   true, if the current browser is IE 5.0x
*/
function isIe5()
{
    var verExtract = /MSIE (\d.\d+);/;
    var matches = verExtract.exec(navigator.appVersion);

    if(matches && matches[1])
    {
        var versionStr = matches[1];
        var ver = parseFloat(versionStr);
        if(ver < 5.5)
        {
            return true;
        }
    }
    
    return false;
}

/**
* Dispatches the current event to it's source element. The event is then bubbled 
* up to each containing element until the element with mouse capture is reached.
* This is a workaround for IE5, which does not automatically dispatch events to 
* the contents of a container that has captured the mouse input. 
*
* @param    container   The element that has captured the mouse input.
*/
function dispatchEvent(container)
{
    var srcElem = window.event.srcElement;

    // Only perform the dispatch to elements within the element that has 
    // captured the mouse input.
    if( container.contains(srcElem) )
    {
        var target = srcElem;

        // Bubble the event out until we hit this level.
        while(target && (target !== container))
        {
            // Build the handler name: "on" + "mouseover", for example.
            var eventName = "on" + event.type;
            
            // If the target has a handler registered for the specified event 
            // name, call it.
            if(target[eventName])
            {
                target[eventName]();
            }
            
            // Move up one level
            target = target.parentElement;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
// Ogre
////////////////////////////////////////////////////////////////////////////////
/**
* An array of parameterless functions that will, when called, cause a particular
* control element to display. These are used during Ogre testing to automate the
* display of menus and tabs.
*/
var ogreArray_ = [];

/**
* Registers a control for use with Ogre. 
*
* @param    id          The id of the control. This will be passed as a 
*                       parameter to the function indicated by funcName
*
* @param    funcName    The name of the function that will be called in order to
*                       show the specified control. 
*/
function registerOgreControl(id, funcName)
{
    // Construct a 0 param function that will cause the appropriate element to 
    // show.
    var func = function(){ window[funcName](id); };

    // Store the function for later use by ogre. (Array.push would be preferable
    // here, however it is not supported on IE5)
    ogreArray_[ogreArray_.length] = func;
}

/**
* Shows the UI element specified by the index.
*
* @param    index   The index into the ogreArray_ of the element that is to be 
*                   displayed.
*/
function showOgreControl(index)
{
    if(0 <= index && index < ogreArray_.length)
    {
        ogreArray_[index]();
    }
}