package software.bernie.geckolib3.renderers.geo;

import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.util.ResourceLocation;
import org.lwjgl.opengl.GL11;
import software.bernie.geckolib3.geo.render.built.GeoBone;
import software.bernie.geckolib3.geo.render.built.GeoCube;
import software.bernie.geckolib3.geo.render.built.GeoModel;
import software.bernie.geckolib3.geo.render.built.GeoQuad;
import software.bernie.geckolib3.geo.render.built.GeoVertex;
import software.bernie.geckolib3.model.provider.GeoModelProvider;
import software.bernie.geckolib3.util.MatrixStack;

import javax.vecmath.Vector3f;
import javax.vecmath.Vector4f;
import java.awt.*;

public interface IGeoRenderer<T>
{
	public static MatrixStack MATRIX_STACK = new MatrixStack();

	default void render(GeoModel model, T animatable, float partialTicks, float red, float green, float blue, float alpha)
	{
		GlStateManager.disableCull();
		GlStateManager.enableRescaleNormal();
		renderEarly(animatable, partialTicks, red, green, blue, alpha);

		renderLate(animatable, partialTicks, red, green, blue, alpha);

		BufferBuilder builder = Tessellator.getInstance().getBuffer();

		builder.begin(GL11.GL_QUADS, DefaultVertexFormats.POSITION_TEX_COLOR_NORMAL);

		//Render all top level bones
		for (GeoBone group : model.topLevelBones)
		{
			renderRecursively(builder, group, red, green, blue, alpha);
		}

		Tessellator.getInstance().draw();

		renderAfter(animatable, partialTicks, red, green, blue, alpha);
		GlStateManager.disableRescaleNormal();
		GlStateManager.enableCull();
	}

	default void renderRecursively(BufferBuilder builder, GeoBone bone, float red, float green, float blue, float alpha)
	{
		MATRIX_STACK.push();

		MATRIX_STACK.translate(bone);
		MATRIX_STACK.moveToPivot(bone);
		MATRIX_STACK.rotate(bone);
		MATRIX_STACK.scale(bone);
		MATRIX_STACK.moveBackFromPivot(bone);

		if (!bone.isHidden)
		{
			for (GeoCube cube : bone.childCubes)
			{
				MATRIX_STACK.push();
				GlStateManager.pushMatrix();
				renderCube(builder, cube, red, green, blue, alpha);
				GlStateManager.popMatrix();
				MATRIX_STACK.pop();
			}
			for (GeoBone childBone : bone.childBones)
			{
				renderRecursively(builder, childBone, red, green, blue, alpha);
			}
		}

		MATRIX_STACK.pop();
	}

	default void renderCube(BufferBuilder builder, GeoCube cube, float red, float green, float blue, float alpha)
	{
		MATRIX_STACK.moveToPivot(cube);
		MATRIX_STACK.rotate(cube);
		MATRIX_STACK.moveBackFromPivot(cube);

		for (GeoQuad quad : cube.quads)
		{
			if (quad == null) {
				continue;
			}
			Vector3f normal = new Vector3f(quad.normal.getX(), quad.normal.getY(), quad.normal.getZ());

			MATRIX_STACK.getNormalMatrix().transform(normal);

			/* Fix shading dark shading for flat cubes + compatibility wish Optifine shaders */
			if ((cube.size.y == 0 || cube.size.z == 0) && normal.getX() < 0)
			{
				normal.x *= -1;
			}
			if ((cube.size.x == 0 || cube.size.z == 0) && normal.getY() < 0)
			{
				normal.y *= -1;
			}
			if ((cube.size.x == 0 || cube.size.y == 0) && normal.getZ() < 0)
			{
				normal.z *= -1;
			}

			for (GeoVertex vertex : quad.vertices)
			{
				Vector4f vector4f = new Vector4f(vertex.position.getX(), vertex.position.getY(), vertex.position.getZ(), 1.0F);

				MATRIX_STACK.getModelMatrix().transform(vector4f);

				builder.pos(vector4f.getX(), vector4f.getY(), vector4f.getZ()).tex(vertex.textureU, vertex.textureV).color(red, green, blue, alpha).normal(normal.getX(), normal.getY(), normal.getZ()).endVertex();
			}
		}
	}

	GeoModelProvider getGeoModelProvider();

	ResourceLocation getTextureLocation(T instance);

	default void renderEarly(T animatable, float ticks, float red, float green, float blue, float partialTicks)
	{
	}

	default void renderLate(T animatable, float ticks, float red, float green, float blue, float partialTicks)
	{
	}

	default void renderAfter(T animatable, float ticks, float red, float green, float blue, float partialTicks)
	{
	}

	default Color getRenderColor(T animatable, float partialTicks)
	{
		return new Color(255, 255, 255, 255);
	}

	default Integer getUniqueID(T animatable)
	{
		return animatable.hashCode();
	}
}
