/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A position in the editor.
 */
var Position = (function () {
	function Position(lineNumber, column) {
		this.lineNumber = lineNumber;
		this.column = column;
	}
	/**
	 * Test if this position equals other position
	 */
	Position.prototype.equals = function (other) {
		return Position.equals(this, other);
	};
	/**
	 * Test if position `a` equals position `b`
	 */
	Position.equals = function (a, b) {
		if (!a && !b) {
			return true;
		}
		return (!!a &&
			!!b &&
			a.lineNumber === b.lineNumber &&
			a.column === b.column);
	};
	/**
	 * Test if this position is before other position.
	 * If the two positions are equal, the result will be false.
	 */
	Position.prototype.isBefore = function (other) {
		return Position.isBefore(this, other);
	};
	/**
	 * Test if position `a` is before position `b`.
	 * If the two positions are equal, the result will be false.
	 */
	Position.isBefore = function (a, b) {
		if (a.lineNumber < b.lineNumber) {
			return true;
		}
		if (b.lineNumber < a.lineNumber) {
			return false;
		}
		return a.column < b.column;
	};
	/**
	 * Test if this position is before other position.
	 * If the two positions are equal, the result will be true.
	 */
	Position.prototype.isBeforeOrEqual = function (other) {
		return Position.isBeforeOrEqual(this, other);
	};
	/**
	 * Test if position `a` is before position `b`.
	 * If the two positions are equal, the result will be true.
	 */
	Position.isBeforeOrEqual = function (a, b) {
		if (a.lineNumber < b.lineNumber) {
			return true;
		}
		if (b.lineNumber < a.lineNumber) {
			return false;
		}
		return a.column <= b.column;
	};
	/**
	 * A function that compares positions, useful for sorting
	 */
	Position.compare = function (a, b) {
		var aLineNumber = a.lineNumber | 0;
		var bLineNumber = b.lineNumber | 0;
		if (aLineNumber === bLineNumber) {
			var aColumn = a.column | 0;
			var bColumn = b.column | 0;
			return aColumn - bColumn;
		}
		return aLineNumber - bLineNumber;
	};
	/**
	 * Clone this position.
	 */
	Position.prototype.clone = function () {
		return new Position(this.lineNumber, this.column);
	};
	/**
	 * Convert to a human-readable representation.
	 */
	Position.prototype.toString = function () {
		return '(' + this.lineNumber + ',' + this.column + ')';
	};
	// ---
	/**
	 * Create a `Position` from an `IPosition`.
	 */
	Position.lift = function (pos) {
		return new Position(pos.lineNumber, pos.column);
	};
	/**
	 * Test if `obj` is an `IPosition`.
	 */
	Position.isIPosition = function (obj) {
		return (obj
			&& (typeof obj.lineNumber === 'number')
			&& (typeof obj.column === 'number'));
	};
	return Position;
}());
exports.Position = Position;
//# sourceMappingURL=position.js.map