{***************************************************************
 * Unit Name: GX_eSampleEditorExpert
 * Purpose  : To show one way of coding an Editor Expert
 *            This expert puts up a modal window and nothing else
 * Author   : Scott Mattes, smattes@erols.com
 ****************************************************************}

unit GX_eSampleEditorExpert;

{$I GX_CondDefine.inc}

interface

uses
  ToolIntf,
  GX_Experts, GX_EditorExpert,
  Windows, Messages, SysUtils, Classes, Controls, Forms, Dialogs, StdCtrls,
  ExtCtrls, ComCtrls;

type
  TGxSampleEditorExpert = class(TEditorExpert)
  private
    FData: string;
  public
    constructor Create; override;
    procedure Configure; override;
    procedure Execute; override;
    procedure GetHelpString(List: TStrings); override;
    procedure LoadSettings; override;
    procedure SaveSettings; override;
  end;

type
  TfmSampleEditorExpertConfig = class(TForm)
    btnCancel: TButton;
    btnOK: TButton;
    lblData: TLabel;
    edtData: TEdit;
    lblNote: TLabel;
  private
  end;

implementation

{$R *.DFM}

uses
  {$IFOPT D+}GX_DbugIntf,{$ENDIF}
  Menus, Registry,
  ToolsApi,
  GX_GenFunc, GX_GExperts, GX_ConfigurationInfo;

{ TGxSampleEditorExpert }

//*********************************************************
//    Name: TGxSampleEditorExpert.Configure
// Purpose: Shows the config form and if the user selects
//          OK then the settings are saved
//    Note: If you have no config items, delete this method
//          (and its declaration from the interface section)
//          and the declaration of the form in the interface
//          section, and remove the and set FHasConfigOptions
//          to False in the Create method
//*********************************************************
procedure TGxSampleEditorExpert.Configure;
begin
  with TfmSampleEditorExpertConfig.Create(nil) do
  try
    edtData.Text := FData;
    if ShowModal = mrOK then
    begin
      FData := edtData.Text;
      SaveSettings;
    end;
  finally
    Free;
  end;
end;

//*********************************************************
//    Name: TGxSampleEditorExpert.Create
// Purpose: Sets up basic information about your editor expert
//   Notes: The value of ShortCut is touchy, use the ShortCut
//          button on the Editor Experts tab of menu item
//          GExperts/GExperts Configuration... on an existing
//          editor expert to see if you can use a specific
//          combination for your expert.
//*********************************************************
constructor TGxSampleEditorExpert.Create;
begin
  inherited Create;
  // The default shortcut key to activate your editor expert
  ShortCut := scCtrl + scAlt + Ord('?');
  // The expert name that appears in Editor Experts box on the Editor tab
  // of menu item GExperts/GExperts Configuration...
  FName := 'Sample Editor Expert';
  // Set to False if you don't have a need for any configuration settings
  FHasConfigOptions := True;
  // Set default values for any data
  FData := 'editor data';
  // LoadSettings is called automatically for the expert upon creation
end;

//*********************************************************
//    Name: TGxSampleEditorExpert.Execute
// Purpose: Called when your hot-key is pressed, this is
//          where you code what the expert should do
//    Note: A real good resource for how to do stuff in
//          the editor or the other parts of the IDE, get
//          Ray Lischner's Hidden Paths of Delphi 3. Next
//          best would be to see the online info on his
//          web site, www.tempest-sw.com
//*********************************************************
procedure TGxSampleEditorExpert.Execute;
begin
  ShowMessage('This is where your editor expert would do something useful.');
end;

//*********************************************************
//    Name: TGxSampleEditorExpert.GetHelpString
// Purpose: To provide your text on what this editor expert
//          does to the expert description box on the Editor
//          Experts tab on menu item GExperts/GExperts Configuration...
//*********************************************************
procedure TGxSampleEditorExpert.GetHelpString(List: TStrings);
resourcestring
  SSampleEditorExpertHelp =
    'This is the text that will appear in the explanation box on the ' +
    'Editor tab of menu item GExperts/GExperts Configuration...' +#13#10#13#10+
    'You can allow the text to wrap automatically, or you can force your ' +
    'own line breaks, or both.';
begin
  List.Text := SSampleEditorExpertHelp;
end;

procedure TGxSampleEditorExpert.LoadSettings;
begin
  inherited;
  // do not localize any of the below items
  with TRegIniFile.Create(BaseRegistryKey) do
  try
    ShortCut := ReadInteger('SampleEditorExpert', 'ShortCut', ShortCut);
    FData := ReadString('SampleEditorExpert', 'TestData', FData);
  finally
    Free;
  end;
end;

procedure TGxSampleEditorExpert.SaveSettings;
begin
  inherited;
  // do not localize any of the below items
  with TRegIniFile.Create(BaseRegistryKey) do
  try
    WriteString('SampleEditorExpert', 'TestData', FData);
    WriteInteger('SampleEditorExpert', 'ShortCut', ShortCut);
  finally
    Free;
  end;
end;

//*******************************************************************
// Purpose: Tells GExperts about the existance of this editor expert
//*******************************************************************
initialization
  RegisterEditorExpert(TGxSampleEditorExpert);
end.

