/* Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.activiti.dmn.engine.impl.cmd;

import java.io.InputStream;
import java.io.Serializable;

import org.activiti.dmn.engine.ActivitiDmnIllegalArgumentException;
import org.activiti.dmn.engine.impl.interceptor.Command;
import org.activiti.dmn.engine.impl.interceptor.CommandContext;
import org.activiti.dmn.engine.impl.persistence.entity.DecisionTableEntity;

/**
 * Gives access to a deployed decision table model, e.g., a DMN XML file, through a stream of bytes.
 * 
 * @author Tijs Rademakers
 */
public class GetDeploymentDmnResourceCmd implements Command<InputStream>, Serializable {

  private static final long serialVersionUID = 1L;
  protected String decisionTableId;

  public GetDeploymentDmnResourceCmd(String decisionTableId) {
    if (decisionTableId == null || decisionTableId.length() < 1) {
      throw new ActivitiDmnIllegalArgumentException("The decision table id is mandatory, but '" + decisionTableId + "' has been provided.");
    }
    this.decisionTableId = decisionTableId;
  }

  public InputStream execute(CommandContext commandContext) {
    DecisionTableEntity decisionTable = commandContext.getDmnEngineConfiguration().getDeploymentManager().findDeployedDecisionById(decisionTableId);
    String deploymentId = decisionTable.getDeploymentId();
    String resourceName = decisionTable.getResourceName();
    InputStream processModelStream = new GetDeploymentResourceCmd(deploymentId, resourceName).execute(commandContext);
    return processModelStream;
  }

}
