﻿// ScribVw.cpp: CScribbleView 类的实现
//
//这是 Microsoft 基础类 C++ 库的一部分。
//版权所有 (C) Microsoft Corporation。保留所有权利。
//
//此源代码仅用作 Microsoft 基础类
//参考及随库提供的相关
//电子文档的补充。
//有关 Microsoft 基础类产品的详细信息，
//请参见这些来源。

#include "stdafx.h"
#include "Scribble.h"

#include "ScribDoc.h"
#include "ScribVw.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CScribbleView

IMPLEMENT_DYNCREATE(CScribbleView, CScrollView)

BEGIN_MESSAGE_MAP(CScribbleView, CScrollView)
	//{{AFX_MSG_MAP(CScribbleView)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_COMMAND(ID_CANCEL_EDIT_SRVR, OnCancelEditSrvr)
	ON_WM_SIZE()
	//}}AFX_MSG_MAP
	//标准打印命令
	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CScribbleView 构造/析构

CScribbleView::CScribbleView()
{
	SetScrollSizes(MM_TEXT, CSize(0, 0));
}

CScribbleView::~CScribbleView()
{
}

BOOL CScribbleView::PreCreateWindow(CREATESTRUCT& cs)
{
	// TODO: 在此处通过修改
	//CREATESTRUCT cs 来修改 Window 类或样式

	return CView::PreCreateWindow(cs);
}

/////////////////////////////////////////////////////////////////////////////
// CScribbleView 绘图

void CScribbleView::OnDraw(CDC* pDC)
{
	CScribbleDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	//获得视图的无效矩形，或在打印的情况下，
	//获得打印机 dc 的剪辑区域。
	CRect rectClip;
	CRect rectStroke;
	pDC->GetClipBox(&rectClip);
	pDC->LPtoDP(&rectClip);
	rectClip.InflateRect(1, 1); //避免舍入为 nothing

	//注意: CScrollView::OnPaint() 会在调用 OnDraw() 前调整
	//视口原点，以反映
	//当前滚动位置。

	//视图将各个笔画的绘制委托给
	// CStroke::DrawStroke().
	CTypedPtrList<CObList,CStroke*>& strokeList = pDoc->m_strokeList;
	POSITION pos = strokeList.GetHeadPosition();
 	while (pos != NULL)
	{
		CStroke* pStroke = strokeList.GetNext(pos);
		rectStroke = pStroke->GetBoundingRect();
		pDC->LPtoDP(&rectStroke);
		rectStroke.InflateRect(1, 1); //避免舍入为 nothing
		if (!rectStroke.IntersectRect(&rectStroke, &rectClip))
			continue;
		pStroke->DrawStroke(pDC);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CScribbleView 打印

BOOL CScribbleView::OnPreparePrinting(CPrintInfo* pInfo)
{
	pInfo->SetMaxPage(2);   //文档长度为两页:
							//第一页为标题页
							//第二页为绘制页
	BOOL bRet = DoPreparePrinting(pInfo);	//默认准备
	pInfo->m_nNumPreviewPages = 2;  //一次预览两页
	//调用 DoPreparePrinting 后设置此值以重写
	//从 .INI 文件读取的值。
	return bRet;
}

void CScribbleView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: 打印前添加额外的初始化代码
}

void CScribbleView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: 打印后添加清理代码
}

/////////////////////////////////////////////////////////////////////////////
// CScribbleView 诊断

#ifdef _DEBUG
void CScribbleView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CScribbleView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CScribbleDoc* CScribbleView::GetDocument() //非调试版本是内联的
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CScribbleDoc)));
	return (CScribbleDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CScribbleView 消息处理程序

void CScribbleView::OnLButtonDown(UINT, CPoint point) 
{
    //在视图窗口中按鼠标按钮开始新笔画

	// CScrollView 更改视口原点和映射模式。
	//需要将点由设备坐标转换为
	//逻辑坐标，如存储在文档中。
	CClientDC dc(this);
	OnPrepareDC(&dc);
	dc.DPtoLP(&point);

	m_pStrokeCur = GetDocument()->NewStroke();
	//向新笔画添加第一个点
	m_pStrokeCur->m_pointArray.Add(point);

	SetCapture();       //捕获鼠标直到按钮弹起为止。
	m_ptPrev = point;   //充当用户拖动鼠标时下一个点的
						//LineTo() 的 MoveTo()
						//定位点。

	return;
}

void CScribbleView::OnLButtonUp(UINT, CPoint point) 
{
	//仅当用户当前正通过拖动捕获的鼠标
	//来绘制新笔画时，鼠标按钮弹起在“随意画图”应用程序中
	//才有意义。

	if (GetCapture() != this)
		return; //如果此窗口(视图)未捕获到鼠标，
				//则用户不在此窗口中绘制。

	CScribbleDoc* pDoc = GetDocument();

	CClientDC dc(this);

	// CScrollView 更改视口原点和映射模式。
	//需要将点由设备坐标转换为
	//逻辑坐标，如存储在文档中。
	OnPrepareDC(&dc);  //设置映射模式和视口原点
	dc.DPtoLP(&point);

	CPen* pOldPen = dc.SelectObject(pDoc->GetCurrentPen());
	dc.MoveTo(m_ptPrev);
	dc.LineTo(point);
	dc.SelectObject(pOldPen);
	m_pStrokeCur->m_pointArray.Add(point);

	//通知笔画项已完成向其添加点的操作。
	//这样，它才能完成对其边框的计算。
	m_pStrokeCur->FinishStroke();

	//通知其他视图已添加此笔画
	//以便其他视图可以在其工作区中使此笔画的区域
	//无效。
	pDoc->UpdateAllViews(this, 0L, m_pStrokeCur);

	ReleaseCapture();   //释放鼠标拖动开始时
						//建立的鼠标捕获。
	pDoc->NotifyChanged();
	return;
}

void CScribbleView::OnMouseMove(UINT, CPoint point) 
{
	//仅当用户当前正通过拖动捕获的鼠标
	//来绘制新笔画时，鼠标移动在“随意画图”应用程序中
	//才有意义。

	if (GetCapture() != this)
		return; //如果此窗口(视图)未捕获到鼠标，
				//则用户不在此窗口中绘制。

	CClientDC dc(this);
	// CScrollView 更改视口原点和映射模式。
	//需要将点由设备坐标转换为
	//逻辑坐标，如存储在文档中。
	OnPrepareDC(&dc);
	dc.DPtoLP(&point);

	m_pStrokeCur->m_pointArray.Add(point);

	//在鼠标拖动中上一个检测到的点到当前点之间
	//绘制一条线。
	CPen* pOldPen = dc.SelectObject(GetDocument()->GetCurrentPen());
	dc.MoveTo(m_ptPrev);
	dc.LineTo(point);
	dc.SelectObject(pOldPen);
	m_ptPrev = point;
	return;
}

void CScribbleView::OnUpdate(CView* /* pSender */, LPARAM /* lHint */, 
	CObject* pHint) 
{
	//文档已通知此视图某些数据已更改。

	if (pHint != NULL)
	{
		if (pHint->IsKindOf(RUNTIME_CLASS(CStroke)))
		{
			//提示是笔画已添加(或更改)。
			//因此，使其矩形无效。
			CStroke* pStroke = (CStroke*)pHint;
			CClientDC dc(this);
			OnPrepareDC(&dc);
			CRect rectInvalid = pStroke->GetBoundingRect();
			dc.LPtoDP(&rectInvalid);
			InvalidateRect(&rectInvalid);
			return;
		}
	}
	//我们不能解释提示，因此假定任何内容
	//都可能已更新。
	Invalidate(TRUE);
	return;
}

void CScribbleView::OnInitialUpdate() 
{
        ResyncScrollSizes();
	CScrollView::OnInitialUpdate();
}

void CScribbleView::ResyncScrollSizes()
{
	CClientDC dc(NULL);
	OnPrepareDC(&dc);
	CSize sizeDoc = GetDocument()->GetDocSize();
	dc.LPtoDP(&sizeDoc);
	SetScrollSizes(MM_TEXT, sizeDoc);
}

void CScribbleView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	if (pInfo->m_nCurPage == 1)  //第一页是标题页
	{
		PrintTitlePage(pDC, pInfo);
		return; //只打印第一页上的页标题
	}
	CString strHeader = GetDocument()->GetTitle();

	PrintPageHeader(pDC, pInfo, strHeader);
	// PrintPageHeader() 从 pInfo->m_rectDraw 中减去
	//标题所使用的量。

	pDC->SetWindowOrg(pInfo->m_rectDraw.left,-pInfo->m_rectDraw.top);

	//现在打印页的其余部分
	OnDraw(pDC);
}

void CScribbleView::PrintTitlePage(CDC* pDC, CPrintInfo* pInfo)
{
	//准备用于显示文件名的字体大小
	LOGFONT logFont;
	memset(&logFont, 0, sizeof(LOGFONT));
	logFont.lfHeight = 75;  //  3/4 英寸高，以 MM_LOENGLISH 为单位
							//(1/100 英寸)
	CFont font;
	CFont* pOldFont = NULL;
	if (font.CreateFontIndirect(&logFont))
		pOldFont = pDC->SelectObject(&font);

	//获取要在标题页上显示的文件名
	CString strPageTitle = GetDocument()->GetTitle();

	//在页顶部下 1 英寸处显示文件名，
	//水平居中
	pDC->SetTextAlign(TA_CENTER);
	pDC->TextOut(pInfo->m_rectDraw.right/2, -100, strPageTitle);

	if (pOldFont != NULL)
		pDC->SelectObject(pOldFont);
}

void CScribbleView::PrintPageHeader(CDC* pDC, CPrintInfo* pInfo,
	CString& strHeader)
{
	//打印由文档名和水平线
	//组成的页标题
	pDC->SetTextAlign(TA_LEFT);
	pDC->TextOut(0,-25, strHeader);  //下方 1/4 英寸

	//在标题下横跨页绘制一条线
	TEXTMETRIC textMetric;
	pDC->GetTextMetrics(&textMetric);
	int y = -35 - textMetric.tmHeight;          //文本下方 1/10 英寸处的线
	pDC->MoveTo(0, y);                          //从左边距
	pDC->LineTo(pInfo->m_rectDraw.right, y);    //到右边距

	//从绘制矩形中减去标题所使用的空间。
	y -= 25;    //线下方(上方)1/4 英寸处的空间
	pInfo->m_rectDraw.top += y;
}

//下面的命令处理程序提供标准键盘
//用户界面以取消就地编辑会话。此处，
//服务器(而不是容器)将导致停用。
void CScribbleView::OnCancelEditSrvr() 
{
	GetDocument()->OnDeactivateUI(FALSE);
}

void CScribbleView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	CScribbleDoc* pDoc = GetDocument();
	CScrollView::OnPrepareDC(pDC, pInfo);

	pDC->SetMapMode(MM_ANISOTROPIC);
	CSize sizeDoc = pDoc->GetDocSize();
	sizeDoc.cy = -sizeDoc.cy;
	pDC->SetWindowExt(sizeDoc);

	CSize sizeNum, sizeDenom;
	pDoc->GetZoomFactor(&sizeNum, &sizeDenom);

	int xLogPixPerInch = pDC->GetDeviceCaps(LOGPIXELSX);
	int yLogPixPerInch = pDC->GetDeviceCaps(LOGPIXELSY);

	long xExt = (long)sizeDoc.cx * xLogPixPerInch * sizeNum.cx;
	xExt /= 100 * (long)sizeDenom.cx;
	long yExt = (long)sizeDoc.cy * yLogPixPerInch * sizeNum.cy;
	yExt /= 100 * (long)sizeDenom.cy;
	pDC->SetViewportExt((int)xExt, (int)-yExt);
}

void CScribbleView::OnSize(UINT nType, int cx, int cy) 
{
        ResyncScrollSizes();        //确保滚动信息最新
	CScrollView::OnSize(nType, cx, cy);
}
