"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VueScriptCache = void 0;
const Parser_1 = require("htmlparser2/lib/Parser");
class VueScriptCache {
    constructor(ts_impl, getHostScriptSnapshot, getScriptVersion) {
        this.ts_impl = ts_impl;
        this.getHostScriptSnapshot = getHostScriptSnapshot;
        this.getScriptVersion = getScriptVersion;
        this.cache = new Map();
    }
    getScriptKind(fileName) {
        return this.getUpToDateInfo(fileName).kind;
    }
    getScriptSnapshot(fileName) {
        return this.getUpToDateInfo(fileName).snapshot;
    }
    getUpToDateInfo(fileName) {
        const fromCache = this.cache.get(fileName);
        const currentVersion = this.getScriptVersion(fileName);
        if ((fromCache === null || fromCache === void 0 ? void 0 : fromCache.version) === currentVersion) {
            return {
                kind: fromCache.kind,
                snapshot: fromCache.snapshot
            };
        }
        const snapshot = this.getHostScriptSnapshot(fileName);
        const result = this.parseVue(snapshot.getText(0, snapshot.getLength()));
        this.cache.set(fileName, Object.assign(Object.assign({}, result), { version: currentVersion }));
        return result;
    }
    parseVue(contents) {
        let lastIndex = 0;
        let level = 0;
        let result = "";
        let isScript = false;
        const ts_impl = this.ts_impl;
        let scriptKind = ts_impl.ScriptKind.JS;
        let inScriptSetup = false;
        let addedScriptSetupPrefix = false;
        let hadScriptSetup = false;
        let hadScriptNormal = false;
        let scriptSetupStartLoc = -1;
        let scriptSetupEndLoc = -1;
        const parser = new Parser_1.Parser({
            onopentag(name, attribs) {
                if (name === "script" && level === 0) {
                    isScript = true;
                    inScriptSetup = false;
                    for (let attr in attribs) {
                        if (attr.toLowerCase() == "lang") {
                            const extension = attribs[attr].toLowerCase();
                            switch (extension) {
                                case "jsx":
                                    if (scriptKind == ts_impl.ScriptKind.JS) {
                                        scriptKind = ts_impl.ScriptKind.JSX;
                                    }
                                    else {
                                        scriptKind = ts_impl.ScriptKind.TSX;
                                    }
                                    break;
                                case "ts":
                                    if (scriptKind == ts_impl.ScriptKind.JS) {
                                        scriptKind = ts_impl.ScriptKind.TS;
                                    }
                                    else if (scriptKind == ts_impl.ScriptKind.JSX) {
                                        scriptKind = ts_impl.ScriptKind.TSX;
                                    }
                                    break;
                                case "tsx":
                                    scriptKind = ts_impl.ScriptKind.TSX;
                                    break;
                            }
                        }
                        if (attr.toLowerCase() == "setup") {
                            inScriptSetup = true;
                            addedScriptSetupPrefix = false;
                            hadScriptSetup = true;
                        }
                    }
                    hadScriptNormal = hadScriptNormal || !inScriptSetup;
                }
                level++;
            },
            ontext(data) {
                if (isScript) {
                    const lineCount = contents.substring(lastIndex, parser.startIndex).split("\n").length - 1;
                    let charsCount = parser.startIndex - lastIndex - lineCount;
                    if (inScriptSetup && !addedScriptSetupPrefix) {
                        addedScriptSetupPrefix = true;
                        scriptSetupStartLoc = result.length;
                        result += ";(()=>{";
                        charsCount -= 7;
                    }
                    result += " ".repeat(charsCount) + "\n".repeat(lineCount) + data;
                    lastIndex = parser.endIndex + 1;
                }
            },
            onclosetag(name) {
                if (inScriptSetup) {
                    scriptSetupEndLoc = result.length;
                    result += "})();";
                    inScriptSetup = false;
                    lastIndex += 5;
                }
                isScript = false;
                level--;
            }
        }, {
            recognizeSelfClosing: true
        });
        parser.write(contents);
        parser.end();
        // Support empty <script> tag
        if (result.trim() === "") {
            result = "import componentDefinition from '*.vue'; export default componentDefinition;";
            scriptKind = ts_impl.ScriptKind.TS;
        }
        // Support <script setup> syntax
        else if (hadScriptSetup && !hadScriptNormal) {
            result = result + "; import __componentDefinition from '*.vue'; export default __componentDefinition;";
            // Remove wrapper for imports to work properly
            if (scriptSetupStartLoc >= 0) {
                result = result.substring(0, scriptSetupStartLoc) + " ".repeat(7) + result.substring(scriptSetupStartLoc + 7);
            }
            if (scriptSetupEndLoc >= 0) {
                result = result.substring(0, scriptSetupEndLoc) + " ".repeat(5) + result.substring(scriptSetupEndLoc + 5);
            }
        }
        else if (hadScriptSetup && hadScriptNormal) {
            // Add imports at the end of the file
            result += "\n;";
            const r = /import[^'"]*['"]([^'"]*)['"]/g;
            const fragmentToMatch = result.substring(scriptSetupStartLoc, scriptSetupEndLoc);
            let match;
            while ((match = r.exec(fragmentToMatch)) !== null) {
                result += `import "${match[1]}";\n`;
            }
        }
        const snapshot = ts_impl.ScriptSnapshot.fromString(result);
        // Allow retrieving script kind from snapshot
        snapshot.scriptKind = scriptKind;
        return {
            snapshot,
            kind: scriptKind
        };
    }
}
exports.VueScriptCache = VueScriptCache;
