"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VueScriptCache = void 0;
const Parser_1 = require("htmlparser2/lib/Parser");
class VueScriptCache {
    constructor(ts_impl, getHostScriptSnapshot, getScriptVersion) {
        this.ts_impl = ts_impl;
        this.getHostScriptSnapshot = getHostScriptSnapshot;
        this.getScriptVersion = getScriptVersion;
        this.cache = new Map();
    }
    getScriptKind(fileName) {
        return this.getUpToDateInfo(fileName).kind;
    }
    getScriptSnapshot(fileName) {
        return this.getUpToDateInfo(fileName).snapshot;
    }
    getUpToDateInfo(fileName) {
        const fromCache = this.cache.get(fileName);
        const currentVersion = this.getScriptVersion(fileName);
        if ((fromCache === null || fromCache === void 0 ? void 0 : fromCache.version) === currentVersion) {
            return {
                kind: fromCache.kind,
                snapshot: fromCache.snapshot
            };
        }
        const snapshot = this.getHostScriptSnapshot(fileName);
        const result = this.parseVue(snapshot.getText(0, snapshot.getLength()));
        this.cache.set(fileName, Object.assign(Object.assign({}, result), { version: currentVersion }));
        return result;
    }
    parseVue(contents) {
        let lastIndex = 0;
        let level = 0;
        let result = "";
        let isScript = false;
        const ts_impl = this.ts_impl;
        let scriptKind = ts_impl.ScriptKind.JS;
        let scriptSetup = false;
        const parser = new Parser_1.Parser({
            onopentag(name, attribs) {
                if (name === "script" && level === 0) {
                    isScript = true;
                    for (let attr in attribs) {
                        if (attr.toLowerCase() == "lang") {
                            const extension = attribs[attr].toLowerCase();
                            switch (extension) {
                                case "js":
                                    scriptKind = ts_impl.ScriptKind.JS;
                                    break;
                                case "jsx":
                                    scriptKind = ts_impl.ScriptKind.JSX;
                                    break;
                                case "ts":
                                    scriptKind = ts_impl.ScriptKind.TS;
                                    break;
                                case "tsx":
                                    scriptKind = ts_impl.ScriptKind.TSX;
                                    break;
                            }
                        }
                        if (attr.toLowerCase() == "setup") {
                            scriptSetup = true;
                        }
                    }
                }
                level++;
            },
            ontext(data) {
                if (isScript) {
                    const lineCount = contents.substring(lastIndex, parser.startIndex).split("\n").length - 1;
                    result += " ".repeat(parser.startIndex - lastIndex - lineCount) + "\n".repeat(lineCount) + data;
                    lastIndex = parser.endIndex + 1;
                }
            },
            onclosetag(name) {
                isScript = false;
                level--;
            }
        }, {
            recognizeSelfClosing: true
        });
        parser.write(contents);
        parser.end();
        // Support empty <script> tag
        if (result.trim() === "") {
            result = "import componentDefinition from '*.vue'; export default componentDefinition;";
            scriptKind = ts_impl.ScriptKind.TS;
        }
        // Support <script setup> syntax
        else if (scriptSetup) {
            result = result + "; import __componentDefinition from '*.vue'; export default __componentDefinition;";
        }
        const snapshot = ts_impl.ScriptSnapshot.fromString(result);
        // Allow to retrieve script kind from snapshot
        snapshot.scriptKind = scriptKind;
        return {
            snapshot,
            kind: scriptKind
        };
    }
}
exports.VueScriptCache = VueScriptCache;
