"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vueScriptCache_1 = require("./vueScriptCache");
let patched = false;
module.exports = function init({ typescript: ts_impl }) {
    if (!patched) {
        patched = true;
        // Don't output compilation results of `.vue` files. The DefaultSessionExtension#getFileWrite should actually be patched
        // to not populate the output list
        const _writeFile = ts_impl.sys.writeFile;
        ts_impl.sys.writeFile = function writeFilePatched(path, data, writeByteOrderMark) {
            if (path.endsWith(".vue.d.ts") || path.endsWith(".vue.js") || path.endsWith(".vue.js.map")) {
                return;
            }
            _writeFile(path, data, writeByteOrderMark);
        };
        // Detect whether script kind has changed and if so, drop the whole program and patch compiler host
        const _createProgram = ts_impl.createProgram;
        ts_impl.createProgram = function createProgram(rootNamesOrOptions) {
            let oldProgram;
            let compilerOptions;
            let compilerHost;
            if (ts_impl.isArray(rootNamesOrOptions)) {
                compilerOptions = arguments[1];
                compilerHost = arguments[2];
                oldProgram = arguments[3];
            }
            else {
                const options = rootNamesOrOptions;
                compilerOptions = options.options;
                compilerHost = options.host;
                oldProgram = options.oldProgram;
            }
            if (compilerHost !== undefined && oldProgram !== undefined) {
                let scriptKindChanged = new Set();
                for (let sourceFile of oldProgram.getSourceFiles()) {
                    if (sourceFile.fileName.endsWith(".vue")) {
                        try {
                            // Check if we can safely acquire source code
                            compilerHost.getSourceFileByPath(sourceFile.fileName, sourceFile.resolvedPath, compilerOptions.target, undefined, false);
                        }
                        catch (e) {
                            // TODO - maybe we could change script kind here to avoid leak below
                            scriptKindChanged.add(sourceFile.resolvedPath);
                            scriptKindChanged.add(sourceFile.fileName);
                        }
                    }
                }
                // Do not reuse old program structure if any of Vue scripts have changed it's script kind
                if (scriptKindChanged.size > 0) {
                    // TODO - forcing shouldCreateNewSourceFile is causing leaks in the document registry,
                    //        as documents with changed script kind are acquired again
                    // Patch compiler host to not fall into fail condition
                    const _getSourceFileByPath = compilerHost.getSourceFileByPath;
                    compilerHost.getSourceFileByPath = function (fileName, path, languageVersion, onError, shouldCreateNewSourceFile) {
                        // shouldCreateNewSourceFile
                        arguments[4] = arguments[4] || scriptKindChanged.has(path) || scriptKindChanged.has(fileName);
                        return _getSourceFileByPath.apply(compilerHost, arguments);
                    };
                    // Patch compiler host to not fall into fail condition
                    const _getSourceFile = compilerHost.getSourceFile;
                    compilerHost.getSourceFile = function (fileName, languageVersion, onError, shouldCreateNewSourceFile) {
                        // shouldCreateNewSourceFile
                        arguments[3] = arguments[3] || scriptKindChanged.has(fileName);
                        return _getSourceFile.apply(compilerHost, arguments);
                    };
                    // Drop old program
                    if (ts_impl.isArray(rootNamesOrOptions)) {
                        arguments[3] = undefined;
                    }
                    else {
                        rootNamesOrOptions.oldProgram = undefined;
                    }
                }
            }
            return _createProgram.apply(undefined, arguments);
        };
        // If script kind is changed for Vue file, we need to create a new source file
        const _updateLanguageServiceSourceFile = ts_impl.updateLanguageServiceSourceFile;
        ts_impl.updateLanguageServiceSourceFile = function (sourceFile, scriptSnapshot, version, textChangeRange, aggressiveChecks) {
            if (sourceFile.fileName.endsWith(".vue")
                && scriptSnapshot.scriptKind !== undefined
                && scriptSnapshot.scriptKind !== sourceFile.scriptKind) {
                return ts_impl.createLanguageServiceSourceFile(sourceFile.fileName, scriptSnapshot, sourceFile.languageVersion, version, /*setNodeParents*/ true, scriptSnapshot.scriptKind);
            }
            return _updateLanguageServiceSourceFile.apply(undefined, arguments);
        };
    }
    function myLoadWithLocalCache(names, containingFile, redirectedReference, loader) {
        if (names.length === 0) {
            return [];
        }
        const resolutions = [];
        const cache = new Map();
        for (const name of names) {
            let result;
            if (cache.has(name)) {
                result = cache.get(name);
            }
            else {
                cache.set(name, result = loader(name, containingFile, redirectedReference));
            }
            resolutions.push(result);
        }
        return resolutions;
    }
    return {
        create(info) {
            const pluginInfo = info;
            const tsLs = info.languageService;
            const tsLsHost = info.languageServiceHost;
            const project = info.project;
            const compilerHost = project;
            const getCanonicalFileName = ts_impl.sys.useCaseSensitiveFileNames ? toFileNameLowerCase : identity;
            // Allow resolve into Vue files
            const vue_sys = Object.assign(Object.assign({}, ts_impl.sys), { fileExists(path) {
                    if (path.endsWith(".vue.ts")
                        && ts_impl.sys.fileExists(path.substring(0, path.length - 3))) {
                        return true;
                    }
                    return ts_impl.sys.fileExists(path);
                } });
            let moduleResolutionCache = ts_impl.createModuleResolutionCache(project.getCurrentDirectory(), (x) => { var _a; return ((_a = compilerHost.getCanonicalFileName) !== null && _a !== void 0 ? _a : getCanonicalFileName)(x); }, project.getCompilerOptions());
            const loader = (moduleName, containingFile, redirectedReference) => {
                const tsResolvedModule = ts_impl.resolveModuleName(moduleName, containingFile, project.getCompilerOptions(), vue_sys, moduleResolutionCache, redirectedReference).resolvedModule;
                if (tsResolvedModule && tsResolvedModule.resolvedFileName.endsWith('.vue.ts')) {
                    const resolvedFileName = tsResolvedModule.resolvedFileName.slice(0, -3);
                    return {
                        resolvedFileName: resolvedFileName,
                        extension: ts_impl.Extension.Ts
                    };
                }
                return tsResolvedModule;
            };
            tsLsHost.resolveModuleNames = (moduleNames, containingFile, _reusedNames, redirectedReference) => (ts_impl.loadWithLocalCache || myLoadWithLocalCache)(moduleNames, containingFile, redirectedReference, loader);
            // Strip non-TS content from the script
            const _getScriptSnapshot = tsLsHost.getScriptSnapshot;
            const vueScriptCache = new vueScriptCache_1.VueScriptCache(ts_impl, fileName => _getScriptSnapshot.call(tsLsHost, fileName), fileName => tsLsHost.getScriptVersion(fileName));
            tsLsHost.getScriptSnapshot = function (fileName) {
                if (fileName.endsWith(".vue")) {
                    return vueScriptCache.getScriptSnapshot(fileName);
                }
                return _getScriptSnapshot.call(this, fileName);
            };
            // Provide script kind based on `lang` attribute
            const _getScriptKind = tsLsHost.getScriptKind;
            tsLsHost.getScriptKind = function (fileName) {
                if (fileName.endsWith(".vue")) {
                    return vueScriptCache.getScriptKind(fileName);
                }
                return _getScriptKind.call(this, fileName);
            };
            return tsLs;
        }
    };
};
/* Copied from TS compiler/core.ts */
function identity(x) { return x; }
function toLowerCase(x) { return x.toLowerCase(); }
const fileNameLowerCaseRegExp = /[^\u0130\u0131\u00DFa-z0-9\\/:\-_. ]+/g;
function toFileNameLowerCase(x) {
    return fileNameLowerCaseRegExp.test(x) ?
        x.replace(fileNameLowerCaseRegExp, toLowerCase) :
        x;
}
