import shlex
import traceback
from enum import Enum

import lldb
import lldb.formatters
from jb_declarative_formatters import *
from jb_declarative_formatters.parsers.type_name_parser import parse_type_name_template
from jb_declarative_formatters.types_storage import TypesStorage
from ._jb_lldb_utils import register_lldb_commands, make_absolute_name

g_types_storage = None

# store all lldb registered types
g_lldb_registered_summaries = set()
g_lldb_registered_synthetics = set()

# options
lldb.formatters.Logger._lldb_formatters_debug_level = 0
g_force_suppress_errors = False
g_max_num_children = 10000


class DiagnosticsLevel(Enum):
    DISABLED = 0
    ERRORS_ONLY = 1
    VERBOSE = 2


def set_diagnostics_level(level):
    global g_force_suppress_errors
    if level == DiagnosticsLevel.VERBOSE:
        lldb.formatters.Logger._lldb_formatters_debug_level = 1
        g_force_suppress_errors = False
    elif level == DiagnosticsLevel.ERRORS_ONLY:
        lldb.formatters.Logger._lldb_formatters_debug_level = 0
        g_force_suppress_errors = False
    elif level == DiagnosticsLevel.DISABLED:
        lldb.formatters.Logger._lldb_formatters_debug_level = 0
        g_force_suppress_errors = True
    else:
        raise Exception('Invalid argument passed, expected level 0, 1 or 2')


# TODO: set_max_num_children

def _log(logger, fmt, *args, **kwargs):
    logger >> fmt.format(*args, **kwargs)


def __lldb_init_module(debugger, internal_dict):
    logger = lldb.formatters.Logger.Logger()
    logger >> '*' * 80
    logger >> 'JetBrains declarative formatters lldb module registered as ' + __name__

    global g_types_storage
    g_types_storage = TypesStorage(logger)

    commands_list = {
        make_absolute_name(__name__, '_cmd_reload_all'): 'jb_renderers_reload_all',
        make_absolute_name(__name__, '_cmd_unload_all'): 'jb_renderers_unload_all',
        make_absolute_name(__name__, '_cmd_set_diagnostics_level'): 'jb_renderers_set_diagnostics_level',
    }
    register_lldb_commands(debugger, commands_list)

    # set errors-only diagnostics level by default
    set_diagnostics_level(DiagnosticsLevel.ERRORS_ONLY)


def _cmd_reload_all(debugger, command, exe_ctx, result, internal_dict):
    reload_all(debugger)


def _cmd_unload_all(debugger, command, exe_ctx, result, internal_dict):
    unload_all(debugger)


def _cmd_set_diagnostics_level(debugger, command, exe_ctx, result, internal_dict):
    cmd = shlex.split(command)
    if len(cmd) != 1:
        result.SetError('Single argument expected.\nUsage: jb_renderers_set_diagnostics_level <level>')
        return

    try:
        level = DiagnosticsLevel(int(cmd[0]))
    except ValueError as e:
        result.SetError('Invalid argument passed, required level as integer in range [0, 2]: {}'.format(str(e)))
        return

    try:
        set_diagnostics_level(level)
    except Exception as e:
        result.SetError(str(e))


def reload_all(debugger):
    unload_all(debugger)

    logger = lldb.formatters.Logger.Logger()

    logger >> 'Registering pretty-printers...'

    for type_name, type_viz, type_viz_name in g_types_storage.iterate_exactly_matched_type_viz():
        debugger.HandleCommand(
            'type summary add "{type_name}" -F {root}.declarative_summary -e --category jb_formatters'
                .format(type_name=type_name, root=__name__))
        g_lldb_registered_summaries.add(type_name)
        logger >> "Summary for {type_viz_name} registered as {type_name}".format(type_viz_name=type_viz_name.type_name,
                                                                                 type_name=type_name)

        if type_viz.item_providers is not None:
            debugger.HandleCommand(
                'type synthetic add "{type_name}" -l {root}.DeclarativeSynthProvider --category jb_formatters'
                    .format(type_name=type_name, root=__name__))
            g_lldb_registered_synthetics.add(type_name)
            logger >> "Synth provider for {type_viz_name} registered as {type_name}".format(
                type_viz_name=type_viz_name.type_name,
                type_name=type_name)

    for type_name_regex, type_viz, type_viz_name in g_types_storage.iterate_wildcard_matched_type_viz():
        debugger.HandleCommand(
            'type summary add "{type_name_regex}" -F {root}.declarative_summary -e -x --category jb_formatters'
                .format(type_name_regex=type_name_regex, root=__name__))
        g_lldb_registered_summaries.add(type_name_regex)
        logger >> "Summary for {type_name} registered as regex {regex}".format(type_name=type_viz_name.type_name,
                                                                               regex=type_name_regex)

        if type_viz.item_providers is not None:
            debugger.HandleCommand(
                'type synthetic add "{type_name_regex}" -x -l {root}.DeclarativeSynthProvider --category jb_formatters'
                    .format(type_name_regex=type_name_regex, root=__name__))
            g_lldb_registered_synthetics.add(type_name_regex)
            logger >> "Synth provider for {type_name} registered as regex {regex}".format(
                type_name=type_viz_name.type_name,
                regex=type_name_regex)


def unload_all(debugger):
    logger = lldb.formatters.Logger.Logger()
    logger >> 'Unregister all previous pretty-printers...'
    for type_name in g_lldb_registered_summaries:
        debugger.HandleCommand('type summary delete "{type_name}" --category jb_formatters'.format(type_name=type_name))
    g_lldb_registered_summaries.clear()

    for type_name in g_lldb_registered_synthetics:
        debugger.HandleCommand(
            'type synthetic delete "{type_name}" --category jb_formatters'.format(type_name=type_name))

    g_lldb_registered_synthetics.clear()


def _strip_references(valobj, logger):
    val_type = valobj.GetNonSyntheticValue().GetType()
    while val_type.IsReferenceType() or val_type.IsPointerType():
        valobj = valobj.GetNonSyntheticValue().Dereference()
        val_type = valobj.GetNonSyntheticValue().GetType()
    return valobj, val_type.GetUnqualifiedType()


class EvaluateError(Exception):
    def __init__(self, error):
        super(Exception, self).__init__(str(error))


class IgnoreSynthProvider(Exception):
    def __init__(self, msg=None):
        super(Exception, self).__init__(str(msg) if msg else None)


def declarative_summary(valobj, internal_dict):
    try:
        return declarative_summary_impl(valobj, internal_dict)
    except:
        global g_force_suppress_errors
        if not g_force_suppress_errors:
            raise
        return ''


def declarative_summary_impl(valobj, internal_dict):
    logger = lldb.formatters.Logger.Logger()
    valobj_non_synth = valobj.GetNonSyntheticValue()
    valobj_type = valobj_non_synth.GetType()
    type_name = valobj_type.GetName()
    _log(logger, "Retrieving summary for type '{}' of value named '{}'...", type_name, valobj_non_synth.GetName())

    valobj_new, valobj_type = _strip_references(valobj, logger)
    if valobj_type.GetName() != type_name:
        valobj = valobj_new
        valobj_non_synth = valobj.GetNonSyntheticValue()
        type_name = valobj_type.GetName()
        _log(logger, "Dereferenced type: '{}'...", type_name)

    try:
        type_name_template = parse_type_name_template(type_name)
    except Exception as e:
        _log(logger, 'Parsing typename {} failed: {}', type_name, e)
        raise

    viz_candidates = _get_matched_type_visualizers(type_name_template)
    if not viz_candidates:
        raise Exception("Can't find any vizualizers: inconsistent type matching for {} (template: {})".format(type_name,
                                                                                                              type_name_template))

    def _try_evaluate_summary(viz, type_viz_name):
        _log(logger, "Trying vizualizer for type '{}'...", str(type_viz_name))

        if not viz.summaries:
            _log(logger, 'No user provided summary found, return default...')
            return ''

        wildcard_matches = []
        if not type_viz_name.type_name_template.match(type_name_template, wildcard_matches):
            raise Exception("Inconsistent type matching for {}".format(type_name))

        wildcard_matches = _fix_wildcard_matches(wildcard_matches)

        # try to choose candidate from ordered display string expressions
        summary_str = _find_first_good_node(viz.summaries, _process_summary_node, valobj_non_synth, wildcard_matches,
                                            logger)
        # if no valid summary found return empty string
        return summary_str if summary_str else ''

    for name_viz_pair in viz_candidates:
        viz, type_viz_name = name_viz_pair
        try:
            res = _try_evaluate_summary(viz, type_viz_name)
        except EvaluateError:
            continue
        return res

    # no candidates
    _log(logger, "No matching display string candidate found")
    return ''


def optional_node_processor(fn):
    def wrapped(node, *args, **kwargs):
        assert isinstance(node, TypeVizItemOptionalNodeMixin)
        try:
            return fn(node, *args, **kwargs)
        except EvaluateError:
            if not node.optional:
                raise

        return None

    return wrapped


def _evaluate_interpolated_string(interp_string, ctx_val, wildcards, logger):
    assert isinstance(interp_string, TypeVizInterpolatedString)
    summary_substrings = []
    for summary_expr in interp_string.expr_list:
        expr = _resolve_wildcards(summary_expr.text, wildcards)
        summary_substring = _eval_display_string_expression(ctx_val, expr, logger) or ''
        summary_substrings.append(summary_substring)

    return interp_string.fmt.format(*summary_substrings)


@optional_node_processor
def _process_summary_node(summary, ctx_val, wildcards, logger):
    assert isinstance(summary, TypeVizSummary)
    if summary.condition:
        condition = _resolve_wildcards(summary.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    return _evaluate_interpolated_string(summary.value, ctx_val, wildcards, logger)


def _get_matched_type_visualizers(type_name_template):
    global g_types_storage
    return [name_match_pair for name_match_pair in g_types_storage.get_matched_types(type_name_template)]


def _fix_wildcard_matches(matches):
    # remove breaking type prefixes from typenames
    def _remove_type_prefix(typename):
        prefix_list = ['enum ', 'struct ', 'class ']
        for prefix in prefix_list:
            if typename.startswith(prefix):
                typename = typename[len(prefix):]
        return typename

    return [_remove_type_prefix(str(t)) for t in matches]


class DeclarativeSynthProvider(object):

    # FOR THE GLORY OF HACKS
    def __new__(cls, valobj, internal_dict):
        logger = lldb.formatters.Logger.Logger()
        try:
            obj = super(DeclarativeSynthProvider, cls).__new__(cls)

            obj.logger = logger

            obj.valobj = valobj
            obj.type_name = valobj.GetNonSyntheticValue().GetTypeName()
            obj.viz = None
            obj.child_providers = None
            obj.child_providers_start_index = None
            obj.wildcard_matches = []

            obj._update(True)

        except IgnoreSynthProvider:
            # valid exception to skip provider and show raw-view
            return None

        except Exception as e:
            # some unexpected error happened
            global g_force_suppress_errors
            if not g_force_suppress_errors:
                _log(logger, "{}", traceback.format_exc())
            return None

        return obj

    def __init__(self, valobj, internal_dict):
        pass

    def update(self):
        # Force do nothing on update
        return False

    def _update(self, is_initial):
        _log(self.logger, "*" * 80)
        _log(self.logger, "{} children for type '{}' of value named '{}'...",
             is_initial and "Initial retrieving" or "Updating", self.type_name, self.valobj.GetName())

        valobj_new, valobj_type = _strip_references(self.valobj, self.logger)
        if valobj_type.GetName() != self.type_name:
            self.valobj = valobj_new
            self.type_name = valobj_type.GetName()
            _log(self.logger, "Dereferenced type: '{}'...", self.type_name)

        try:
            type_name_template = parse_type_name_template(self.type_name)
        except Exception as e:
            _log(self.logger, 'Parsing typename {} failed: {}', self.type_name, e)
            raise e

        viz_candidates = _get_matched_type_visualizers(type_name_template)
        if not viz_candidates:
            raise Exception("Inconsistent type matching for {}".format(self.type_name))

        valobj_non_synth = self.valobj.GetNonSyntheticValue()

        _self = self

        def _try_update(viz, type_viz_name):
            _log(_self.logger, "Found vizualizer for type '{}'...", str(type_viz_name))

            wildcard_matches = []
            if not type_viz_name.type_name_template.match(type_name_template, wildcard_matches):
                raise Exception("Inconsistent type matching for {}".format(_self.type_name))

            _self.wildcard_matches = _fix_wildcard_matches(wildcard_matches)

            _self.child_providers = _build_child_providers(viz.item_providers, valobj_non_synth,
                                                           _self.wildcard_matches,
                                                           _self.logger) if viz.item_providers else None
            if _self.child_providers:
                start_idx = 0
                _self.child_providers_start_index = []
                for prov in _self.child_providers:
                    _self.child_providers_start_index.append(start_idx)
                    start_idx += prov.num_children(self.logger)

        for name_viz_pair in viz_candidates:
            viz, type_viz_name = name_viz_pair
            try:
                _try_update(viz, type_viz_name)
            except EvaluateError:
                continue

            self.viz = viz
            return

        _log(self.logger, "No viz matched for for {}", self.type_name)
        raise IgnoreSynthProvider()

    def num_children(self):
        return sum(
            child_prov.num_children(self.logger) for child_prov in self.child_providers) if self.child_providers else 0

    def has_children(self):
        has_children = self.viz and bool(self.viz.item_providers)
        return has_children

    def get_child_index(self, name):
        if not self.child_providers:
            return -1

        for prov in self.child_providers:
            try:
                index = prov.get_child_index(name, self.logger)
            except Exception as e:
                # some unexpected error happened
                global g_force_suppress_errors
                if not g_force_suppress_errors:
                    raise
                return -1

            if index != -1:
                return index

        return -1

    def get_child_at_index(self, index):
        if not self.child_providers:
            return None

        child_provider, relative_index = self._find_child_provider(index)
        if not child_provider:
            return None
        try:
            return child_provider.get_child_at_index(relative_index, self.logger)
        except Exception as e:
            # some unexpected error happened
            global g_force_suppress_errors
            if not g_force_suppress_errors:
                raise
            return None

    def _find_child_provider(self, index):
        # TODO: binary search, not linear
        for i, start_idx in enumerate(self.child_providers_start_index):
            if start_idx > index:
                # return previous provider
                prov_index = i - 1
                break
        else:
            # last provider
            prov_index = len(self.child_providers) - 1

        if prov_index == -1:
            return None, index

        prov = self.child_providers[prov_index]
        child_start_idx = self.child_providers_start_index[prov_index]

        return prov, (index - child_start_idx)


def _check_condition(val, condition, logger):
    res = _eval_expression(val, '(bool)(' + condition + ')', None, logger)
    if not res.GetValueAsUnsigned():
        return False
    return True


def _resolve_wildcards(expr, wildcards):
    for i, wildcard in enumerate(wildcards):
        template = '$T{d}'.format(d=i + 1)
        expr = expr.replace(template, wildcard)
    return expr


def _eval_expression(val, expr, value_name, logger):
    _log(logger, "Evaluate {} in context of {} of {}", expr, val.GetName(), val.GetTypeName())

    options = lldb.SBExpressionOptions()
    options.SetSuppressPersistentResult(True)
    result = val.EvaluateExpression(expr, options, value_name)
    if result is None:
        err = lldb.SBError()
        err.SetErrorString("evaluation setup failed")
        _log(logger, "Evaluate failed: {}", str(err))
        raise EvaluateError(err)

    result_non_synth = result.GetNonSyntheticValue()
    err = result_non_synth.GetError()
    if err.Fail():
        _log(logger, "Evaluate failed: {}", str(err))
        raise EvaluateError(err)

    _log(logger, "Evaluate succeed: result type - {}", str(result_non_synth.GetTypeName()))
    return result


def _eval_display_string_expression(ctx, expr, logger):
    result = _eval_expression(ctx, expr, None, logger)
    return result.GetSummary() or result.GetValue() or ''


def _process_node_condition(condition, ctx_val, logger):
    result = _eval_expression(ctx_val, '(bool)(' + condition + ')', None, logger)
    if not result.GetValueAsUnsigned():
        return False
    return True


@optional_node_processor
def _node_processor_display_value(node, ctx_val, wildcards, logger):
    assert isinstance(node, TypeVizItemConditionalNodeMixin)
    if node.condition:
        condition = _resolve_wildcards(node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    assert isinstance(node, TypeVizItemFormattedExpressionNodeMixin)
    expression = _resolve_wildcards(node.expr.text, wildcards)

    name = node.name if isinstance(node, TypeVizItemNamedNodeMixin) else None

    return _eval_expression(ctx_val, expression, name, logger)


class SingleItemProvider(object):
    def __init__(self, value):
        self.value = value

    def num_children(self, logger):
        return 1

    def get_child_index(self, name, logger):
        if self.value.GetName() == name:
            return 0
        return -1

    def get_child_at_index(self, index, logger):
        assert index == 0
        return self.value


def _process_item_provider_single(item_provider, val, wildcards, logger):
    item_value = _node_processor_display_value(item_provider, val, wildcards, logger)
    if not item_value:
        return None

    return SingleItemProvider(item_value)


class ExpandedItemProvider(object):
    def __init__(self, value):
        self.value = value

    def num_children(self, logger):
        return self.value.GetNumChildren()

    def get_child_index(self, name, logger):
        return self.value.GetIndexOfChildWithName(name)

    def get_child_at_index(self, index, logger):
        return self.value.GetChildAtIndex(index)


def _process_item_provider_expanded(item_provider, val, wildcards, logger):
    item_value = _node_processor_display_value(item_provider, val, wildcards, logger)
    if not item_value:
        return None

    return ExpandedItemProvider(item_value)


def _find_first_good_node(nodes, node_proc, ctx_val, wildcards, logger):
    # NOTE: next() can be used
    for node in nodes:
        item_value = node_proc(node, ctx_val, wildcards, logger)
        if item_value is not None:
            return item_value
    return None


@optional_node_processor
def _node_processor_size(size_node, ctx_val, wildcards, logger):
    assert isinstance(size_node, TypeVizItemSizeTypeNode)
    if size_node.condition:
        condition = _resolve_wildcards(size_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    expression = size_node.text
    expression = _resolve_wildcards(expression, wildcards)
    value = _eval_expression(ctx_val, expression, None, logger)
    if value is None:
        return None

    result_value = value.GetValueAsSigned()
    if not isinstance(result_value, int):
        raise EvaluateError('Size value must be of integer type')

    return result_value


def _node_processor_array_items_value_pointer(value_pointer_node, ctx_val, wildcards, logger):
    assert isinstance(value_pointer_node, TypeVizItemValuePointerTypeNode)
    if value_pointer_node.condition:
        condition = _resolve_wildcards(value_pointer_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    expr = value_pointer_node.expr
    expression = expr.text
    expression = _resolve_wildcards(expression, wildcards)
    return _eval_expression(ctx_val, expression, None, logger)


class ArrayItemsProvider(object):
    def __init__(self, size, value_pointer, elem_type):
        self.size = size
        self.value_pointer = value_pointer
        self.elem_type = elem_type
        self.elem_byte_size = elem_type.GetByteSize()

    def num_children(self, logger):
        return self.size

    def get_child_index(self, name, logger):
        try:
            return int(name.lstrip('[').rstrip(']'))
        except ValueError:
            return -1

    def get_child_at_index(self, index, logger):
        child_name = "[{}]".format(index)
        offset = index * self.elem_byte_size
        return self.value_pointer.CreateChildAtOffset(child_name, offset, self.elem_type)


@optional_node_processor
def _node_processor_array_items(array_items_node, ctx_val, wildcards, logger):
    assert isinstance(array_items_node, TypeVizItemProviderArrayItems)
    if array_items_node.condition:
        condition = _resolve_wildcards(array_items_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    size = _find_first_good_node(array_items_node.size_nodes,
                                 _node_processor_size,
                                 ctx_val, wildcards, logger)
    # ???
    if size is None:
        raise EvaluateError('No valid Size node found')

    value_pointer_value = _find_first_good_node(array_items_node.value_pointer_nodes,
                                                _node_processor_array_items_value_pointer,
                                                ctx_val, wildcards, logger)
    # ???
    if value_pointer_value is None:
        raise EvaluateError('No valid ValuePointerType node found')

    value_pointer_type = value_pointer_value.GetNonSyntheticValue().GetType()
    if value_pointer_type.IsPointerType():
        elem_type = value_pointer_type.GetPointeeType()
    elif value_pointer_type.IsArrayType():
        elem_type = value_pointer_type.GetArrayElementType()
        value_pointer_value = value_pointer_value.GetNonSyntheticValue().AddressOf()
    else:
        raise EvaluateError('Value pointer is not of pointer or array type ({})'.format(str(value_pointer_type)))

    return ArrayItemsProvider(size, value_pointer_value, elem_type)


def _process_item_provider_array_items(item_provider, val, wildcards, logger):
    return _node_processor_array_items(item_provider, val, wildcards, logger)


def _node_processor_index_list_items_value_node(idx_str, name, index_list_value_node, ctx_val, wildcards, logger):
    assert isinstance(index_list_value_node, TypeVizItemIndexNodeTypeNode)
    if index_list_value_node.condition:
        condition = index_list_value_node.condition.replace('$i', idx_str)
        condition = _resolve_wildcards(condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    expression = index_list_value_node.expr.text.replace('$i', idx_str)
    expression = _resolve_wildcards(expression, wildcards)
    return _eval_expression(ctx_val, expression, name, logger)


class IndexListItemsProvider(object):
    def __init__(self, size, index_list_node, ctx_val, wildcards):
        self.size = size
        self.index_list_node = index_list_node
        self.ctx_val = ctx_val
        self.wildcards = wildcards

    def num_children(self, logger):
        return self.size

    def get_child_index(self, name, logger):
        try:
            return int(name.lstrip('[').rstrip(']'))
        except ValueError:
            return -1

    def get_child_at_index(self, index, logger):
        name = "[{}]".format(index)
        value = None
        for value_node_node in self.index_list_node.value_node_nodes:
            value = _node_processor_index_list_items_value_node(str(index), name, value_node_node, self.ctx_val,
                                                                self.wildcards, logger)
            if value:
                break

        # TODO: show some error value on None
        return value


@optional_node_processor
def _node_processor_index_list_items(index_list_node, ctx_val, wildcards, logger):
    assert isinstance(index_list_node, TypeVizItemProviderIndexListItems)
    if index_list_node.condition:
        condition = _resolve_wildcards(index_list_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    size = _find_first_good_node(index_list_node.size_nodes,
                                 _node_processor_size,
                                 ctx_val, wildcards, logger)
    # ????
    if size is None:
        raise EvaluateError('No valid Size node found')

    return IndexListItemsProvider(size, index_list_node, ctx_val, wildcards)


def _process_item_provider_index_list_items(item_provider, val, wildcards, logger):
    return _node_processor_index_list_items(item_provider, val, wildcards, logger)


def _is_valid_node_ptr(node):
    if node is None:
        return False

    if not node.TypeIsPointerType():
        return False

    return True


def _get_ptr_value(node):
    val = node.GetNonSyntheticValue()
    return val.GetValueAsUnsigned() if _is_valid_node_ptr(val) else 0


class NodesProvider(object):
    def __init__(self):
        self.cache = []
        self.has_more = False
        self.names = None
        self.name2index = None


class CustomItemsProvider(object):
    def __init__(self, nodes_provider, value_expression, wildcards, logger):
        assert isinstance(nodes_provider, NodesProvider)

        self.nodes_cache = nodes_provider.cache
        self.has_more = nodes_provider.has_more
        self.custom_names = nodes_provider.names
        self.custom_name_to_index = nodes_provider.name2index
        self.value_expression = value_expression
        self.wildcards = wildcards
        self.logger = logger

        self.size = len(self.nodes_cache)

    def num_children(self, logger):
        return self.size

    def get_child_index(self, name, logger):
        if self.custom_name_to_index:
            return self.custom_name_to_index.get(name, -1)

        try:
            return int(name.lstrip('[').rstrip(']'))
        except IndexError:
            return -1

    def get_child_at_index(self, index, logger):
        if index < 0 or index >= self.size:
            return None

        node_value = self.nodes_cache[index]
        if node_value is None:
            return None

        if self.custom_names:
            name = self.custom_names[index]
        else:
            name = "[{}]".format(index)
        value = _eval_expression(node_value, self.value_expression, name, logger)
        return value


class LinkedListIterator(object):
    def __init__(self, node_value, next_expression, logger):
        self.node_value = node_value
        self.next_expression = next_expression
        self.logger = logger

    def __bool__(self):
        return _get_ptr_value(self.node_value) != 0

    def __eq__(self, other):
        return _get_ptr_value(self.node_value) == _get_ptr_value(other.node_value)

    def cur_value(self):
        return self.node_value.GetNonSyntheticValue().Dereference()

    def cur_ptr(self):
        return self.node_value.GetNonSyntheticValue().GetValueAsUnsigned()

    def move_to_next(self):
        self.node_value = self._next()

    def _next(self):
        return _eval_expression(self.cur_value(), self.next_expression, None, self.logger)


class LinkedListIndexedNodesProvider(NodesProvider):
    def __init__(self, size, head_pointer, next_expression, logger):
        super(LinkedListIndexedNodesProvider, self).__init__()

        it = LinkedListIterator(head_pointer, next_expression, logger)

        cache = []
        has_more = False

        # iterate all list nodes and cache them
        start = it.cur_ptr() if _is_valid_node_ptr(it.node_value) else 0
        max_size = size if size is not None else g_max_num_children
        idx = 0
        while it and idx < max_size:
            cache.append(it.cur_value())
            idx += 1
            it.move_to_next()

            if it and it.cur_ptr() == start:
                # check for cycled
                break

        if size is None:
            if it and idx >= max_size:
                has_more = True
        else:
            if idx < size:
                cache.extend([None] * (size - idx))

        self.cache = cache
        self.has_more = has_more
        self.names = None
        self.name2index = None


class LinkedListCustomNameNodesProvider(NodesProvider):
    def __init__(self, size, head_pointer, next_expression, custom_value_name, wildcards, logger):
        super(LinkedListCustomNameNodesProvider, self).__init__()

        it = LinkedListIterator(head_pointer, next_expression, logger)

        cache = []
        has_more = False
        names = []
        name2index = {}

        # iterate all list nodes and cache them
        max_size = size if size is not None else g_max_num_children
        idx = 0
        start = it
        while it and idx < max_size:
            cur_val = it.cur_value()
            name = _evaluate_interpolated_string(custom_value_name, cur_val, wildcards, logger)
            names.append(name)
            name2index[name] = idx

            cache.append(cur_val)
            idx += 1
            it.move_to_next()

            if it == start:
                # check for cycled
                break

        if size is None:
            if it and idx >= max_size:
                has_more = True
        else:
            if idx < size:
                cache.extend([None] * (size - idx))

        self.cache = cache
        self.has_more = has_more
        self.names = names
        self.name2index = name2index


def _node_processor_linked_list_items_head_pointer(head_pointer_node, ctx_val, wildcards, logger):
    assert isinstance(head_pointer_node, TypeVizItemListItemsHeadPointerTypeNode)
    expression = _resolve_wildcards(head_pointer_node.text, wildcards)
    return _eval_expression(ctx_val, expression, None, logger)


@optional_node_processor
def _node_processor_linked_list_items(linked_list_node, ctx_val, wildcards, logger):
    assert isinstance(linked_list_node, TypeVizItemProviderLinkedListItems)
    if linked_list_node.condition:
        condition = _resolve_wildcards(linked_list_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    size = _find_first_good_node(linked_list_node.size_nodes,
                                 _node_processor_size,
                                 ctx_val, wildcards, logger)
    # size can be None

    head_pointer_value = _node_processor_linked_list_items_head_pointer(linked_list_node.head_pointer_node, ctx_val,
                                                                        wildcards, logger)

    next_pointer_node = linked_list_node.next_pointer_node
    assert isinstance(next_pointer_node, TypeVizItemListItemsNextPointerTypeNode)
    next_pointer_expression = _resolve_wildcards(next_pointer_node.text, wildcards)

    value_node = linked_list_node.value_node_node
    assert isinstance(value_node, TypeVizItemListItemsIndexNodeTypeNode)
    value_expression = _resolve_wildcards(value_node.expr.text, wildcards)

    if value_node.name is None:
        nodes_provider = LinkedListIndexedNodesProvider(size, head_pointer_value, next_pointer_expression,
                                                        logger)
    else:
        nodes_provider = LinkedListCustomNameNodesProvider(size, head_pointer_value, next_pointer_expression,
                                                           value_node.name, wildcards,
                                                           logger)

    return CustomItemsProvider(nodes_provider, value_expression, wildcards, logger)


def _process_item_provider_linked_list_items(item_provider, val, wildcards, logger):
    return _node_processor_linked_list_items(item_provider, val, wildcards, logger)


class BinaryTreeIndexedNodesProvider(NodesProvider):
    def __init__(self, size, head_pointer, left_expression, right_expression, node_condition, logger):
        super(BinaryTreeIndexedNodesProvider, self).__init__()

        cache = []
        has_more = False

        # iterate all list nodes and cache them
        max_size = size if size is not None else g_max_num_children
        idx = 0
        cur = head_pointer
        stack = []  # parents

        def check_condition(node):
            if node_condition is None:
                return True
            return _check_condition(node.GetNonSyntheticValue().Dereference(), node_condition, logger)

        while (_get_ptr_value(cur) != 0 and check_condition(cur) or stack) and idx < max_size:
            while _get_ptr_value(cur) != 0 and check_condition(cur):
                if len(stack) > 100:  # ~2^100 nodes can't be true - something went wrong
                    raise Exception("Invalid tree")

                stack.append(cur)
                cur = _eval_expression(cur.GetNonSyntheticValue().Dereference(), left_expression, None, logger)

            cur = stack.pop()
            cache.append(cur.GetNonSyntheticValue().Dereference())
            idx += 1

            cur = _eval_expression(cur.GetNonSyntheticValue().Dereference(), right_expression, None, logger)

        if size is None:
            if _get_ptr_value(cur) != 0 and check_condition(cur) or stack and idx >= max_size:
                has_more = True
        else:
            if idx < size:
                cache.extend([None] * (size - idx))

        self.cache = cache
        self.has_more = has_more
        self.names = None
        self.name2index = None


class BinaryTreeCustomNamesNodesProvider(NodesProvider):
    def __init__(self, size, head_pointer, left_expression, right_expression, node_condition, custom_value_name,
                 wildcards, logger):
        super(BinaryTreeCustomNamesNodesProvider, self).__init__()

        cache = []
        has_more = False
        names = []
        name2index = {}

        # iterate all list nodes and cache them
        max_size = size if size is not None else g_max_num_children
        idx = 0
        cur = head_pointer
        stack = []  # parents

        def check_condition(node):
            if node_condition is None:
                return True
            return _check_condition(node.GetNonSyntheticValue().Dereference(), node_condition, logger)

        while (_get_ptr_value(cur) != 0 and check_condition(cur) or stack) and idx < max_size:
            while _get_ptr_value(cur) != 0 and check_condition(cur):
                if len(stack) > 100:  # ~2^100 nodes can't be true - something went wrong
                    raise Exception("Invalid tree")

                stack.append(cur)
                cur = _eval_expression(cur.GetNonSyntheticValue().Dereference(), left_expression, None, logger)

            cur = stack.pop()
            cur_val = cur.GetNonSyntheticValue().Dereference()
            name = _evaluate_interpolated_string(custom_value_name, cur_val, wildcards, logger)
            names.append(name)
            name2index[name] = idx
            cache.append(cur_val)
            idx += 1

            cur = _eval_expression(cur_val, right_expression, None, logger)

        if size is None:
            if _get_ptr_value(cur) != 0 and check_condition(cur) or stack and idx >= max_size:
                has_more = True
        else:
            if idx < size:
                cache.extend([None] * (size - idx))

        self.cache = cache
        self.has_more = has_more
        self.names = names
        self.name2index = name2index


def _node_processor_tree_items_head_pointer(head_pointer_node, ctx_val, wildcards, logger):
    assert isinstance(head_pointer_node, TypeVizItemTreeHeadPointerTypeNode)
    expression = _resolve_wildcards(head_pointer_node.text, wildcards)
    return _eval_expression(ctx_val, expression, None, logger)


@optional_node_processor
def _node_processor_tree_items(tree_node, ctx_val, wildcards, logger):
    assert isinstance(tree_node, TypeVizItemProviderTreeItems)
    if tree_node.condition:
        condition = _resolve_wildcards(tree_node.condition, wildcards)
        if not _process_node_condition(condition, ctx_val, logger):
            return None

    size = _find_first_good_node(tree_node.size_nodes,
                                 _node_processor_size,
                                 ctx_val, wildcards, logger)
    # size can be None

    head_pointer_value = _node_processor_tree_items_head_pointer(tree_node.head_pointer_node, ctx_val,
                                                                 wildcards, logger)

    left_pointer_node = tree_node.left_pointer_node
    right_pointer_node = tree_node.right_pointer_node
    assert isinstance(left_pointer_node, TypeVizItemTreeChildPointerTypeNode)
    assert isinstance(right_pointer_node, TypeVizItemTreeChildPointerTypeNode)
    left_pointer_expression = _resolve_wildcards(left_pointer_node.text, wildcards)
    right_pointer_expression = _resolve_wildcards(right_pointer_node.text, wildcards)

    value_node = tree_node.value_node_node
    assert isinstance(value_node, TypeVizItemTreeNodeTypeNode)
    value_expression = _resolve_wildcards(value_node.expr.text, wildcards)

    value_condition = _resolve_wildcards(value_node.condition, wildcards) if value_node.condition else None

    if value_node.name is None:
        nodes_provider = BinaryTreeIndexedNodesProvider(size, head_pointer_value,
                                                        left_pointer_expression, right_pointer_expression,
                                                        value_condition,
                                                        logger)
    else:
        nodes_provider = BinaryTreeCustomNamesNodesProvider(size, head_pointer_value,
                                                            left_pointer_expression, right_pointer_expression,
                                                            value_condition, value_node.name, wildcards,
                                                            logger)

    return CustomItemsProvider(nodes_provider, value_expression, wildcards, logger)


def _process_item_provider_tree_items(item_provider, val, wildcards, logger):
    return _node_processor_tree_items(item_provider, val, wildcards, logger)


def _build_child_providers(item_providers, valobj_non_synth, wildcards, logger):
    provider_handlers = {
        TypeVizItemProviderTypeKind.Single: _process_item_provider_single,
        TypeVizItemProviderTypeKind.Expanded: _process_item_provider_expanded,
        TypeVizItemProviderTypeKind.ArrayItems: _process_item_provider_array_items,
        TypeVizItemProviderTypeKind.IndexListItems: _process_item_provider_index_list_items,
        TypeVizItemProviderTypeKind.LinkedListItems: _process_item_provider_linked_list_items,
        TypeVizItemProviderTypeKind.TreeItems: _process_item_provider_tree_items,
    }
    child_providers = []
    for item_provider in item_providers:
        handler = provider_handlers.get(item_provider.kind)
        if not handler:
            continue
        child_provider = handler(item_provider, valobj_non_synth, wildcards, logger)
        if not child_provider:
            continue
        child_providers.append(child_provider)

    return child_providers
