/** This file contains common JavaScript constructs used by the Photoshop
native extension implementation.
*/

// --------------------------------------------------------------------------------
// Convert en error object to a form that can be interpreted by Photoshop
function _internalConvertError(error) {
	if (error instanceof Error) {
		return error.toString();
		}
	return error;
}

// --------------------------------------------------------------------------------

/** Create list of arguments to be used when invoking a JavaScript target function.
*/
function _internalCreateJSArguments(theArgs, fixedArgCount) {
    let jsArgs = [];
    let varArgCount = theArgs.length - fixedArgCount;
    for (index = 0; index < varArgCount; ++index) {
        jsArgs.push(theArgs[index + fixedArgCount]);
    }
    return jsArgs;
}

// --------------------------------------------------------------------------------

/** Wrapper function used by javascript callbacks from Photoshop.
Main purpose is to catch and discard top level errors.
*/
async function _internalCallbackWrapper(jsFunction, errorHandler) {
    try {
        let jsArgs = _internalCreateJSArguments(arguments, 2);
        await jsFunction(...jsArgs);
    }
    catch (error) {
        console.warn("Uncaught exception discarded");
        
        if (errorHandler != undefined) {
            errorHandler(_internalConvertError(error))
        }
    }
}

// --------------------------------------------------------------------------------

/** Wrapper function used by executeAsModal to track the state of a target
*/
async function _internalExecuteAsModalWrapper(jsFunction, executionControl) {
    try {
        let jsArgs = _internalCreateJSArguments(arguments, 2);
        jsArgs.unshift(executionControl);
        let result = await jsFunction(...jsArgs);
        executionControl.resolve(result)
    }
    catch (error) {
        executionControl.reject(_internalConvertError(error));
    }
}

// --------------------------------------------------------------------------------

function _internalCreateStackTrace() {
    let result = "";
    try {
        let e = new Error;
        let stackTxt = e.stack;
		let frameList = stackTxt.split("\n");
		for (line of frameList) {
			if (!line.includes("_internalCreateStackTrace")) {
				if (result != "")
					result += "\n"
				result += line;
			}
		}
    }
    catch (error) {
    }
    return result;
}

// --------------------------------------------------------------------------------

/** Wrapper function used to route messages from core to the console of the active context
*/
async function _internalLogMessageWrapper(category, message) {
    try {
        if (category == "warning") {
            console.warn(message);
        }
        else if (category == "error") {
            console.error(message);
        }
        else {
            console.log(message);
        }
    }
    catch (e) {
    }
}
