/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr: true*/
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false*/

var sinon = require("sinon"),
    Q = require("q"),
    FS = require("fs"),
    PathUtils = require("../PathUtils");

describe("PathUtils", function () {
    "use strict";

    var sandbox = sinon.sandbox.create(),
        pathExistsStub;

    beforeEach(function () {
        pathExistsStub = sandbox.stub(FS, "existsSync").returns(false);
    });

    afterEach(function () {
        sandbox.restore();
    });

    describe("addExtIfNeeded", function () {
        it("adds the correct extension if there is no extension", function () {
            expect(PathUtils.addExtIfNeeded("/path/file", "png")).to.equal("/path/file.png");
        });

        it("adds the correct extension if there is an incorrect extension", function () {
            expect(PathUtils.addExtIfNeeded("/path/file.jpg", "png")).to.equal("/path/file.jpg.png");
        });

        it("recognizes the correct extension in a case-insensitive way", function () {
            expect(PathUtils.addExtIfNeeded("/path/file.PnG", "png")).to.equal("/path/file.PnG");
        });
    }); // addExtIfNeeded

    describe("addExtIfNeededWithoutCausingConflicts", function () {
        beforeEach(function () {
            sandbox.spy(PathUtils, "addExtIfNeeded");
        });

        it("increments the file name numerically if adding the correct file extension conflicts with an existing file", function () {
            pathExistsStub.withArgs("/path/file.png").returns(true);
            expect(PathUtils.addExtIfNeededWithoutCausingConflicts("/path/file", "png")).to.equal("/path/file-1.png");
            expect(PathUtils.addExtIfNeeded.calledOnce).to.equal(true);
        });

        it("does not increment the file name numerically if there is no file conflict", function () {
            expect(PathUtils.addExtIfNeededWithoutCausingConflicts("/path/file", "png")).to.equal("/path/file.png");
            expect(PathUtils.addExtIfNeeded.calledOnce).to.equal(true);
        });

        it("does not increment the file name numerically if no extension needs to be added", function () {
            expect(PathUtils.addExtIfNeededWithoutCausingConflicts("/path/file.png", "png")).to.equal("/path/file.png");
            expect(PathUtils.addExtIfNeeded.calledOnce).to.equal(true);
        });
    }); // addExtIfNeededWithoutCausingConflicts
    
    describe("suffixPathNumericallyIfNeeded", function() {
        it("does not suffix paths if not needed", function () {
            var originalPath = "/Users/Batman/gotham-skyline.png";
            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath)).to.equal(originalPath);
        });

        it("suffixes paths if needed", function () {
            var originalPath = "/Users/Batman/gotham-skyline.png";

            pathExistsStub.onCall(0).returns(true); // Original path conflicts.
            pathExistsStub.onCall(1).returns(false); // Suffixed path does not conflict.

            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath)).to.equal("/Users/Batman/gotham-skyline-1.png");
        });

        it("does not increment any trailing numbers in the basename", function () {
            var originalPath = "/Users/Batman/gotham-skyline-1.png";

            pathExistsStub.onCall(0).returns(true); // Original path conflicts.
            pathExistsStub.onCall(1).returns(false); // Suffixed path does not conflict.

            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath)).to.equal("/Users/Batman/gotham-skyline-1-1.png");
        });

        it("does not increment numbers that are not preceded by a dash", function () {
            var originalPath = "/Users/Batman/gotham-skyline2048x2048.png";

            pathExistsStub.onCall(0).returns(true); // Original path conflicts.
            pathExistsStub.onCall(1).returns(false); // Suffixed path does not conflict.

            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath)).to.equal("/Users/Batman/gotham-skyline2048x2048-1.png");
        });

        it("keeps incrementing path suffixes if needed", function () {
            var originalPath = "/Users/Batman/gotham-skyline.png";

            pathExistsStub.onCall(0).returns(true); // Original path conflicts.
            pathExistsStub.onCall(1).returns(true); // First suffixed path conflicts.
            pathExistsStub.onCall(2).returns(false); // Second suffixed path does not conflict.
 
            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath)).to.equal("/Users/Batman/gotham-skyline-2.png");
        });

        it("increments path suffixes based on a predicate", function () {
            var originalPath = "/Users/Batman/gotham-skyline.png",
                predicate = function (attemptedPath) {
                    if (attemptedPath == "/Users/Batman/gotham-skyline.png") {
                        // The predicate says that this path conflicts.
                        return true;
                    }
                    // Otherwise, the predicate says there is not a conflict.
                    return false;
                };

            expect(PathUtils.suffixPathNumericallyIfNeeded(originalPath, predicate)).to.equal("/Users/Batman/gotham-skyline-1.png");
        });
    }); // suffixPathNumericallyIfNeeded

    describe("buildPath", function () {
        it("builds a path with a folder, basename, and extension correctly", function () {
            expect(PathUtils.buildPath("/path", "file", "png")).to.equal("/path/file.png");
        });
    }); // buildPath

    describe("resolveInitialTilde", function () {
        it("expands a tilde at the beginning of a file path to the user home directory", function () {
            sandbox.stub(PathUtils, "getHomePath").returns("/Users/Batman");
            expect(PathUtils.resolveInitialTilde("~/Desktop")).to.equal("/Users/Batman/Desktop");
        });

        it("does not touch tildes not at the beginning of the path", function () {
            var path = "/Users/Batman/Got~ham";
            expect(PathUtils.resolveInitialTilde(path)).to.equal(path);
        });
    }); // resolveInitialTilde

    describe("getHomePath", function () {
        var getEnvVarStub;

        beforeEach(function () {
            getEnvVarStub = sandbox.stub(PathUtils, "getEnvVar").returns(undefined);
        });

        it("returns HOME if avaiable", function () {
            getEnvVarStub.withArgs("HOME").returns("/Users/Home");
            getEnvVarStub.withArgs("USERPROFILE").returns("/Users/UserProfile");

            expect(PathUtils.getHomePath()).to.equal("/Users/Home");
        });

        it("returns USERPROFILE if HOME isn't available", function () {
            getEnvVarStub.withArgs("USERPROFILE").returns("/Users/UserProfile");

            expect(PathUtils.getHomePath()).to.equal("/Users/UserProfile");
        });

        it("returns empty string if no environment variables are available", function () {
            expect(PathUtils.getHomePath()).to.equal("");
        });
    });//getHomePath

    describe("getPhotoshopPreferencesPath", function () {
        var getEnvVarStub;

        beforeEach(function () {
            getEnvVarStub = sandbox.stub(PathUtils, "getEnvVar").returns(undefined);
        });
        it("returns expected path on windows", function () {
            getEnvVarStub.withArgs("APPDATA").returns("/U");


            expect(PathUtils.getPhotoshopPreferencesPath()).to.equal("/U/Adobe/Adobe Photoshop CC 2015/Adobe Photoshop CC 2015 Settings");
        });
        it("returns expected path on mac", function () {
            getEnvVarStub.withArgs("APPDATA").returns(undefined);
            getEnvVarStub.withArgs("HOME").returns("/Users/Home");

            expect(PathUtils.getPhotoshopPreferencesPath()).to.equal("/Users/Home/Library/Preferences/Adobe Photoshop CC 2015 Settings");
        });

    });//getPhotoshopPreferencesPath
    describe("getUserSettingPath", function () {
        it("returns expected path", function () {
            PathUtils.getPhotoshopPreferencesPath = sinon.stub().returns("/path");
            expect(PathUtils.getUserSettingPath()).to.equal("/path/ExtractSettings.json");
            expect(PathUtils.getPhotoshopPreferencesPath).to.be.called;
        });
    });//getPhotoshopPreferencesPath
});
