/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

// IMPORTANT: 

//////////////////////////////////////////////////////////////////////////////
//
// The WorkflowController object is a controller for the whole system
// and entry point to start/stop workflow processing
//
function WorkflowController()
{
	var kDefaultGUID = '00000000-0000-0000-0000-00000000';

	var workflows = null;	// { 'GUID' : {'file' : the_filename, 'workflow' : WorkflowObject}, ...}
	var currentWorkflow = null;
	var currentProcessor = null;
	var thisObj = this;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Start workflow processing for workflow with given guid
	//
	this.startWorkflow = function(/*String*/ inGUID)
	{
		logInfo('Attempt to start Workflow, id "' + inGUID + '"');

		var guid = inGUID;

		if (!StartupController.get().isInitialized())
		{
			logInfo('System not completely initialized, delay starting workflow');

			var thisObj = this;
			StartupController.get().registerTask(function()
			{
				thisObj.startWorkflow(guid);
			});
		}
		else
		{
			this.stopWorkflow();

			getWorkflow(guid, function(/*Workflow*/ inWorkflow)
			{
				var wf = inWorkflow;

				if (Utils.isValidProperty(wf))
				{
					currentWorkflow = wf;

					try
					{
						currentProcessor = new WorkflowProcessor(currentWorkflow, onProcessingFinished);
					}
					catch (exc)
					{
						clearCurrentWorkflow();
						wf = null;

						logExc(exc);
					}

					if (Utils.isValidProperty(currentProcessor))
					{
						currentProcessor.start();
					}
					else
					{
						wf = null;
					}
				}

				if (!Utils.isValidProperty(wf))
				{
					// something went wrong, announce end of processing
					if (Utils.isValidProperty(currentWorkflow))
					{
						ScriptedWorkflowSupport.instance.stopWorkflow(currentWorkflow.getID());
					}
					else
					{
						ScriptedWorkflowSupport.instance.stopWorkflow(guid);
					}

					clearCurrentWorkflow();
				}
			});
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Stop current workflow processing
	//
	this.stopWorkflow = function()
	{
		if (Utils.isValidProperty(currentWorkflow) && Utils.isValidProperty(currentProcessor))
		{
			logInfo('Stop Workflow processing, id "' + currentWorkflow.getID() + '"');

			currentProcessor.cancel();
			clearCurrentWorkflow();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Incoming event to start workflow processing
	//
	function onStartWorkflow(/*EventObject*/ inEventObj)
	{
		var guid = inEventObj.string;
		thisObj.startWorkflow(guid);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Incoming event to stop workflow processing
	//
	function onStopWorkflow(/*EventObject*/ inEventObj)
	{
		thisObj.stopWorkflow();
		// send acknowledge to backend
		inEventObj.acknowledge(true);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve workflow description with guid
	//
	function getWorkflow(/*String*/ inGUID, /*Function*/ inCallback)
	{
		var guid = inGUID;
		var callback = inCallback;

		function retrieveWorkflow()
		{
			if (Utils.isValidProperty(workflows))
			{
				if (Utils.isValidProperty(workflows[guid]))
				{
					if (Utils.isValidProperty(workflows[guid].workflow))
					{
						callback(workflows[guid].workflow);
					}
					else
					{
						loadWorkflow(guid, callback);
					}
				}
				else
				{
					logError('Unknown Workflow identifier "' + guid + '"');
				}
			}
			else
			{
				logError('Can not load workflow list, ' + ScriptedWorkflowSupport.instance.workflowRoot + '/workflows.json');
			}
		}

		if (Utils.isValidProperty(workflows))
		{
			retrieveWorkflow();
		}
		else
		{
			var path = ScriptedWorkflowSupport.instance.workflowRoot + '/workflows.json';
			loadWorkflowList(path, retrieveWorkflow);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Load workflow description file
	//
	function loadWorkflow(/*String*/ inGUID, /*Function*/ inCallback)
	{
		var callback = inCallback;
		var guid = inGUID;

		if (Utils.isValidProperty(workflows) && Utils.isValidProperty(workflows[guid]))
		{
			if (Utils.isValidProperty(workflows[guid].workflow))
			{
				callback(workflows[guid].workflow);
			}
			else
			{
				try
				{
					DataRequestManager.requestData(workflows[guid].file, function(/*Boolean*/ inSuccess, /*String*/ inData)
					{
						if (inSuccess && Utils.isValidProperty(inData))
						{
							workflows[guid].workflow = Workflow.create(inData);
							callback(workflows[guid].workflow);
						}
					});
				}
				catch(exc)
				{
					logExc(exc);
					callback(null);
				}
			}
		}
		else
		{
			logError('Unknown Workflow identifier "' + guid + '"');
			callback(null);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Load workflow description file list
	//
	function loadWorkflowList(/*String*/ inPath, /*Function*/ inCallback)
	{
		var callback = inCallback;

		try
		{
			DataRequestManager.requestJSONData(inPath, function(/*Boolean*/ inSuccess, /*Object*/ inJSON)
			{
				if (inSuccess && Utils.isValidProperty(inJSON, Array))
				{
					/*	File format
					 [
					 { guid : theGUID, file : theFilename },
					 { guid : theGUID, file : theFilename },
					 { guid : theGUID, file : theFilename },
					 :
					 :
					 ]

					 */

					workflows               = {};

					Utils.forEach(inJSON, function(/*Object*/ inEntry)
					{
						if (Utils.isValidProperty(inEntry, Object) && Utils.isValidProperty(inEntry.file))
						{
							var filepath = inEntry.file;

							if (inEntry.file.indexOf('/') < 0)
							{
								filepath = ScriptedWorkflowSupport.instance.workflowRoot + '/' + inEntry.guid + '/' +  filepath;
							}

							workflows[inEntry.guid] = {file: filepath, workflow: null};

							if (!Utils.isValidProperty(workflows[kDefaultGUID]))
							{
								workflows[kDefaultGUID] = workflows[inEntry.guid];
							}
						}
					});

					callback();
				}
			});
		}
		catch (exc)
		{
			logExc(exc);

			callback();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Current workflow processing has finished
	//
	function onProcessingFinished(/*String*/ inFinishedState)
	{
		// tell the rest of the world we are done
		if (Utils.isValidProperty(currentWorkflow))
		{
			ScriptedWorkflowSupport.instance.finishedWorkflow(currentWorkflow.getID(), inFinishedState);
		}

		clearCurrentWorkflow();
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Clear current workflow processing data
	//
	function clearCurrentWorkflow()
	{
		currentWorkflow  = null;
		currentProcessor = null;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	ScriptedWorkflowSupport.instance.addEventListener('Tour.start', onStartWorkflow);
	ScriptedWorkflowSupport.instance.addEventListener('Tour.stop', onStopWorkflow);
}

WorkflowController.instance = new WorkflowController();
