/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// A Workflow object represents a Workflow Description
//
function Workflow()
{
    var workflowID = '';
	var type = '';
    var name = '';
    var date = new Date();
    var appVersion = '';
    var title = null;
    var preActions = [];
    var postActions = [];
    var steps = [];
	var stepIdentifierMap = {};
	var range = [0];

    var initialized = false;
    var formatVersion = '';

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return workflow identifier
	//
	this.getID = function()
	{
		return workflowID;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return workflow type
	//
	this.getType = function()
	{
		return type;
	}

	//////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        if (!initialized)
        {
            Utils.throwInvalid(inJSON);

            // dispatch format
            //
            var jsonVersion_1 = inJSON[Workflow.FORMAT_VERSION_1];
            if (Utils.isValidProperty(jsonVersion_1))
            {
                formatVersion = Workflow.FORMAT_VERSION_1;
                initializeJSON_V1(jsonVersion_1, this);
            }
            else
            {
                logError('Format not supported, missing version entry "' + Workflow.FORMAT_VERSION_1 + '"');
				throw new Error("Invalid description format");
            }
        }
        else
        {
            logWarn('Workflow object already initialized.');
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Return WorkflowStep for index
    //
    this.getStep = function(/*Number*/ inStepIndex)
    {
        var ret = null;

        if (!isNaN(inStepIndex) && inStepIndex >= 0 && inStepIndex < steps.length)
        {
            ret = steps[inStepIndex];
        }
        else
        {
            logError('getStep: Invalid step index: ' + inStepIndex);
        }

        return ret;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get step index for step identifier
	//
    this.getStepIndex = function(/*String*/ inStepID)
	{
		var ret = NaN;

		if (stepIdentifierMap.hasOwnProperty(inStepID))
		{
			ret = stepIdentifierMap[inStepID];
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get number of WorkflowStep's
	//
	this.getStepLength = function()
	{
		return steps.length;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get steps range
	//
	this.getStepRange = function()
	{
		return range;
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Execute PreActions in the order of their definition
    //
	this.executePreActions = function(/*Function*/ inCallback)
	{
		try
		{
			Executable.executeMultiple(preActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Execute PostActions in the order of their definition
    //
    this.executePostActions = function(/*Function*/ inCallback)
	{
		try
		{
			Executable.executeMultiple(postActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*[Object]*/ inJSON, /*Workflow*/ inWorkflow)
    {
        Utils.throwInvalid(inJSON);

        // Workflow identifier (mandatory)
        //
		workflowID = Utils.getPropertyValue(inJSON, 'id', Utils.REQUIRED);

		// Workflow type (optional)
		//
		type = Utils.getPropertyValue(inJSON, 'type', Utils.OPTIONAL, 'guidedtour');

		// Plain Workflow name (optional)
        //
        name = Utils.getPropertyValue(inJSON, 'name', Utils.OPTIONAL, workflowID);

        // Minimal application version (optional)
        //
        appVersion = Utils.getPropertyValue(inJSON, 'appVersion', Utils.OPTIONAL, '');

		// Validate the minimal app version, throw if not match
		//
		validateAppVersion(appVersion);

        // Title content (optional)
        //
        var titleObj = Utils.getPropertyValue(inJSON, 'title', Utils.OPTIONAL);

        if (Utils.isValidProperty(titleObj))
        {
            // mandatory
            var contentObj = Utils.getPropertyValue(titleObj, 'content', Utils.REQUIRED);
            var content = new WorkflowContent(formatVersion, inWorkflow);
            content = content.initializeJSON(contentObj);

            if (Utils.isValidProperty(content))
            {
                title = content;
            }
        }

        // Pre action array (optional)
        //
        var preActionsObj = Utils.getPropertyValue(inJSON, 'preActions', Utils.OPTIONAL, []);

        if (Utils.isType(preActionsObj, Array))
        {
            preActions = initializeActions(preActionsObj, inWorkflow);
        }

        // Post action array (optional)
        //
        var postActionsObj = Utils.getPropertyValue(inJSON, 'postActions', Utils.OPTIONAL, []);

        if (Utils.isType(postActionsObj, Array))
        {
            postActions = initializeActions(postActionsObj, inWorkflow);
        }

        // trigger components (optional)
        // always load before loading steps!
        //
        var triggerCmpObj = Utils.getPropertyValue(inJSON, 'triggerComponents', Utils.OPTIONAL);

        if (Utils.isValidProperty(triggerCmpObj, Object))
        {
            var triggerCmps = [];

            for (var cmp in triggerCmpObj)
            {
				if (triggerCmpObj.hasOwnProperty(cmp))
				{
					var triggerObj = triggerCmpObj[cmp];
					var triggerCmp = new TriggerComponent(formatVersion, inWorkflow);
					triggerCmp.initializeJSON(cmp, triggerObj);

					if (Utils.isValidProperty(triggerCmp))
					{
						triggerCmps.push(triggerCmp);
					}
				}
            }

            TriggerComponentManager.add(triggerCmps, workflowID);
        }

        // Step definitions (mandatory)
        //
        var stepsObj = Utils.getPropertyValue(inJSON, 'steps', Utils.REQUIRED);
        Utils.throwInvalid(stepsObj);

        if (Utils.isValidProperty(stepsObj, Array))
        {
            var stepDefinitions = [];
			var stepIDs = {};

			for (var stepIndex=0; stepIndex<stepsObj.length; stepIndex++)
			{
				var step = new WorkflowStep(formatVersion, inWorkflow);
				step = step.initializeJSON(stepsObj[stepIndex]);

				if (Utils.isValidProperty(step))
				{
					stepDefinitions.push(step);

					// add step identifier to ID/Index map if available
					var id = step.getID();

					if (id.length)
					{
						stepIDs[id] = stepIndex;
					}
				}
				else
				{
					logError('Invalid Workflow Step format');
				}
			}

            steps = stepDefinitions;
			stepIdentifierMap = stepIDs;
        }

		// Step range
		//
		var rangeArray = Utils.getPropertyValue(inJSON, 'range', Utils.OPTIONAL);

		if (Utils.isValidProperty(rangeArray, Array))
		{
			if (rangeArray.length > 0 && !isNaN(rangeArray[0]))
			{
				range[0] = rangeArray[0];
			}
			else
			{
				range[0] = 0;
			}

			if (rangeArray.length > 1 && !isNaN(rangeArray[1]))
			{
				range[1] = rangeArray[1];
			}
			else
			{
				range[1] = steps.length - 1;
			}
		}
		else
		{
			range = [0, steps.length - 1];
		}
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with executable blocks
    //
    function initializeActions(/*Array of Object*/ inActions, /*Workflow*/ inWorkflow)
    {
        Utils.throwInvalid(inActions);

        var actions = [];

        Utils.forEach(inActions, function(entry)
        {
            var exe = new Executable(formatVersion, Executable.TYPE_ACTION);
            exe = exe.initializeJSON(entry);

            if (Utils.isValidProperty(exe))
            {
                actions.push(exe);
            }
            else
            {
                logError('Invalid Executable format');
            }
        });

        return actions;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Validate minimal required app version for this workflow
	// inMinVersion might be an empty string
	// Throws an exception if current app version doesn't fit
	//
    function validateAppVersion(/*String*/ inMinVersion)
	{
		function parseVersionString(/*String*/ inVersionStr)
		{
			var ret = [];

			var parts = inVersionStr.split('.');

			if (parts.length > 0)
			{
				var res = true;
				var values = [];

				Utils.forEach(parts, function(/*String*/ inPart)
				{
					var value = parseInt(inPart);

					if (isNaN(value))
					{
						res = false;
					}
					else
					{
						values.push(value);
					}
				});

				if (res)
				{
					ret = values;
				}
			}

			return ret;
		}

		var minVersionStr = Utils.getParamValue(inMinVersion, Utils.OPTIONAL, '');

		if (minVersionStr.length > 0)
		{
			var minVersion = parseVersionString(minVersionStr);
			var appVersion = parseVersionString(ScriptedWorkflowSupport.instance.appVersion);

			if (minVersion.length > 0 && appVersion.length > 0)
			{
				var diff = minVersion.length - appVersion.length;

				if (diff != 0)
				{
					for (var a=0; a<Math.abs(diff); a++)
					{
						if (diff > 0)
						{
							appVersion.push(0);
						}
						else
						{
							minVersion.push(0);
						}
					}
				}

				for (var c=0; c<minVersion.length; c++)
				{
					if (appVersion[c] < minVersion[c])
					{
						logWarn("Application version [" + ScriptedWorkflowSupport.instance.appVersion + "] doesn't comply to required workflow version[" + minVersionStr + "]")
						throw new Error("Application version doesn't comply to required workflow version");
					}
				}
			}
		}

		return true;
	}
}

// Supported format versions
Workflow.FORMAT_VERSION_1 = 'V1';

//////////////////////////////////////////////////////////////////////////////
//
// Executable objects represents executable definitions of a workflow description.
// These a pre- and post-actions as well as conditions.
// Script based Executable are littel (Javascript-) scripts which are evaluated.
//
function Executable(/*String*/ inFormatVersion, /*Boolean*/ inCondition)
{
    var type    = Executable.EVALTYPE_SCRIPT;
    var evalSrc = '';
	var condition = Utils.getParamValue(inCondition, Utils.OPTIONAL, false);

    var formatVersion = inFormatVersion;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Execute
    //
    this.execute = function(/*Function*/ inCallback)
    {
		ExecutorManager.execute(type, evalSrc, condition, inCallback);
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON (format version V1)
	//
	function initializeJSON_V1(/*[Object]*/ inJSON)
	{
		Utils.throwInvalid(inJSON);

		// executable type (optional)
		//
		var typeStr = Utils.getPropertyValue(inJSON, 'type', Utils.OPTIONAL, Executable.EVALTYPE_SCRIPT);

		if (!ExecutorManager.exists(typeStr))
		{
			logError('Invalid Executable type');
			throw new Error("Invalid Executable type");
		}

		type = typeStr;

		// executable eval source (mandatory)
		//
		evalSrc = Utils.getPropertyValue(inJSON, 'eval', Utils.REQUIRED);
	}
}

//////////////////////////////////////////////////////////////////////////////
//
// Execute multiple Executables and call callback after all were executed
//
Executable.executeMultiple = function(/*Array of Executable*/ inExecutables, /*Function*/ inCallback)
{
	var exes = Utils.getParamValue(inExecutables, Utils.REQUIRED);

	var exeCount = exes.length;

	if (exeCount > 0)
	{
		var callback = inCallback;
		var actionResult = null;

		Utils.forEach(exes, function(/*Executable*/ inExe)
		{
			inExe.execute(function(/*ExecutionResult*/inResult)
			{
				if (!Utils.isValidProperty(actionResult) && inResult.success)
				{
					actionResult = inResult.result;
				}

				if (--exeCount <= 0)
				{
					callback(actionResult);
				}
			});
		});
	}
	else
	{
		inCallback(null);
	}
}

Executable.EVALTYPE_SCRIPT = 'script';

Executable.TYPE_CONDITION = true;
Executable.TYPE_ACTION = false;

//////////////////////////////////////////////////////////////////////////////
//
// A WorkflowStep object describe a single step of a workflow description
//
function WorkflowStep(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    var stepName = null; // For debugging purposes
	var stepID = '';
    var display = [];
    var continuation = null;
    var trigger = {};		// { 'trigger type' : [WorkflowTrigger, ... ], ... }
    var preActions = [];
    var posActions = [];
    var negActions = [];
    var conditions = [];

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Returns the step id
	//
	this.getID = function()
	{
		return (Utils.isValidProperty(stepID) ? stepID : '');
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Returns the step name (useful for debugging)
	//
	this.getName = function()
	{
		return stepName;
  	}
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return Array of WorkflowDisplay
	//
	this.getDisplayTypes = function()
	{
		return display;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return WorkflowStepContinuation
	//
	this.getContinuation = function()
	{
		return Utils.cloneObject(continuation);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return all available trigger types
	//
    this.getTriggerTypes = function()
	{
		var types = [];

		for (var tt in trigger)
		{
			types.push(tt);
		}

		return types;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return all expanded trigger for inTriggerType
	//
	this.expandTrigger = function(/*String*/ inTriggerType)
	{
		var ret = [];
		var typedtrigger = trigger[inTriggerType];

		if (Utils.isValidProperty(typedtrigger, Array))
		{
			Utils.forEach(typedtrigger, function(/*WorkflowTrigger*/ inTrigger)
			{
				var expanded = inTrigger.expandTrigger(workflow);
				ret = ret.concat(expanded);
			});
		}

		return ret;
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Execute pre-actions in the order of their definition
    //
    this.executePreActions = function(/*Function*/ inCallback)
    {
		try
		{
			Executable.executeMultiple(preActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
    }

	//////////////////////////////////////////////////////////////////////////////
    //
    // Execute positive actions in the order of their definition
    //
    this.executePositiveActions = function(/*Function*/ inCallback)
    {
		try
		{
			Executable.executeMultiple(posActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Execute negative actions in the order of their definition
    //
    this.executeNegativeActions = function(/*Function*/ inCallback)
    {
		try
		{
			Executable.executeMultiple(negActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Evaluate all conditions
    // Return the first return value which is neither '#' nor 'undefined' nor 'true'
	//
	// Condition return values:
	// undefined, true, '#'		- condition evaluation successfull, just move on
	// #STEP_INDEX				- continue with step index STEP_INDEX
	// #next					- continue with next step
	// #previous				- continue with previous step
	// #restart					- restart this step
	// #cancel					- cancel the whole workflow
    //
    this.evalConditions = function(/*Function*/ inCallback)
    {
        var count = conditions.length;

        if (count > 0)
        {
            // execute each condition executable in the order they were defined
            //
			var results = [];

			Utils.forEach(conditions, function(/*Executable*/ inCondition)
            {
                inCondition.execute(function(ret)
                {
                    // store each result which is neither '#' nor 'undefined'
                    //
                    if (Utils.isValidProperty(ret) &&
						ret.success &&
						Utils.isValidProperty(ret.result) &&
						ret.result != '#' &&
						ret.result != true)
                    {
                        results.push(ret.result);
                    }

                    // if all conditions were evaluated then call back the result
                    //
                    if (--count <= 0 && Utils.isValidProperty(inCallback))
                    {
                        inCallback(results.length > 0 ? results[0] : true);
                    }
                });
            });
        }
        else if (Utils.isValidProperty(inCallback))
        {
            // no conditions available, call back immediately
            //
            inCallback(true);
        }
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

		// Optional - set the step identifier if defined in JSON
		stepID = Utils.getPropertyValue(inJSON, 'id', Utils.REQUIRED);

		// Optional - set the step name if defined in JSON
        stepName = Utils.getPropertyValue(inJSON, 'stepName', Utils.OPTIONAL);

        // display settings (mandatory)
        //
        var displayArray = Utils.getPropertyValue(inJSON, 'display', Utils.OPTIONAL);

        if (Utils.isValidProperty(displayArray, Array))
        {
            var displays = [];

            Utils.forEach(displayArray, function(entry)
            {
                var display = new WorkflowDisplay(formatVersion, workflow);
                displays.push(display.initializeJSON(entry));
            });

            display = displays;
        }

        // trigger definitions (optional)
        //
        var triggerArray = Utils.getPropertyValue(inJSON, 'trigger', Utils.OPTIONAL);

        if (Utils.isValidProperty(triggerArray, Array))
        {
            var triggers = {};

            Utils.forEach(triggerArray, function(entry)
            {
                var trigger = new WorkflowTrigger(formatVersion, workflow);
				trigger = trigger.initializeJSON(entry);
				Utils.forEach(trigger, function(/*WorkflowTrigger*/ inTrigger)
				{
					var type = inTrigger.type;

					if (!Utils.isValidProperty(triggers[type]))
					{
						triggers[type] = [];
					}

					triggers[type].push(inTrigger);
				})
            });

            trigger = triggers;
        }

        // step continuation (optional)
        //
		var kDefaultContinuation = { type : 'manually', delay : 0 };
		var continuationObj = Utils.getPropertyValue(inJSON, 'continuation', Utils.OPTIONAL, kDefaultContinuation);

        if (Utils.isValidProperty(continuationObj))
        {
            continuation = new WorkflowStepContinuation(formatVersion, workflow);
            continuation.initializeJSON(continuationObj);
        }

        // step conditions (optional)
        //
        var conditionsArray = Utils.getPropertyValue(inJSON, 'conditions', Utils.OPTIONAL);

        if (Utils.isValidProperty(conditionsArray, Array))
        {
            Utils.forEach(conditionsArray, function(entry)
            {
                var condition = new Executable(formatVersion, Executable.TYPE_CONDITION);
                conditions.push(condition.initializeJSON(entry));
            });
        }

        // pre actions (optional)
        //
        var preActionsArray = Utils.getPropertyValue(inJSON, 'preActions', Utils.OPTIONAL);

        if (Utils.isValidProperty(preActionsArray, Array))
        {
            Utils.forEach(preActionsArray, function(entry)
            {
                var action = new Executable(formatVersion, Executable.TYPE_ACTION);
                preActions.push(action.initializeJSON(entry));
            });
        }

		// positive actions (optional)
        //
        var positiveActionsArray = Utils.getPropertyValue(inJSON, 'positiveActions', Utils.OPTIONAL);

        if (Utils.isValidProperty(positiveActionsArray, Array))
        {
            Utils.forEach(positiveActionsArray, function(entry)
            {
                var action = new Executable(formatVersion, Executable.TYPE_ACTION);
                posActions.push(action.initializeJSON(entry));
            });
        }

        // negative actions (optional)
        //
        var negativeActionsArray = Utils.getPropertyValue(inJSON, 'negativeActions', Utils.OPTIONAL);

        if (Utils.isValidProperty(negativeActionsArray, Array))
        {
            Utils.forEach(negativeActionsArray, function(entry)
            {
                var action = new Executable(formatVersion, Executable.TYPE_ACTION);
                negActions.push(action.initializeJSON(entry));
            });
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// A WorkflowTrigger object defines a user action.
// A WorkflowTrigger object consists of one or more definitions based
// on ITriggerDefinition and optional actions
//
function WorkflowTrigger(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
	this.component = null;

    this.type = '';
    this.triggerDefinition = null;
	this.positiveActions = [];
	this.negativeActions = [];
	this.matchActions = [];

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

		var ret = [this];

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                ret = initializeJSON_V1(this, inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return ret;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with ITriggerDefinition object
	//
    this.initialize = function (/*String*/ inType, /*ITriggerDefinition*/ inDefinition)
	{
		var type = Utils.getParamValue(inType, Utils.REQUIRED);
		var def = Utils.getParamValue(inDefinition, Utils.REQUIRED);

		this.type = type;
		this.triggerDefinition = def;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Clone WorkflowTrigger
	//
	this.clone = function()
	{
		var ret 				= new WorkflowTrigger(formatVersion, workflow);
		ret.component			= this.component;
		ret.type				= this.type;
		ret.triggerDefinition	= this.triggerDefinition.clone();
		ret.positiveActions		= this.positiveActions;
		ret.negativeActions		= this.negativeActions;
		ret.matchActions		= this.matchActions;

		return ret;
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Copy and expand this trigger. Return one or more new trigger instances
    //
    this.expandTrigger = function(/*Workflow*/ inWorkflow)
    {
        var ret = [];
		var wf = Utils.getParamValue(inWorkflow, Utils.OPTIONAL, workflow);

		if (Utils.isValidProperty(wf))
		{
			var type = this.type;
			var posActions = this.positiveActions;
			var negActions = this.negativeActions;
			var matchActions = this.matchActions;

			var triggerDefs = [];

			if (Utils.isValidProperty(this.component, String) &&
				TriggerComponentManager.exists(this.component, wf))
			{
				var cmp = TriggerComponentManager.getComponent(this.component, wf);

				type 			= cmp.type;
				posActions		= cmp.positiveActions;
				negActions		= cmp.negativeActions;
				matchActions	= cmp.matchActions;

				Utils.forEach(cmp.triggerDefinitions, function (/*ITriggerDefinition*/ inTriggerDef)
				{
					var defs = inTriggerDef.expandTrigger(wf.getID());
					triggerDefs = triggerDefs.concat(defs);
				});
			}
			else
			{
				triggerDefs = this.triggerDefinition.expandTrigger(wf.getID());
			}

			Utils.forEach(triggerDefs, function(def)
			{
				var cp               = new WorkflowTrigger(formatVersion, wf);
				cp.type              = type;
				cp.positiveActions	 = posActions;
				cp.negativeActions	 = negActions;
				cp.matchActions		 = matchActions;
				cp.triggerDefinition = def;

				ret.push(cp);
			});
		}

        return ret;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Execute positive actions in the order of their definition
	//
	this.executePositiveActions = function(/*Function*/ inCallback)
	{
		try
		{
			Executable.executeMultiple(this.positiveActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Execute negative actions in the order of their definition
	//
	this.executeNegativeActions = function(/*Function*/ inCallback)
	{
		try
		{
			Executable.executeMultiple(this.negativeActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Execute match actions in the order of their definition
	//
	this.executeMatchActions = function(/*Function*/ inCallback)
	{
		try
		{
			Executable.executeMultiple(this.matchActions, inCallback);
		}
		catch(exc)
		{
			logExc(exc);
		}
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*WorkflowTrigger*/ inThis, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inThis);
        Utils.throwInvalid(inJSON);

		var ret = [inThis];

		// refering to a component?
		//
		var componentName = Utils.getPropertyValue(inJSON, 'component', Utils.OPTIONAL);

		if (Utils.isValidProperty(componentName, String))
		{
			if (TriggerComponentManager.exists(componentName, workflow))
			{
				var component = TriggerComponentManager.getComponent(componentName, workflow);

				inThis.type			= component.type;
				inThis.component 	= componentName;
			}
			else
			{
				logError('Unknown trigger component name "' + componentName + '"');
				throw new Error("Unknown trigger component name");
			}
		}
		else
		{
			// trigger type (optional)
			//
			var type = Utils.getPropertyValue(inJSON, 'type', Utils.OPTIONAL, WorkflowTrigger.TYPE_SCRIPTEVENT);

			if (!TriggerFactory.exists(type))
			{
				logError('Unknown trigger type "' + type + '"');
				throw new Error("Invalid trigger type");
			}

			inThis.type = type;

			// trigger definition (mandatory)
			//
			var defObj = Utils.getPropertyValue(inJSON, 'definition', Utils.REQUIRED);

			if (Utils.isValidProperty(defObj))
			{
				var def = TriggerFactory.create(inThis.type, formatVersion, workflow);
				Utils.throwInvalid(def);

				inThis.triggerDefinition = def.initializeJSON(defObj);
			}
		}

		// match actions (optional)
		//
		var matchActionsArray = Utils.getPropertyValue(inJSON, 'matchActions', Utils.OPTIONAL);

		if (Utils.isValidProperty(matchActionsArray, Array))
		{
			var matchActions = [];

			Utils.forEach(matchActionsArray, function(entry)
			{
				var action = new Executable(formatVersion, Executable.TYPE_ACTION);
				matchActions.push(action.initializeJSON(entry));
			});

			if (matchActions.length)
			{
				for (var t=0; t<ret.length; t++)
				{
					ret[t].matchActions = matchActions;
				}
			}
		}

		// positive actions (optional)
        //
        var positiveActionsArray = Utils.getPropertyValue(inJSON, 'positiveActions', Utils.OPTIONAL);

        if (Utils.isValidProperty(positiveActionsArray, Array))
        {
            var posActions = [];

            Utils.forEach(positiveActionsArray, function(entry)
            {
                var action = new Executable(formatVersion, Executable.TYPE_ACTION);
                posActions.push(action.initializeJSON(entry));
            });

			if (posActions.length)
			{
				for (var t=0; t<ret.length; t++)
				{
					ret[t].positiveActions = posActions;
				}
			}
        }

        // negative actions (optional)
        //
        var negativeActionsArray = Utils.getPropertyValue(inJSON, 'negativeActions', Utils.OPTIONAL);

        if (Utils.isValidProperty(negativeActionsArray, Array))
        {
            var negActions = [];

            Utils.forEach(negativeActionsArray, function(entry)
            {
                var action = new Executable(formatVersion, Executable.TYPE_ACTION);
                negActions.push(action.initializeJSON(entry));
            });

			if (negActions.length)
			{
				for (var t=0; t<ret.length; t++)
				{
					ret[t].negativeActions = negActions;
				}
			}
        }

        return ret;
    }
}

WorkflowTrigger.TYPE_SCRIPTEVENT = 'scriptevent';

//////////////////////////////////////////////////////////////////////////////
//
// WorkflowDisplay defines the display surfaces and the according content
//
function WorkflowDisplay(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    var type = '';
    var content = null;
    var params = {};

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve display type
	//
	this.getType = function()
	{
		return type;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve display content
	//
	this.getContent = function()
	{
		return content;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve display params (depends on display type)
	//
	this.getParams = function()
	{
		return Utils.cloneObject(params);
	}

	//////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        // display type (optional)
        //
        var kSupportedTypes = DisplayAdapterFactory.getTypes();
        var typeStr = Utils.getPropertyValue(inJSON, 'type', Utils.OPTIONAL, TourCoachmarkDisplayAdapter.TYPE_ID);

        if (kSupportedTypes.indexOf(typeStr) < 0)
        {
            logError('Unknown Display type "' + typeStr + '"');
			throw new Error("Invalid Display type");
        }

        type = typeStr;

        // content (mandatory)
        //
        var contentObj = Utils.getPropertyValue(inJSON, 'content', Utils.REQUIRED);

        if (Utils.isValidProperty(contentObj))
        {
            content = new WorkflowContent(formatVersion, workflow);
            content.initializeJSON(contentObj);
        }

        // parameter object (optional)
        //
        params = Utils.getPropertyValue(inJSON, 'params', Utils.OPTIONAL, {});
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// Defines the content which is displayed for a single step
//
function WorkflowContent(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    var content = {};
	var locators = [];

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve available locators
	//
	this.getLocators = function()
	{
		return locators;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve WorkflowContentPart for locator
	//
	this.getContent = function(/*String*/ inLocator)
	{
		return content[inLocator];
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        var parts = {};

        for (var ci in inJSON)
        {
			if (inJSON.hasOwnProperty(ci))
			{
				var contentPartObj = inJSON[ci];
				Utils.throwInvalid(contentPartObj);

				var contentPart = new WorkflowContentPart(formatVersion, workflow);
				parts[ci]       = contentPart.initializeJSON(contentPartObj);
				locators.push(ci);
			}
        }

        content = parts;
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// Defines a single content part like e.g. a localized string
//
function WorkflowContentPart(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    this.contentID = '';
    this.mimeType = '';

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return related Workflow identifier
	//
	this.getWorkflowID = function()
	{
		return (Utils.isValidProperty(workflow) ? workflow.getID() : '');
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return related Workflow type
	//
	this.getWorkflowType = function()
	{
		return (Utils.isValidProperty(workflow) ? workflow.getType() : '');
	}

	//////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(this, inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*WorkflowContentPart*/ inThis, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inThis);
        Utils.throwInvalid(inJSON);

        // content id (mandatory)
        //
		inThis.contentID = Utils.getPropertyValue(inJSON, 'id', Utils.REQUIRED);

        // content mime type (optional)
        //
        inThis.mimeType = Utils.getPropertyValue(inJSON, 'mime', Utils.OPTIONAL, 'text/plain');
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// WorkflowStepContinuation defines how to continue to the next step if the
// current step was accomplished (automatically or by user action)
//
function WorkflowStepContinuation(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    this.type = '';
    this.delay = NaN;
	this.handler = '';

	var workflow = inWorkflow;
    var formatVersion = inFormatVersion;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(this, inJSON);
                break;

            default:
				logError('Invalid format version "' + formatVersion + '"');
				throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*WorkflowStepContinuation*/ inThis, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inThis);
        Utils.throwInvalid(inJSON);

        inThis.type = Utils.getPropertyValue(inJSON, 'type', Utils.OPTIONAL, WorkflowStepContinuation.TYPE_MANUALLY);
		inThis.handler = Utils.getPropertyValue(inJSON, 'handler', Utils.OPTIONAL, CoachmarkContinuationHandler.NAME);
        inThis.delay = Utils.getPropertyValue(inJSON, 'delay', Utils.OPTIONAL, 0);
    }
}

WorkflowStepContinuation.TYPE_MANUALLY  = 'manually';
WorkflowStepContinuation.TYPE_AUTO      = 'automatically';

//////////////////////////////////////////////////////////////////////////////
//
// Parse Workflow Description
//
Workflow.create = function(inJSON)
{
    var wf = null;

    try
    {
        var obj = JSON.parse(inJSON);
        wf = new Workflow();
        wf = wf.initializeJSON(obj);
    }
    catch(exc)
    {
		logExc(exc);

        wf = null;
    }

    return wf;
}
